import { Dsn } from '@sentry/core';
import { SentryEvent, SentryResponse } from '@sentry/types';
/**
 * Payload for a minidump request comprising a persistent file system path and
 * event metadata.
 */
export interface MinidumpRequest {
    /** Path to the minidump file. */
    path: string;
    /** Associated event data. */
    event: SentryEvent;
}
/**
 * A service that discovers Minidump crash reports and uploads them to Sentry.
 */
export declare class MinidumpUploader {
    private readonly crashesDirectory;
    private readonly cacheDirectory;
    /** The minidump ingestion endpoint URL. */
    private readonly url;
    /** The type of the Electron CrashReporter used to search for Minidumps. */
    private readonly type;
    /** List of minidumps that have been found already. */
    private readonly knownPaths;
    /**
     * Store to persist queued Minidumps beyond application crashes or lost
     * internet connection.
     */
    private readonly queue;
    /**
     * Creates a new uploader instance.
     *
     * @param dsn The Sentry DSN.
     * @param crashesDirectory The directory Electron stores crashes in.
     * @param cacheDirectory A persistent directory to cache minidumps.
     */
    constructor(dsn: Dsn, crashesDirectory: string, cacheDirectory: string);
    /**
     * Returns the minidump endpoint in Sentry
     * @param dsn Dsn
     */
    static minidumpUrlFromDsn(dsn: Dsn): string;
    /**
     * Uploads a minidump file to Sentry.
     *
     * @param path Absolute path to the minidump file.
     * @param event Event data to attach to the minidump.
     * @returns A promise that resolves when the upload is complete.
     */
    uploadMinidump(request: MinidumpRequest): Promise<SentryResponse>;
    /**
     * Searches for new, unknown minidump files in the crash directory.
     * @returns A promise that resolves to absolute paths of those dumps.
     */
    getNewMinidumps(): Promise<string[]>;
    /** Flushes locally cached minidumps from the queue. */
    flushQueue(): Promise<void>;
    /** Scans the Crashpad directory structure for minidump files. */
    private scanCrashpadFolder;
    /** Scans the Breakpad directory structure for minidump files. */
    private scanBreakpadFolder;
    /**
     * Enqueues a minidump with event information for later upload.
     * @param request The request containing a minidump and event info.
     */
    private queueMinidump;
}
