"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : new P(function (resolve) { resolve(result.value); }).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const path_1 = require("path");
const util_1 = require("util");
const types_1 = require("@sentry/types");
const async_1 = require("@sentry/utils/async");
const fs_1 = require("@sentry/utils/fs");
const store_1 = require("@sentry/utils/store");
const electron_fetch_1 = require("electron-fetch");
const FormData = require("form-data");
const readdir = util_1.promisify(fs.readdir);
const rename = util_1.promisify(fs.rename);
const stat = util_1.promisify(fs.stat);
const unlink = util_1.promisify(fs.unlink);
/** Status code returned by Sentry to retry event submission later. */
const CODE_RETRY = 429;
/** Maximum number of days to keep a minidump before deleting it. */
const MAX_AGE = 30;
/** Maximum number of requests that we store/queue if something goes wrong. */
const MAX_REQUESTS_COUNT = 10;
/**
 * A service that discovers Minidump crash reports and uploads them to Sentry.
 */
class MinidumpUploader {
    /**
     * Creates a new uploader instance.
     *
     * @param dsn The Sentry DSN.
     * @param crashesDirectory The directory Electron stores crashes in.
     * @param cacheDirectory A persistent directory to cache minidumps.
     */
    constructor(dsn, crashesDirectory, cacheDirectory) {
        this.crashesDirectory = crashesDirectory;
        this.cacheDirectory = cacheDirectory;
        /**
         * Store to persist queued Minidumps beyond application crashes or lost
         * internet connection.
         */
        this.queue = new store_1.Store(this.cacheDirectory, 'queue', []);
        this.type = process.platform === 'darwin' ? 'crashpad' : 'breakpad';
        this.knownPaths = [];
        this.url = MinidumpUploader.minidumpUrlFromDsn(dsn);
    }
    /**
     * Returns the minidump endpoint in Sentry
     * @param dsn Dsn
     */
    static minidumpUrlFromDsn(dsn) {
        const { host, path, projectId, port, protocol, user } = dsn;
        return `${protocol}://${host}${port !== '' ? `:${port}` : ''}${path !== '' ? `/${path}` : ''}/api/${projectId}/minidump?sentry_key=${user}`;
    }
    /**
     * Uploads a minidump file to Sentry.
     *
     * @param path Absolute path to the minidump file.
     * @param event Event data to attach to the minidump.
     * @returns A promise that resolves when the upload is complete.
     */
    uploadMinidump(request) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                const body = new FormData();
                body.append('upload_file_minidump', fs.createReadStream(request.path));
                body.append('sentry', JSON.stringify(request.event));
                const response = yield electron_fetch_1.default(this.url, { method: 'POST', body });
                // Too many requests, so we queue the event and send it later
                if (response.status === CODE_RETRY) {
                    yield this.queueMinidump(request);
                    return {
                        status: types_1.Status.RateLimit,
                    };
                }
                // We either succeeded or something went horribly wrong. Either way, we
                // can remove the minidump file.
                yield unlink(request.path);
                // Forget this minidump in all caches
                this.queue.update(queued => queued.filter(stored => stored !== request));
                this.knownPaths.splice(this.knownPaths.indexOf(request.path), 1);
                // If we were successful, we can try to flush the remaining queue
                if (response.ok) {
                    yield this.flushQueue();
                }
                return {
                    status: types_1.Status.fromHttpCode(response.status),
                };
            }
            catch (err) {
                // User's internet connection was down so we queue it as well
                const error = err ? err : { code: '' };
                if (error.code === 'ENOTFOUND') {
                    yield this.queueMinidump(request);
                }
                return {
                    status: types_1.Status.Failed,
                };
            }
        });
    }
    /**
     * Searches for new, unknown minidump files in the crash directory.
     * @returns A promise that resolves to absolute paths of those dumps.
     */
    getNewMinidumps() {
        return __awaiter(this, void 0, void 0, function* () {
            const minidumps = this.type === 'crashpad' ? yield this.scanCrashpadFolder() : yield this.scanBreakpadFolder();
            const oldestMs = new Date().getTime() - MAX_AGE * 24 * 3600 * 1000;
            return async_1.filterAsync(minidumps, (path) => __awaiter(this, void 0, void 0, function* () {
                // Skip files that we have seen before
                if (this.knownPaths.indexOf(path) >= 0) {
                    return false;
                }
                // Lock this minidump until we have uploaded it or an error occurs and we
                // remove it from the file system.
                this.knownPaths.push(path);
                // We do not want to upload minidumps that have been generated before a
                // certain threshold. Those old files can be deleted immediately.
                const stats = yield stat(path);
                if (stats.birthtimeMs < oldestMs) {
                    yield unlink(path);
                    this.knownPaths.splice(this.knownPaths.indexOf(path), 1);
                    return false;
                }
                return true;
            }));
        });
    }
    /** Flushes locally cached minidumps from the queue. */
    flushQueue() {
        return __awaiter(this, void 0, void 0, function* () {
            yield Promise.all(this.queue.get().map((request) => __awaiter(this, void 0, void 0, function* () { return this.uploadMinidump(request); })));
        });
    }
    /** Scans the Crashpad directory structure for minidump files. */
    scanCrashpadFolder() {
        return __awaiter(this, void 0, void 0, function* () {
            // Crashpad moves minidump files directly into the completed/ folder. We can
            // load them from there, upload to the server, and then delete it.
            const dumpDirectory = path_1.join(this.crashesDirectory, 'completed');
            const files = yield readdir(dumpDirectory);
            return files.filter(file => file.endsWith('.dmp')).map(file => path_1.join(dumpDirectory, file));
        });
    }
    /** Scans the Breakpad directory structure for minidump files. */
    scanBreakpadFolder() {
        return __awaiter(this, void 0, void 0, function* () {
            // Breakpad stores all minidump files along with a metadata file directly in
            // the crashes directory.
            const files = yield readdir(this.crashesDirectory);
            // Remove all metadata files (asynchronously) and forget about them.
            files
                .filter(file => file.endsWith('.txt') && !file.endsWith('log.txt'))
                .forEach((file) => __awaiter(this, void 0, void 0, function* () { return unlink(path_1.join(this.crashesDirectory, file)); }));
            return files.filter(file => file.endsWith('.dmp')).map(file => path_1.join(this.crashesDirectory, file));
        });
    }
    /**
     * Enqueues a minidump with event information for later upload.
     * @param request The request containing a minidump and event info.
     */
    queueMinidump(request) {
        return __awaiter(this, void 0, void 0, function* () {
            const filename = path_1.basename(request.path);
            // Only enqueue if this minidump hasn't been enqueued before. Compare the
            // filename instead of the full path, because we will move the file to a
            // temporary location later on.
            if (this.queue.get().some(req => path_1.basename(req.path) === filename)) {
                return;
            }
            // Move the minidump file to a separate cache directory and enqueue it. Even
            // if the Electron CrashReporter's cache directory gets wiped or changes,
            // this will allow us to retry uploading the file later.
            const queuePath = path_1.join(this.cacheDirectory, filename);
            yield fs_1.mkdirp(this.cacheDirectory);
            yield rename(request.path, queuePath);
            // Remove stale minidumps in case we go over limit. Note that we have to
            // re-fetch the queue as it might have changed in the meanwhile. It is
            // important to store the queue value again immediately to avoid phantom
            // reads.
            const requests = [...this.queue.get(), Object.assign({}, request, { path: queuePath })];
            const stale = requests.splice(-MAX_REQUESTS_COUNT);
            this.queue.set(requests);
            yield Promise.all(stale.map((req) => __awaiter(this, void 0, void 0, function* () { return unlink(req.path); })));
        });
    }
}
exports.MinidumpUploader = MinidumpUploader;
//# sourceMappingURL=uploader.js.map