(function (RongIM, dependencies, components) {
'use strict';

var utils = RongIM.utils;
var KEYCODE = utils.keyCode;
var $ = dependencies.jQuery;
var RongIMEmoji = RongIMLib.RongIMEmoji;
var isEmojiOverlap = utils.isEmojiOverlap;
var emojiReg = getEmojiEditReg();
var MaxLen = 10000;
var Cache = RongIM.dataModel._Cache;

var EmojiRegExp = (function () {
    var names = RongIMEmoji.list.map(function (data) {
        var symbol = data.symbol;
        return '\\[' + symbol.substring(1, symbol.length - 1) + '\\]';
    });
    var nameReg = names.join('|');
    return new RegExp(nameReg, 'ig');
}());

/*
说明： 输入文本框
*/
components.getEditBox = function (resolve, reject) {
    var im = RongIM.instance;
    var conversationApi = im.dataModel.Conversation;
    var options = {
        name: 'edit-box',
        template: '#rong-template-edit-box',
        props: {
            atMembers: {
                type: Array,
                required: false
            },
            autoFocus: {
                type: Boolean,
                required: false,
                default: true
            },
            placeHolder: '',
            disabled: false,
            notSaveDraft: false,
        },
        data: function () {
            return {
                value: '',
                at: {},
                atPanel: {},
                // @ 按字符搜索成员列表结果
                atFilterMembers: [],
                // 已选 @ 成员
                atSelectedMembers: []
            };
        },
        computed: {
            status: function () {
                return im.status;
            }
        },
        mounted: function () {
            mounted(this, conversationApi);
        },
        created: function () {
            this.addAltUse ()
        },
        watch: {
            atFilterMembers: function () {
                // @ 搜索结果改变更新 @ panel 展示
                var params = {
                    atPanel: this.atPanel,
                    atFilterMembers: this.atFilterMembers,
                    at: this.at
                };
                renderAtpanel(params);
            },
            value: function () {
                var context = this;
                this.scrollToInput();
                if (this.value.length > MaxLen) {
                    this.value = this.value.substr(0, MaxLen);
                }
                context.$emit('editBoxChange', context.getValue().text);
                // Mac 下输入框中 emoji 字符会重叠占位较窄拼接空格拉开距离
                dealWithEmoji(context, function (value, index) {
                    var isValid = true;
                    if (index || index === 0) {
                        index += value.length;
                        var space = context.value.substring(index, index + 1);
                        isValid = space !== ' ';
                    }
                    return isValid ? value + ' ' : value;
                }, emojiReg.native);
            },
            $route: function () {
                var context = this;
                var params = context.$route.params;
                var draft = conversationApi.getDraft(params.conversationType, params.targetId);
                context.atSelectedMembers = draft.atMembers || [];
                context.value = draft.content || '';
                Vue.nextTick(function () {
                    context.$el.focus();
                });
            },
            status: function () {
                var context = this;
                context.$emit('editBoxChange', context.getValue().text);
            },
            notSaveDraft: function () {
                var params = this.$route.params;
                var conversationType = params.conversationType;
                var targetId = params.targetId;
                if (this.notSaveDraft) {
                    conversationApi.clearDraft(conversationType, targetId);
                }
            }
        },
        methods: getMethods(conversationApi, im.auth.id)
    };
    utils.asyncComponent(options, resolve, reject);
};

function mounted(context, conversationApi) {
    var params = context.$route.params;
    var draft = conversationApi.getDraft(params.conversationType, params.targetId);
    context.atSelectedMembers = draft.atMembers || [];
    context.value = draft.content || '';

    if (context.autoFocus) {
        Vue.nextTick(function () {
            context.$el.focus();
        });
    }

    $(context.$el).on('paste', function (event) {
        context.$emit('paste', event.originalEvent);
    });

    context.at = new At(context.$el);

    components.getAtPanel(function (option) {
        var AtPanel = Vue.extend(option);
        var atPanelInstance = new AtPanel();
        atPanelInstance.$on('atPanelClickSelect', function (data) {
            context.selectAtMember(data);
        });
        atPanelInstance.$mount();
        context.$el.parentNode.insertBefore(atPanelInstance.$el, context.$el);
        context.atPanel = atPanelInstance;
    });
}

/*
说明： 计算 @ 列表显示位置调用 render 方法渲染到页面
*/
function renderAtpanel(params) {
    var height = document.body.clientHeight;
    // console.time('get cursor pos');
    var cursorPostion = params.at.getCursorPos();
    // console.timeEnd('get cursor pos');
    var left = cursorPostion.left;
    var bottom = height - cursorPostion.top;
    var atPanelStyle = {
        position: 'fixed',
        left: left + 'px',
        bottom: bottom + 'px'
    };
    var showAtall = !params.at.matchStr.text || params.at.matchStr.text.length === 0;
    params.atPanel.render(params.atFilterMembers, atPanelStyle, showAtall);
}

function getMethods(conversationApi, authId) {
    return {
        // 右键头像@成员
        addAltUse: function () {
            eventBus.$on('altUse', params => {
                if(params != null && params.user != null && JSON.stringify(params.user) != '{}') {
                    let isAddAt = this.atSelectedMembers.map((item,index) => item.id).indexOf (params.user.id) != -1;
                    params.user.targetId = params.targetId;
                    if(!isAddAt) {
                        let groups = Cache.group[params.user.targetId].groupMembers;
                        for(let i = 0; i < groups.length; i++ ) {
                            if( groups[i].id == params.user.id && groups[i].alias && groups[i].alias.length) {
                                params.user.groupAlias = groups[i].alias;
                                break;
                            }else {
                                params.user.groupAlias = undefined;
                            }
                        };
                        this.selectAtMember(params.user,"addAtText");
                    }
                }
            });   
        },
        focus: function () {
            var context = this;
            Vue.nextTick(function () {
                context.$el.focus();
            });
        },
        getValue: function () {
            return {
                text: this.value,
                at: this.atSelectedMembers
            };
        },
        setValue: function (value) {
            this.value = value.text;
            this.atSelectedMembers = value.at;
        },
        appendValue: function (value) {
            this.value += value.text;
            Array.prototype.push.apply(this.atSelectedMembers, value.at);
            this.focus();
        },
        insertText: function (str) {
            var el = this.$el;
            var cursorOffset = getCursorOffset(el);
            var text = el.value;

            var beforeCursorStr = text.substring(0, cursorOffset);
            var afterCursorStr = text.substring(el.selectionEnd);

            text = beforeCursorStr + str + afterCursorStr;
            el.value = text;

            cursorOffset += str.length;
            el.focus();
            el.setSelectionRange(cursorOffset, cursorOffset);
            this.value = text;
        },
        saveDraft: function () {
            var params = this.$route.params;
            var conversationType = params.conversationType;
            var targetId = params.targetId;
            var draftContent = this.value;
            if (utils.isEmpty((this.value || '').trim())) {
                draftContent = '';
            }
            var draft = {
                content: draftContent,
                atMembers: draftContent ? this.atSelectedMembers : []
            };
            conversationApi.setDraft(conversationType, targetId, draft);
        },
        selectAtMember: function (member,isAddAtText) {
            selectAtMember(this, member,isAddAtText);
        },
        findIndexSelectedMemberByName: function (name) {
            return findIndexSelectedMemberByName(this, name);
        },
        searchAtRemove: function (event) {
            searchAtRemove(this, event);
        },
        enter: function () {
            this.$emit('enter', this.getValue());
        },
        keydown: function (event) {
            keydown(this, event, authId);
        },
        keyup: function (event) {
            keyup(this, event, authId);
        },
        prepareinput: function () {
            this.$emit('prepareinput');
        },
        clear: function () {
            this.atSelectedMembers = [];
            this.value = '';
        },
        scrollToInput: function () {
            var textNode = this.$el;
            textNode.blur();
            textNode.focus();
        }
    };
}

/*
说明： 根据 @ 字符串搜索成员
*/
function searchAtShowMembers(context, authId) {
    var atStr = context.at.inputMatch();
    if (utils.isEmpty(atStr)) {
        context.atFilterMembers = [];
        return;
    }
    var members = searchMember(context.atMembers, atStr.text);
    context.atFilterMembers = members.filter(function (item) {
        return item.id !== authId;
    });
}

/*
说明： 确认选择 @ 成员，添加到已选择列表
*/
function selectAtMember(context, member,isAddAtText) {
    // 右键@
    if(isAddAtText && isAddAtText == "addAtText") {
        console.log('seleMember ==', member)
        let newVal = context.at.rightClickInsert(member.groupAlias || member.name);
        console.log(newVal)
        context.value = newVal;
    }else { //输入@
        var text = context.at.insert(member.name);
        context.value = text;
    }
    context.atSelectedMembers.push(member);
    context.atFilterMembers = [];
}

/*
说明： 根据名称查找是否存在已选择列表当中（目前用于删除）
*/
function findIndexSelectedMemberByName(context, name) {
    var arr = context.atSelectedMembers;
    for (var i = 0, len = arr.length; i < len; i += 1) {
        if (name === arr[i].name || name === arr[i].groupAlias) {
            return i;
        }
    }
    return -1;
}

/*
说明： 删除单个 @ 成员
*/
function searchAtRemove(context, event) {
    var atStr = context.at.removeMatch();
    console.log(atStr)
    if (utils.isEmpty(atStr)) {
        return;
    }
    if (event.keyCode === KEYCODE.backspace && atStr.cursorPos === 'before') {
        return;
    }
    if (event.keyCode === KEYCODE.delete && atStr.cursorPos === 'after') {
        return;
    }
    var index = context.findIndexSelectedMemberByName(atStr.text);
    if (index > -1) {
        event.preventDefault();
        context.atSelectedMembers.splice(index, 1);
        context.value = context.at.remove(atStr);
    }
}

/*
说明： 删除选中区域（多个文字）时查找删除字符串当中是否包含已选 @ 成员
*/
function delSelection(context) {
    var el = context.$el;
    var startP = el.selectionStart;
    var endP = el.selectionEnd;
    if (startP !== endP) {
        var delCount = endP - startP;
        var deleteValue = context.value.substr(startP, delCount);
        if(context.value.length == 0) deleteValue = el.value.substr(startP, delCount);
        context.value.slice(startP, delCount);
        for (var i = context.atSelectedMembers.length - 1; i > -1; i -= 1) {
            var item = context.atSelectedMembers[i];
            if(deleteValue.indexOf('@' + item.name) > -1 || deleteValue.indexOf('@' + item.groupAlias) > -1) {
                context.atSelectedMembers.splice(i, 1);
            }
        }
        return true;
    }
    return false;
}

function delValue(context, event) {
    if (delSelection(context)) {
        return;
    }
    context.searchAtRemove(event);
}

var debounceSearch = utils.debounce(searchAtShowMembers, 200);
// TODO: 使用键值对优化算法，避免频繁比对
function keydown(context, event, authId) {
    var params;
    switch (event.keyCode) {
        case KEYCODE.enter:
            enter(context, event);
            break;
        case KEYCODE.up:
            up(context, event);
            break;
        case KEYCODE.down:
            if (context.atFilterMembers.length) {
                event.preventDefault();
                context.atPanel.next();
            }
            break;
        case KEYCODE.left:
            params = {
                value: context.value,
                el: context.$el,
                keycode: KEYCODE.left
            };
            moveEmojiTag(params);
            break;
        case KEYCODE.right:
            params = {
                value: context.value,
                el: context.$el,
                keycode: KEYCODE.right
            };
            moveEmojiTag(params);
            break;
        case KEYCODE.esc:
            context.atFilterMembers = [];
            break;
        case KEYCODE.backspace:
            dealWithEmoji(context, function () {
                return ' ';
            }, emojiReg.delete);
            delValue(context, event);
            break;
        case KEYCODE.delete:
            delValue(context, event);
            break;
        case 229:
            debounceSearch(context, authId);
            break;
        default:
            break;
    }
}

function up(context, event) {
    if (context.atFilterMembers.length) {
        event.preventDefault();
        context.atPanel.prev();
    }
}

function enter(context, event) {
    if (!event.ctrlKey && !event.shiftKey) {
        event.preventDefault();
        if (context.atFilterMembers.length) {
            context.selectAtMember(context.atPanel.getValue());
        } else {
            context.enter();
        }
    }
}

function keyup(context, event, authId) {
    switch (event.keyCode) {
        case KEYCODE.up:
        case KEYCODE.down:
        case KEYCODE.esc:
            break;
        default:
            var searchKeyCode = [50, KEYCODE.backspace, KEYCODE.enter, KEYCODE.shift];
            if (searchKeyCode.indexOf(event.keyCode) > -1 || context.atFilterMembers.length) {
                debounceSearch(context, authId);
            }
            convertTextToEmoji(context);
            break;
    }
}

/*
说明： 根据 @ 字符串 搜索成员
*/
function searchMember(members, keyword) {
    if (!$.isArray(members) || $.type(keyword) !== 'string') {
        return [];
    }
    // 深拷贝一个数组防止修改原来数组内对象
    members = $.extend(true, [], members);
    if (utils.isEmpty(keyword)) {
        return members;
    }

    var resultArr = [];
    // eg.  As中文 => as中文
    keyword = keyword.toLowerCase();
    for (var i = 0, len = members.length; i < len; i += 1) {
        var user = members[i];
        if (matchUser(user, keyword)) {
            resultArr.push(user);
        }
    }
    return resultArr;
}
// 此 at 字符串是否可以匹配到此用户
function matchUser(user, keyword) {
    var name = user.name;
    var nameLetter = utils.convertToABC(name);
    var alias = user.alias || '';
    var aliasLetter = utils.convertToABC(alias);
    var sign = '\u0000';
    var matchList = [
        name,
        nameLetter.pinyin,
        nameLetter.first,
        alias,
        aliasLetter.pinyin,
        aliasLetter.first
    ];
    var str = matchList.join(sign);
    return str.toLowerCase().indexOf(keyword) !== -1;
}

function getCursorOffset(el) {
    return el.selectionStart;
}

/*
说明： 处理 emoji 字符 根据传入 replaceReg 替换调用 repeatFunc
*/
function dealWithEmoji(context, repeatFunc, replaceReg) {
    var isDeal = context.value && isEmojiOverlap() && RongIMEmoji.isSupportEmoji;
    if (!isDeal) {
        return;
    }
    var cursorOffset = getCursorOffset(context.$el);
    var beforeValue = context.value.substring(0, cursorOffset);
    var afterValue = context.value.substring(cursorOffset);
    var replaceValue = beforeValue.replace(replaceReg, function (value, p, index) {
        return repeatFunc(value, index);
    });
    var changeCount = replaceValue.length - beforeValue.length;
    var value = replaceValue + afterValue;
    if (changeCount) {
        cursorOffset += changeCount;
        context.$el.value = value;
        context.$el.focus();
        context.$el.setSelectionRange(cursorOffset, cursorOffset);
        context.value = value;
    }
}

function moveEmojiTag(params) {
    if (!isEmojiOverlap() || !RongIMEmoji.isSupportEmoji) {
        return;
    }
    var keycode = params.keycode;
    var el = params.el;
    var cursorOffset = getCursorOffset(el);
    var beforeValue = params.value.substring(0, cursorOffset);
    var afterValue = params.value.substring(cursorOffset);
    var leftReg = emojiReg.moveLeft;
    var rightReg = emojiReg.moveRight;
    if (keycode === KEYCODE.left && beforeValue.match(leftReg)) {
        cursorOffset -= 1;
        el.setSelectionRange(cursorOffset, cursorOffset);
    }
    if (keycode === KEYCODE.right && afterValue.match(rightReg)) {
        cursorOffset += 1;
        el.setSelectionRange(cursorOffset, cursorOffset);
    }
}

function convertTextToEmoji(context) {
    if (!RongIMEmoji.isSupportEmoji) {
        return;
    }
    var value = context.value;
    if (value.length === 0) {
        return;
    }
    var reg = EmojiRegExp;
    if (!reg.test(value)) {
        return;
    }
    var matchValue = value.match(reg)[0];
    var beforeIndex = value.indexOf(matchValue);
    var afterIndex;
    value = value.replace(matchValue, function (name) {
        var emoji = RongIMEmoji.symbolToEmoji(name);
        afterIndex = beforeIndex + emoji.length;
        return emoji;
    });
    if (afterIndex) {
        var el = context.$el;
        el.value = value;
        el.focus();
        el.setSelectionRange(afterIndex, afterIndex);
        context.value = value;
    }
}

function At(el) {
    this.el = el;
    this.matchStr = {};
}

At.prototype.inputMatch = function () {
    var cursorOffset = getCursorOffset(this.el);
    var text = this.el.value;
    var beforeCursorStr = text.slice(0, cursorOffset);
    var reg = new RegExp('(?:[^0-9a-z]|^)@([^\u0020@]*)$', 'i');
    var atMatch = reg.exec(beforeCursorStr);
    if (atMatch) {
        var atText = atMatch[1];
        var start = cursorOffset - atText.length;
        this.matchStr = {
            text: atText,
            start: start,
            end: cursorOffset
        };
        return this.matchStr;
    }
    return undefined;
};

// 获取光标 相对页面位置
At.prototype.getCursorPos = function () {
    var cursorPostion = $(this.el).caret('offset', this.matchStr.start  - 1);
    return cursorPostion;
};

At.prototype.insert = function (str) {
    var text = this.el.value;
    var end = Math.max(this.matchStr.start, 0);
    var beforeAtStr = text.substring(0, end);
    var atText = str + ' ';
    var afterAtStr = text.substring(this.matchStr.end);
    text = beforeAtStr + atText + afterAtStr;
    this.el.value = text;
    this.el.focus();
    var cursorOffset = beforeAtStr.length + atText.length;
    $(this.el).caret('pos', cursorOffset);
    return text;
};

At.prototype.rightClickInsert = function (str) {
    var cursorOffset = getCursorOffset(this.el);
    var text = this.el.value;
    var beforeCursorStr = text.slice(0, cursorOffset);
    var end = Math.max(this.matchStr.start, 0);
    console.log(beforeCursorStr);
    text = beforeCursorStr + "@" + str + " " + text.substring(beforeCursorStr.length);
    this.el.value = text;
    this.el.focus();
    
    $(this.el).caret('pos', cursorOffset+str.length + 2);
    return text;
};

At.prototype.removeMatch = function () {
    var text = this.el.value;
    var cursorOffset = getCursorOffset(this.el);
    var cursorScroll = cursorOffset;

    // 匹配光标在 atText 中间或之前
    var afterCursorStr = text.substring(cursorScroll);
    var afterReg = /(^[^\u0020@]*\u0020)|(^@[^\u0020@]+\u0020)/;
    var afterCursorAtMatch = afterReg.exec(afterCursorStr);

    // 将游标移至 atText 之后
    if (afterCursorAtMatch) {
        cursorScroll += afterCursorAtMatch[0].length;
    }

    // 截取游标之前的字符串
    var beforeCursorStr = text.substring(0, cursorScroll);

    // 匹配游标前的 atText
    var allReg = /@([^\u0020@]+)\u0020$/;
    var atMatch = allReg.exec(beforeCursorStr);
    if (atMatch) {
        var atText = atMatch[1];
        var start = cursorScroll - atMatch[0].length;
        var cursorPos = '';
        if (cursorOffset === start) {
            cursorPos = 'before';
        } else if (cursorOffset === cursorScroll) {
            cursorPos = 'after';
        } else {
            cursorPos = 'amongst';
        }
        return {
            cursorPos: cursorPos,
            text: atText,
            start: start,
            end: cursorScroll
        };
    }
    return undefined;
};

At.prototype.remove = function (matchStr) {
    var text = this.el.value;
    var beforeCursorStr = text.substring(0, matchStr.start);
    var afterCursorStr = text.substring(matchStr.end);
    var value = beforeCursorStr + afterCursorStr;

    this.el.value = value;
    this.el.focus();
    $(this.el).caret('pos', matchStr.start);

    return value;
};

function getEmojiEditReg() {
    var regMark = 'ig';
    var tagReg = utils.emojiNativeReg.toString();
    tagReg = tagReg.substring(1, tagReg.length - 3);
    var nativeReg = new RegExp(tagReg, regMark);
    // 删除时, 匹配 emoji + 空格
    var deleteReg = tagReg + '([ ])(?=$)';
    deleteReg = new RegExp(deleteReg, regMark);
    // 添加时, 匹配 emoji && 后面无空格
    var addReg = tagReg + '(?=[^ ]|$)';
    addReg = new RegExp(addReg, regMark);
    // 向右移动时, 匹配 emoji && 处于开始位置
    var moveRightReg = '^{{reg}}';
    moveRightReg = moveRightReg.replace('{{reg}}', tagReg);
    moveRightReg = new RegExp(moveRightReg, regMark);
    return {
        native: nativeReg,
        add: addReg,
        delete: deleteReg,
        moveLeft: deleteReg,
        moveRight: moveRightReg
    };
}
}(RongIM, {
    jQuery: jQuery
}, RongIM.components));
