(function (RongIM, dependencies, components) {
'use strict';

var common = RongIM.common;
var utils = RongIM.utils;
var moment = dependencies.moment;

/*
说明： 群公告，查看发布群公告
*/
components.getGroupNotice = function (resolve, reject) {
    var im = RongIM.instance;
    var dataModel = im.dataModel;
    var groupPermission = utils.groupPermission;
    var options = {
        name: 'group-notice',
        template: '#rong-template-group-notice',
        props: ['group'],
        data: function () {
            var params = im.$route.params;
            return {
                backupConent: '',
                user: {},
                time: 0,
                isModifying: false,
                content: '',
                conversationType: +params.conversationType,
                targetId: params.targetId
            };
        },
        components: {
            avatar: components.getAvatar
        },
        mounted: function () {
            var context = this;
            im.$on('imclick', function (event) {
                close(context, event);
            });
            var groupNoticeApi = dataModel.GroupNotice;
            var params = {
                id: context.targetId
            };
            // 获取群公告信息： 发布人 发布时间 内容。缓存内容（ backupConent ）取消编辑时恢复原内容。
            groupNoticeApi.get(params, function (errorCode, notice) {
                if (errorCode) {
                    notice = {
                        user: {},
                        content: '',
                        createDt: 0
                    };
                }
                context.user = notice.user;
                var content = notice.content;
                context.content = content;
                context.backupConent = content;
                context.time = moment(notice.create_dt).format('YYYY/M/D HH:mm');
            });
        },
        directives: {
            focus: {
                inserted: function (el) {
                    el.focus();
                }
            }
        },
        computed: {
            // 判断自己在对应群中是否被禁言，若被禁言则不可修改群公告
            isBanned: function () {
                var group = this.group;
                var members = group.members;
                var authId = im.auth.id;
                var selfInfo = members.filter(function (member) {
                    return member.id === authId;
                })[0] || {};
                var isCantNotSpeak = group.is_all_mute && selfInfo.mute_status !== 2;
                var isBanned = selfInfo.mute_status === 1;
                var isRole = selfInfo.role === 1;
                return (isCantNotSpeak || isBanned) && !isRole;
            },
            /*
            发布按钮是否可以点击
            当 发布内容为空 或 只包含空格和回车时 不可点击
            */
            publishDisabled: function () {
                var content = this.content;
                return content.replace(/\s/g, '').length === 0;
            },
            isSpecialGroup: function () {
                var group = this.group || {};
                var type = common.getGroupType(group.type);
                return !!type;
            },
            isOwnerManage: function () {
                var group = this.group || {};
                return group.invite_member === groupPermission.Owner && group.publish_group_notice === groupPermission.Owner;
            },
            ownerManageCondition: function () {
                return (this.isOwnerManage && this.group.is_creator) || !this.isOwnerManage;
            }
        },
        methods: getMethods(im)
    };
    utils.asyncComponent(options, resolve, reject);
};

function getMethods(im) {
    var dataModel = im.dataModel;
    var groupNoticeApi = dataModel.GroupNotice;
    return {
        close: function () {
            this.$emit('hidepanel');
            im.$off('imclick', this.close);
        },
        isShow: function (type) {
            var context = this;
            var content = context.content || '';
            var hasNotice = content.length;
            var isModifying = context.isModifying;
            var ownerManageCondition = context.ownerManageCondition;

            var showMap = {
                clear: function () {
                    return hasNotice && !isModifying && ownerManageCondition;
                },
                edit: function () {
                    return !isModifying && ownerManageCondition;
                },
                cancel: function () {
                    return isModifying;
                },
                publish: function () {
                    return isModifying;
                },
                empty: function () {
                    return !hasNotice && !isModifying;
                },
                'edit-box': function () {
                    return isModifying;
                },
                notice: function () {
                    return !isModifying && hasNotice;
                }
            };
            return showMap[type]();
        },
        getGroupUsername: function (user) {
            return common.getGroupUsername(user, this.group.id);
        },
        edit: function () {
            var context = this;
            context.isModifying = true;
        },
        /*
        说明： 取消编辑群公告
        */
        cancel: function () {
            var context = this;
            var locale = context.locale;
            var tips = locale.tips;
            var confirmCancel = function () {
                context.isModifying = false;
                context.content = context.backupConent;
                context.close();
            };
            /*
            当公告为空(或只包含空格或者回车)时, 不弹框提示, 直接取消
             */
            if (this.publishDisabled) {
                confirmCancel();
                return;
            }
            common.messagebox({
                type: 'confirm',
                message: locale.cancel,
                submitText: tips.msgboxSubmitText,
                callback: confirmCancel
            });
        },
        /*
        说明： 清除群公告，弹框提示确认清除
        */
        clear: function () {
            var context = this;
            var locale = context.locale;
            var tips = locale.tips;
            common.messagebox({
                type: 'confirm',
                message: locale.clear,
                submitText: tips.msgboxSubmitText,
                callback: function () {
                    context.content = '';
                    context.backupConent = '';
                    var params = {
                        id: context.targetId
                    };
                    groupNoticeApi.remove(params, function (errorCode) {
                        if (errorCode) {
                            common.toastError(errorCode);
                        }
                    });
                    context.close();
                }
            });
        },
        /*
        说明： 发布群公告，弹框提示确认发布
        */
        publish: function () {
            var context = this;
            var content = context.content;
            if (utils.isEmpty(content)) {
                return;
            }
            if (this.isOwnerManage && !this.group.is_creator) {
                common.messageToast({
                    message: context.locale.onlyOwnerManage,
                    type: 'error'
                });
                return;
            }
            var locale = context.locale;
            var tips = locale.tips;
            common.messagebox({
                type: 'confirm',
                message: locale.confirm,
                submitText: tips.publish,
                callback: function () {
                    context.isModifying = false;
                    context.backupConent = content;
                    var params = {
                        id: context.targetId,
                        content: content
                    };
                    groupNoticeApi.publish(params, function (errorCode) {
                        if (errorCode) {
                            common.toastError(errorCode);
                        }
                    });
                    context.close();
                }
            });
        }
    };
}

function close(context, event) {
    var $target = $(event.target);
    var wrap = '.rong-dialog, .rong-group-create, .rong-group-remove';
    var inBody = $target.closest('body').length > 0;
    var inWrap = $target.closest(wrap).length < 1;
    var isOuter = inBody && inWrap;
    if (isOuter) context.close();
}
}(RongIM, {
    jQuery: jQuery,
    moment: moment
}, RongIM.components));
