(function (RongIM, dependencies) {
    'use strict';

    var $ = dependencies.jQuery;
    var QRCode = dependencies.QRCode;
    var Http = RongIM.dataModel._Http;
    var Cache = RongIM.dataModel._Cache;
    var ObserverList = RongIM.dataModel._ObserverList;
    var httpRequest = RongIM.dataModel._httpRequest;

    var utils = RongIM.utils;
    var common = RongIM.common;

    var orgApi = null;
    var userApi = null;
    var conversationApi = null;
    var messageApi = null;

    var Group = {
        observerList: new ObserverList()
    };

    var groupObserverList = Group.observerList;

    Cache.group = {
        _defer: {}
    };

    Group.cleanCache = function () {
        Cache.group = {
            _defer: {}
        };
    };

    Group.loadApi = function () {
        var dataModel = RongIM.dataModel;
        orgApi = dataModel.Organization;
        userApi = dataModel.User;
        conversationApi = dataModel.Conversation;
        messageApi = dataModel.Message;
    };

    Group.registerMessage = function () {
        // 群成员变化消息
        var messageName = 'GroupMemChangedNotifyMessage';
        var objectName = 'RCE:GrpMemChanged';
        var messageTag = new RongIMLib.MessageTag(false, true);
        var properties = ['action', 'operatorUser', 'targetGroup', 'targetUsers', 'extra'];
        RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

        // 群信息更新消息
        messageName = 'GroupNotifyMessage';
        objectName = 'RCE:GrpNtfy';
        messageTag = new RongIMLib.MessageTag(false, true);
        properties = ['action', 'data', 'operatorUser', 'targetGroup', 'targetUsers'];
        RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

        // 群组命令消息
        messageName = 'GroupCmdMessage';
        objectName = 'RCE:GrpCmd';
        messageTag = new RongIMLib.MessageTag(false, false);
        properties = ['action', 'operatorUser', 'targetGroup'];
        RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

        messageName = 'GroupVerifyNotifyMessage';
        objectName = 'RCE:GrpRcvNtfy';
        messageTag = new RongIMLib.MessageTag(false, false);
        properties = ['operatorUser', 'targetGroup', 'action', 'targetUser', 'lastReceiver'];
        RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);
    };

    /* 收到被踢, 被禁言消息时, clear草稿 */
    function clearDraft(message) {
        var conversationType = message.conversationType;
        var targetId = message.targetId;
        conversationApi = conversationApi || RongIM.dataModel.Conversation;
        var targetIds;
        var isSelf;
        switch (message.messageType) {
            case 'GroupMemChangedNotifyMessage':
                var action = message.content.action;
                var isQuit = action === 3;
                targetIds = message.content.targetUsers.map(function (item) {
                    return item.id;
                });
                if (RongIM.instance.auth) {
                    isSelf = targetIds.indexOf(RongIM.instance.auth.id);
                } else {
                    isSelf = false;
                }
                if (isQuit && isSelf) {
                    conversationApi.clearDraft(conversationType, targetId);
                }
                break;
            case 'GroupNotifyMessage':
                action = message.content.action;
                var isBanned = action === 23;
                var isClear = isBanned;
                if (isBanned) {
                    targetIds = message.content.targetUsers.map(function (item) {
                        return item.id;
                    });
                    isSelf = targetIds.indexOf(Cache.auth.id);
                    isClear = isClear && isSelf;
                }
                var isBannedAll = action === 21;
                if (isClear || isBannedAll) {
                    conversationApi.clearDraft(conversationType, targetId);
                }
                break;
            default:
                break;
        }
    }

    Group.messageCtrol = {
        GroupMemChangedNotifyMessage: function (message) {
            clearDraft(message);
            updateGroupNotify(message);
        },
        GroupNotifyMessage: function (message) {
            clearDraft(message);
            updateGroup(message);
            messageApi.observerList.notify(message);
            conversationApi.notifyConversation();
        },
        GroupCmdMessage: function (message) {
            if (message.offLineMessage) {
                return;
            }
            updateGroup(message);
        }
    };
    Group.create = function (params, callback) {
        callback = callback || $.noop;
        var data = {
            type: params.type,
            name: params.name,
            portrait_url: '',
            member_ids: params.member_ids
        };
        Http.post('/groups', data).done(function (result) {
            Group.getOne(result.id, function (errorCode, group) {
                if (errorCode) {
                    callback(errorCode);
                    return;
                }
                var arg = {
                    conversationType: RongIMLib.ConversationType.GROUP,
                    targetId: group.id
                };
                conversationApi.add(arg);
                callback(null, group);
            });
        }).fail(callback);
    };

    Group.addToFav = function (idList, callback) {
        var data = {
            ids: idList
        };
        Http.post('/favgroups', data, callback);
    };

    Group.removeFromFav = function (idList, callback) {
        var data = {
            ids: idList
        };
        Http.del('/favgroups', data, callback);
    };

    Group.rename = function (groupId, name, callback) {
        var data = {
            name: name
        };
        Http.put('/groups/' + groupId + '/name', data, callback);
    };

    Group.getMember = function (memberId) {
        return Cache.user[memberId] ? Cache.user[memberId].name : 'user<' + memberId + '>';
    };

    Group.getNewGroup = function (groupId, callback) {
        callback = callback || $.noop;
        if (utils.isEmpty(groupId)) {
            callback('params error: groupId invalid');
            return;
        }
        getGroups([groupId], function (errorCode, list) {
            if (errorCode) {
                callback(errorCode);
                return;
            }
            var group = list[0] || {};
            var memberIdList = group.members && group.members.map(function (item) {
                return item.id;
            });
            userApi.getUsers(memberIdList, function (error, memberList) {
                if (error) {
                    callback(error);
                    return;
                }
                group.groupMembers = memberList.map(function (item) {
                    var newItem = group.members.find(function (ele) {
                        return ele.id === item.id;
                    });
                    var endItem = $.extend({}, item, newItem);
                    return endItem;
                });
                callback(null, group);
            });
            updateConversationGroup(group);
        }, {
            isRemote: true
        });
    };

    // 更新会话中的群信息，会话中群信息为啥没有保持引用不不清楚
    function updateConversationGroup(group) {
        var _conversation = conversationApi.getLocalOne(RongIMLib.ConversationType.GROUP, group.id);
        if (_conversation) {
            _conversation.group = group;
        }
    }

    // 获取群组内所有成员
    Group.getAllMembers = function (groupId, callback, isRefresh) {
        if (isRefresh) {
            delete Cache.group[groupId];
        }
        getGroups([groupId], function (errorCode, groupList) {
            if (errorCode) {
                callback(errorCode);
                return;
            }
            var group = groupList[0] || {};
            if (group.groupMembers) {
                callback(null, group.groupMembers, group);
                return;
            }
            Http.get('/groups/' + groupId + '/members', function (error, result) {
                if (error) {
                    callback(error);
                    return;
                }
                group.groupMembers = result.data.map(function (item) {
                    return {
                        id: item.id,
                        name: item.name,
                        avatar: item.portrait_url,
                        state: item.state,
                        type: item.user_type,
                        groupAlias: item.alias,
                        portrait_big_url: item.portrait_big_url,
                        mute_status: item.mute_status
                    };
                });
                callback(null, group.groupMembers, group);
            });
        });
    };

    Group.addMembers = function (groupId, memberIdList, callback) {
        var data = {
            ids: memberIdList
        };
        Http.post('/groups/' + groupId + '/invite', data, callback);
    };

    Group.removeMembers = function (groupId, memberIdList, callback) {
        var data = {
            ids: memberIdList
        };
        Http.post('/groups/' + groupId + '/remove', data, callback);
    };

    /*
    设置群昵称 PUT
    删除群昵称 DELETE
    */
    Group.modifyMemberAlias = function (groupId, memberId, name, callback) {
        var data = {
            alias: name
        };
        var url = '/groups/' + groupId + '/members/' + memberId + '/alias';
        var method = name === '' ? 'DELETE' : 'PUT';
        httpRequest(method, url, data, function (errorCode) {
            if (errorCode) {
                callback(errorCode);
                return;
            }
            Cache.group[groupId].members.forEach(function (member) {
                if (member.id === memberId) {
                    member.groupAlias = name;
                }
            });
            callback(null);
        });
    };

    Group.watch = function (handle) {
        groupObserverList.add(handle);
    };

    Group.unwatch = function (handle) {
        groupObserverList.remove(handle);
    };

    Group.quit = function (groupId, callback) {
        Http.post('/groups/' + groupId + '/quit').done(function (result) {
            callback(null, result);
        }).fail(callback);
    };

    Group.dismiss = function (groupId, callback) {
        Http.del('/groups/' + groupId).done(function (result) {
            callback(null, result);
        }).fail(callback);
    };

    Group.getOne = function (groupId, callback) {
        callback = callback || $.noop;
        var error = null;
        var group = Cache.group[groupId];
        if (group) {
            callback(error, group);
            return;
        }
        getGroups([groupId], function (errorCode, list) {
            if (errorCode) {
                callback(errorCode);
                return;
            }
            callback(error, list[0]);
        });
    };

    Group.getList = function (callback) {
        callback = callback || $.noop;
        /**
         * 实现过程
         * 1. 获取收藏群组列表
         * 2. 从`Cache.group`中获取群组信息
         * 3. 缓存中没有从服务器获取群组信息，群成员信息，更新缓存`Cache.group`
         * 4. 根据 memberId 获取群成员信息 userApi.getBatch([userId], callback)
         */
        Http.get('/favgroups', function (errorCode, result) {
            if (errorCode) {
                callback(errorCode);
                return;
            }
            var idList = result.data
                .sort(function (a, b) {
                    return a.create_dt < b.create_dt;
                })
                .map(function (group) {
                    return group.id;
                });
            getGroups(idList, callback);
        });
    };

    Group.getBatch = function (idsList, callback) {
        callback = callback || $.noop;
        getGroups(idsList, callback);
    };

    Group.search = function (keyword, callback) {
        keyword = keyword.replace('%', function (item) {
            return encodeURI(item);
        });
        Http.post('/groups/search', {
            keywords: [keyword]
        }).done(function (result) {
            var ids = result.map(function (item) {
                return item.id;
            });
            Group.getBatchGroups(ids).then(function () {
                getGroups(ids, callback, null, result);
            });
        }).fail(callback);
    };

    Group.bannedAll = function (params, callback) {
        var id = params.id;
        var url = '/groups/' + id + '/gag/set_all_mute';
        Http.post(url, {
            mute_status: params.status
        }).done(function (result) {
            var error = null;
            callback(error, result);
        }).fail(callback);
    };
    /*
        params = {
            id: '',
            members = [{
                memberId: 'dka12d',
                status: 1
            }]
        }
    */
    Group.banned = function (params, callback) {
        var members = params.members.map(function (item) {
            return {
                member_id: item.memberId,
                mute_status: item.status
            };
        });
        Http.post('/groups/' + params.id + '/gag/set_member_mute', {
            data: members
        }).done(function (result) {
            var error = null;
            callback(error, result);
        }).fail(callback);
    };

    Group.getApproveList = function (callback) {
        var url = '/groups/receivers';
        Http.get(url).done(function (result) {
            addUserToList(result, 'receiver_id').done(function () {
                callback(null, result);
            });
        }).fail(callback);
    };

    Group.setPermission = function (groupId, params, callback) {
        var url = '/groups/' + groupId + '/permission';
        Http.put(url, params).done(function () {
            callback(null);
            RongIM.syncdata.groupById(groupId);
        }).fail(callback);
    };

    /* 审批, status: 1为通过, 5为删除 */
    Group.approve = function (groupId, userId, status, callback) {
        var url = '/groups/' + groupId + '/approve_receiver/' + userId;
        var data = {
            approve_status: status
        };
        Http.post(url, data).done(function () {
            callback(null);
        }).fail(callback);
    };

    Group.clearApprove = function (callback) {
        var url = '/groups/clear_receiver';
        Http.post(url, callback);
    };

    Group.getApproveUnread = function (callback) {
        var url = '/groups/receiver_unread';
        Http.get(url).done(function (result) {
            var unreadCount = result.unread_count;
            callback(null, unreadCount);
        }).fail(callback);
    };

    Group.clearApproveUnRead = function (callback) {
        var url = '/groups/clear_receiver_unread';
        Http.post(url).done(function () {
            callback(null);
        }).fail(callback);
    };

    /*
        params.id
        params.manager
    */
    Group.transfer = function (params, callback) {
        var id = params.id;
        var manager = params.manager;
        var url = '/groups/' + id + '/transfer_manager';
        Http.put(url, {
            id: manager
        }).done(function (result) {
            callback(null, result);
        }).fail(callback);
    };

    Group.createQRCode = function (node, userId, groupId, size) {
        var text = utils.templateFormat('rce://group/join?code&{{0}}&{{1}}', groupId, userId);
        text = encodeURIComponent(text);
        var shareUrls = RongIM.serverConfig.qrcode.share_urls;
        var isHttps = location.href.startsWith('https');
        var imUrl = shareUrls[0];
        var configIsHttps = imUrl.startsWith('https');
        if ((isHttps && !configIsHttps) || (!isHttps && configIsHttps)) {
            imUrl = shareUrls[1];
        }
        text = imUrl + '?key=' + text;
        $(node).empty();
        // eslint-disable-next-line no-new
        new QRCode(node, {
            text: text,
            width: size.width,
            height: size.height,
            correctLevel: QRCode.CorrectLevel.L
        });
    };

    function addUserToList(list, key) {
        var def = $.Deferred();
        var count = 0;
        if (list.length === 0) {
            def.resolve();
        } else {
            var ids = list.map(function (item) {
                return item[key];
            });
            userApi.get(ids, function (errorCode, userList) {
                userList = [].concat(userList);
                userList.forEach(function (user) {
                    count += 1;
                    list.forEach(function (data) {
                        if (user && user.id === data[key]) {
                            data.user = user;
                        }
                    });
                    if (count === userList.length) def.resolve();
                });
            });
        }
        return def;
    }

    // 返回不在缓存里的keys
    function filterCache(cache, keys) {
        return keys.filter(function (key) {
            return !cache[key];
        });
    }

    /**
     * getGroups
     * @param  {array}   idList
     * @param  {function} callback
     * @param  {object}   option: isRemote(是否从服务器获取)
     * @param  {array}   searchInfo: 搜索群组结果
     */
    function getGroups(idList, callback, option, searchInfo) {
        callback = callback || $.noop;
        option = option || {};
        var isRemote = option.isRemote;
        if (idList.length <= 0) {
            return callback(null, []);
        }
        var newIdList = idList;
        if (!isRemote) {
            newIdList = filterCache(Cache.group, idList);
        }
        if (newIdList <= 0) {
            var groups = [];
            if (searchInfo) {
                groups = searchInfo.map(function (item) {
                    var _group = Cache.group[item.id];
                    _group.user_list = item.user_list;
                    return _group;
                });
            } else {
                groups = idList.map(function (id) {
                    return Cache.group[id];
                });
            }
            return callback(null, groups);
        }
        var promiseList = newIdList.map(function (id) {
            return getOneGroup(id);
        });
        return $.when.apply(null, promiseList).then(function () {
            var groupList = idList.map(function (id) {
                return Cache.group[id];
            });
            callback(null, groupList);
            return groupList;
        });
    }
    Group.getGroups = getGroups;

    var groupUpdateTimer = {};

    function updateGroupNotify(message) {
        var requireClear = requireClearGroup(message);

        if (requireClear.group) {
            Group.removeFromFav([message.targetId]);
        }
        if (requireClear.conversation) {
            conversationApi.remove(RongIMLib.ConversationType.GROUP, message.targetId);
        }

        messageApi = messageApi || RongIM.dataModel.Message;

        var authId = Cache.auth.id;
        var isTargetUsers = false;
        message.content.targetUsers.forEach(function (item) {
            if (item.id === authId) {
                isTargetUsers = true;
            }
        });
        if (message.offLineMessage) {
            return;
        }
        var groupId = message.targetId;
        clearTimeout(groupUpdateTimer[groupId]);
        groupUpdateTimer[groupId] = setTimeout(function () {
            RongIM.syncdata.groupById(groupId, function () {
                if (message.content.action === 3 && isTargetUsers) {
                    Group.getNewGroup(groupId, function (errorCode, group) {
                        if (!errorCode) groupObserverList.notify(group);
                    });
                } else {
                    Group.getAllMembers(groupId, function (errorCode, members, group) {
                        if (errorCode) {
                            return;
                        }
                        groupObserverList.notify(group);
                        // pc端会话列表群组信息不更新，web端有更新，不知道为啥 zhaokeyang
                        updateConversationGroup(group);
                        // messageApi._push(message);
                    }, true);
                }
            });
        }, 1000);
    }

    // 群通知消息单独处理
    Group.getNotifyGroupMsg = function (message) {
        if (message.messageType !== 'GroupMemChangedNotifyMessage') {
            return false;
        }

        var isMe = RongIM.dataModel._Cache.auth.id;
        var group = message.content.targetGroup;
        // 操作行为类型
        var messageAction = message.content.action;
        // 群组减员通知：3，被移除；4，主动退出
        if (messageAction === 3 || messageAction === 4) {
            // 非操作者，后台操作时，操作者 id 为群主
            var isNotOperatorUser = message.content.operatorUser.id !== isMe;
            // 非被操作者
            var isNotTargetUsers = !message.content.targetUsers.some(function (item) {
                return item.id === isMe;
            });
            var isManager = group.manager_id === isMe;
            var params = {
                conversation: message.conversationType,
                targetId: message.targetId,
                messageId: message.messageId,
                notNotify: true
            };
            // 无关消息，不显示
            if (isNotOperatorUser && isNotTargetUsers && !isManager) {
                delayRemoveLocalMessage(params);
                return false;
            }
            // 群组减员消息，只通知群主且群必须为自建群，部门群等不通知
            if (!isManager || group.type !== 0) {
                delayRemoveLocalMessage(params);
                return false;
            }
        }
        return true;
    };

    var delayRemoveList = [];

    function deayHandel() {
        removeLocalMessageBatch(delayRemoveList);
        delayRemoveList = [];
        messageApi.unwatchOfflineReceivefinish(deayHandel);
    }
    // 收离线消息时延时执行当收完离线消息时执行
    function delayRemoveLocalMessage(params) {
        if (RongIM.offlineMessageReceiving) {
            if (delayRemoveList.length === 0) {
                messageApi.watchOfflineReceivefinish(deayHandel);
            }
            delayRemoveList.push(params);
            return;
        }
        removeLocalMessageBatch([params]);
    }

    function removeLocalMessageBatch(paramsList) {
        var messageIds = [];
        paramsList.forEach(function (params) {
            messageIds.push(params.messageId);
            var key = params.conversationType + '_' + params.targetId;
            var list = messageApi._cache[key];
            if (!list || list.length === 0) {
                return;
            }
            for (var i = 0, length = list.length; i < length; i += 1) {
                var msg = list[i];
                if (msg.messageId === params.messageId) {
                    list.splice(i, 1);
                    return;
                }
            }
        });
        RongIMClient.getInstance().deleteLocalMessages('', '', messageIds, {
            onSuccess: function () {},
            onError: function () {}
        });
    }

    function updateGroup(message) {
        var requireClear = requireClearGroup(message);

        if (message.offLineMessage && requireClear.conversation) {
            conversationApi.remove(RongIMLib.ConversationType.GROUP, message.targetId);
            return;
        }

        if (requireClear.group) {
            Group.removeFromFav([message.targetId]);
        }
        if (requireClear.conversation) {
            conversationApi.remove(RongIMLib.ConversationType.GROUP, message.targetId);
        }
        if (message.messageType !== 'GroupCmdMessage') {
            messageApi._push(message);
        }
        if (message.offLineMessage) {
            return;
        }
        var groupId = message.targetId;
        clearTimeout(groupUpdateTimer[groupId]);
        groupUpdateTimer[groupId] = setTimeout(function () {
            RongIM.syncdata.groupById(groupId, function () {
                // 23 群组禁言黑名单添加操作 24 群组禁言黑名单删除操作
                var isUpdateMember = message.messageType === 'GroupNotifyMessage' && [23, 24].indexOf(message.content.action) > -1;
                if (isUpdateMember) {
                    Group.getAllMembers(groupId, function (errorCode, members, group) {
                        if (errorCode) {
                            return;
                        }
                        groupObserverList.notify(group);
                    }, true);
                } else {
                    Group.getNewGroup(groupId, function (errorCode, group) {
                        if (!errorCode) groupObserverList.notify(group);
                    });
                }
            });
        });
    }

    function requireClearGroup(message) {
        var actionMap = {
            GroupMemChangedNotifyMessage: {
                1: 'Invite',
                2: 'Join',
                3: 'Kick',
                4: 'Quit'
            },
            GroupNotifyMessage: {
                1: 'Create',
                2: 'Dismiss',
                4: 'Rename'
            },
            GroupCmdMessage: {
                1: 'UpdPortrait',
                2: 'UpdManager'
            }
        };
        var action = message.content.action;
        var actionText = actionMap[message.messageType][action];
        var list = ['Kick', 'Quit', 'Dismiss'];
        var targetUsers = message.content.targetUsers;
        var includeMe = false;
        if (targetUsers) {
            includeMe = targetUsers.filter(function (item) {
                return item.id === Cache.auth.id;
            }).length > 0;
        } else {
            includeMe = true;
        }

        var clearGroup = list.indexOf(actionText) >= 0 && includeMe;

        var isOperator = message.content.operatorUser.id === Cache.auth.id;
        var clearConversation = false;
        if (actionText === 'Quit' && isOperator) {
            clearConversation = true;
        }
        var isSender = (message.messageDirection === RongIMLib.MessageDirection.SEND);
        if (actionText === 'Dismiss' && isSender) {
            clearConversation = true;
        }
        // 如果是部门群解散，直接删除群组
        // getGroups([message.targetId], function (errorCode, groupList) {
        //     if (errorCode) {
        //         return;
        //     }
        //     var group = groupList[0];
        //     if (+group.type !== 0 && actionText === 'Dismiss') {
        //         conversationApi.remove(RongIMLib.ConversationType.GROUP, message.targetId);
        //     }
        // });
        Group.concatRequest(message.targetId).then(function (group) {
            if (!group) {
                console.error('group is undefined!!', message);
                return;
            }
            if (+group.type !== 0 && actionText === 'Dismiss') {
                conversationApi.remove(RongIMLib.ConversationType.GROUP, message.targetId);
            }
        });
        // clearConversation = clearConversation || (['Kick'].indexOf(actionText) >=0 && includeMe);
        return {
            group: clearGroup,
            conversation: clearConversation
        };
    }

    Group.batchMembers = function (idList) {
        var url = '/groups/batch/members';
        var data = {
            ids: idList
        };
        return Http.post(url, data).done(function (result) {
            return result;
        });
    };

    // 不提供回调，获取后直接在缓存中获取
    Group.getBatchGroups = function (idList, isCover) {
        var requestIdList = [];
        var promiseList = [];
        idList.forEach(function (id) {
            var defer = Cache.group._defer[id];
            var group = Cache.group[id];
            if (defer) {
                promiseList.push(defer.promise());
            } else if (utils.isEmpty(group) || isCover) {
                defer = $.Deferred();
                promiseList.push(defer.promise());
                requestIdList.push(id);
                Cache.group._defer[id] = defer;
            }
        });

        if (requestIdList.length > 0) {
            var url = '/groups/batch';
            // 复制一份防止被修改
            var data = {
                ids: requestIdList.concat()
            };
            Http.post(url, data).done(function (groups) {
                var groupIds = [];
                groups.forEach(function (group) {
                    group.avatar = group.portrait_url;
                    group.admin_id = group.manager_id;
                    groupIds.push(group.id);
                    var organization = orgApi.getLocalDept(group.organization_id);
                    if (organization.type === common.OrgType.COMPANY) {
                        var co = orgApi.getLocalCompany(group.organization_id) || {};
                        organization.fullName = co.fullName;
                    }
                    group.organization = organization;
                });
                // 缓存群组前9个人id
                var _members = {};
                var bindMembers = function (group) {
                    group.member_brief = group.members.map(function (member) {
                        return {
                            id: member.id,
                            createDt: member.create_dt
                        };
                    });

                    var groupMemberInfo = _members[group.id];
                    group.member_id_list = group.members.map(function (member) {
                        return member.id;
                    });
                    group.member_names = groupMemberInfo.memberInfos.map(function (item) {
                        return item.name;
                    });
                    group.member_avatars = groupMemberInfo.memberInfos.map(function (item) {
                        return item.portrait_url;
                    });
                    var id = group.id;
                    Cache.group[id] = group;
                };
                Group.batchMembers(groupIds).then(function (result) {
                    result.forEach(function (item) {
                        _members[item.group_id] = {
                            memberIds: item.member_ids,
                            memberInfos: item.member_infos
                        };
                    });
                    groups.forEach(bindMembers);
                    requestIdList.forEach(function (id) {
                        Cache.group._defer[id].resolve();
                        delete Cache.group._defer[id];
                    });
                }, function (error) {
                    requestIdList.forEach(function (id) {
                        Cache.group._defer[id].reject(error);
                        delete Cache.group._defer[id];
                    });
                });
            }).fail(function (error) {
                requestIdList.forEach(function (id) {
                    Cache.group._defer[id].reject(error);
                    delete Cache.group._defer[id];
                });
            });
        }

        var groupDefer = $.Deferred();
        $.when.apply(null, promiseList).then(function () {
            groupDefer.resolve();
        }).fail(function (errorCode) {
            groupDefer.reject(errorCode);
        });
        return groupDefer.promise();
    };

    function getOneGroup(id) {
        if (Cache.group._defer[id]) {
            return Cache.group._defer[id].promise();
        }

        var defer = $.Deferred();
        Cache.group._defer[id] = defer;

        var getGroupFailed = function () {
            var group = {
                id: id,
                member_avatars: [],
                member_names: [],
                member_id_list: []
            };
            var thisGroup = Cache.group[id] || group;
            // Cache.group[id] = thisGroup;
            defer.resolve(thisGroup);
        };

        var url = '/groups/' + id;
        Http.get(url).then(function (group) {
            if (!group) {
                getGroupFailed();
                return;
            }

            group.avatar = group.portrait_url;
            group.admin_id = group.manager_id;
            group.member_id_list = group.members.map(function (member) {
                return member.id;
            });
            group.member_brief = group.members.map(function (member) {
                return {
                    id: member.id,
                    createDt: member.create_dt
                };
            });
            var organization = orgApi.getLocalDept(group.organization_id);
            if (organization.type === common.OrgType.COMPANY) {
                var co = orgApi.getLocalCompany(group.organization_id) || {};
                organization.fullName = co.fullName;
            }
            group.organization = organization;
            Group.batchMembers([group.id]).then(function (result) {
                var groupMemberInfo = ((result[0] || {}).member_infos || []);
                group.member_names = groupMemberInfo.map(function (userItem) {
                    return userItem.name;
                });
                group.member_avatars = groupMemberInfo.map(function (userItem) {
                    return userItem.portrait_url;
                });
                Cache.group[id] = group;
                defer.resolve(group);
            }, function (error) {
                defer.reject(error);
            });
        }).fail(getGroupFailed).always(function () {
            delete Cache.group._defer[id];
        });
        return defer.promise();
    }

    // 获取群昵称
    Group.getGroupAlias = function (userId, members) {
        var groupAlias = '';
        members.forEach(function (member) {
            if (member.id === userId) {
                groupAlias = member.alias;
            }
        });
        return groupAlias;
    };

    // 更新某个群成员的信息
    Group.updateGroupMember = function (user) {
        var groups = Cache.group;
        Object.keys(groups).forEach(function (item) {
            if (item !== '_defer') {
                var group = groups[item];
                var index = group.member_id_list.indexOf(user.id);
                if (index > -1 && group.groupMembers) {
                    $.extend(group.groupMembers[index], user);
                }
            }
        });
    };

    var timeoutConcatRequest = null;
    var delayGroupIdList = [];
    var concatRequestDefer = {};
    Group.concatRequest = function (groupId) {
        var defer = concatRequestDefer[groupId];
        if (!defer) {
            concatRequestDefer[groupId] = $.Deferred();
            defer = concatRequestDefer[groupId];
        }
        clearTimeout(timeoutConcatRequest);
        delayGroupIdList.push(groupId);
        timeoutConcatRequest = setTimeout(function () {
            var idList = [].concat(delayGroupIdList);
            delayGroupIdList = [];
            Group.getBatchGroups(idList).then(function () {
                idList.forEach(function (id) {
                    if (concatRequestDefer[id]) {
                        concatRequestDefer[id].resolve(Cache.group[id]);
                        concatRequestDefer[id] = null;
                    }
                });
            }, function () {
                idList.forEach(function (id) {
                    if (concatRequestDefer[id]) {
                        concatRequestDefer[id].reject();
                        concatRequestDefer[id] = null;
                    }
                });
            });
        }, 500);
        return defer.promise();
    };
    Group.getPinItems = function (id, callback) {
        var url = '/pins/pinGroup/' + id + '/items';
        Http.get(url).done(function (result) {
            callback(null, result);
        }).fail(callback);
    };
    //.通讯组名称唯一性校验
    Group.checkPinGroupName = function (params, callback) {
        let val = {
            'pinGroup_name': params
        }
        Http.post('/pins/checkPinGroupName', val).then(function (errorCode, result) {
            console.log('*******', errorCode, result)
            callback(result);
        }).fail(callback);

    }

    //新建通讯组
    Group.createContact = function (params, callback) {
        Http.post('/pins/pinGroup', params)
            .then(function (result) {
                callback(null, result);
            }).fail(callback);
    };
    // //获取通讯组详情
    Group.getPinGroupsDetailByid = function (id, callback) {
        let paStr = "?uid=" + id
        Http.get('/pins/pinGroups' + paStr, function (errorCode, result) {
            if (!result.data) {
                callback(errorCode);
                return;
            }
            Group.getPinItems(id, function (errorCode, res) {
                let item = {};
                var groupIds = [];
                res.data.forEach(function (group) {
                    group.avatar = group.portrait_url;
                    group.admin_id = group.manager_id;
                    groupIds.push(group.id);
                    var organization = orgApi.getLocalDept(group.organization_id);
                    if (organization.type === common.OrgType.COMPANY) {
                        var co = orgApi.getLocalCompany(group.organization_id) || {};
                        organization.fullName = co.fullName;
                    }
                    group.organization = organization;
                });
                item.members = res.data;
                item.member_count = res.count
                item.member_id_list = res.data.map(res => res.id)
                item.member_names = res.data.map(res => res.name)
                item.member_brief = res.data.map(res => {
                    return {
                        id: res.id,
                        createDt: res.create_dt
                    }
                })
                item.member_avatars = res.data.map(res => res.portrait_url)
                result.data[0].member = item
                return callback(null, result);
            })

        }).fail(callback);

    }
    //给群聊设置通讯组
    Group.changePersonalGroup = function (param, callback) {
        Http.put('/groups/changePersonalGroup', param, function (errorCode, result) {
            callback(errorCode, result);
        }).fail(callback);
    }

    function getGroupMembers(id, callback) {
        let result = null;
        Group.getPinItems(id, function (errorCode, res) {
            var groupIds = [];
            res.data.forEach(function (group) {
                group.avatar = group.portrait_url;
                group.admin_id = group.manager_id;
                groupIds.push(group.id);
                var organization = orgApi.getLocalDept(group.organization_id);
                if (organization.type === common.OrgType.COMPANY) {
                    var co = orgApi.getLocalCompany(group.organization_id) || {};
                    organization.fullName = co.fullName;
                }
                group.organization = organization;
            });
            result.members = res.data;
            result.member_count = res.count
            result.member_id_list = res.data.map(res => res.id)
            result.member_names = res.data.map(res => res.name)
            result.member_brief = res.data.map(res => {
                return {
                    id: res.id,
                    createDt: res.create_dt
                }
            })
            result.member_avatars = res.data.map(res => res.portrait_url)
            return result
        })
    }
    Group.OpenInvita = function (id,params,callback) {
        var url = '/staffs/' + id +'/openInvita';
        Http.put(url,{
            invita_state : params
        }).done(function (result) {
            callback(result);
        }).fail(callback);
    };
    Group.groupInfo = function (id,callback) {
        var url = '/groups/'+id
        Http.get(url).done(function (result) {
            callback(result);
        }).fail(callback);
    };
    Group.inviteAccept = function (gid,callback) {
        var url = '/groups/'+ gid + '/invitee_accept'
        Http.get(url).done(function (result) {
            callback(null,result);
        }).fail(callback);
    };

    RongIM.dataModel.Group = Group;
}(RongIM, {
    jQuery: jQuery,
    QRCode: window.QRCode
}));