(function (RongIM, dependencies) {
    'use strict';

    var $ = dependencies.jQuery;
    var QRCode = dependencies.QRCode;
    var Http = RongIM.dataModel._Http;
    var Cache = RongIM.dataModel._Cache;
    var ObserverList = RongIM.dataModel._ObserverList;
    var request = RongIM.dataModel._request;
    var store = RongIM.utils.cache;
    var config = RongIM.dataModel.config;

    var utils = RongIM.utils;
    var common = RongIM.common;

    var conversationApi = RongIM.dataModel.Conversation;
    var fileApi = RongIM.dataModel.File;
    var friendApi = RongIM.dataModel.Friend;

    var User = {
        observerList: new ObserverList()
    };

    var userObserverList = User.observerList;

    Cache.user = {
        _defer: {}
    };
    Cache.alias = {};

    User.cleanCache = function () {
        Cache.user = {
            _defer: {}
        };
        Cache.alias = {};
    };

    User.loadApi = function () {
        var dataModel = RongIM.dataModel;
        conversationApi = dataModel.Conversation;
        fileApi = dataModel.File;
        friendApi = dataModel.Friend;
        config = dataModel.config;
        utils = RongIM.utils;
    };

    function afterLogin(params, errorCode, result, callback) {
        if (errorCode) {
            callback(errorCode, result);
            return;
        }
        var staff = result.staff;
        staff.deptId = staff.departId || staff.depart_id || staff.dept_id;
        staff.deptName = staff.departName || staff.depart_name || staff.dept_name;
        staff.companyId = staff.companyId || staff.company_id;
        staff.orgsInfo = staff.orgsInfo || staff.orgs_info;
        Cache.auth = {
            token: result.token,
            id: staff.id,
            deptId: staff.deptId,
            companyId: staff.companyId,
            orgsInfo: staff.orgsInfo,
            enable_care:staff.enable_care
        };
        if (staff.enable_care == 1) {
            document.documentElement.style.setProperty('--font-size','18px')
            document.documentElement.style.setProperty('--height-big','80px')
        } else {
            document.documentElement.style.setProperty('--font-size','14px')
            document.documentElement.style.setProperty('--height-big','50px')

        }
        resetDeviceState();

        function finishLogin(err, res) {
            store.set('loginInfo', {
                zkxToken: (res && res.st) ? res.st : result.zkxToken,
                zkxEmpId: result.zkxEmpId,
                RCESESSIONID: result.RCESESSIONID,
                staffId: staff.id,
                phone: staff.mobile,
                password: utils.compileStr(params.password),
                timestamp: (new Date()).getTime(),
                isRememberMe: params.isRememberMe
            });
            callback(errorCode, result);
        }
        if (result.zkxToken == null) {
            User.refreshZkxToken(staff.mobile, finishLogin);
        } else {
            finishLogin();
        }
    }

    function requestLogin(params, isSecurity, callback) {
        callback = callback || $.noop;
        var loginType = RongIM.config.modules.zkxLoginType;
        console.log("loginType-----", loginType);
        var zkxextra = {
            validataCode: params.captcha,
            loginType: loginType
        };
        var extra = JSON.stringify(zkxextra);
        // var publicKey = 'MIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQDXfDruD0oJZDThj5VNiEaVPi8HorV6p5nxmngqsKQCgt0Kvh1LG/2ebJTfC9JB/GVL0bQ6ukCpMekKifL54QB+HdzdfuNOj9j8+bnD1kaBRadMnXR0GTmXz3QNl3CeAOjHZyMDFlnk5zx03ptg1RdC/mMRGxUcls5G8meRzT637QIDAQAB'
        // var publicKey = localStorage.getItem('publicKey')
        var publicKey = RongIM.serverConfig.security.password_public_key_rsa
        var encryptor = new JSEncrypt()
        encryptor.setPublicKey(publicKey)
        var data = {
            // username: params.phone || params.ldapAccount,
            username: encryptor.encrypt(params.phone),
            // password: params.password || params.ldapPassword,
            password: encryptor.encrypt(params.password),
            agent: params.agent,
            extra: encryptor.encrypt(extra),
            status: params.status,
            isRememberMe: params.isRememberMe,
            validate_code: params.verificationCode,
            verifyId: params.verifyId
        };
        var url = '/user/login';
        // if (isSecurity) {
        //     // 加密算法的密码
        //     data.password = common.encryptPassword(data.password);
        //     url = '/user/security_login';
        // } else {
        //     url = '/user/login';
        // }

        var deviceInfo = getDeviceInfo();
        data.device_info = deviceInfo;
        // 设备信息
        Http.post(url, data, function (errorCode, result) {
            afterLogin(params, errorCode, result, callback);
        });
    }

    User.securityLogin = function (params, callback) {
        requestLogin(params, true, callback);
    };

    User.login = function (params, callback) {
        requestLogin(params, false, callback);
    };

    // 自动登录
    User.autoLogin = function (params, callback) {
        callback = callback || $.noop;
        var data = {
            username: params.phone,
            password: params.password,
            agent: params.agent,
            status: params.status,
            isRememberMe: params.isRememberMe
        };
        var deviceInfo = getDeviceInfo();
        data.device_info = deviceInfo;
        // 设备信息
        var url = '/user/login_refresh';
        Http.post(url, data, function (errorCode, result) {
            afterLogin(params, errorCode, result, callback);
        });
    };

    // 获取登录时传给后台的设备信息
    function getDeviceInfo() {
        return {
            app_version: config.product.version,
            app_language: getLanguage(),
            timezone: getTimeZone(),
            network: 'WIFI'
        };
    }

    function getLanguage() {
        var localeList = {
            zh: 'zh-CN',
            en: 'en-US'
        };
        var locale = store.get('locale');
        return localeList[locale];
    }

    function getTimeZone() {
        var timeNum = (new Date().getTimezoneOffset() / 60);
        var timeStr = 'UTC-';
        if (timeNum < 0) {
            timeStr = 'UTC+';
        }
        timeStr += timeNum > 9 ? timeNum : '0' + Math.abs(timeNum);
        timeStr += ':00';
        return timeStr;
    }

    function resetDeviceState() {
        var url = '/usetting/multiclient/pc_lock_screen';
        Http.put(url, { value: false }).done(function () { });
    }

    User.logout = function () {
        Cache.clean();
        User.setStatus('offline');
        fileApi.downloadManage.abortAll();
        fileApi.uploadManage.abortAll();
        Http.post('/user/logout');
        RongIM.system.appLogger('info', '退出登录');
    };

    User.refreshToken = function (callback) {
        Http.post('/user/refresh_token', callback);
    };

    User.refreshZkxToken = function (loginId, callback) {
        // var publicKey = 'MIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQDXfDruD0oJZDThj5VNiEaVPi8HorV6p5nxmngqsKQCgt0Kvh1LG/2ebJTfC9JB/GVL0bQ6ukCpMekKifL54QB+HdzdfuNOj9j8+bnD1kaBRadMnXR0GTmXz3QNl3CeAOjHZyMDFlnk5zx03ptg1RdC/mMRGxUcls5G8meRzT637QIDAQAB'
        // var publicKey = localStorage.getItem('publicKey')
        var publicKey = RongIM.serverConfig.security.password_public_key_rsa
        var encryptor = new JSEncrypt()
        encryptor.setPublicKey(publicKey)
        // var encryptedData = encryptor.encrypt('plaintext')
        Http.post('/user/refresh_ZKXToken', {
            username: encryptor.encrypt(loginId),
            logintype: encryptor.encrypt('5')
        }, callback);
    };

    User.getCasTicket = function (loginId, appService, callback) {
        Http.get('/user/getCasTicket', {
            appService: appService,
            userId: loginId
        }, callback);
    };

    User.securityChangePassword = function (params, callback) {
        var data = {
            old_password: common.encryptPassword(params.oldPassword),
            new_password: common.encryptPassword(params.newPassword)
        };
        Http.post('/user/security_change_password', data, callback);
    };

    User.changePassword = function (params, callback) {
        var data = {
            old_password: params.oldPassword,
            new_password: params.newPassword
        };
        Http.post('/user/change_password', data, callback);
    };

    User.sendCode = function (type, params, callback) {
        // type: 'resetpwd' or 'register'
        var url = utils.templateFormat('/user/{{0}}/send_code/{{1}}', type, params.phone);
        Http.post(url, callback);
    };
    User.zkxsendCode = function (type, params, callback) {
        // type: 'resetpwd' or 'register'
        var url = utils.templateFormat('/user/{{0}}/{{1}}', type, params.phone);
        Http.post(url, callback);
    };

    User.checkCode = function (params, callback) {
        var url = utils.templateFormat('/user/verify_code/{{0}}/{{1}}', params.phone, params.code);
        Http.post(url, callback);
    };

    User.securityRegister = function (params, callback) {
        var data = {
            name: params.name,
            zip: params.zip,
            tel: params.tel,
            verify_token: params.verifyToken,
            password: common.encryptPassword(params.password)
        };
        var url = '/user/security_register';
        Http.post(url, data, callback);
    };

    // http://gitlab.rongcloud.net/RCE/RCE-Doc/blob/master/docs/design/subsystem/contact_service.md
    User.register = function (params, callback) {
        var data = {
            name: params.name,
            zip: params.zip,
            tel: params.tel,
            verify_token: params.verifyToken,
            password: params.password
        };
        var url = '/user/register';
        Http.post(url, data, callback);
    };

    User.securityResetPassword = function (params, callback) {
        var data = {
            user_name: params.phone,
            new_password: common.encryptPassword(params.password),
            verify_token: params.verifyToken
        };
        Http.post('/user/security_reset_password', data, callback);
    };

    User.resetPassword = function (params, callback) {
        var loginType = RongIM.config.modules.zkxLoginType;
        var zkxextra = {
            validataCode: params.captcha,
            loginType: loginType
        };
        var extra = JSON.stringify(zkxextra);
        var data = {
            extra: extra,
            user_name: params.phone,
            new_password: params.password,
            verify_token: params.verifyToken
        };
        Http.post('/user/reset_password', data, callback);
    };

    User.setAlias = function (targetId, alias, callback) {
        var data = {
            alias: alias
        };
        //  ;
        Http.put('/userrelation/alias/' + targetId, data).done(function (result) {
            Cache.alias[targetId] = alias;
            User.get(targetId, function (errorCode, user) {
                if (errorCode) {
                    return;
                }
                userObserverList.notify(user);
                callback(null, result);
            });
            var friend = Cache.friendList.filter(function (item) {
                return item.id === targetId;
            });
            if (friend.length !== 0) {
                if (friendApi) friendApi.notifyFriend();
            }
            conversationApi.getList(function (errorCode, list) {
                if (errorCode) {
                    return;
                }
                conversationApi.observerList.notify(list);
            });
        }).fail(callback);
    };

    User.setAvatar = function (src, big_src, callback) {
        var data = {
            portrait_url: src,
            portrait_big_url: big_src
        };
        Http.put('/user/portrait', data).done(function () {
            callback();
            var auth = RongIM.instance.auth;
            if (auth) {
                var user = Cache.user[auth.id];
                userObserverList.notify(user);
            }
        }).fail(callback);
    };

    User.getBatch = function (idsList, callback) {
        callback = callback || $.noop;
        getUsers(idsList, function (errorCode, list) {
            if (errorCode) {
                callback(errorCode);
                return;
            }
            callback(null, list);
        });
    };

    User.get = function (id, callback) {
        callback = callback || $.noop;
        var idList = [];
        idList = idList.concat(id);
        getUsers(idList, function (errorCode, list) {
            if (errorCode) {
                callback(errorCode);
                return;
            }
            var result = list;
            if (!$.isArray(id)) {
                result = list[0];
            }
            callback(null, result);
        });
    };

    User.searchByMobile = function (keyword, callback) {
        callback = callback || $.noop;
        var deferred = $.Deferred();
        Http.post('/staffs/search/mobile', {
            keywords: [keyword],
            partial_match: true
        }).done(function (result) {
            result.forEach(function (item) {
                item.userType = item.user_type;
                item.isExecutive = item.is_executive;
                item.avatar = item.portrait_url;
            });
            callback(null, result);
            deferred.resolve(result);
        }).fail(function (errorCode) {
            callback(errorCode);
            deferred.reject(errorCode);
        });
        return deferred.promise();
    };

    User.searchByEmail = function (keyword, callback) {
        callback = callback || $.noop;
        var deferred = $.Deferred();
        Http.post('/staffs/search/email', {
            keywords: [keyword]
        }).done(function (result) {
            result.forEach(function (item) {
                item.userType = item.user_type;
                item.isExecutive = item.is_executive;
                item.avatar = item.portrait_url;
            });
            callback(null, result);
            deferred.resolve(result);
        }).fail(function (errorCode) {
            callback(errorCode);
            deferred.reject(errorCode);
        });
        return deferred.promise();
    };

    User.searchByDuty = function (duty_name, callback) {
        var url = '/staffs?duty_name=' + encodeURIComponent(duty_name);
        callback = callback || $.noop;
        return Http.get(url).done(function (result) {
            if (result.data && result.data.length > 0) {
                result.data.forEach(function (item) {
                    item.userType = item.user_type;
                    item.isExecutive = item.is_executive;
                    item.avatar = item.portrait_url;
                });
            }
            callback(null, result);
        }).fail(callback);
    };

    User.updateMajorCompany = function (companyId, callback) {
        var auth = RongIM.instance.auth;
        if (utils.isEmpty(auth)) {
            return callback('nologin');
        }
        var authId = auth.id;
        return Http.put('/staffs/' + authId, {
            company_id: companyId
        }).done(function (result) {
            var user = Cache.user[authId];
            if (user) {
                user.companyId = companyId;
                user.detail = null;
            }
            callback(null, result);
        }).fail(callback);
    };
    //给个人设置通讯组
    User.changePersonalGroup = function (param, callback) {
        Http.put('/staffs/changePersonalGroup', param, function (errorCode, result) {
            callback(errorCode, result);
        }).fail(callback);
    }
    User.getUsers = getUsers;

    function getUsers(idList, callback) {
        /**
         * 实现过程
         * 1. 从`Cache.user` 匹配, 未找到的从服务器批量获取。
         * 2. 从`Cache.alias`里补充字段`alias`
         * 3. 从`Cache.orgTree`里补充字段`path`（所在部门的路径：由`deptId`构成，用逗号分隔）
         * 4. 从`Cache.starList`里补充字段`star`
         * 5. 从`Cache.friendList`里补充字段`isFriend`
         */

        callback = callback || $.noop;

        idList = idList || [];
        var invalidIdList = [];

        // 排除重复 id 检查传入参数 为 null、 undefined、空字符串输出错误信息
        idList.forEach(function (id) {
            var unexist = invalidIdList.indexOf(id) === -1;
            if (!utils.isEmpty(id) && unexist) {
                invalidIdList.push(id);
            }
        });

        if (invalidIdList.length > 0) {
            // utils.console.error('getUsers parameter contains invalid userId', idList);
        }

        var len = idList.length;

        var deferred = $.Deferred();

        if (len === 0) {
            callback(null, []);
            deferred.resolve([]);
            return deferred.promise();
        }

        var getExtDuties = function (strVal) {
            if (strVal) {
                var duties = {};
                strVal.split(',').forEach(function (dutyItem) {
                    var dutyPair = dutyItem.trim().split(':');
                    if (dutyPair.length > 1) {
                        duties[dutyPair[0].trim()] = dutyPair[1].trim();
                    }
                });
                return duties;
            }
            return null;
        }

        var fetchUsers = function (idArr) {
            var userMap = {};
            var userPromiseList = [];
            var notCacheDeferList = [];

            for (var j = idArr.length - 1; j >= 0; j -= 1) {
                var userId = idArr[j];
                var user = Cache.user[userId];
                if (friendApi.isFileHelper(userId)) {
                    user = friendApi.getFileHelper();
                }
                var userPromise = Cache.user._defer[userId];
                if (!$.isEmptyObject(user)) {
                    user = compositeUser(user);
                    userMap[user.id] = user;
                } else if (!$.isEmptyObject(userPromise)) {
                    userPromiseList.push({
                        id: userId,
                        promise: userPromise
                    });
                } else {
                    var defer = $.Deferred();
                    notCacheDeferList.push({
                        id: userId,
                        defer: defer
                    });
                    var promise = defer.promise();
                    Cache.user._defer[userId] = promise;
                    userPromiseList.push({
                        id: userId,
                        promise: promise
                    });
                }
            }
            if (notCacheDeferList.length !== 0) {
                var notCacheIdList = notCacheDeferList.map(function (item) {
                    return item.id;
                });
                var data = {
                    ids: notCacheIdList
                };
                Http.post('/staffs/batch?withOrgs=1', data).done(function (result) {
                    result.data.forEach(function (item) {
                        var userInfo = {
                            id: item.id,
                            name: item.name,
                            avatar: item.portrait_url,
                            deptId: item.depart_id,
                            dutyName: item.duty_name,
                            extDuties: getExtDuties(item.vchar_p3),
                            state: item.state,
                            isExecutive: item.is_executive,
                            type: item.user_type,
                            companyId: item.company_id,
                            orgsInfo: item.orgs_info,
                            portrait_big_url: item.portrait_big_url
                        };
                        Cache.user[item.id] = userInfo;
                    });
                    for (var i = notCacheDeferList.length - 1; i >= 0; i -= 1) {
                        var deferObj = notCacheDeferList[i];
                        var userCache = Cache.user[deferObj.id];
                        if (!userCache) {
                            utils.console.warn('getUsers 用户信息未获取到：', deferObj.id);
                            userCache = {};
                            // Cache.user[deferObj.id] = user;
                        }
                        deferObj.defer.resolve(userCache);
                    }
                }).fail(function (errorCode) {
                    for (var i = notCacheDeferList.length - 1; i >= 0; i -= 1) {
                        var deferObj = notCacheDeferList[i];
                        deferObj.defer.reject(errorCode);
                    }
                });
            }

            if (userPromiseList.length === 0) {
                var userList = getUserList(idArr, userMap);
                callback(null, userList);
                deferred.resolve(userList);
                return;
            }

            var promiseList = userPromiseList.map(function (item) {
                return item.promise;
            });
            $.when.apply(null, promiseList).done(function () {
                for (var i = arguments.length - 1; i >= 0; i -= 1) {
                    var userInfo = arguments[i];
                    if (!userInfo) {
                        utils.console.log(userPromiseList);
                    }
                    userInfo = compositeUser(userInfo);
                    userMap[userInfo.id] = userInfo;
                }
                var userArr = getUserList(idArr, userMap);
                callback(null, userArr);
                deferred.resolve(userArr);
            }).fail(function (errorCode) {
                deferred.reject(errorCode);
                callback(errorCode);
            }).always(function () {
                userPromiseList.forEach(function (item) {
                    delete Cache.user._defer[item.id];
                });
            });
        };

        fetchUsers(invalidIdList);

        return deferred.promise();
    }

    User.getDetail = function (id, callback) {
        callback = callback || $.noop;
        Http.get('/staffs/' + id).done(function (result) {
            var cacheuser = Cache.user[id];
            if (cacheuser) {
                cacheuser.isExecutive = result.isExecutive;
            }
            var userInfo = {
                id: result.id,
                name: result.name,
                avatar: result.portrait_url,
                portrait_big_url: result.portrait_big_url,
                deptId: result.depart_id,
                state: result.state,
                companyId: result.company_id,
                isExecutive: result.is_executive,
                type: result.user_type,
                staff_no: result.staff_no,
                orgsInfo: result.orgs_info
            };
            if (cacheuser) {
                $.extend(cacheuser, userInfo);
            }
            getUsers([id], function (errorCode, list) {
                if (errorCode) {
                    callback(errorCode);
                    return;
                }
                var user = list[0];
                // 删除用户详情 防止循环引用
                user.detail = null;
                if (user.userType === common.UserType.VISITOR) {
                    callback(null, user);
                    return;
                }
                var detail = $.extend(result, user);
                if (cacheuser) {
                    cacheuser.detail = detail;
                }
                callback(null, detail);
            });
        }).fail(callback);
    };

    User.getVisitors = function (id, callback) {
        callback = callback || $.noop;
        Http.get('/user/' + id).done(function (result) {
            var cacheuser = Cache.user[id];
            if (cacheuser) {
                cacheuser.isExecutive = result.isExecutive;
            }
            var userInfo = {
                id: result.id,
                name: result.name,
                avatar: result.portrait_url,
                portrait_big_url: result.portrait_big_url,
                deptId: result.depart_id,
                state: result.state,
                companyId: result.company_id,
                isExecutive: result.is_executive,
                type: result.user_type,
                staff_no: result.staff_no,
                orgsInfo: result.orgs_info
            };
            if (cacheuser) {
                $.extend(cacheuser, userInfo);
            }
            getUsers([id], function (errorCode, list) {
                if (errorCode) {
                    callback(errorCode);
                    return;
                }
                var user = list[0];
                // 删除用户详情 防止循环引用
                user.detail = null;
                if (user.userType === common.UserType.VISITOR) {
                    callback(null, user);
                    return;
                }
                var detail = $.extend(result, user);
                if (cacheuser) {
                    cacheuser.detail = detail;
                }
                callback(null, detail);
            });
        }).fail(callback);
    };
    User.getAvatarToken = function (callback) {
        Http.get('/user/get_image_token', callback);
    };
    // 查询个人所在的通讯组
    User.getPersonalGroup = function (id, callback) {
        Http.get('/staffs/getPersonalGroup/' + id)
            .then(function (result) {
                return callback(null, result);
            }).fail(callback);
    };
    // 获取当前登录人是否是学会管理员
    User.getXhManager = function (callback) {
        Http.get('/staffs/getXhManager')
            .then(function (result) {
                return callback(null, result);
            }).fail(callback);
    };
    var subscribeTitle = ['Login_Status_PC', 'Login_Status_Mobile', 'Login_Status_Web'];

    /**
     * @params.userIds
     * @params.duration
     */
    User.subscribe = function (userId, duration, callback) {
        callback = callback || $.noop;
        Http.post('/presence/subscribe', {
            type: 0,
            target_ids: [userId],
            titles: subscribeTitle,
            duration: duration,
            fetch_data: true
        }).done(function (result) {
            callback(null, result);
            User.get(userId, function (errorCode, user) {
                if (errorCode) {
                    callback(errorCode);
                    return;
                }
                user.onlineStatus = {};
                result.datas.forEach(function (item) {
                    user.onlineStatus[item.title] = item;
                });
                userObserverList.notify(user);
            });
        }).fail(callback);
    };

    // params.userIds
    User.unsubscribe = function (userId, callback) {
        callback = callback || $.noop;
        Http.post('/presence/unsubscribe', {
            type: 0,
            target_ids: [userId],
            titles: subscribeTitle
        }).done(function (result) {
            callback(null, result);
        }).fail(callback);
    };

    User.setStatus = function (status, callback) {
        callback = callback || $.noop;
        var title = RongIM.client.userStatusTitle;
        Http.put('/presence/publish', {
            title: title,
            value: status,
            persist: true
        }).done(function (result) {
            callback(null, result);
        }).fail(callback);
    };

    User.getAlias = function () {
        return Cache.alias;
    };

    User.watch = function (listener) {
        userObserverList.add(listener);
    };

    User.unwatch = function (listener) {
        userObserverList.remove(listener);
    };

    User.executiveLimit = function (user) {
        var isFriend = Cache.friendList.some(function (friend) {
            return friend.id === user.id;
        });
        if (isFriend || RongIM.instance.auth.isExecutive) {
            return false;
        }
        var isExecutive = !!user.isExecutive;
        return isExecutive;
    };

    function compositeUser(user) {
        user.alias = Cache.alias[user.id];
        user.star = Cache.starList.indexOf(user.id) !== -1;
        var friend = friendApi.getCacheFriend(user.id);
        user.isFriend = !!friend;
        user.bothFriend = (friend || {}).bothFriend;
        return user;
    }

    function getUserList(idList, userMap) {
        return idList.map(function (id) {
            return userMap[id];
        });
    }

    User.qrcodeLogin = function login(ele, callback) {
        callback = callback || $.noop;
        var timeout = false;
        if (login.timer) clearTimeout(login.timer);

        getQRCodeToken()
            .then(render)
            .then(polling)
            .then(qrcodeLogin)
            .then(function (result) {
                var staff = result.staff;
                User.refreshZkxToken(staff.mobile, function (err, res) {
                    store.set('loginInfo', {
                        zkxToken: (res && res.st) ? res.st : result.zkxToken,
                        zkxEmpId: result.zkxEmpId,
                        RCESESSIONID: result.RCESESSIONID,
                        staffId: staff.id,
                        phone: staff.mobile,
                        timestamp: (new Date()).getTime()
                    });
                    callback(null, result);
                });
            })
            .fail(callback);

        function render(result) {
            login.timer = setTimeout(function () {
                timeout = true;
            }, result.timeout);

            var token = result.token;
            qrcodeRender(ele, token);
            return token;
        }

        function polling(token) {
            return qrcodePolling(ele, token).then(function (result) {
                return {
                    token: token,
                    ticket: result.ticket
                };
            });
        }

        function qrcodePolling(el, token) {
            var url = utils.templateFormat('/user/qrcode/login/polling/{{0}}', token);
            // var POLLING = 0;
            // var VERIFIED = 1;
            var LOGINED = 2;
            var wait = 1000;
            var defer = $.Deferred();

            function loop(node) {
                Http.get(url).then(function (result) {
                    if (result.state === LOGINED) {
                        defer.resolve(result);
                        return;
                    }
                    var isInPage = document.body.contains(node);
                    if (timeout) {
                        defer.reject(common.ErrorCode.INVALID_TOKEN);
                    } else if (isInPage) {
                        login.timer = setTimeout(function () {
                            loop(node);
                        }, wait);
                    }
                }).fail(function (errorCode) {
                    if (errorCode === common.ErrorCode.QR_INVALID_TOKEN) {
                        login(node, callback);
                        return;
                    }
                    defer.reject(errorCode);
                });
            }

            loop(el);
            return defer.promise();
        }
    };

    function getQRCodeToken() {
        var params = {
            agent: {
                platform: utils.getPlatform(),
                device_id: common.getDeviceId()
            }
        };
        return Http.post('/user/qrcode/login/create', params);
    }

    function qrcodeRender(node, token) {
        if (!node || !token) return;
        var platform = utils.getPlatform();
        var text = utils.templateFormat('RCE_LOGIN@{{0}}@{{1}}', token, platform);
        $(node).empty();
        // eslint-disable-next-line no-new
        new QRCode(node, {
            text: text,
            width: 162,
            height: 162
        });
    }

    function qrcodeLogin(params) {
        // 携带设备信息
        var deviceInfo = getDeviceInfo();
        params.device_info = deviceInfo;
        return Http.post('/user/qrcode/login', params)
            .then(function (result) {
                var staff = result.staff;
                staff.deptId = staff.departId || staff.depart_id || staff.dept_id;
                staff.deptName = staff.departName || staff.depart_name || staff.dept_name;
                staff.companyId = staff.companyId || staff.company_id;
                staff.orgsInfo = staff.orgsInfo || staff.orgs_info;
                Cache.auth = {
                    token: result.token,
                    id: staff.id,
                    deptId: staff.deptId,
                    companyId: staff.companyId,
                    orgsInfo: staff.orgsInfo
                };
                resetDeviceState();
                return result;
            });
    }

    User.getNewestAlias = function () {
        //  ;
        return request('/userrelation/alias', 'GET').then(function (result) {
            var alias = {};
            result.data.forEach(function (user) {
                alias[user.id] = user.alias;
            });
            return alias;
        });
    };

    User.batchFromServer = function (idList, callback) {
        request('/staffs/batch', 'post', {
            ids: idList
        }, function (error, result) {
            callback(error, result);
        }, true);
    };

    User.getNewUser = function (id, callback) {
        if (utils.isEmpty(id)) {
            callback('params error: id invalid');
            return;
        }
        delete Cache.user[id];
        getUsers([id], function (errorCode, userList) {
            if (errorCode) {
                callback(errorCode, userList);
                return;
            }
            var user = (userList || [])[0];
            callback(null, user);
        });
    };

    User.validateCanChat = function (user) {
        if (utils.isEmpty(user)) {
            return false;
        }
        var fileHelperId = user.id;
        if (typeof user === 'string') {
            fileHelperId = user;
            user = Cache.user[user] || { bothFriend: false };
        }
        if (friendApi.isFileHelper(fileHelperId)) {
            return true;
        }
        var auth = RongIM.instance.loginUser || {};
        var isVisitor = user.type === common.UserType.VISITOR;
        var selfIsVisitor = auth.type === common.UserType.VISITOR;
        if (isVisitor || selfIsVisitor) {
            return user.bothFriend;
        }
        return true;
    };
    // 修改外部用户姓名
    User.setUsername = function (targetId, name, callback) {
        var data = {
            name: name
        };

        Object.keys(Cache.user).forEach(function (key) {
            if (key === targetId) {
                // 更新Cache中的name
                Cache.user[key].name = name;
            }
        });
        callback();
        // TODO 根据后端接口传参
        var url = '/visitors/' + targetId + '/name';

        Http.put(url, data).done(function (result) {
            Cache.name[targetId] = name;
            User.get(targetId, function (errorCode, user) {
                if (errorCode) {
                    return;
                }
                userObserverList.notify(user);
                callback(null, result);
            });
            var friend = Cache.friendList.filter(function (item) {
                return item.id === targetId;
            });
            if (friend.length !== 0 && friendApi) {
                friendApi.notifyFriend();
            }
            conversationApi.getList(function (errorCode, list) {
                if (errorCode) {
                    return;
                }
                conversationApi.observerList.notify(list);
            });
        }).fail(callback);
    };

    // 伯鑫star
    // 绑定邮箱
    User.bindEmail = function (data, callback) {
        Http.post('/user/change_email_two', data, function (errorCode, result) {
            if (errorCode && errorCode != 10000) {
                common.messageToast({
                    message: '绑定邮箱失败',
                    type: 'error'
                });
            };
            if (callback) callback(errorCode, result);
        })
    };

    // 重新发送验证邮箱
    User.resetSendEmailCode = function (data, callback) {
        Http.post('/user/again_send_email', data, function (errorCode, result) {
            if (errorCode && errorCode != 10000) {
                common.messageToast({
                    message: '验证码发送失败',
                    type: 'error'
                });
            };
            if (!errorCode) {
                common.messageToast({
                    message: '验证码发送成功',
                    type: 'success'
                });
            }
            if (callback) callback(errorCode, result);
        })
    };

    // 解绑邮箱
    User.unBindEmail = function (data, callback) {
        Http.del('/user/relieve_email_bind', data, function (errorCode, result) {
            console.log(errorCode, result);
            if (errorCode && errorCode != 10000) {
                common.messageToast({
                    message: 'oa 修改异常, 请联系相关人员！',
                    type: 'error'
                });
            } else if (errorCode && errorCode != 10000) {
                common.messageToast({
                    message: '解绑邮箱失败',
                    type: 'error'
                });
            }
            callback(errorCode, result);
        })
    }
    // 伯鑫 end

    // 高英豪 star
    // 姚肖洋star
    // 给待修改的手机号发送验证码
    User.changePhone_send_code = function (new_phone, uid, callback) {
        Http.post(`/user/changePhone/send_code/${new_phone}/${uid}`, {}, function (errorCode, result) {
            if (errorCode && errorCode != 10000) {
                if (errorCode == 10161) {
                    common.messagebox({
                        message: "该手机号已被其他账号绑定，如仍需绑定请联系管理员，电话010-68587825 内线 38681",
                        type: 'alert'
                    });
                } else if (errorCode == 10160) {
                    common.messagebox({
                        message: "请输入新的手机号",
                        type: 'alert'
                    });
                } else {
                    common.messageToast({
                        message: '获取验证码失败',
                        type: 'error'
                    });
                }
            }
            if (callback) callback(errorCode, result);
        })
    };
    // 用户修改手机号
    User.change_phone = function (uid, new_phone, sms_code, callback) {
        var params = { uid, new_phone, sms_code }
        Http.post(`/user/change_phone`, params, function (errorCode, result) {
            if (errorCode && errorCode != 10000) {
                common.messageToast({
                    message: '验证码错误',
                    type: 'error'
                });
            };
            if (callback) callback(errorCode, result);
        })
    };
    // 高英豪 end

    RongIM.dataModel.User = User;
}(RongIM, {
    jQuery: jQuery,
    QRCode: window.QRCode
}));

