(function (RongIM, dependencies, components) {
    'use strict';

    var common = RongIM.common;
    var $ = dependencies.jQuery;
    var dataModel = RongIM.dataModel;
    var pinApi = dataModel.Pin;
    var im;

    RongIM.dialog.addReceivers = function (defaultMembers, pinDetail) {
        im = RongIM.instance;
        var maxReceiverCount = im.serverConfig.pin.max_receiver_count;
        var options = {
            name: 'add-receiver',
            template: 'modules/pin/templates/add-receiver.html',
            data: function () {
                return {
                    maxReceiverCount: maxReceiverCount,
                    tab: 'org',
                    show: true,
                    selected: $.extend(true, [], defaultMembers),
                    defaultSelected: $.extend(true, [], defaultMembers),
                    isStaff: im.auth.isStaff
                };
            },
            components: {
                avatar: components.getAvatar,
                org: components.group.getOrg,
                star: components.group.getStar,
                friend: components.group.getFriend
            },
            computed: {
                isStarSelected: function () {
                    return this.tab === 'star';
                },
                isOrgSelected: function () {
                    return this.tab === 'org';
                },
                isFriendSelected: function () {
                    return this.tab === 'friend';
                },
                canNotSelected: function () {
                    var loginUser = RongIM.instance.loginUser || { id: '' };
                    return [loginUser];
                },
                defaultCount: function () {
                    return defaultMembers.length;
                }
            },
            created: function () {
                if (!im.auth.isStaff) {
                    this.tab = 'friend';
                }
            },
            methods: getMethods(defaultMembers, pinDetail)
        };
        common.mountDialog(options);
    };

    function getMethods(defaultMembers, pinDetail) {
        return {
            reset: function () {
                this.selected.push({});
                this.selected.pop();
            },
            toastError: function (errorCode) {
                common.toastError(errorCode, this.$el.firstChild);
            },
            toast: function (params) {
                params.el = this.$el.firstChild;
                common.messageToast(params);
            },
            getUsername: common.getUsername,
            close: function () {
                this.selected = [];
                this.show = false;
            },
            selectTab: function (tab) {
                this.tab = tab;
            },
            added: function (members) {
                added(this, members, defaultMembers);
            },
            removed: function (members) {
                removed(this, members);
            },
            removeMembers: function (member) {
                removed(this, [member]);
            },
            addReceivers: function () {
                addReceivers(this, pinDetail, defaultMembers);
            },
            showRemoveBtn: function (item) {
                var sameUserList = defaultMembers.filter(function (user) {
                    return user.id === item.id;
                });
                return sameUserList.length === 0;
            },
            maxCountLimit: function () {
                var mostReceiveFormat = im.locale.components.newPin.mostReceive;
                var hintMessage = this.localeFormat(mostReceiveFormat, this.maxReceiverCount);
                this.toast({
                    type: 'error',
                    message: hintMessage
                });
                this.reset();
            }
        };
    }

    function addReceivers(context, pinContext, defaultMembers) {
        var defaultIdList = defaultMembers.map(function (item) {
            return item.id;
        });
        var newReciverIds = context.selected.map(function (item) {
            return item.id;
        });
        newReciverIds = newReciverIds.filter(function (id) {
            return defaultIdList.indexOf(id) === -1;
        });
        pinApi.addReceivers(pinContext.pinUid, newReciverIds, function (errorCode) {
            if (errorCode) {
                context.toastError(errorCode);
                return;
            }
            sendNewReceiverMessage(pinContext.pinUid);
            context.close();
        });
    }

    function sendNewReceiverMessage(uid) {
        var message = {
            messageType: pinApi.MessageType.PinNewReciverMessage,
            content: { pinUid: uid }
        };
        pinApi.observerList.notify(message);
    }

    function added(context, members) {
        var instance = RongIM.instance;
        var selectedIdList = context.selected.map(function (item) {
            return item.id;
        });
        var addedList = members.filter(function (item) {
            var hasSelected = selectedIdList.indexOf(item.id) < 0;
            var notSelf = item.id !== instance.loginUser.id;
            return hasSelected && notSelf;
        });
        var totalList = context.selected.concat(addedList);
        context.selected = totalList;
        // PIN 最大人数由服务端下发
        var maxReceiverCount = context.maxReceiverCount;
        var mostReceiveFormat = instance.locale.components.newPin.mostReceive;
        var hintMessage = context.localeFormat(mostReceiveFormat, maxReceiverCount);
        if (totalList.length > maxReceiverCount) {
            context.toast({
                type: 'error',
                message: hintMessage
            });
            removed(context, addedList);
        }
    }

    function removed(context, members) {
        var idList = members.map(function (item) {
            return item.id;
        });
        var reservedIdList = context.defaultSelected.map(function (item) {
            return item.id;
        });
        context.selected = context.selected.filter(function (item) {
            var reserved = reservedIdList.indexOf(item.id) >= 0;
            return reserved || idList.indexOf(item.id) < 0;
        });
    }
}(RongIM, {
    jQuery: jQuery
}, RongIM.components));
