import { Scope } from '@sentry/hub/esm';
import { Breadcrumb, Integration, IntegrationClass, SdkInfo, SentryBreadcrumbHint, SentryEvent, SentryEventHint, SentryResponse, Severity } from '@sentry/types/esm';
import { BackendClass } from './basebackend';
import { Dsn } from './dsn';
import { IntegrationIndex } from './integration';
import { Backend, Client, Options } from './interfaces';
import { PromiseBuffer } from './promisebuffer';
/**
 * Base implementation for all JavaScript SDK clients.
 *
 * Call the constructor with the corresponding backend constructor and options
 * specific to the client subclass. To access these options later, use
 * {@link Client.getOptions}. Also, the Backend instance is available via
 * {@link Client.getBackend}.
 *
 * If a Dsn is specified in the options, it will be parsed and stored. Use
 * {@link Client.getDsn} to retrieve the Dsn at any moment. In case the Dsn is
 * invalid, the constructor will throw a {@link SentryException}. Note that
 * without a valid Dsn, the SDK will not send any events to Sentry.
 *
 * Before sending an event via the backend, it is passed through
 * {@link BaseClient.prepareEvent} to add SDK information and scope data
 * (breadcrumbs and context). To add more custom information, override this
 * method and extend the resulting prepared event.
 *
 * To issue automatically created events (e.g. via instrumentation), use
 * {@link Client.captureEvent}. It will prepare the event and pass it through
 * the callback lifecycle. To issue auto-breadcrumbs, use
 * {@link Client.addBreadcrumb}.
 *
 * @example
 * class NodeClient extends BaseClient<NodeBackend, NodeOptions> {
 *   public constructor(options: NodeOptions) {
 *     super(NodeBackend, options);
 *   }
 *
 *   // ...
 * }
 */
export declare abstract class BaseClient<B extends Backend, O extends Options> implements Client<O> {
    /**
     * The backend used to physically interact in the enviornment. Usually, this
     * will correspond to the client. When composing SDKs, however, the Backend
     * from the root SDK will be used.
     */
    private readonly backend;
    /** Options passed to the SDK. */
    private readonly options;
    /**
     * The client Dsn, if specified in options. Without this Dsn, the SDK will be
     * disabled.
     */
    private readonly dsn?;
    /**
     * Stores whether installation has been performed and was successful. Before
     * installing, this is undefined. Then it contains the success state.
     */
    private installed?;
    /** Array of used integrations. */
    private readonly integrations;
    /** A simple buffer holding all requests. */
    protected readonly buffer: PromiseBuffer<SentryResponse>;
    /**
     * Initializes this client instance.
     *
     * @param backendClass A constructor function to create the backend.
     * @param options Options for the client.
     */
    protected constructor(backendClass: BackendClass<B, O>, options: O);
    /**
     * @inheritDoc
     */
    install(): boolean;
    /**
     * @inheritDoc
     */
    captureException(exception: any, hint?: SentryEventHint, scope?: Scope): Promise<SentryResponse>;
    /**
     * @inheritDoc
     */
    captureMessage(message: string, level?: Severity, hint?: SentryEventHint, scope?: Scope): Promise<SentryResponse>;
    /**
     * @inheritDoc
     */
    captureEvent(event: SentryEvent, hint?: SentryEventHint, scope?: Scope): Promise<SentryResponse>;
    /**
     * @inheritDoc
     */
    addBreadcrumb(breadcrumb: Breadcrumb, hint?: SentryBreadcrumbHint, scope?: Scope): void;
    /**
     * @inheritDoc
     */
    getDsn(): Dsn | undefined;
    /**
     * @inheritDoc
     */
    getOptions(): O;
    /** Returns the current backend. */
    protected getBackend(): B;
    /** Determines whether this SDK is enabled and a valid Dsn is present. */
    protected isEnabled(): boolean;
    /**
     * Adds common information to events.
     *
     * The information includes release and environment from `options`,
     * breadcrumbs and context (extra, tags and user) from the scope.
     *
     * Information that is already present in the event is never overwritten. For
     * nested objects, such as the context, keys are merged.
     *
     * @param event The original event.
     * @param hint May contain additional informartion about the original exception.
     * @param scope A scope containing event metadata.
     * @returns A new event with more information.
     */
    protected prepareEvent(event: SentryEvent, scope?: Scope, hint?: SentryEventHint): Promise<SentryEvent | null>;
    /**
     * This function adds all used integrations to the SDK info in the event.
     * @param sdkInfo The sdkInfo of the event that will be filled with all integrations.
     */
    protected _addIntegrations(sdkInfo?: SdkInfo): void;
    /**
     * Processes an event (either error or message) and sends it to Sentry.
     *
     * This also adds breadcrumbs and context information to the event. However,
     * platform specific meta data (such as the User's IP address) must be added
     * by the SDK implementor.
     *
     * The returned event status offers clues to whether the event was sent to
     * Sentry and accepted there. If the {@link Options.shouldSend} hook returns
     * `false`, the status will be {@link SendStatus.Skipped}. If the rate limit
     * was exceeded, the status will be {@link SendStatus.RateLimit}.
     *
     * @param event The event to send to Sentry.
     * @param send A function to actually send the event.
     * @param scope A scope containing event metadata.
     * @param hint May contain additional informartion about the original exception.
     * @returns A Promise that resolves with the event status.
     */
    protected processEvent(event: SentryEvent, send: (finalEvent: SentryEvent) => Promise<SentryResponse>, hint?: SentryEventHint, scope?: Scope): Promise<SentryResponse>;
    /**
     * @inheritDoc
     */
    flush(timeout?: number): Promise<boolean>;
    /**
     * @inheritDoc
     */
    close(timeout?: number): Promise<boolean>;
    /**
     * @inheritDoc
     */
    getIntegrations(): IntegrationIndex;
    /**
     * @inheritDoc
     */
    getIntegration<T extends Integration>(integration: IntegrationClass<T>): T | null;
}
