"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : new P(function (resolve) { resolve(result.value); }).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const electron_1 = require("electron");
const path_1 = require("path");
const types_1 = require("@sentry/types");
const core_1 = require("@sentry/core");
const node_1 = require("@sentry/node");
const async_1 = require("@sentry/utils/async");
const store_1 = require("@sentry/utils/store");
const common_1 = require("../common");
const sdk_1 = require("../sdk");
const normalize_1 = require("./normalize");
const uploader_1 = require("./uploader");
/** Gets the path to the Sentry cache directory. */
function getCachePath() {
    return path_1.join(electron_1.app.getPath('userData'), 'sentry');
}
/**
 * Retruns a promise that resolves when app is ready.
 */
function isAppReady() {
    return __awaiter(this, void 0, void 0, function* () {
        return (electron_1.app.isReady() ||
            new Promise(resolve => {
                electron_1.app.once('ready', resolve);
            }));
    });
}
exports.isAppReady = isAppReady;
/** Backend implementation for Electron renderer backends. */
class MainBackend extends core_1.BaseBackend {
    /** Creates a new Electron backend instance. */
    constructor(options) {
        super(options);
        this.inner = new node_1.NodeBackend(options);
        this.scopeStore = new store_1.Store(getCachePath(), 'scope', new core_1.Scope());
    }
    /**
     * @inheritDoc
     */
    install() {
        let success = true;
        // We refill the scope here to not have an empty one
        core_1.configureScope(scope => {
            // tslint:disable:no-unsafe-any
            const loadedScope = core_1.Scope.clone(this.scopeStore.get());
            if (loadedScope.user) {
                scope.setUser(loadedScope.user);
            }
            if (loadedScope.tags) {
                Object.keys(loadedScope.tags).forEach(key => {
                    scope.setTag(key, loadedScope.tags[key]);
                });
            }
            if (loadedScope.extra) {
                Object.keys(loadedScope.extra).forEach(key => {
                    scope.setExtra(key, loadedScope.extra[key]);
                });
            }
            if (loadedScope.breadcrumbs) {
                loadedScope.breadcrumbs.forEach((crumb) => {
                    scope.addBreadcrumb(crumb);
                });
            }
            // tslint:enable:no-unsafe-any
        });
        if (this.isNativeEnabled()) {
            success = this.installNativeHandler() && success;
        }
        this.installIPC();
        return success;
    }
    /**
     * @inheritDoc
     */
    eventFromException(exception) {
        return __awaiter(this, void 0, void 0, function* () {
            return this.inner.eventFromException(exception);
        });
    }
    /**
     * @inheritDoc
     */
    eventFromMessage(message) {
        return __awaiter(this, void 0, void 0, function* () {
            return this.inner.eventFromMessage(message);
        });
    }
    /**
     * @inheritDoc
     */
    sendEvent(event) {
        return __awaiter(this, void 0, void 0, function* () {
            yield isAppReady();
            return this.inner.sendEvent(event);
        });
    }
    /**
     * Uploads the given minidump and attaches event information.
     *
     * @param path A relative or absolute path to the minidump file.
     * @param event Optional event information to add to the minidump request.
     * @returns A promise that resolves to the status code of the request.
     */
    uploadMinidump(path, event = {}) {
        return __awaiter(this, void 0, void 0, function* () {
            if (this.uploader) {
                return this.uploader.uploadMinidump({ path, event });
            }
            return { status: types_1.Status.Success };
        });
    }
    /**
     * @inheritDoc
     */
    storeScope(scope) {
        const cloned = core_1.Scope.clone(scope);
        cloned.eventProcessors = [];
        // tslint:disable-next-line:no-object-literal-type-assertion
        this.scopeStore.update((current) => (Object.assign({}, current, cloned)));
    }
    /** Returns whether native reports are enabled. */
    isNativeEnabled() {
        // Mac AppStore builds cannot run the crash reporter due to the sandboxing
        // requirements. In this case, we prevent enabling native crashes entirely.
        // https://electronjs.org/docs/tutorial/mac-app-store-submission-guide#limitations-of-mas-build
        if (process.mas) {
            return false;
        }
        return this.options.enableNative !== false;
    }
    /** Activates the Electron CrashReporter. */
    installNativeHandler() {
        // We are only called by the frontend if the SDK is enabled and a valid DSN
        // has been configured. If no DSN is present, this indicates a programming
        // error.
        const dsnString = this.options.dsn;
        if (!dsnString) {
            throw new core_1.SentryError('Invariant exception: install() must not be called when disabled');
        }
        const dsn = new core_1.Dsn(dsnString);
        // We will manually submit errors, but CrashReporter requires a submitURL in
        // some versions. Also, provide a productName and companyName, which we will
        // add manually to the event's context during submission.
        electron_1.crashReporter.start({
            companyName: '',
            ignoreSystemCrashHandler: true,
            productName: electron_1.app.getName(),
            submitURL: uploader_1.MinidumpUploader.minidumpUrlFromDsn(dsn),
            uploadToServer: false,
        });
        // The crashReporter has an undocumented method to retrieve the directory
        // it uses to store minidumps in. The structure in this directory depends
        // on the crash library being used (Crashpad or Breakpad).
        const reporter = electron_1.crashReporter;
        const crashesDirectory = reporter.getCrashesDirectory();
        this.uploader = new uploader_1.MinidumpUploader(dsn, crashesDirectory, getCachePath());
        // Flush already cached minidumps from the queue.
        async_1.forget(this.uploader.flushQueue());
        // Start to submit recent minidump crashes. This will load breadcrumbs and
        // context information that was cached on disk prior to the crash.
        async_1.forget(this.sendNativeCrashes({}));
        // Every time a subprocess or renderer crashes, start sending minidumps
        // right away.
        electron_1.app.on('web-contents-created', (_, contents) => {
            contents.on('crashed', () => __awaiter(this, void 0, void 0, function* () {
                try {
                    yield this.sendNativeCrashes(this.getRendererExtra(contents));
                }
                catch (e) {
                    console.error(e);
                }
                core_1.addBreadcrumb({
                    category: 'exception',
                    level: types_1.Severity.Critical,
                    message: 'Renderer Crashed',
                    timestamp: new Date().getTime() / 1000,
                });
            }));
            if (this.options.enableUnresponsive !== false) {
                contents.on('unresponsive', () => {
                    core_1.captureMessage('BrowserWindow Unresponsive');
                });
            }
        });
        return true;
    }
    /** Installs IPC handlers to receive events and metadata from renderers. */
    installIPC() {
        electron_1.ipcMain.on(common_1.IPC_PING, (event) => {
            event.sender.send(common_1.IPC_PING);
        });
        electron_1.ipcMain.on(common_1.IPC_CRUMB, (_, crumb) => {
            core_1.addBreadcrumb(crumb);
        });
        electron_1.ipcMain.on(common_1.IPC_EVENT, (ipc, event) => {
            event.extra = Object.assign({}, this.getRendererExtra(ipc.sender), event.extra);
            core_1.captureEvent(event);
        });
        electron_1.ipcMain.on(common_1.IPC_SCOPE, (_, rendererScope) => {
            // tslint:disable:no-unsafe-any
            const sentScope = core_1.Scope.clone(rendererScope);
            core_1.configureScope(scope => {
                if (sentScope.user) {
                    scope.setUser(sentScope.user);
                }
                if (sentScope.tags) {
                    Object.keys(sentScope.tags).forEach(key => {
                        scope.setTag(key, sentScope.tags[key]);
                    });
                }
                if (sentScope.extra) {
                    Object.keys(sentScope.extra).forEach(key => {
                        scope.setExtra(key, sentScope.extra[key]);
                    });
                }
            });
            // tslint:enable:no-unsafe-any
        });
    }
    /** Loads new native crashes from disk and sends them to Sentry. */
    sendNativeCrashes(extra) {
        return __awaiter(this, void 0, void 0, function* () {
            // Whenever we are called, assume that the crashes we are going to load down
            // below have occurred recently. This means, we can use the same event data
            // for all minidumps that we load now. There are two conditions:
            //
            //  1. The application crashed and we are just starting up. The stored
            //     breadcrumbs and context reflect the state during the application
            //     crash.
            //
            //  2. A renderer process crashed recently and we have just been notified
            //     about it. Just use the breadcrumbs and context information we have
            //     right now and hope that the delay was not too long.
            const uploader = this.uploader;
            if (uploader === undefined) {
                throw new core_1.SentryError('Invariant violation: Native crashes not enabled');
            }
            const currentCloned = core_1.Scope.clone(node_1.getCurrentHub().getScope());
            const fetchedScope = this.scopeStore.get();
            const storedScope = core_1.Scope.clone(fetchedScope);
            let event = { extra };
            event = yield storedScope.applyToEvent(event);
            event = event && (yield currentCloned.applyToEvent(event));
            const paths = yield uploader.getNewMinidumps();
            paths.map(path => {
                sdk_1.captureMinidump(path, Object.assign({}, event));
            });
        });
    }
    /** Returns extra information from a renderer's web contents. */
    getRendererExtra(contents) {
        const customName = this.options.getRendererName && this.options.getRendererName(contents);
        return {
            crashed_process: customName || `renderer[${contents.id}]`,
            crashed_url: normalize_1.normalizeUrl(contents.getURL()),
        };
    }
}
exports.MainBackend = MainBackend;
//# sourceMappingURL=backend.js.map