"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const node_1 = require("@sentry/node");
const electron_1 = require("electron");
/** Electron integration that cleans up the event. */
class Electron {
    constructor() {
        /**
         * @inheritDoc
         */
        this.name = Electron.id;
    }
    /**
     * @inheritDoc
     */
    setupOnce() {
        this.instrumentBreadcrumbs('app', electron_1.app);
        electron_1.app.once('ready', () => {
            // We can't access these until 'ready'
            this.instrumentBreadcrumbs('Screen', electron_1.screen);
            this.instrumentBreadcrumbs('PowerMonitor', electron_1.powerMonitor);
        });
        electron_1.app.on('web-contents-created', (_, contents) => {
            // SetImmediate is required for contents.id to be correct
            // https://github.com/electron/electron/issues/12036
            setImmediate(() => {
                const options = node_1.getCurrentHub().getClient().getOptions();
                const customName = options.getRendererName && options.getRendererName(contents);
                this.instrumentBreadcrumbs(customName || `WebContents[${contents.id}]`, contents, [
                    'dom-ready',
                    'load-url',
                    'destroyed',
                ]);
            });
        });
    }
    /**
     * Hooks into the Electron EventEmitter to capture breadcrumbs for the
     * specified events.
     */
    instrumentBreadcrumbs(category, emitter, events = []) {
        const emit = emitter.emit.bind(emitter);
        emitter.emit = (event, ...args) => {
            if (events.length === 0 || events.indexOf(event) > -1) {
                const breadcrumb = {
                    category: 'electron',
                    message: `${category}.${event}`,
                    timestamp: new Date().getTime() / 1000,
                    type: 'ui',
                };
                const self = node_1.getCurrentHub().getIntegration(Electron);
                if (self) {
                    node_1.getCurrentHub().addBreadcrumb(breadcrumb);
                }
            }
            return emit(event, ...args);
        };
    }
}
/**
 * @inheritDoc
 */
Electron.id = 'Electron';
exports.Electron = Electron;
//# sourceMappingURL=electron.js.map