"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var tslib_1 = require("tslib");
var core_1 = require("@sentry/core");
var types_1 = require("@sentry/types");
var object_1 = require("@sentry/utils/object");
var util = require("util");
/** Console module integration */
var Console = /** @class */ (function () {
    function Console() {
        /**
         * @inheritDoc
         */
        this.name = Console.id;
    }
    /**
     * @inheritDoc
     */
    Console.prototype.setupOnce = function () {
        var nativeModule = require('module');
        object_1.fill(nativeModule, '_load', loadWrapper(nativeModule));
        // special case: since console is built-in and app-level code won't require() it, do that here
        require('console');
    };
    /**
     * @inheritDoc
     */
    Console.id = 'Console';
    return Console;
}());
exports.Console = Console;
/**
 * Wrapper function for internal _load calls within `require`
 */
function loadWrapper(nativeModule) {
    // We need to use some functional-style currying to pass values around
    // as we cannot rely on `bind`, because this has to preserve correct
    // context for native calls
    return function (originalLoad) {
        return function (moduleId) {
            var originalModule = originalLoad.apply(nativeModule, arguments);
            if (moduleId !== 'console' || originalModule.__sentry__) {
                return originalModule;
            }
            ['debug', 'info', 'warn', 'error', 'log'].forEach(consoleWrapper(originalModule));
            originalModule.__sentry__ = true;
            return originalModule;
        };
    };
}
/**
 * Wrapper function that'll be used for every console level
 */
function consoleWrapper(originalModule) {
    return function (level) {
        if (!(level in originalModule)) {
            return;
        }
        object_1.fill(originalModule, level, function (originalConsoleLevel) {
            var sentryLevel;
            switch (level) {
                case 'debug':
                    sentryLevel = types_1.Severity.Debug;
                    break;
                case 'error':
                    sentryLevel = types_1.Severity.Error;
                    break;
                case 'info':
                    sentryLevel = types_1.Severity.Info;
                    break;
                case 'warn':
                    sentryLevel = types_1.Severity.Warning;
                    break;
                default:
                    sentryLevel = types_1.Severity.Log;
            }
            return function () {
                if (core_1.getCurrentHub().getIntegration(Console)) {
                    core_1.getCurrentHub().addBreadcrumb({
                        category: 'console',
                        level: sentryLevel,
                        message: util.format.apply(undefined, arguments),
                    }, {
                        input: tslib_1.__spread(arguments),
                        level: level,
                    });
                }
                originalConsoleLevel.apply(originalModule, arguments);
            };
        });
    };
}
//# sourceMappingURL=console.js.map