"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var tslib_1 = require("tslib");
var core_1 = require("@sentry/core");
var types_1 = require("@sentry/types");
var fs = require("fs");
var version_1 = require("../version");
/** Base Transport class implementation */
var BaseTransport = /** @class */ (function () {
    /** Create instance and set this.dsn */
    function BaseTransport(options) {
        this.options = options;
        /** A simple buffer holding all requests. */
        this.buffer = new core_1.PromiseBuffer(30);
        this.api = new core_1.API(options.dsn);
    }
    /** Returns a build request option object used by request */
    BaseTransport.prototype.getRequestOptions = function () {
        var headers = tslib_1.__assign({}, this.api.getRequestHeaders(version_1.SDK_NAME, version_1.SDK_VERSION), this.options.headers);
        var dsn = this.api.getDsn();
        var options = {
            agent: this.client,
            headers: headers,
            hostname: dsn.host,
            method: 'POST',
            path: this.api.getStoreEndpointPath(),
            port: dsn.port,
            protocol: dsn.protocol + ":",
        };
        if (this.options.caCerts) {
            options.ca = fs.readFileSync(this.options.caCerts);
        }
        return options;
    };
    /** JSDoc */
    BaseTransport.prototype.sendWithModule = function (httpModule, body) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var _this = this;
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.buffer.add(new Promise(function (resolve, reject) {
                        var req = httpModule.request(_this.getRequestOptions(), function (res) {
                            res.setEncoding('utf8');
                            if (res.statusCode && res.statusCode >= 200 && res.statusCode < 300) {
                                resolve({
                                    status: types_1.Status.fromHttpCode(res.statusCode),
                                });
                            }
                            else {
                                if (res.headers && res.headers['x-sentry-error']) {
                                    var reason = res.headers['x-sentry-error'];
                                    reject(new core_1.SentryError("HTTP Error (" + res.statusCode + "): " + reason));
                                }
                                else {
                                    reject(new core_1.SentryError("HTTP Error (" + res.statusCode + ")"));
                                }
                            }
                            // force the socket to drain
                            res.on('data', function () {
                                // Drain
                            });
                            res.on('end', function () {
                                // Drain
                            });
                        });
                        req.on('error', reject);
                        req.end(body);
                    }))];
            });
        });
    };
    /**
     * @inheritDoc
     */
    BaseTransport.prototype.sendEvent = function (_) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                throw new core_1.SentryError('Transport Class has to implement `sendEvent` method.');
            });
        });
    };
    /**
     * @inheritDoc
     */
    BaseTransport.prototype.close = function (timeout) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, this.buffer.drain(timeout)];
            });
        });
    };
    return BaseTransport;
}());
exports.BaseTransport = BaseTransport;
//# sourceMappingURL=base.js.map