const dateFormat = require('dateformat');
const winston = require('winston');
const DailyRotateFile = require('winston-daily-rotate-file');
const path = require('path');
const mkdirp = require('mkdirp');

const app = process.type === 'browser'
    ? require('electron').app
    : require('electron').remote.app;

const logDir = path.resolve(app.getPath('userData'), 'logs');

// const logLevel = process.env.NODE_ENV === 'production' ? 'info' : 'debug';
const logLevel = 'debug';

const levels = {
    debug: 0,
    info: 1,
    warn: 2,
    error: 3,
};

const d = (level, message) => {
    if (levels[level] >= levels[logLevel]) {
        const consoleLevel = level === 'debug' ? 'log' : level;
        console[consoleLevel](message);
    }
};

// ensure the log dir exists
try {
    mkdirp.sync(logDir);
} catch (e) {
    d('error', e);
}

function createLogger(filename) {
    const logger = new (winston.Logger)({
        transports: [
            new (DailyRotateFile)({
                level: logLevel,
                filename,
                timestamp() {
                    const _date = new Date();
                    return dateFormat(_date, 'yyyy-mm-dd HH:MM:ss');
                },
                maxSize: '5m',
            }),
        ],
    });
    return logger;
}

// 应用进程
function ProcessLogger() {
    const processLogFileName = path.resolve(logDir, 'processLogs%DATE%.log');
    this.logger = createLogger(processLogFileName);
    return this;
}

ProcessLogger.prototype.debug = function debug(message) {
    d('debug', message);
    return this.logger.debug(message);
};

ProcessLogger.prototype.info = function info(message) {
    d('info', message);
    return this.logger.info(message);
};

ProcessLogger.prototype.warn = function warn(message) {
    d('warn', message);
    return this.logger.warn(message);
};

ProcessLogger.prototype.error = function error(message) {
    d('error', message);
    return this.logger.error(message);
};

// App进程
function AppLogger() {
    const appLogFileName = path.resolve(logDir, 'appLogs%DATE%.log');
    this.logger = createLogger(appLogFileName);
    return this;
}

AppLogger.prototype.debug = function debug(message) {
    d('debug', message);
    return this.logger.debug(message);
};

AppLogger.prototype.info = function info(message) {
    d('info', message);
    return this.logger.info(message);
};

AppLogger.prototype.warn = function warn(message) {
    d('warn', message);
    return this.logger.warn(message);
};

AppLogger.prototype.error = function error(message) {
    d('error', message);
    return this.logger.error(message);
};

// voip 进程
function VoipLogger() {
    const appLogFileName = path.resolve(logDir, 'voipLogs%DATE%.log');
    this.logger = createLogger(appLogFileName);
    return this;
}

VoipLogger.prototype.info = function info(message) {
    d('info', message);
    return this.logger.info(message);
};

const processLogger = new ProcessLogger();
const appLogger = new AppLogger();
const voipLogger = new VoipLogger();

module.exports = {
    processLogger,
    appLogger,
    voipLogger,
};
