/* eslint-disable  */

const { remote } = require('electron');
const { app, dialog } = remote;
const fs = require('fs');
const path = require('path');
const request = require('../../common/httpRequest');
const pieceCacheConfig = require('../../common/cachePath').download;
const { appLogger } = require('../../common/logger');

/**
 * 文件分片最大值 单位 M
 */
const CHUNK_MAX_SIZE = 10;
/**
 * 文件分片最小值 单位 M
 */
const CHUNK_MIN_SIZE = 1;
const pieceCacheDir = pieceCacheConfig.tmpPath;
const downloadLimit = 5;

const utils = {
    mkdir: function mkdir(targetPath) {
        const parent = path.join(targetPath, '..');
        if (!fs.existsSync(parent)) {
            mkdir(parent);
        }
        if (!fs.existsSync(targetPath)) {
            fs.mkdirSync(targetPath);
        }
    },
    delDir: function delDir(targetPath) {
        if (fs.existsSync(targetPath)) {
            const files = fs.readdirSync(targetPath);
            files.forEach((file) => {
                const curPath = `${targetPath}/${file}`;
                if (fs.statSync(curPath).isDirectory()) {
                    delDir(curPath);
                } else {
                    fs.unlinkSync(curPath);
                }
            });
            fs.rmdirSync(targetPath);
        }
    },
    isFunction(fun) {
        return typeof fun === 'function';
    },
    extend(target, ext) {
        if (!ext) {
            return target;
        }
        Object.keys(ext).forEach((key) => {
            target[key] = ext[key];
        });
        return target;
    },
    concatFile(rangeList, dist, callback, writable) {
        // if (dist != "") {
        //     var re = /[\u4e00-\u9fa5A-Za-z0-9`~!@#$%^&*()_\-+=<>?:"{}|,.\/;'\\[\]·~！@#￥%……&*（）——\-+={}|《》？:，。‘“’”【】]/g;
        //     dist = Array.from(dist.match(re)).join('')
        // }
        if (!writable) {
            writable = fs.createWriteStream(dist);
            writable.once('error', (error) => {
                writable.end();
                // 写入失败无权限或者文件被占用，删除也会失败此方法无需调用
                // fs.unlink(dist);
                callback(error);
            });
        }
        if (rangeList.length === 0) {
            writable.end();
            callback(null, { path: dist });
            return;
        }
        // 创建可读流
        const readable = fs.createReadStream(rangeList.shift());
        // 监听 pip end 事件，读取下一个文件
        readable.once('end', () => {
            utils.concatFile(rangeList, dist, callback, writable);
        });
        readable.once('error', (error) => {
            writable.end();
            fs.unlink(dist, () => { });
            callback(error);
        });
        readable.pipe(writable, { end: false });
    },
    getUId() {
        let date = Date.now();
        const uuid = 'xxxxxx4xxxyxxxxxxx'.replace(/[xy]/g, (c) => {
            const r = (date + Math.random() * 16) % 16 | 0;
            date = Math.floor(date / 16);
            return (c === 'x' ? r : (r & 0x3 | 0x8)).toString(16);
        });
        return uuid;
    },
};

const pieceCache = {
    get(uid) {
        return pieceCacheConfig.rangeConf[uid];
    },
    set(uid, value) {
        value = utils.extend({}, value);
        pieceCacheConfig.rangeConf[uid] = value;
        pieceCacheConfig.flush();
    },
    remove(uid) {
        delete pieceCacheConfig.rangeConf[uid];
        pieceCacheConfig.flush();
    },
};

class Piece {
    // 属性
    // url = ''; // 资源地址
    // sessionId = ''; // 下载唯一标识
    // totalSize = 0; // 文件总大小
    // range = [];
    // size = 0; // 分片大小
    // current = 0; // 下载进度

    // 事件
    // onprogress = null;
    // onsuccess = null;
    // onerror = null;

    constructor(options) {
        this.url = options.url;
        this.sessionId = options.sessionId;
        this.totalSize = options.totalSize;
        this.range = options.range;
        this.size = (options.range[1] - options.range[0]);
        this.current = 0;
        this.timeoutRetry = 0;
    }

    start() {
        const self = this;

        const pieceFilePath = path.join(pieceCacheDir, self.sessionId);
        utils.mkdir(pieceFilePath);
        const pieceFileName = path.join(pieceFilePath, self.range.join('-'));

        appLogger.info(`>>> download: ${self.range[0]}-${self.range[1]}`);
        self.req = request({
            method: 'get',
            url: self.url,
            encoding: null,
            timeout: 180000,
            headers: {
                Range: `bytes=${self.range[0]}-${self.range[1]}`,
                'X-File-TransactionId': self.sessionId,
                'X-File-Total-Size': self.totalSize,
            },
        }, (error, response) => {
            if (error || !response || [200, 206].indexOf(response.statusCode) === -1) {
                if (response) appLogger.info(`<<< statusCode: ${response.statusCode}`);
                if (error && error.code === 'ESOCKETTIMEDOUT' && ++self.timeoutRetry < 3) {
                    appLogger.info(`!!! timeout: ${self.range[0]}-${self.range[1]}`);
                    self.current = 0;
                    self.start();
                } else if (utils.isFunction(self.onerror)) {
                    self.onerror(error || response || {
                        statusCode: 504,
                        message: error.message || 'NET_ERROR',
                    });
                }
                return;
            }
            if (utils.isFunction(self.onsuccess)) {
                fs.writeFile(pieceFileName, response.body, (err) => {
                    if (err) {
                        self.error(err);
                        return;
                    }
                    self.onsuccess();
                });
            }
        });
        // const pieceSize = self.range[1] - self.range[0] + 1;
        self.req.on('data', (chunk) => {
            // appLogger.info(`<<< received: ${self.current}/${pieceSize}:${self.range[0]}-${self.range[1]}`);
            self.current += chunk.length;
            if (utils.isFunction(self.onprogress)) {
                self.onprogress();
            }
        });
    }

    pause() {
        this.req.pause();
    }

    resume() {
        // check if abort
        if (this.req) {
            this.req.resume();
        } else {
            this.start();
        }
    }

    abort() {
        this.req.abort();
        this.req = null;
        this.current = 0;
        if (this.onprogress) {
            this.onprogress();
        }
    }
}

class PieceManage {
    // 属性
    // waitingList = [];
    // loadingList = [];
    // loadedList = [];

    // 事件
    // onprogress = null;
    // onsuccess = null;
    // onerror = null;

    /**
     *
     * @param {object} options { url: '',sessionId: '',totalSize: 0,rangeList: [] }
     */
    constructor(options) {
        this.waitingList = [];
        this.loadingList = [];
        this.loadedList = [];
        this._init(options);
    }

    _init(options) {
        const self = this;
        options.rangeList.forEach((range) => {
            const piece = new Piece({
                url: options.url,
                sessionId: options.sessionId,
                totalSize: options.totalSize,
                range,
            });
            piece.onprogress = function () {
                let size = 0;
                (self.loadedList.concat(self.loadingList)).forEach((item) => {
                    size += item.current;
                });
                if (utils.isFunction(self.onprogress)) {
                    self.onprogress(size);
                }
            };
            piece.onsuccess = function () {
                const index = self.loadingList.indexOf(this);
                self.loadingList.splice(index, 1);
                self.loadedList.push(this);

                // appLogger.info(`    waiting: ${self.waitingList.length}`)
                // appLogger.info(`    loading: ${self.loadingList.length}`)
                // appLogger.info(`    loaded: ${self.loadedList.length}`)
                if (self.waitingList.length > 0) {
                    const next = self.waitingList.shift();
                    self.loadingList.push(next);
                    next.start();
                } else if (self.loadingList.length === 0) {
                    if (utils.isFunction(self.onsuccess)) {
                        self.onsuccess();
                    }
                }
            };
            piece.onerror = function (error) {
                self.loadingList.forEach((item) => {
                    item.abort();
                });
                if (utils.isFunction(self.onerror)) {
                    self.onerror(error);
                }
            };
            self.waitingList.push(piece);
        });
    }

    start() {
        const length = Math.min(downloadLimit, this.waitingList.length);
        for (let i = 0; i < length; i++) {
            const piece = this.waitingList.shift();
            this.loadingList.push(piece);
            piece.start();
        }
    }

    pause() {
        this.loadingList.forEach((item) => {
            item.pause();
        });
    }

    resume() {
        this.loadingList.forEach((item) => {
            item.resume();
        });
    }

    abort() {
        this.loadingList.forEach((item) => {
            item.abort();
        });
    }
}

function getChunksizeByToalsize(size) {
    // 换算成兆按比例分片
    const sizeM = size / 1000000;
    const chunkSize = Math.min(Math.max(Math.ceil(sizeM / downloadLimit), CHUNK_MIN_SIZE), CHUNK_MAX_SIZE);
    return chunkSize * 1000000;
}

class Download {
    // stats 只读外部不允许改变
    // stats = {
    //     id: '',
    //     url: '',
    //     name: '',
    //     size: 0,
    //     path: '',
    //     chunkSize: 3 * 1024
    // }
    // rangeList = [];
    // manage = null;
    // isPause = false;

    // 事件
    // onReady = null;
    // onProgress = null;
    // onComplete = null;
    // onError = null;
    // onCancel = null;

    /**
     *
     * @param {object} options {id: 唯一标识, url: 下载资源地址, name: 文件名称, size: 文件大小, path: 保存位置}
     */
    constructor(options) {
        this.stats = {
            id: options.uId || utils.getUId(),
            url: encodeURI(options.url),
            name: options.name,
            size: options.size,
            path: options.path,
            chunkSize: getChunksizeByToalsize(options.size),
        };
        if (this.stats.id) {
            utils.extend(this.stats, Download.getStats(this.stats.id));
        }
        this.rangeList = [];
        this.manage = null;
        this.isPause = false;
        this._init();
    }

    /**
     * 根据创建对象传入参数初始化 rangeList
     */
    _init() {
        const { stats } = this;
        const { size } = stats;
        const { chunkSize } = stats;
        const pieceCount = Math.ceil(size / chunkSize);
        for (let i = 0; i < pieceCount; i += 1) {
            const start = i * chunkSize;
            const end = i === pieceCount - 1 ? size - 1 : (i + 1) * chunkSize - 1;
            this.rangeList.push([start, end]);
        }
    }

    /**
     * 检查并返回需要下载的分片
     */
    _check() {
        const pieceFilePath = path.join(pieceCacheDir, this.stats.id);
        // appLogger.info(`*** piece file path: ${pieceFilePath}`)
        let fileList = [];
        try {
            fileList = fs.readdirSync(pieceFilePath);
        } catch (e) {
            // 文件不存在
        }
        return this.rangeList.filter((item) => {
            const fileName = item.join('-');
            return fileList.indexOf(fileName) === -1;
        });
    }

    /**
     * 开始分片下载
     * 检查需要下载的分片,创建分片管理对象开始下载
     */
    _start(message) {
        console.log("_start", message);
        const self = this;
        const concatPiece = function concatPiece(id, rangeList, savePath, complete) {
            const pieceFilePath = path.join(pieceCacheDir, id);
            const pathList = rangeList.map(item => path.join(pieceFilePath, item.join('-')));
            // 下载时选择覆盖需要删除原文件
            if (message) {
                var re = /[\u4e00-\u9fa5A-Za-z0-9`~!@#$%^&*()_\-+=<>?:"{}|,.\/;'\\[\]·~！@#￥%……&*（）——\-+={}|《》？:，。‘“’”【】]/g;
                message.content.name = Array.from(message.content.name.match(re)).join('')
                if (savePath.indexOf(message.content.name) == -1) {
                    savePath = savePath +message.content.name;
                }
            }
            fs.unlink(savePath, (err) => {
                console.log(savePath);
                // 删除成功或文件不存在时进行合并文件分片
                if (!err || err.code === 'ENOENT') {
                    utils.concatFile(pathList, savePath, (error, data) => {
                        if (error) {
                            if (utils.isFunction(self.onError)) {
                                self.onError(error.code.toLowerCase());
                            }
                            return;
                        }
                        pieceCache.remove(id);
                        utils.delDir(pieceFilePath);
                        complete(data);
                    });
                } else if (utils.isFunction(self.onError)) {
                    self.onError(err.code.toLowerCase());
                }
            });
        };
        const unexistRangeList = self._check();
        const manage = new PieceManage({
            url: this.stats.url,
            sessionId: this.stats.id,
            totalSize: this.stats.size,
            rangeList: unexistRangeList,
        });
        self.manage = manage;
        let unloadSize = 0;
        unexistRangeList.forEach((range) => {
            unloadSize += (range[1] - range[0]);
        });
        if (unexistRangeList.length === 0) {
            concatPiece(self.stats.id, self.rangeList, self.stats.path, self.onComplete);
            return;
        }
        const preloadSize = self.stats.size - unloadSize;
        manage.onprogress = function onprogress(size) {
            self.stats.offset = preloadSize + size;
            if (utils.isFunction(self.onProgress)) {
                self.onProgress({
                    loaded: self.stats.offset,
                    total: self.stats.size,
                });
            }
        };
        manage.onsuccess = function onsuccess() {
            concatPiece(self.stats.id, self.rangeList, self.stats.path, self.onComplete);
        };
        manage.onerror = function onerror(error) {
            if (error.statusCode >= 400 && error.statusCode < 500) {
                const { id } = self.stats;
                pieceCache.remove(id);
                const pieceFilePath = path.join(pieceCacheDir, id);
                utils.delDir(pieceFilePath);
            }
            if (utils.isFunction(self.onError)) {
                self.onError('error');
            }
        };
        manage.start();
        pieceCache.set(self.stats.id, self.stats);
        if (utils.isFunction(self.onReady)) {
            self.onReady({
                total: this.stats.size,
                path: this.stats.path,
            });
        }
    }

    /**
     * 首次下载选择保存位置
     * TODO: 兼容之前调用，showSaveDialog 方法不应出现在这个类中
     */
    saveAs() {
        appLogger.info(`>>> saveFolder: ${pieceCacheConfig.saveFolder}`);
        if (pieceCacheConfig.saveFolder) {
            this.stats.path = pieceCacheConfig.saveFolder + this.stats.name;
        }

        const self = this;
        // self.stats.path = encodeURIComponent(self.stats.path )
        console.log('SELF====>',self);
        const file = {
            name: self.stats.name,
            path: self.stats.path || app.getPath('downloads'),
        };
        const distPath = path.join(file.path, file.name);
        const dialogOptions = {
            title: file.name,
            defaultPath: distPath,
        };
        dialogOptions.filters = [{
            name: '*.*',
            extensions: ['*'],
        }];
        let ext = '';
        const fileName = self.stats.name;
        const extIndex = fileName.lastIndexOf('.');
        if (extIndex !== -1) {
            ext = fileName.substring(extIndex + 1, fileName.length);
        }
        if (ext) {
            dialogOptions.filters.unshift({
                name: `[文件](*.${ext})`,
                extensions: [ext],
            });
        }
        if (self.stats.path) {
            console.log("self.stats.path",self.stats.path);
            // RongIM.common.messageToast({
            //     message: "文件保存为"+self.stats.path,
            //     type: 'success'
            // });
            self._start();
        } else {
            dialog.showSaveDialog(remote.getCurrentWindow(), dialogOptions, function (savePath) {
                console.log("savePath",savePath);
                if (savePath) {
                    const l = savePath.lastIndexOf(self.stats.name);
                    pieceCacheConfig.saveFolder = savePath.substring(0, l);
                    self.stats.path = savePath;
                    self._start();
                } else {
                    self.onCancel();
                }
            });
        }
    }

    /**
     * 继续上次断点续传的位置开始下载
     */
    continue(message) {
        if (this.stats.path) {
            this._start(message);
        } else {
            this.saveAs();
        }
    }

    /**
     * 暂停下载
     */
    pause() {
        this.isPause = true;
        if (this.manage) {
            this.manage.pause();
        }
    }

    /**
     * 暂停后的开始下载
     */
    resume() {
        this.isPause = false;
        if (this.manage) {
            this.manage.resume();
        }
    }

    /**
     * 取消下载
     */
    abort() {
        if (this.manage) {
            this.manage.abort();
        }
    }

    static getStats(downloadUid) {
        const downloadStats = pieceCache.get(downloadUid);
        if (!downloadStats) {
            return null;
        }
        // 获取准确的下载文件大小，暂停下载再次打开应用时获取 range.json 中已下载大小不正确
        let downloadSize = 0;
        try {
            const fileList = fs.readdirSync(path.join(pieceCacheDir, downloadUid));
            fileList.forEach((fileName) => {
                const range = fileName.split('-');
                downloadSize += (Number(range[1]) - Number(range[0]));
            });
        } catch (e) {
            // 文件不存在
        }
        if (!Number.isNaN(downloadSize)) {
            downloadStats.offset = downloadSize;
        }
        return downloadStats;
    }
}

module.exports = {
    getSaveFolder() {
        return pieceCacheConfig.saveFolder;
    },
    setSaveFolder(callback) {
        let self = this;
        dialog.showOpenDialog({
            title: '',
            defaultPath: pieceCacheConfig.saveFolder,
        }, (savePath) => {
            if (savePath && savePath.length > 0) {
                let l = savePath[0].lastIndexOf('\\');
                if (l < 0) l = savePath[0].lastIndexOf('/');
                pieceCacheConfig.saveFolder = savePath[0].substring(0, l + 1);
                if (callback) callback(pieceCacheConfig.saveFolder);
            }
        });
    },
    removeSaveFolder() {
        console.log('remove')
        pieceCacheConfig.saveFolder = "";
    },
    load(file, config) {
        return new Download(file, config);
    },
    getProgress(uId) {
        return Download.getStats(uId) || {};
    },
    /**获取文件列表 */
    getFileList(pathName) {
        return fs.readdirSync(pathName, function (err, files) {
            let dirs = [];
            (function iterator(i) {
                if (i == files.length) {
                    return;
                }
                fs.stat(path.join(pathName, files[i]), function (err, data) {
                    if (data.isFile()) {
                        dirs.push(files[i]);
                    }
                    iterator(i + 1);
                });
                return dirs;
            })(0);
        })
    },

    /** 检测本地是否存在某个目录地址 */
    doesItExistLocaPath(locaPath) {
        return new Promise((resovle, reject) => {
            fs.exists(locaPath, function (exists) {
                if (exists) {
                    resovle(exists)
                } else {
                    reject(exists)
                }
            })
        })
    }
};
