const path = require('path');
const express = require('express');
const net = require('net');

const ssi = require('./middleware/ssi');
const router = require('./router');
const { appLogger } = require('../../common/logger');

// TODO: 整合 logger 工具，封装等级常量
const INFO = 'info';
const ERROR = 'error';
const WARN = 'warn';
/**
 * 写日志
 * @param {String} level 日志登录
 * @param {String} message 日志内容
 */
function writeLog(level, message) {
    appLogger[level.toLowerCase()](`[ProxyServer ${level.toUpperCase()}] ${message}`);
}

/**
 * 查找可用端口号
 * @param {Number} port 起始端口
 * @returns {Promise<Number>} 可用的端口号
 */
async function portIsOccupied(port) {
    const server = net.createServer().listen(port);
    return new Promise((resolve, reject) => {
        server.on('listening', () => {
            server.close();
            resolve(port);
        });
        server.on('error', (err) => {
            if (err.code === 'EADDRINUSE') {
                const nextPort = port + 1;
                writeLog(INFO, `端口 ${port} 已被占用，尝试端口 ${nextPort}`);
                // 端口号 + 1 重试
                resolve(portIsOccupied(nextPort));
            } else {
                reject(err);
            }
        });
    });
}


/**
 * 创建代理服务
 * @param {Number} defaultPort 默认监听端口
 * @param {String} userDir 用户缓存目录
 * @returns {Promise<Number>} 所监听的端口
 */
async function createLocalServer(defaultPort, userDir) {
    writeLog(INFO, `查找可用端口，初始端口：${defaultPort}`);
    const port = await portIsOccupied(defaultPort);
    const app = express();
    // 拒绝所有非本机请求
    app.use((req, res, next) => {
        const { remoteAddress, remoteFamily } = req.socket;
        if (!remoteAddress.includes('127.0.0.1')) {
            writeLog(WARN, `警告!!! ${remoteFamily}-${req.socket} 正试图向代理服务发起请求！`);
            res.sendStatus(403);
            return;
        }
        next();
    });
    const staticRoot = path.join(__dirname, '../../../webapp');
    // Server Side Include 功能中间件，需放在 static 之前
    app.use(ssi(staticRoot));
    // 应用所需静态资源
    app.use(express.static(staticRoot));
    // 代理服务路由解析
    app.use(router(userDir));
    // 记录异常并处理，异常处理中间件需定义 4 个形参
    // eslint-disable-next-line no-unused-vars
    app.use((error, req, res, next) => {
        writeLog(ERROR, `${req.method} ${req.url} ${error.stack}`);
        res.status(500).send(error.stack);
    });
    return new Promise((resolve) => {
        app.listen(port, () => resolve(port));
    });
}
module.exports = createLocalServer;
