(function (RongIM, dependencies, components) {
'use strict';

var common = RongIM.common;
var utils = RongIM.utils;
var Base64Util = utils.Base64;
var moment = dependencies.moment;
var RongIMClient = dependencies.RongIMClient;
var allImageList = [];
var system = RongIM.system;

/*
说明： 获取历史消息
       桌面版增加文件分类，由于涉及到按文件名称搜索 SDK 目前不支持所以获取所有文件类消息内存中做分页。
*/
components.getHistory = function (resolve, reject) {
    var im = RongIM.instance;
    var dataModel = im.dataModel;
    // 模块化屏蔽入口

    var options = {
        name: 'history',
        template: 'templates/conversation/history.html',
        props: ['conversation'],
        data: function () {
            return {
                busy: true,
                messageType: '',
                keyword: '',
                count: 20,
                currentPage: 1,
                pageCount: '',
                total: 0,
                messageList: [],
                notOnlyImage: true,
                hasMoreLast: true
            };
        },
        computed: {
            // 支持在历史消息中展示的消息
            // filterList: function () {
            //     var list = [
            //         'TextMessage',
            //         'ImageMessage',
            //         'FileMessage',
            //         'VoiceMessage',
            //         'LocationMessage',
            //         'CardMessage',
            //         'SightMessage',
            //         'RichContentMessage',
            //         'GroupNoticeNotifyMessage',
            //         'ReferenceMessage'
            //     ];

            //     return this.messageList.filter(function (item) {
            //         return list.indexOf(item.messageType) >= 0 && item.sentStatus !== RongIMLib.SentStatus.FAILED;
            //     });
            // },
            imageList: function () {
                var list = ['ImageMessage', 'SightMessage'];
                var imageMessageList = this.messageList.filter(function (item) {
                    return list.indexOf(item.messageType) >= 0 && item.sentStatus !== RongIMLib.SentStatus.FAILED;
                });
                var imageList = getImageList(imageMessageList);
                return imageList;
            },
            // 是否支持消息搜索（仅桌面版 C++ SDK 支持消息搜索）
            supportSearch: function () {
                return im.config.support.search;
            },
            messageTypeClassName: function () {
                if (this.messageType === 'ImageMessage') {
                    this.notOnlyImage = false;
                }
                if (this.messageType === 'FileMessage') {
                    this.notOnlyImage = true;
                }
                if (utils.isEmpty(this.messageType)) {
                    this.notOnlyImage = true;
                    return 'all';
                }
                var map = {};
                map[RongIMClient.MessageType.FileMessage] = 'file';
                map[RongIMClient.MessageType.ImageMessage] = 'image';
                return map[this.messageType];
            },
            imgViewerSource: function () {
                if (this.messageType === 'ImageMessage') {
                    return this.messageList;
                } if (utils.isEmpty(this.messageType)) {
                    return this.blockFilterList.data;
                }
                return undefined;
            }
        },
        components: {
            avatar: components.getAvatar,
            TextMessage: components.getTextMessage,
            ImageMessage: components.getImageMessage,
            FileMessage: components.getFileMessage,
            VoiceMessage: components.getVoiceMessage,
            LocationMessage: components.getLocationMessage,
            CardMessage: components.getCardMessage,
            RichContentMessage: components.getRichContentMessage,
            SightMessage: components.getSightMessage,
            UnknownMessage: components.getUnknownMessage,
            GroupNoticeNotifyMessage: components.getGroupNoticeNotifyMessage,
            ReferenceMessage: components.getQuoteMessage
        },
        directives: {
            autoScrolltotop: function (ele, binding, vnode) {
                var context = vnode.context;
                if (context.keyword || context.$refs.content) {
                    ele.scrollTop = 0;
                } else {
                    Vue.nextTick(function () {
                        ele.scrollTop = ele.scrollHeight;
                    });
                }
            }
        },
        watch: {
            keyword: function () {
                toggle(this, dataModel.Message);
            },
            messageType: function () {
                toggle(this, dataModel.Message);
            }
        },
        mounted: function () {
            var context = this;
            im.$on('imclick', context.close);
            toggle(this, dataModel.Message);
            var params = {
                route: context.$route,
                count: context.count,
                messageApi: dataModel.Message
            };
            if (system.platform.indexOf('web') === -1) {
                getAllFileList(params);
                getAllImageList(context, params);
            }
        }
    };

    options.methods = {
        clear: function () {
            this.keyword = '';
        },
        getUsername: function (user) {
            var conversation = this.conversation || {};
            var group = conversation.group || {};
            return common.getGroupUsername(user, group.id);
        },
        dateFormat: function (timestamp, format) {
            return moment(timestamp).format(format);
        },
        getMessageType: function (item) {
            var supported = options.components[item.messageType];
            return supported ? item.messageType : 'UnknownMessage';
        },
        showFileMessage: function () {
            this.messageType = RongIMClient.MessageType.FileMessage;
        },
        showImageMessage: function () {
            this.messageType = RongIMClient.MessageType.ImageMessage;
        },
        scrollToMessage: function (messageId, alignToTop) {
            var $content = $(this.$refs.content);
            Vue.nextTick(function () {
                var el = document.getElementById('rong-history-message-' + messageId);
                if (el) el.scrollIntoView(alignToTop);
                if ($content) $content.css('visibility', '');
            });
        },
        scroll: utils.throttle(function (event) {
            // if(im.status === utils.status.CONNECTED) {
            if (RongIM.isAvailableData()) {
                scroll(this, event);
            }
        }, 500),
        next: function () {
            if (this.pageCount === this.currentPage) {
                return;
            }
            this.currentPage += 1;
            getCurrentPageMessage(this, dataModel.Message);
        },
        prev: function () {
            if (this.currentPage === 1) {
                return;
            }
            this.currentPage -= 1;
            getCurrentPageMessage(this, dataModel.Message);
        },
        close: function () {
            this.$emit('hidepanel');
            im.$off('imclick', this.close);
        },
        showImage: function (message) {
            common.showImage(getImageMessageList(this.messageList), message.messageUId, im.config.locale);
        },
        showSight: function (message) {
            this.showImage(message);
        }
    };

    utils.asyncComponent(options, resolve, reject);
};

var cacheMessageList = [];

function toggle(context, messageApi) {
    cacheMessageList = [];
    context.pageCount = 0;
    context.currentPage = 1;
    getCurrentPageMessage(context, messageApi);
}

/*
说明： 获取所有文件类型的消息，在内存中分页查找。（SDK 不支持只搜索文件消息）
*/
var allFileList = [];
function getAllFileList(params) {
    var routeParams = params.route.params;
    allFileList = [];
    var messageParams = {
        conversationType: Number(routeParams.conversationType),
        targetId: routeParams.targetId,
        count: params.count,
        position: 2,
        type: 'FileMessage',
        timestamp: 0
    };
    function getFile() {
        params.messageApi.get(messageParams, function (errorCode, list, hasMore) {
            if (errorCode) {
                common.toastError(errorCode);
                return;
            }
            list.reverse();
            allFileList = allFileList.concat(list);
            if (hasMore) {
                messageParams.timestamp = allFileList.slice(-1)[0].sentTime;
                getFile();
            }
        });
    }
    getFile();
}

/*
说明：获取图片历史消息列表
参数：
    @param {array<object>}  context
    @param {object}         params  获取历史消息所需参数和 API
    @param {boolean}        last    判断是否为获取更多图片历史消息
*/
function getAllImageList(context, params, last) {
    var routeParams = params.route.params;
    var sentTime = 0;
    var firstMessage;
    allImageList = [];
    if (last) {
        sentTime = context.messageList[0].sentTime;
        firstMessage = context.messageList[0];
        allImageList = context.messageList.reverse();
    }
    var $content = $(context.$refs.content);
    $content.css('visibility', 'hidden');
    var messageParams = {
        conversationType: Number(routeParams.conversationType),
        targetId: routeParams.targetId,
        count: 20,
        position: 2,
        type: 'ImageMessage',
        timestamp: sentTime
    };
    params.messageApi.get(messageParams, function (errorCode, list, hasMore) {
        if (errorCode) {
            common.toastError(errorCode);
            return;
        }
        context.hasMoreLast = hasMore;
        list.reverse();
        allImageList = allImageList.concat(list);
        if (last) {
            var imgList = [];
            imgList = imgList.concat(allImageList);
            context.messageList = imgList.reverse();
            if (firstMessage) context.scrollToMessage(firstMessage.messageId);
        }
    }, true);
}

/*
说明： 根据当前页码 currentPage 当前消息类型 messageType 获取对应消息
*/
function getCurrentPageMessage(context, messageApi) {
    var currentPage = context.currentPage;
    var count = context.count;
    var start = (currentPage - 1) * count;
    var end = currentPage * count;
    if (context.messageType === 'FileMessage') {
        // C++ SDK 不支持文件消息关键字搜索。全部获取在内存中搜索。
        var notSearch = utils.isEmpty(context.keyword);
        if (notSearch) {
            context.pageCount = Math.ceil(allFileList.length / context.count) || 1;
            context.messageList = allFileList.slice(start, end).reverse();
        } else {
            searchFile(context, start, end);
        }
    } else if (context.messageType === 'ImageMessage') {
        var imageList = [];
        imageList = imageList.concat(allImageList);
        context.messageList = imageList.reverse();
    } else {
        var cacheHas = cacheMessageList.length > end && context.pageCount !== 0;
        if (cacheHas) {
            if (context.keyword) {
                context.messageList = cacheMessageList.slice(start, end);
            } else {
                context.messageList = cacheMessageList.slice(start, end).reverse();
            }
        } else {
            var earliest = cacheMessageList.slice(-1)[0];
            var timestamp = earliest ? earliest.sentTime : 0;
            getMessage(context, messageApi, timestamp, start, end);
        }
    }
}

/*
说明： 获取所有类型的历史消息，关键字 keyword 存在则根据关键字查找消息
       文本消息在缓存中搜索
*/
function getMessage(context, messageApi, timestamp, start, end) {
    var routeParams = context.$route.params;
    var params = {
        conversationType: Number(routeParams.conversationType),
        targetId: routeParams.targetId,
        timestamp: timestamp,
        count: context.count
    };
    if (context.keyword) {
        params.keyword = context.keyword;
        messageApi.search(params, function (errorCode, list, total) {
            context.busy = false;
            if (errorCode) {
                common.toastError(errorCode);
                return;
            }
            list.reverse();
            if (context.conversation.conversationType === utils.conversationType.GROUP) {
                addGroupAlias(context.conversation, list);
            }
            cacheMessageList = cacheMessageList.concat(list);
            context.messageList = list;
            context.pageCount = Math.ceil(total / context.count) || 1;
        });
    } else {
        params.position = 2;
        params.type = context.messageType;
        var pageList = [];
        var leftCount = cacheMessageList.length - start;
        getPageMessages(messageApi, params, leftCount, pageList, context, function (errorCode, list, hasMore) {
            context.busy = false;
            if (errorCode) {
                common.toastError(errorCode);
                return;
            }
            if (context.conversation.conversationType === utils.conversationType.GROUP) {
                addGroupAlias(context.conversation, list);
            }
            cacheMessageList = cacheMessageList.concat(list);
            context.messageList = cacheMessageList.slice(start, end).reverse();

            leftCount = cacheMessageList.length - end;
            if (hasMore) {
                context.pageCount = 0;
            } else if (leftCount > 0) {
                context.pageCount = context.currentPage + 1;
            } else {
                context.pageCount = context.currentPage;
            }
        });

        // messageApi.get(params, function (errorCode, list, hasMore) {
        //     context.busy = false;
        //     if(errorCode) {
        //         return common.toastError(errorCode);
        //     }
        //     list.reverse();
        //     if(context.conversation.conversationType === utils.conversationType.GROUP) {
        //         addGroupAlias(context.conversation, list);
        //     }
        //     cacheMessageList = cacheMessageList.concat(list);
        //     context.messageList = list.reverse();

        //     if (hasMore) {
        //         context.pageCount = 0;
        //     } else {
        //         context.pageCount = context.currentPage;
        //     }
        // });
    }
}

// 支持在历史消息中展示的消息
var filterMessages = [
    'TextMessage',
    'ImageMessage',
    'FileMessage',
    'VoiceMessage',
    'LocationMessage',
    'CardMessage',
    'SightMessage',
    'RichContentMessage',
    'GroupNoticeNotifyMessage',
    'ReferenceMessage'
];
/*
    递归 filter 不显示的消息类型
*/
function getPageMessages(messageApi, params, leftCount, pageList, context, callback) {
    messageApi.get(params, function (errorCode, list, hasMore) {
        if (errorCode) {
            callback(errorCode);
            return;
        }
        if (!list.length) {
            callback(null, [], hasMore);
            return;
        }
        var latestTimestamp = list[0].sentTime;
        var tmpList = list.filter(function (item) {
            return filterMessages.indexOf(item.messageType) >= 0
                && item.sentStatus !== RongIMLib.SentStatus.FAILED;
        });
        tmpList.reverse();
        pageList = pageList.concat(tmpList);

        if (hasMore && leftCount + pageList.length < context.count) {
            params.timestamp = latestTimestamp;
            setTimeout(getPageMessages, 0, messageApi, params, leftCount, pageList, context, callback);
        } else {
            callback(null, pageList, hasMore);
        }
    });
}

/*
说明： 根据关键字 keyword 从缓存中匹配文件名包含关键字的消息
*/
function searchFile(context, start, end) {
    var arr = allFileList.filter(function (item) {
        return item.content.name.indexOf(context.keyword) !== -1;
    });
    context.pageCount = Math.ceil(arr.length / context.count) || 1;
    arr = arr.slice(start, end);
    context.messageList = arr;
}

function addGroupAlias(conversation, list) {
    if (conversation.conversationType !== utils.conversationType.GROUP) {
        return;
    }
    var objGroup = {};
    conversation.group.members.forEach(function (group) {
        objGroup[group.id] = group.groupAlias;
    });
    list.forEach(function (_message) {
        var userId = _message.user.id;
        if (objGroup[userId]) {
            _message.user.groupAlias = objGroup[userId];
        }
    });
}

/*
说明：将图片历史消息按本周、本月、其他月的方式转换成所需格式
参数：
    @param {array<object>}    imageMessageList    图片历史消息数组
*/
function getImageList(imageMessageList) {
    var imageList = [];
    var imageListInfo = {};
    var isNowWeek;
    var isNowMonth;
    var ImageMessageTime = '';

    for (var i = 0; i < imageMessageList.length; i += 1) {
        isNowWeek = isSameWeek(imageMessageList[i].sentTime);
        isNowMonth = isSameMonth(imageMessageList[i].sentTime);
        // 根据消息的 sentTime 判断消息是否为本周、本月或其他月份
        if (isNowWeek) {
            imageListInfo['本周'] = imageListInfo['本周'] ? imageListInfo['本周'] : [];
            imageListInfo['本周'].push(imageMessageList[i]);
        } else if (isNowMonth) {
            imageListInfo['本月'] = imageListInfo['本月'] ? imageListInfo['本月'] : [];
            imageListInfo['本月'].push(imageMessageList[i]);
        } else {
            // 如果消息不是本周，同时也不是本月的话，在数组中添加该消息月份的分类
            ImageMessageTime = timestampToTime(imageMessageList[i].sentTime);
            imageListInfo[ImageMessageTime] = imageListInfo[ImageMessageTime] ? imageListInfo[ImageMessageTime] : [];
            imageListInfo[ImageMessageTime].push(imageMessageList[i]);
        }
    }
    Object.keys(imageListInfo).forEach(function (imageMessageItem) {
        imageList.push({
            name: imageMessageItem,
            data: imageListInfo[imageMessageItem]
        });
    });
    return imageList;
}

/*
说明：判断滚动条是否滚动到最顶部，获取更多图片历史消息
*/
function scroll(context, event) {
    var up = event.deltaY < 0;
    var $content = $(context.$refs.content);
    var im = RongIM.instance;
    var dataModel = im.dataModel;
    var params = {
        route: context.$route,
        count: context.count,
        messageApi: dataModel.Message
    };
    if (up && $content.scrollTop() <= 0 && context.hasMoreLast) {
        getAllImageList(context, params, true);
    }
}

/*
说明：根绝消息 sentTime 判断是否为本周消息
参数：
    @param {string}    old    图片历史消息 sentTime 时间戳
*/
function isSameWeek(old) {
    var now = new Date();
    var oneDayTime = 1000 * 60 * 60 * 24;
    var old_count = parseInt(+old / oneDayTime);
    var now_other = parseInt(+now / oneDayTime);
    return parseInt((old_count + 4) / 7) === parseInt((now_other + 4) / 7);
}

/*
说明：根绝消息 sentTime 判断是否为本月消息
参数：
    @param {string}    old    图片历史消息 sentTime 时间戳
*/
function isSameMonth(old) {
    var now = new Date();
    var nowMonth = now.getMonth() + 1;
    var oldMonth = (new Date(old)).getMonth() + 1;
    return nowMonth === oldMonth;
}

/*
说明：根绝消息 sentTime 获取消息的年、月
参数：
    @param {string}    timestamp    图片历史消息 sentTime 时间戳
*/
function timestampToTime(timestamp) {
    var date = new Date(timestamp);
    var year = date.getFullYear() + ' 年 ';
    var month = (date.getMonth() + 1 < 10 ? '0' + (date.getMonth() + 1) : date.getMonth() + 1) + ' 月';
    return year + month;
}
function getImageMessageList(list) {
    var imageMsgList = list.filter(function (item) {
        if (item.messageType === 'LocalImageMessage') {
            item.content.imageUri = Base64Util.concat(item.content.content);
        }
        var url = item.content.imageUri || item.content.sightUrl || (item.content.content || {}).imageUri;
        var isImage = item.messageType === 'ImageMessage' || item.messageType === 'LocalImageMessage';
        var isSight = item.messageType === 'SightMessage';
        var isQuoteimage = item.messageType === 'ReferenceMessage' && item.content.objName === 'RC:ImgMsg';
        if (!url && !isImage) {
            return false;
        }
        return isImage || isSight || isQuoteimage;
    });
    return imageMsgList;
}
}(RongIM, {
    jQuery: jQuery,
    moment: moment,
    RongIMClient: RongIMClient
}, RongIM.components));
