/*
说明： 消息列表
功能： 会话界面处理各种消息的展示、复制、删除、转发，图片文件的下载等
*/
(function (RongIM, dependencies, components) {
    'use strict';

    function transId(messageId) {
        return 'rong-message-' + messageId;
    }
    var Http = RongIM.dataModel._Http;
    var common = RongIM.common;
    var utils = RongIM.utils;
    var Base64Util = utils.Base64;
    var dialog = RongIM.dialog;
    var $ = dependencies.jQuery;
    var copyToClipboard = utils.copyToClipboard;
    var UploadClient = dependencies.UploadClient;
    var dataModel = RongIM.dataModel;
    var userApi = dataModel.User;
    var messageApi = dataModel.Message;
    var clipboard = RongIM.clipboard;
    var Cache = RongIM.dataModel._Cache;
    Cache.collectList = [];
    var supportMessageTypeList = [
        'TextMessage',
        'ImageMessage',
        'FileMessage',
        'VoiceMessage',
        'GroupMemChangedNotifyMessage',
        'GroupNotifyMessage',
        'GroupCmdMessage',
        'LocationMessage',
        'NotificationMessage',
        'RecallCommandMessage',
        'LocalFileMessage',
        'LocalImageMessage',
        'InformationNotificationMessage',
        'CardMessage',
        'SightMessage',
        'ContactNotifyMessage',
        'RichContentMessage',
        'ApprovalMessage',
        'GroupNoticeNotifyMessage',
        'JrmfRedPacketMessage',
        'JrmfRedPacketOpenedMessage',
        'PublicServiceRichContentMessage',
        'PublicServiceMultiRichContentMessage',
        'PublicServiceCommandMessage',
        'VideoSummaryMessage',
        'RealTimeLocationStartMessage',
        /* 远程控制结束的提示 */
        'RemoteControlSummaryMessage',
        'ReferenceMessage',
        'RequestFriendVerificationMessage'
    ];

    components.getMessageList = function (resolve, reject) {
        // 模块化屏蔽入口

        var im = RongIM.instance;
        var options = {
            name: 'message-list',
            template: '#rong-template-message-list',
            props: ['conversation', 'append-message', 'status', 'group', 'in-group', 'isBanned'],
            data: function () {
                var titleBarHeight = 55;
                return {
                    busy: false,
                    autoScroll: true,
                    fixOffset: {
                        top: titleBarHeight
                    },
                    messageList: [],
                    lastMessage: {},
                    hasMoreLast: true,
                    showAlt: false,
                    altVal: "",
                    altParams: null,
                    authId: im.auth.id,
                    showSmsMeu: false,
                    checkboxModel: [], //转发所选数组
                    isforwards: false //是否展示多选
                };
            },
            components: {
                avatar: components.getAvatar,
                RecallCommandMessage: components.getRecallCommandMessage,
                TextMessage: components.getTextMessage,
                ImageMessage: components.getImageMessage,
                LocalImageMessage: components.getImageMessage,
                FileMessage: components.getFileMessage,
                LocalFileMessage: components.getFileMessage,
                VoiceMessage: components.getVoiceMessage,
                LocationMessage: components.getLocationMessage,
                CardMessage: components.getCardMessage,
                SightMessage: components.getSightMessage,
                RichContentMessage: components.getRichContentMessage,
                ApprovalMessage: components.getApprovalMessage,
                UnknownMessage: components.getUnknownMessage,
                GroupNoticeNotifyMessage: components.getGroupNoticeNotifyMessage,
                JrmfRedPacketMessage: components.getJrmfRedPacketMessage,
                PublicServiceRichContentMessage: components.getPSImageTextMessage,
                PublicServiceMultiRichContentMessage: components.getPSMultiImageTextMessage,
                VideoSummaryMessage: components.getSummaryMessage,
                RemoteControlSummaryMessage: components.getRemoteSummaryMessage,
                ReferenceMessage: components.getQuoteMessage,
                RequestFriendVerificationMessage: components.getRequestFriendVerificationMessage
            },
            computed: {
                // 文件助手
                isFileHelper: function () {
                    var conversation = this.conversation || {};
                    var user = conversation.user || {};
                    return user.type === 3;
                },
                conversationType: function () {
                    return +this.$route.params.conversationType;
                },
                isPublic: function () {
                    return +this.$route.params.conversationType === utils.conversationType.APP_PUBLIC_SERVICE;
                },
                targetId: function () {
                    return this.$route.params.targetId;
                },
                /*
                显示新消息提示
                消息列表滚动条不在最底部时有新消息给出提示
                */
                hasNewMessageTip: function () {
                    return (!this.autoScroll) && (this.conversation.unreadMessageCount > 0);
                },
                disabled: function () {
                    return !this.inGroup;
                },
                /*
                url query 参数包含 messageUId 是由搜索消息跳转
                */
                isSearch: function () {
                    var query = this.$route.query;
                    return !utils.isEmpty(query.messageUId);
                },
                filtedMessageList: function () {
                    var authId = im.auth.id;
                    var messageList = this.messageList;
                    var group = this.conversation.group;
                    var now = Date.now().toString().substr(-6);
                    var that = this;
                    // 排除部分群通知 成员被踢出和退出时其他成员不显示
                    var list = messageList.filter(function (message, index) {
                        that.$set(message, "showAltName", false);
                        var isGroupMemberMessage = message.messageType === 'GroupMemChangedNotifyMessage';
                        if (isGroupMemberMessage) {
                            if (!group) {
                                return false;
                            }
                            // 操作行为类型
                            var messageAction = message.content.action;
                            // 群组减员通知：3，被移除；4，主动退出
                            if (messageAction === 3 || messageAction === 4) {
                                // 非操作者，后台操作时，操作者 id 为群主
                                var isNotOperatorUser = message.content.operatorUser.id !== authId;
                                // 非被操作者
                                var isNotTargetUsers = !message.content.targetUsers.some(function (item) {
                                    return item.id === authId;
                                });
                                var isManager = group.admin_id === authId;
                                // 无关消息，不显示
                                if (isNotOperatorUser && isNotTargetUsers && !isManager) {
                                    return false;
                                }
                                // 群组减员消息，只通知群主且群必须为自建群，部门群等不通知
                                return isManager && group.type === 0;
                            }
                            return true;
                        }
                        // 移动端仅有的消息类型
                        var isMobileMessage = message.messageType === 'RealTimeLocationStartMessage';
                        // 如果是发送给公众号 commandMessage，不显示
                        // 移动端进入应用公众号会发一条 ClickMenuMessage 消息
                        var isPublicServiceCommandMessage = message.messageType === 'PublicServiceCommandMessage' || message.objectName === 'RC:PSCmd';
                        // 排除 '可以在手机与电脑间互传文件与消息' 消息
                        var isInformationNotificationMessage = message.messageType === 'InformationNotificationMessage';
                        var isDeletePrompt = false;
                        if (isInformationNotificationMessage) {
                            var content = message.content.message || '';
                            var enDesc = RongIM.locale.en.components.getFileHelper.desc;
                            var zhDesc = RongIM.locale.zh.components.getFileHelper.desc;
                            isDeletePrompt = content === enDesc || content === zhDesc;
                        }
                        if (isPublicServiceCommandMessage || isMobileMessage || isDeletePrompt) {
                            return false;
                        }
                        if (!message.messageId) {
                            // 偶现 messageId 为 0 值导致页面渲染出现多个 id 相同的 DOM 元素，致使 Vue 渲染报错
                            // 故增加日志用于分析排查
                            // 修复撤回消息 messageId 为空的问题打开日志看是否有其他消息影响
                            RongIM.system.appLogger('error', 'messageId invalied!!!\nmessage: ' + JSON.stringify(message));
                            message.showId = message.showId || transId(now + index);
                        } else {
                            message.showId = transId(message.messageId);
                        }
                        return true;
                    });
                    console.log('messageList ====> ', this.messageList);
                    return list;
                },
                isWebPlatform: function () {
                    var platform = utils.getPlatform();
                    return platform.indexOf('web') !== -1;
                },
                // 未读消息提示的位置
                bottom: function () {
                    var node = im.resizeNode.messageInput;
                    var height = node.height;
                    return (height || 120) + 8;
                }
            },
            mixins: [getContextMenu()],
            watch: {
                $route: function () {
                    this.busy = false;
                    this.getMessage();
                },
                messageList: function () {
                    if (this.messageList.length === 0) {
                        return;
                    }
                    var newMessage = this.messageList.slice(-1)[0] || {};
                    newMessage.messageUId = newMessage.messageUId || (new Date()).getTime();
                    var hasNewMessage = this.lastMessage.messageUId !== newMessage.messageUId;
                    if (hasNewMessage && this.autoScroll && this.messageList.length > 300) {
                        this.messageList.splice(0, 100);
                    }
                    messageListChanged(this, im);
                },
                appendMessage: function (newMessage, oldMessage) {
                    console.log('appendMessage ===> ', newMessage, oldMessage);
                    var changValid = utils.isEmpty(oldMessage) || (newMessage.messageUId !== oldMessage.messageUId);
                    if (changValid) {
                        this.messageList.push(newMessage);
                    }
                }
            },
            created: function () {
                created(this);
                window.addEventListener('resize', this.handleResize);
            },
            beforeDestroy: function () {
                window.removeEventListener('resize', this.handleResize);
            },
            filters: {
                dateFormat: function (timestamp) {
                    // console.log("timestamp",timestamp);
                    return utils.dateFormat(timestamp, {
                        alwaysShowTime: true
                    });
                }
            }
        };
        options.methods = getMethods(im, options.components);
        utils.asyncComponent(options, resolve, reject);
    };
    /**
     * 查询消息内容是否可复制
     * @param {Message} message 消息内容
     * @param {(boolean) => void} callback 结果回调
     */
    function checkCopyEnable(message, callback) {
        var messageType = message.messageType;
        var isTxtMsg = [
            'TextMessage',
            'ReferenceMessage'
        ].indexOf(messageType) > -1;
        if (isTxtMsg) {
            callback(true);
            return;
        }
        var file = RongIM.file;
        var content = message.content;
        var isImgMsg = messageType === 'ImageMessage';
        if (isImgMsg) {
            common.localfileInDesk(content.imageUri, 'media', function (localPath) {
                if (localPath) {
                    message.localPath = localPath;
                }
                callback(!!localPath);
            });
            return;
        }
        var isFileMsg = messageType === 'FileMessage';
        if (isFileMsg) {
            var localPath = content.localPath;
            var fileExist = file.checkExist(localPath);
            callback(localPath && fileExist);
            return;
        }
        callback(false);
    }

    /*
    说明： 消息的右键菜单
    备注： 多图文消息比较特殊，需要解析每一项进行转发
    */
    function getContextMenu() {
        var im = RongIM.instance;
        var enableCollect = RongIM.config.modules.collect;
        var options = {
            template: 'templates/conversation/message-contextmenu.html',
            data: function () {
                return {
                    showCopy: false
                };
            },
            props: ['isBanned', 'context'],
            computed: {
                support: function () {
                    return im.config.support;
                },
                message: function () {
                    var context = this;
                    var message = this.context.message;
                    // 异步检查 copy 键是否显示
                    checkCopyEnable(message, function (bool) {
                        context.showCopy = bool;
                    });
                    return message;
                },
                messageContent: function () {
                    return this.message.content;
                },
                imageURL: function () {
                    return this.messageContent.imageUri;
                },
                fileURL: function () {
                    return this.messageContent.fileUrl;
                },
                filename: function () {
                    return this.messageContent.name;
                },
                canOpenFile: function () {
                    var localUrl = this.message.content.localPath || this.message.content.fileUrl;
                    return getFileExists(localUrl);
                },
                isSuccess: function () {
                    var sentStatus = this.message.sentStatus;
                    var read = sentStatus === utils.sentStatus.READ;
                    var unread = sentStatus === utils.sentStatus.SENT;
                    return utils.isEmpty(sentStatus) || read || unread;
                },
                showSave: function () {
                    var isImageMessage = this.message.messageType === 'ImageMessage';
                    return this.isSuccess && isImageMessage;
                },
                showRemind: function () {
                    return true
                },
                showDownload: function () {
                    var isFileMessage = this.message.messageType === 'FileMessage';
                    return this.isSuccess && isFileMessage;
                },
                showSaveAs: function () {
                    var isFileMessage = this.message.messageType === 'FileMessage';
                    var isImageMessage = this.message.messageType === 'ImageMessage';
                    return this.isSuccess && (isFileMessage || isImageMessage);
                },
                showForward: function () {
                    var excludeList = [
                        'VoiceMessage',
                        'VideoSummaryMessage',
                        'ApprovalMessage',
                        'JrmfRedPacketMessage'
                    ];
                    var show = excludeList.indexOf(this.message.messageType) < 0;
                    return this.isSuccess && show;
                },
                showJieLong: function () {
                    var show = this.message.messageType == 'RichContentMessage';
                    let jielongid = this.message.content.extra && this.message.content.extra.jielongid
                    return this.isSuccess && show && jielongid != '';
                },
                showRecall: function () {
                    var excludeList = [
                        'VideoSummaryMessage',
                        'JrmfRedPacketMessage'
                    ];
                    var show = excludeList.indexOf(this.message.messageType) < 0;
                    var isSent = this.message.messageDirection === 1;
                    var time = (new Date().getTime() - this.message.sentTime);
                    var isValidTime = time < RongIM.config.recallMessageTimeout;
                    var isPublic = this.message.conversationType === 7;
                    var isUploading = this.message.uploadStatus === utils.uploadStatus.UPLOADING;
                    return /*this.isSuccess &&*/ isValidTime && isSent && show && !isPublic && !isUploading;
                },
                showQuote: function () {
                    var message = this.message;
                    // 有效消息类型检测
                    var validMessage = [
                        'TextMessage',
                        'ImageMessage',
                        'FileMessage',
                        'RichContentMessage',
                        'ReferenceMessage'
                    ].some(function (type) {
                        return type === message.messageType;
                    });
                    if (!validMessage) {
                        return false;
                    }
                    // 有效会话类型检测
                    var conversationType = +this.message.conversationType;
                    // 私聊
                    if (conversationType === utils.conversationType.PRIVATE) {
                        var conversation = this.context.conversation;
                        // 文件小助手不显示引用按钮
                        var isRobot = (conversation.user.type === common.UserType.ROBOT);
                        return this.isSuccess && !isRobot;
                    }
                    // 群聊
                    if (conversationType === utils.conversationType.GROUP) {
                        // 被禁言时不显示引用按钮
                        return this.isSuccess && !this.isBanned;
                    }
                    return false;
                },
                showCollect: function () {
                    var excludeList = [
                        'BQMMEmojiMessage',
                        'BQMMGifMessage',
                        'VideoSummaryMessage',
                        'BQMMGifMessage',
                        'ApprovalMessage',
                        'JrmfRedPacketMessage',
                        'JrmfRedPacketOpenedMessage',
                        'GroupNoticeNotifyMessage',
                        'StickerMessage',
                        // 'PublicServiceRichContentMessage',
                        // 'PublicServiceMultiRichContentMessage',
                        'UnknownMessage',
                        'CardMessage',
                        'ReferenceMessage',
                        'RCBQMM:GifMsg',
                        'BQMMEmojiMessage'
                    ];
                    var show = excludeList.indexOf(this.message.messageType) < 0;
                    return this.isSuccess && show && enableCollect;
                },
                // 技术限制，暂时去掉语音类型的右键寻呼
                showPin: function () {
                    var excludeList = [
                        'TextMessage',
                        'ImageMessage',
                        'FileMessage',
                        'RichContentMessage',
                        'PublicServiceMultiRichContentMessage',
                        'PublicServiceCommandMessage'
                    ];
                    var show = excludeList.indexOf(this.message.messageType) != -1;

                    return show;
                },
                showOnePin: function () {
                    var excludeList = [
                        'TextMessage',
                        'ImageMessage',
                        'FileMessage',
                        'RichContentMessage',
                        'PublicServiceMultiRichContentMessage',
                        'PublicServiceCommandMessage'
                    ];
                    var show = excludeList.indexOf(this.message.messageType) != -1;
                    var isRobot = false
                    // 有效会话类型检测
                    var conversationType = +this.message.conversationType;
                    if (conversationType === utils.conversationType.PRIVATE) {
                        isRobot = (this.context.conversation.user.type === common.UserType.ROBOT);
                    }
                    return show && !isRobot;
                }
            },
            methods: {

                getDownload: function () {
                    if (this.message.messageType === 'ImageMessage') {
                        let index = this.messageContent.imageUri.lastIndexOf('/')
                        return this.messageContent.imageUri.slice(index)
                    } else {
                        return this.messageContent.name;
                    }

                },
                saveAs(event, url) {
                    this.close()
                    if (this.message.messageType === 'ImageMessage') {
                        event.target.href = this.message.content.imageUri;
                    } else {
                        event.target.href = url;
                    }
                },
                close: function () {
                    this.$emit('close');
                },
                open: function () {
                    this.$emit('open', this.context.message);
                },
                recall: function () {
                    this.$emit('recall', this.context.message);
                },
                copy: function () {
                    this.$emit('copy', this.context.message);
                },
                forward: function () {
                    console.log("转发内容===>", this.context.message || this.checkboxModel);
                    this.$emit('forward', this.context.message || this.checkboxModel);
                },
                forwards: function () {
                    this.$emit('forwards');
                },
                collect: function () {
                    this.$emit('collect', this.context.message);
                },
                oneClickSMS: function () {
                    console.log(123, this.context.message)
                },
                remove: function () {
                    this.$emit('remove', this.context.message);
                },
                download: function () {
                    this.$emit('download', this.context.message);
                },
                uploadMethod: function () {
                    var fileUrl
                    if(this.context.message.content.messageName == 'FileMessage'){
                      fileUrl = this.context.message.content.fileUrl;
                    }else if(this.context.message.content.messageName == "ImageMessage"){
                        fileUrl = this.context.message.content.imageUri;
                    }
                    var params = { fileUrl:fileUrl}
                    Http.get(`/user/uploadFileToYunPan`, params, function (ErrorCode) {
                        // console.log(ErrorCode)
                        if (ErrorCode && ErrorCode != 10000) {
                            common.messageToast({
                                message: '上传失败',
                                type: 'error'
                            });  
                        }else{
                            common.messageToast({
                                message: '上传成功',
                                type: 'success'
                            }); 
                        }
                    })
                },
                quote: function () {
                    this.$emit('quote', this.context.message);
                },
                remind: function () {
                    var dialog = RongIM.dialog;
                    const dataObj = {
                        data: this.context.message,
                        type: 'message'
                    }
                    dialog.addChooseDate(dataObj)
                },
                addPin: function () {

                    console.log(this.context.message)
                    this.$emit('addPin', this.context.message);
                },

                onePin: function () {
                    console.log(this.context.message)
                    this.$emit('onePin', this.context.message);

                    // this.$emit('onePin', this.context.message);
                },
                addapproval: function () {
                    this.$emit('addapproval', this.context.message);
                    var link = 'https://xh.cast.org.cn/default/oa/taskcenter/taskquery/sendRedirect.jsp?taskCenterRedirect=/freeflow/wap/launch&processdefId=1';
                    let url = ''
                    var loginInfo = utils.cache.get('loginInfo');
                    if (loginInfo && loginInfo.zkxToken) {
                        var token = decodeURIComponent(loginInfo.zkxToken) + ';path=/';
                        if (loginInfo.RCESESSIONID) {
                            // let urlParts = link.split('#');
                            // link += urlParts[0] + (
                            //     urlParts[0].indexOf('?') > 0 ? '&' : '?'
                            // ) + "sessionid=" + encodeURIComponent(loginInfo.RCESESSIONID) + "&uid=RY" + loginInfo.zkxEmpId+(
                            //     urlParts[1] ? ('#' + urlParts[1]) : ''
                            // );
                            link += "&sessionid=" + encodeURIComponent(loginInfo.RCESESSIONID) +
                                "&uid=RY" + loginInfo.zkxEmpId

                        }
                        // https: //xh.cast.org.cn/default/oa/taskcenter/taskquery/sendRedirect.jsp?taskCenterRedirect=/freeflow/wap/launch&processdefId=1
                        let append = ''

                        if (this.context.message.messageType == "TextMessage") {
                            //文字消息
                            let content = this.context.message.content.content
                            if (this.context.message.content.content.length > 200) {
                                content = content.slice(0, 200) + '...'
                            }
                            append += '&content=' + encodeURIComponent(content)
                        } else if (this.context.message.messageType == "FileMessage") {
                            //文件消息
                            append += '&enclosureUrl=' + encodeURIComponent(this.context.message.content.fileUrl)
                            append += '&enclosureName=' + encodeURIComponent(this.context.message.content.name)
                        } else if (this.context.message.messageType == "ImageMessage") {
                            //图片消息
                            append += '&enclosureUrl=' + encodeURIComponent(this.context.message.content.imageUri)
                            let start = this.context.message.content.imageUri.lastIndexOf('/') + 1
                            let imgName = this.context.message.content.imageUri.substring(start)
                            append += '&enclosureName=' + encodeURIComponent(imgName)
                        }
                        link += append;
                        url = RongIM.config.loginServer + '/sign-bridge.html' + '?target=' + encodeURIComponent(link) +
                            '&token=' + encodeURIComponent(token);
                    }
                    RongIM.browserWindow.openPSArticle(url)
                    // RongDesktop.browserWin.create({
                    //     width:1000,
                    //     minWidth: 800,
                    //     path: url,
                    // });
                    // window.open(url);
                }
            }
        };
        return components.getContextMenu(options);
    }

    /*
    说明： 消息列表改变事件 （收到新消息，获取历史消息时）
    功能： 判断是否需要展示消息时间戳
           如果消息列表滚动条在最底部立即清除未读消息
    */
    function messageListChanged(context, im) {
        var newMessage = context.messageList.slice(-1)[0] || {};
        newMessage.messageUId = newMessage.messageUId || (new Date()).getTime();
        var hasNewMessage = context.lastMessage.messageUId !== newMessage.messageUId;
        var fromMe = newMessage.messageDirection === 1;
        context.lastMessage = newMessage;
        if (hasNewMessage && fromMe) {
            context.autoScroll = true;
        }

        // 接收到新消息并且消息列表滚动条在最底部并且会话窗口处于激活状态，清除未读
        if (hasNewMessage && !fromMe && context.autoScroll && !im.hidden) {
            var conversationApi = im.dataModel.Conversation;
            conversationApi.clearUnReadCount(context.conversationType, context.targetId);
        }

        var otherInterval = 3 * 60 * 1000;
        var myInterval = 5 * 60 * 1000;
        var myAccountId = im.auth.id;
        var oldSentTime;
        context.messageList.forEach(function (item, index) {
            var support = supportMessageTypeList.indexOf(item.messageType) >= 0;
            if (!support) {
                item.messageType = 'UnknownMessage';
            }
            if (index === 0) {
                oldSentTime = item.sentTime;
                context.$set(item, '_showTime', true);
                return;
            }

            var interval = item.senderUserId === myAccountId ? myInterval : otherInterval;
            var sentTime = item.sentTime || (new Date()).getTime();
            if (sentTime - oldSentTime > interval) {
                oldSentTime = item.sentTime;
                context.$set(item, '_showTime', true);
            } else {
                context.$set(item, '_showTime', false);
            }
        });

        if (context.autoScroll) context.scrollToMessage('bottom', false);
    }

    function created(context) {
        var unwatch = context.$watch('status', function (status) {
            console.log('watch status ===> ', status);
            if (status === utils.status.CONNECTED) {
                unwatch();
                context.getMessage();
            }
        });

        context.getMessage();
    }

    function getMethods(im, childComponents) {
        return {
            userAlt: function (userItem) {
                if (this.altParams != null && this.altParams.showAltName) this.altParams.showAltName = false;
                userItem.showAltName = true;
                this.altParams = userItem;
                this.altVal = "@" + (userItem.user.alias || userItem.user.name);
            },
            NoForward() {
                console.log("取消");
                this.isforwards = false;
                this.checkboxModel = [];
            },
            YesForward() {
                if (!this.checkboxModel.length) {
                    return
                }
                console.log("发送");
                console.log(this.checkboxModel);
                this.forward(this.checkboxModel)
            },
            choice() {
                console.log(" this.checkboxModel", this.checkboxModel);
            },
            setAltName() {
                this.$emit("setSendVal", this.altParams)
            },
            setMessageList: function (messageList) {
                console.log('setMessageList ===> ', messageList);
                this.messageList = messageList;
            },
            reset: function () {
                this.setMessageList([]);
            },
            getMessage: function () {
                getMessage(this, dataModel.Message);
            },
            getHtmlUsername: function (user, msg) {
                var conversation = this.conversation || {};
                var group = conversation.group || {};
                return common.getHtmlGroupUsername(user, group.id, 16, msg.senderUserId);
            },
            getUsername: function (user) {
                var conversation = this.conversation || {};
                var group = conversation.group || {};
                return common.getGroupUsername(user, group.id);
                // if (group.id) {
                //     return common.getGroupUsername(user, group.id);
                // }
                // return common.getUsername(user);
            },
            getMessageType: function (item) {
                var messageType = item.messageType;
                var supported = childComponents[messageType];
                if (!supported) {
                    return 'UnknownMessage';
                }
                var map = {
                    LocalImageMessage: 'ImageMessage',
                    LocalFileMessage: 'FileMessage'
                };
                return map[item.messageType] || messageType;
            },
            isGroupNotificationMessage: function (message) {
                var NOTICE_ACTION = 3;
                var content = message.content || {};
                var isNotice = content.action === NOTICE_ACTION;
                var isGroupNotification = message.messageType === 'GroupNotifyMessage';
                if (isGroupNotification && isNotice) {
                    return false;
                }
                var list = [
                    'GroupMemChangedNotifyMessage',
                    'GroupNotifyMessage'
                ];
                return list.indexOf(message.messageType) >= 0;
            },
            getNotification: function (item) {
                return item.content.content;
            },
            getGroupNotification: function (item) {
                return common.getGroupNotification(item, im.auth.id);
            },
            isGroupSummary: function (message) {
                var isGroup = message.conversationType === utils.conversationType.GROUP;
                var isSummary = message.messageType === 'VideoSummaryMessage';
                return isGroup && isSummary;
            },
            getGroupSummary: function (message) {
                return common.groupSummaryFormat(message.content, this.locale);
            },
            getJrmfRedPacket: function (item) {
                return common.getJrmfRedPacket(item);
            },
            getJrmfRedPacketOpened: function (item) {
                return common.getJrmfRedPacketOpened(item, im.auth.id);
            },
            getContactNotification: function (item) {
                return common.getContactNotification(item.content, im.auth.id);
            },
            getInformationNotificationMessage: function (item) {
                var content = item.content.message;
                if (content.messageName === 'ContactNotifyMessage') {
                    return common.getContactNotification(content, im.auth.id);
                }
                return content;
            },
            /*
            获取消息的发送状态
            10 'SENDING'
            20 'FAILED'
            30 'SENT'
            40 'RECEIVED'
            50 'READ'
            60 'DESTROYED'
            */
            getMessageStatus: function (message) {
                if (utils.isEmpty(message.sentStatus)) {
                    // 接收消息的 sentStatus 为空
                    return undefined;
                }
                var status = utils.sentStatus[message.sentStatus];
                if (common.isCanceled(message)) {
                    status = 'SENT';
                }
                return status.toLowerCase();
            },
            isImageMessage: function (item) {
                return item.messageType === 'ImageMessage';
            },
            recall: function (message) {
                recall(message, dataModel.Message);
                this.$refs.contextmenu.close();
            },
            copy: function (message) {
                copy(this, message);
            },
            collect: function (message) {
                collect(this, message);
            },
            forward: function (message) {
                forward(this, message);
            },

            remove: function (message) {
                remove(this, message, dataModel.Message);
            },
            /* 下载图片 */
            download: function (message) {
                var imgUrl = message.content.imageUri;
                var _file = utils.getFilename(imgUrl);
                var filename = '';
                if (_file.ext === '') {
                    filename = utils.getDateId() + '.png';
                }
                var file = {
                    url: imgUrl,
                    name: filename
                };
                var downloader = RongIM.download(file);

                this.selectedTime = new Date().setSeconds(60);
                downloader.saveAs();

                this.closeContextmenu();
            },
            /* 文件消息打开文件所在文件夹 */
            open: function (message) {
                openFolder(message);
                this.closeContextmenu();
            },
            quote: function (message) {
                // 引用消息再次引用，只引用文本部分。
                var msg = message;
                if (msg.messageType === 'ReferenceMessage') {
                    var text = message.content.text;
                    msg = $.extend({}, message, {
                        messageType: 'TextMessage',
                        content: {
                            content: text
                        }
                    });
                }
                this.$emit('quote', msg);
                this.closeContextmenu();
            },
            userProfile: dialog.user,
            scrollToRecord: function () {
                var $content = $(this.$refs.content);
                var recordTop = this.recordTop;
                setTimeout(function () {
                    $content[0].scrollTop = recordTop;
                }, 300);
            },
            /*
            当input被拖动, 且会话也在底部时, 消息滚动到底部
            调用位置: message-input.js(inputBoxResize方法)
             */
            scrollWhenInputResize: function () {
                if (this.autoScroll) {
                    this.scrollToMessage('bottom');
                }
            },
            scrollToMessage: function (messageId, alignToTop) {
                var $content = $(this.$refs.content);
                console.log($content)

                function waitImageLoad() {
                    var el = document.getElementById(transId(messageId));
                    if (el) el.scrollIntoView(alignToTop);
                    if ($content) $content.css('visibility', 'visible');
                }
                setTimeout(waitImageLoad, 200);
            },
            scrollToNewMessage: function (unreadMessageCount) {
                scrollToNewMessage(this, unreadMessageCount);
            },
            scroll: utils.throttle(function (event) {
                // if(im.status === utils.status.CONNECTED) {
                if (RongIM.isAvailableData()) {
                    scroll(this, event);
                }
            }, 500),
            getHistory: utils.debounce(function (last) {
                getHistory(this, dataModel.Message, last);
            }, 500, true),
            clearUnReadCount: function () {
                if (this.altParams != null && this.altParams.showAltName) this.altParams.showAltName = false;
                clearUnReadCount(this, dataModel.Conversation, this.conversation, im);
            },
            /* 发送失败的消息点击重新发送 */
            reSendMessage: function (message) {
                if (message.sentStatus === utils.sentStatus.FAILED && this.inGroup) {
                    var api = {
                        message: dataModel.Message,
                        file: dataModel.File
                    };
                    reSendMessage(api, message, this);
                }
            },
            uploadCancel: function (message, messageList) {
                var _messageList = messageList || this.messageList;
                var index = _messageList.indexOf(message);
                _messageList.splice(index, 1);
            },
            isValidMessage: function (message) {
                return message.messageType !== 'UnknownMessage';
            },
            isLastMessage: function (message) {
                var lastMessage = this.messageList.slice(-1)[0] || message;
                return message.messageUId === lastMessage.messageUId;
            },
            /* 是否显示群消息回执 */
            showGroupResp: function (message) {
                var send = message.messageDirection === utils.messageDirection.SEND;
                var isGroup = message.conversationType === utils.conversationType.GROUP;
                return this.isValidMessage(message) && send && isGroup;
            },
            /* 是否显示单聊消息已读回执状态 */
            showPrivateResp: function (message) {
                if (message.senderUserId !== message.targetId) {
                    var excludeList = [
                        'VideoSummaryMessage',
                        'RemoteControlSummaryMessage'
                    ];
                    var show = excludeList.indexOf(message.messageType) === -1;
                    var send = message.messageDirection === utils.messageDirection.SEND;
                    var isPrivate = +message.conversationType === utils.conversationType.PRIVATE;
                    var conversation = this.conversation || {};
                    var user = conversation.user || {};
                    var isFileHelper = user.type === 3;
                    return this.isValidMessage(message) && send && isPrivate && !isFileHelper && show;
                } else {
                    return false
                }

            },
            fromMe: function (message) {
                return message.messageDirection === 1;
            },
            isUnRead: function (message) {
                return message.sentStatus === utils.sentStatus.SENT;
            },
            isRead: function (message) {
                return message.sentStatus === utils.sentStatus.READ;
            },
            /* 可发送群已读回执请求 显示 '查看未读' */
            ableSendGroupReq: function (message) {
                if (this.$route.query.messageUId) {
                    return false;
                }
                // 30 发送成功 40 对方已接收 50 对方已读 60 对方已销毁。 大于 30 都记为发送成功
                var isSendSuccess = message.sentStatus >= utils.sentStatus.SENT;
                var isSendLastMessage = isLatestSentMessage(message, this.messageList);
                return utils.isEmpty(message.receiptResponse) && isSendSuccess && isSendLastMessage;
            },
            sendGroupReq: function (message) {
                Vue.set(message, 'receiptResponse', []);
                dataModel.Message.sendGroupRequest(message);
            },
            /* 存在群消息已读回执数据 */
            hasGroupResp: function (message) {
                return $.isArray(message.receiptResponse);
            },
            /* 获取群消息未读具体人数 */
            getUnreadCount: function (message) {
                var unreadMember = getUnreadMember(this.conversation.group, message, im.auth.id);
                uploadUnReadMember(message, unreadMember, im);
                var unreadCount = unreadMember ? unreadMember.length : -1;
                return unreadCount;
            },
            /* 展示群消息已读回执详情 */
            showUnreadMember: function (message) {
                var readMember = (message.receiptResponse).filter(function (item) {
                    return item !== im.auth.id;
                });
                var unreadMember = getUnreadMember(this.conversation.group, message, im.auth.id).map(function (item) {
                    return item.id;
                });
                var arr = unreadMember.concat(readMember);
                var group = this.conversation.group;
                userApi.getBatch(arr, function (errorCode, userList) {
                    dialog.ack(userList, readMember, message.messageId, group.id);
                });
            },
            handleResize: function () {
                this.closeContextmenu();
            },
            showImage: function (message) {
                common.showImage(getImageMessageList(this.filtedMessageList), message.messageUId, im.config.locale);
            },
            showSight: function (message) {
                this.showImage(message);
            },
            sendRichContentMessage: function (message) {
                console.log(message);

                this.$emit("sendRichContentMessage", message)

            },
            /* 自动播放未读语音的回调 */
            autoPlay: function (context) {
                var msg = getNextUnreadVoice(this, context);
                im.$emit('voicemessage.autoplay', msg);
            },
            addPin: function (msg) {
                if (msg.conversationType === utils.conversationType.GROUP) {
                    msg.groupMembers = this.conversation.group.groupMembers;
                    dialog.addPin(msg);
                    // dialog.newAddPin(msg);
                    // dialog.pinDialogBox(msg);
                } else {
                    userApi.get(msg.targetId, function (err, user) {
                        msg.targetUser = user;
                        dialog.addPin(msg);
                        // dialog.newAddPin(msg);
                        // dialog.pinDialogBox(msg);
                    });
                }
            },
            // 获取附件
            getMsgFile: function (msg) {
                if (msg.attachments) {
                    return msg.attachments;
                }

                if (msg.content) {
                    var uri = msg.content.fileUrl || msg.content.imageUri;
                    if (uri) {
                        return [{
                            name: msg.content.name || uri.substr(uri.lastIndexOf('/') + 1),
                            size: msg.content.size || msg.content.content.length * 0.75,
                            mime_type: msg.content.type || uri.substr(uri.lastIndexOf('.') + 1),
                            url: uri
                        }];
                    }
                }
                return [];
            },
            // 获取content
            getMsgText: function (msg) {
                if (msg && msg.content) {
                    if (msg.messageType === 'ReferenceMessage') {
                        return msg.content.text;
                    }

                    if (typeof msg.content.content === 'string') {
                        var isInvalidType = [
                            'ImageMessage',
                            'LocationMessage',
                            'VoiceMessage',
                            'SightMessage'
                        ].some(function (invalidType) {
                            return msg.messageType === invalidType;
                        });

                        if (!isInvalidType) {
                            return msg.content.content;
                        }
                    }
                }
                return '';
            },
            forwards: function () {
                this.isforwards = true
                this.closeContextmenu();
            },
            // 一键寻呼
            onePin: function (msg) {
                console.log(msg);
                this.closeContextmenu();
                if (msg.conversationType == 3) {
                    var message = '是否确认发送寻呼 (含短信) 给群内所有人员？'
                } else {
                    var message = '是否确认寻呼(含短信)？'
                }
                common.messagebox({
                    type: 'confirm',
                    message: message,
                    callback: () => {
                        let users = []
                        if (msg.conversationType === utils.conversationType.GROUP) {
                            users = this.conversation.group.members;
                        } else {
                            userApi.get(msg.targetId, function (err, user) {
                                users.push(user);
                            });
                        }
                        let data = {
                            "title": "",
                            "content": this.getMsgText(msg) || '仅附件',
                            "send_see": 0,
                            "send_sms": true,
                            "send_email": false,
                            "delayed": false,
                            "pin_type": 0,
                            "receiver_ids": users.map(obj => obj.id),
                            "attachments": this.getMsgFile(msg)
                        }
                        var pinApi = dataModel.Pin;
                        pinApi.create(data, function (errorCode, result) {
                            if (errorCode) {
                                common.toastError(errorCode);
                                return;
                            }
                            common.messagebox({
                                message: RongIM.locale.zh.components.newPin.successToPinPreview,
                                callback: function () {
                                    this.show = false;
                                },
                                closeCallback: function () {
                                    this.show = false;
                                },
                                hashchangeClose: false
                            });
                        });
                    }
                });

            }
        };
    }

    function getImageMessageList(list) {
        var imageMsgList = list.filter(function (item) {
            if (item.messageType === 'LocalImageMessage') {
                item.content.imageUri = Base64Util.concat(item.content.content);
            }
            var url = item.content.imageUri || item.content.sightUrl || (item.content.content || {}).imageUri;
            var isImage = item.messageType === 'ImageMessage' || item.messageType === 'LocalImageMessage';
            var isSight = item.messageType === 'SightMessage';
            var isQuoteimage = item.messageType === 'ReferenceMessage' && item.content.objName === 'RC:ImgMsg';
            if (!url && !isImage) {
                return false;
            }
            return isImage || isSight || isQuoteimage;
        });
        return imageMsgList;
    }
    // 获取下一条未读语音
    function getNextUnreadVoice(currentContext, context) {
        var list = currentContext.filtedMessageList;
        var message = context.message;

        var result = null;
        var after = false;
        for (var i = 0, len = list.length; i < len; i += 1) {
            var item = list[i];
            if (after) {
                var RECEIVE = utils.messageDirection.RECEIVE;
                var LISTENED = utils.receivedStatus.LISTENED;
                var isReceiver = item.messageDirection === RECEIVE;
                var isVoice = item.messageType === 'VoiceMessage';
                var unListened = item.receivedStatus !== LISTENED;
                if (isReceiver && isVoice && unListened) {
                    result = item;
                    break;
                }
            } else if (common.equalMessage(item, message)) {
                after = true;
            }
        }
        return result;
    }
    /*
    说明： 判断传入消息是否为自己发送的最后一条消息
    参数：
        @param {object} message     需要判断的消息
        @param {array}  messageList 从此消息列表中匹配
    */
    function isLatestSentMessage(message, messageList) {
        var latestSentMessage = getLatestSentMessage(messageList);
        if (!latestSentMessage) {
            return false;
        }
        return common.equalMessage(latestSentMessage, message);
    }
    /*
    说明： 获取自己发送的最后一条消息
    */
    function getLatestSentMessage(list) {
        var sentList = list.filter(function (item) {
            var isSend = item.messageDirection === utils.messageDirection.SEND;
            var notNotify = ['GroupMemChangedNotifyMessage',
                'GroupNotifyMessage',
                'GroupCmdMessage',
                'GroupVerifyNotifyMessage'
            ].indexOf(item.messageType) === -1;
            return isSend && notNotify;
        });
        return sentList[sentList.length - 1];
    }

    /*
    说明： 获取群消息回执未读成员
    参数：
        @param {object} group   群组信息
        @param {object} message 群消息回执指定消息（回执详情保存在 消息的 receiptResponse 属性中）
        @param {string} selfId  自己的 id （需要排除自己）
    */
    function getUnreadMember(group, message, selfId) {
        var unreadMember = null;
        if (group) {
            var readMember = message.receiptResponse || [];
            var timestamp = message.sentTime;
            var briefMember = group.member_brief || [];
            unreadMember = briefMember.filter(function (item) {
                var valid = item.createDt < timestamp;
                var unread = readMember.indexOf(item.id) === -1;
                var notSelf = selfId !== item.id;
                return valid && unread && notSelf;
            });
        }
        return unreadMember;
    }

    function uploadUnReadMember(message, unreadMember, im) {
        var readMember = message.receiptResponse;
        unreadMember = (unreadMember || []).map(function (item) {
            return item.id;
        });
        var arr = unreadMember.concat(readMember);
        userApi.getBatch(arr, function (errorCode, userList) {
            im.$emit('messageUnRead' + message.messageId, {
                memberList: userList,
                readIdList: readMember
            });
        });
    }

    /*
    说明： 获取会话消息 (进入会话第一次获取消息)
    */
    function getMessage(context, messageApi) {
        /**
         * PC 中断开连接任可获取历史消息。 8, 30010 为重定向 navi 不可用重新获取 navi
         * web 中断开连接状态为 utils.status.NETWORK_UNAVAILABLE 不可继续获取消息
         */
        var validStatus = [
            utils.status.CONNECTED,
            utils.status.DISCONNECTED,
            8,
            30010
        ];
        var invalid = validStatus.indexOf(context.status) === -1;
        if (invalid || context.busy) {
            return;
        }
        context.autoScroll = true;
        context.busy = true;
        context.hasMoreLast = true;

        // 切换会话后加载消息列表，等消息列表滚动后再显示，避免视觉上的抖动
        var $content = $(context.$refs.content);

        $content.css('visibility', 'hidden');

        var query = context.$route.query;
        if (!context.isSearch) {
            messageApi.get({
                conversationType: context.conversationType,
                targetId: context.targetId,
                position: 1,
                timestamp: 0
            }, function (errorCode, list) {
                context.busy = false;
                if (errorCode) {
                    common.toastError(errorCode);
                    return;
                }
                context.$nextTick(function () {
                    console.log(messageApi);
                    console.log("list", list);
                    context.setMessageList(list);
                    context.lastMessage = context.messageList.slice(-1)[0] || {};
                    context.scrollToMessage('bottom');
                });
            });
        } else {
            messageApi.getMessageNearList(query.messageUId, function (errorCode, list, msg) {
                context.busy = false;
                if (errorCode) {
                    common.toastError(errorCode);
                    return;
                }
                context.$nextTick(function () {
                    context.setMessageList(list);
                    context.lastMessage = context.messageList.slice(-1)[0] || {};
                    context.autoScroll = false;
                    context.scrollToMessage(msg.messageId);
                })
            });
        }
    }

    function scroll(context, event) {
        var up = event.deltaY < 0;
        var $content = $(context.$refs.content);
        if (up && $content.scrollTop() <= 0) {
            context.getHistory();
            return;
        }
        var bottom = ($content.scrollTop() + $content.height()) >= $content[0].scrollHeight;
        if (context.isSearch && bottom) {
            context.recordTop = $content.scrollTop();
            context.getHistory(true);
            return;
        }
        if (!context.isSearch) {
            var DISTANCE_TO_BOTTOM = 50;
            var autoScroll = $content[0].scrollHeight - $content.height() - $content.scrollTop() < DISTANCE_TO_BOTTOM;
            if (autoScroll) context.clearUnReadCount();
            context.autoScroll = autoScroll;
        }
    }

    /*
    说明： 获取历史消息（向上滚动向下滚动时）
    参数：
        @param {object}  context    上下文
        @param {object}  messageApi Message 相关操作 API
        @param {boolean} last       true 获取最新的 （向下滚动获取） false 获取更早的消息 （向上滚动获取）
    */
    function getHistory(context, messageApi, last) {
        // var disconnected = context.status !== utils.status.CONNECTED;
        if (context.busy || !RongIM.isAvailableData()) {
            return;
        }
        if (last && !context.hasMoreLast) {
            context.$router.push({
                name: 'conversation',
                params: {
                    targetId: context.targetId,
                    conversationType: context.conversationType
                }
            });
        }

        context.busy = true;
        context.autoScroll = false;
        var firstMessage = context.messageList[0];
        var sentTime = context.messageList.length > 0 ? context.messageList[0].sentTime : 0;
        if (last) {
            sentTime = context.messageList[context.messageList.length - 1].sentTime;
        }
        var params = {
            conversationType: context.conversationType,
            targetId: context.targetId,
            position: 2,
            timestamp: sentTime,
            before: !last
        };
        messageApi.get(params, function (errorCode, list, hasMore) {
            context.busy = false;
            if (last) {
                context.hasMoreLast = hasMore;
                list.reverse().forEach(function (item) {
                    context.messageList.push(item);
                });
                context.scrollToRecord();
            } else if (list.length !== 0) {
                list.reverse().forEach(function (item) {
                    context.messageList.unshift(item);
                });
                if (firstMessage) context.scrollToMessage(firstMessage.messageId);
            }
        });
    }

    /*
    说明： 撤回消息
    参数：
        @param {object} message    要撤回的消息
        @param {object} messageApi Message 相关 API
    */
    function recall(message, messageApi) {
        var time = new Date().getTime() - message.sentTime;
        var isValidTime = time < RongIM.config.recallMessageTimeout;
        if (isValidTime) {
            messageApi.recall(message, function () {
                // 撤回成功
                // 通知图片查看器
                if (message.messageType === utils.messageType.ImageMessage) {
                    if (RongIM.imageViewer) RongIM.imageViewer.recall(message.messageUId);
                }
            });
        } else {
            var errMsg = common.getErrorMessage('message-recall-timeout');
            common.messageToast({
                type: 'error',
                message: errMsg
            });
        }
    }

    /*
    说明： 判断页面是否有选中的区域
    */
    function isValidSelect(context, message) {
        var sel = window.getSelection();
        var selText = window.getSelection().toString();
        var hasContextMenu = sel.containsNode(context.$refs.contextmenu.$el);
        var isValid = false;
        var copyContentText = ['TextMessage', 'GroupNoticeNotifyMessage', 'ReferenceMessage'];
        if (copyContentText.indexOf(message.messageType) !== -1 && selText.length && !hasContextMenu) {
            isValid = true;
        }
        return isValid;
    }

    /*
    说明： 右键复制消息
           判断页面是否有选中文字区域有则复制选中区域到剪切板
    */
    function copy(context, message) {
        message.content.messageName = message.messageType;
        if (isValidSelect(context, message)) {
            document.execCommand('copy');
        } else {
            var str = JSON.stringify(message.content);
            var type = message.messageType;
            var localPath = '';
            if (type === 'TextMessage') {
                str = message.content.content;
                copyToClipboard(str);
            } else if (type === 'ReferenceMessage') {
                str = message.content.text;
                copyToClipboard(str);
            } else if (type === 'GroupNoticeNotifyMessage') {
                str = common.groupNoticeFormat(message.content.content, context.locale);
                copyToClipboard(str);
            } else if (type === 'ImageMessage') {
                localPath = message.localPath;
                clipboard.writePath(localPath);
            } else if (type === 'FileMessage') {
                localPath = message.content.localPath;
                clipboard.writePath(localPath);
            } else {
                copyToClipboard(str);
            }

            // var copyContentText = ['TextMessage', 'GroupNoticeNotifyMessage'];
            // if (copyContentText.indexOf(message.messageType) !== -1) {
            //     str = message.content.content;
            // }
        }
        context.closeContextmenu();
    }

    /*
    说明： 右键消息转发
    */
    function forward(context, message) {
        // RongIM.dialog.newForward(message);
        // RongIM.dialog.forward(message);
        // if (message.length) {
        //     for ( var i = 0,l = message.length; i < l; i++ ){
        //         RongIM.dialog.forwardDialog(message[i], "forward");
        //         } 
        // }
        console.log('forward =====================> ', message);
        // let msgArr = [
        //   {
        //     messageName: 'TextMessage',
        //     content: '111222'
        //   },
        //   {
        //     messageName: 'TextMessage',
        //     content: '444455555'
        //   }
        // ];
        // let msgContent = msgArr.map(m => {
        //   let msg = messageApi.create({
        //     messageType: m.messageName,
        //     content: m
        //   });
        //   if (msg.messageName === utils.messageType.FileMessage) {
        //     msg.localPath = m.localPath;
        //   }
        //   return msg;
        // });
        // let extMsg = {
        //   content: {
        //     messageName: utils.messageType.ExtendMessage,
        //     content: JSON.stringify(msgContent)
        //   }
        // };
        // RongIM.dialog.forwardDialog(extMsg, "forward");
        RongIM.dialog.forwardDialog(message, "forward");
        context.isforwards = false;
        context.checkboxModel = [];
        context.closeContextmenu();
    }

    /*
    说明：右键消息收藏
    */
    function collect(context, message) {
        var param = {
            message: context.locale.tips.collected,
            callback: function () { },
            el: context.$el
        };
        var type = message.objectName || message.messageType;

        var senderId = message.senderUserId;

        var content = message.content;

        var targetId = message.conversationType === 3 ? message.targetId : message.senderUserId;

        var content_id = message.messageUId;
        var searchContent = '';
        if (message.objectName === 'RC:TxtMsg') {
            searchContent = message.content.content;
        } else if (message.objectName === 'RC:FileMsg') {
            searchContent = message.content.name;
        } else if (message.objectName === 'RC:LBSMsg') {
            searchContent = message.content.poi;
        } else if (message.objectName === 'RC:ImgTextMsg') {
            searchContent = message.content.title;
        }
        var sourceType;
        if (+message.conversationType === 1) {
            sourceType = 0;
        } else if (+message.conversationType === 3) {
            sourceType = 1;
        } else if (+message.conversationType === 7) {
            sourceType = 3;
        }
        if (type === 'ImageMessage') {
            type = 'RC:ImgMsg';
        }
        var params = {
            scope: 'message',
            type: type,
            search_content: searchContent,
            fav_content: {
                sender_id: senderId,
                source_type: sourceType,
                target_id: targetId,
                content_id: content_id,
                url: '',
                content: JSON.stringify(content)
            }
        };
        if (message.type === 'RC:ImgTextMsg') {
            params = message;
        }
        dataModel.Collect.add(params, function (errorCode, result) {
            if (errorCode) {
                common.toastError(errorCode);
                context.closeContextmenu();
                return;
            }
            if (result) {
                common.messageToast(param);
                context.closeContextmenu();
                var typeList = dataModel.Collect.typeList;
                if (typeList.length === 0) {
                    params = {
                        version: -1,
                        scope: 'message'
                    };
                    dataModel.Collect.getList(params, function (errcode, results) {
                        if (!errcode) {
                            var arr = results.map(function (item) {
                                return item.objectName;
                            });
                            dataModel.Collect.typeList = arr.filter(function (x, index, self) {
                                return self.indexOf(x) === index;
                            });
                        }
                    });
                } else {
                    var iType = type || message.type;
                    if (typeList.indexOf(iType) === -1) {
                        typeList.push(iType);
                    }
                }
            }
        });
    }
    /*
    说明： 右键删除消息
    */
    function remove(context, message, messageApi) {
        // 上传文件消息删除时取消上传
        var isUploadFileMessage = [
            'LocalFileMessage',
            'LocalImageMessage',
            'FileMessage',
            'SightMessage',
            'ImageMessage'
        ].indexOf(message.messageType) > -1 && $.isFunction(message.cancel);
        if (isUploadFileMessage) {
            message.cancel();
        }

        if (message.messageId) {
            var params = {
                conversationType: message.conversationType,
                targetId: message.targetId,
                messageIds: [message.messageId]
            };
            messageApi.removeLocal(params, function () {
                // 删除成功  搜索跳转的消息不是同一个数据源,需要单独删除
                spliceMessage(context.messageList, message.messageId);
                messageApi.saveRemovedEarliestMessageTime(message);
            });
        } else if (message.messageUId && message.objectName === 'LRC:fileMsg') {
            spliceMessage(context.messageList, message.messageUId);
        } else {
            utils.console.log('没有 messageId web 端不支持删除');
        }
        context.closeContextmenu();
    }

    /*
    说明： 根据 messageId 删除缓存中对应消息
    参数：
        @param {array}  cacheList 缓存消息列表
        @param {string} messageId 消息 messageId
    */
    function spliceMessage(cacheList, messageId) {
        if (!cacheList) {
            return;
        }
        var index = null;
        for (var i = 0, len = cacheList.length; i < len; i += 1) {
            var cacheMsg = cacheList[i];
            if (cacheMsg.messageId === messageId || cacheMsg.messageUId === messageId) {
                index = i;
            }
        }

        if (index === null) {
            return;
        }
        cacheList.splice(index, 1);
    }

    function clearUnReadCount(context, conversationApi, conversation) {
        if (conversation.unreadMessageCount > 0) {
            conversationApi.clearUnReadCount(context.conversationType, context.targetId);
        }
    }

    /*
    说明： 新消息提示，点击时消息列表滚动到最底部
    */
    function scrollToNewMessage(context, unreadMessageCount) {
        context.clearUnReadCount();
        if (context.isSearch) {
            context.$router.push({
                name: 'conversation',
                params: {
                    targetId: context.targetId,
                    conversationType: context.conversationType
                }
            });
        } else {
            var index = context.messageList.length - unreadMessageCount;
            var messageId = context.messageList[index].messageId;
            context.scrollToMessage(messageId);
            context.autoScroll = true;
        }
    }

    /*
    说明： 重新发送消息(消息发送失败重发)
    参数：
        @param {object} api         包含 File 和 Message 的 API
        @param {object} message     需要重发的消息
        @param {object} context     发送消息失败不在群组中时通知上层组件
    */
    function reSendMessage(api, message, context) {
        var messageApi = api.message;
        var content = reMessage(message, api, context);
        if (utils.isEmpty(content)) {
            // utils.console.log('此消息暂不支持重发:', message);
            return;
        }

        var params = {
            notNotify: true,
            targetId: message.targetId,
            conversationType: message.conversationType
        };
        params.messageIds = [message.messageId];
        messageApi.removeLocal(params, function () {
            messageApi.saveRemovedEarliestMessageTime(message);
        });
        messageApi.send({
            conversationType: message.conversationType,
            targetId: message.targetId,
            content: content,
            mentiondMsg: !utils.isEmpty(content.mentionedInfo)
        }, function (errcode) {
            if (errcode) {
                var errMsg = common.getErrorMessage(errcode);
                if (errcode === 'lib-' + RongIMLib.ErrorCode.NOT_IN_GROUP) {
                    var notificationMsg = common.createNotificationMessage(message.conversationType, message.targetId, errMsg);
                    messageApi.insertMessage(notificationMsg);
                    context.$emit('setInGroup', false);
                }
            }
        });
    }

    /*
    说明： 重发消息
    功能： 不同消息类型重发操作不同
           文本消息直接重新发送文本
           图片消息判断是否已上传成功未上传成功需要先上传再发送
           文件消息和图片消息一样需要先判断是否上传成功
    */
    function reMessage(message, api, context) {
        var msg = null;
        var isValidUrl = function (url) {
            if (url.indexOf('http') > -1) {
                return true;
            }
            return false;
        };
        var _url;
        switch (message.messageType) {
            case 'TextMessage':
                msg = new RongIM.dataModel.Message.TextMessage(message.content);
                break;
            case 'ImageMessage':
                _url = message.content.imageUri;
                if (_url && isValidUrl(_url)) {
                    msg = new RongIM.dataModel.Message.ImageMessage(message.content);
                } else if (typeof message.data === 'string') {
                    uploadBase64(message, context);
                } else {
                    message.content.localPath = message.data.path;
                    reSendLocalFile(context, message, api);
                }
                break;
            case 'LocalImageMessage':
                reSendLocalFile(context, message, api);
                break;
            case 'FileMessage':
                _url = message.content.fileUrl;
                if (_url && isValidUrl(_url)) {
                    msg = new RongIM.dataModel.Message.FileMessage(message.content);
                } else {
                    reSendLocalFile(context, message, api);
                }
                break;
            case 'LocalFileMessage':
                reSendLocalFile(context, message, api);
                break;
            default:
                $.noop();
                break;
        }
        return msg;
    }

    /*
    说明： 消息发送失败，文件为上传成功时重新上传
    */
    function reSendLocalFile(context, message, api) {
        var path = message.content.localPath || message.localPath;
        var file = RongIM.file;
        // var uploadFile = file.getBlobByPath(path);
        var uploadFile = file.getBlobs([path])[0];

        var messageApi = api.message;
        var params = {};
        if (uploadFile) {
            // 删除原来消息
            params = {
                targetId: message.targetId,
                conversationType: message.conversationType
            };
            params.messageIds = [message.messageId];
            messageApi.removeLocal(params, function () {
                messageApi.saveRemovedEarliestMessageTime(message);
            });

            // 重新上传
            params = {
                targetId: message.targetId,
                conversationType: message.conversationType,
                data: uploadFile,
                localPath: path
            };
            var config = RongIM.config.upload.file;
            var uploadMessage = api.file.createUploadMessage(params);
            uploadMessage.content.localPath = path;
            upload(context, uploadMessage, config, api);
        } else {
            var type = message.conversationType;
            var id = message.targetId;
            var fileUnexist = context.locale.tips.fileUnexist;
            params = common.createNotificationMessage(type, id, fileUnexist);
            messageApi.insertMessage(params);
        }
    }

    /*
    说明： 上传 base64 图片消息
    */
    function uploadBase64(message, context) {
        var base64Str = message.data;
        var fileApi = dataModel.File;
        var messageApi = dataModel.Message;

        var im = RongIM.instance;
        var base64Config = $.extend({}, im.config.upload.base64);
        var size = utils.getBase64Size(base64Str);
        var base64Size = base64Config.size;
        if (size > base64Size) {
            var tip = utils.templateFormat(context.locale.components.messageInput.screenshotMaxSize, parseInt(base64Size / 1024) + 'KB');
            common.messageToast({
                type: 'error',
                message: tip
            });
            return;
        }
        // 删除原消息
        messageApi.removeLocal({
            targetId: message.targetId,
            conversationType: message.conversationType,
            messageIds: [message.messageId]
        }, function () {
            messageApi.saveRemovedEarliestMessageTime(message);
        });
        // 重新上传
        var params = {
            targetId: message.targetId,
            conversationType: message.conversationType,
            data: base64Str
        };
        var uploadMessage = fileApi.createUploadMessage(params);
        base64Config.data = UploadClient.dataType.data;
        var api = {
            file: fileApi,
            message: messageApi
        };
        upload(context, uploadMessage, base64Config, api);
    }

    /*
    说明： 上传
    */
    function upload(context, uploadMessage, config, api) {
        var fileApi = api.file;
        var messageApi = api.message;
        fileApi.upload(uploadMessage, config, function (errorCode, _uploadMessage, data) {
            if (errorCode) {
                return;
            }
            fileApi.addFileUrl(_uploadMessage, data, function (error, message) {
                fileApi.send(message, function (err, msg) {
                    if (err) {
                        var errMsg = common.getErrorMessage(err);
                        if (err === RongIMLib.ErrorCode.NOT_IN_GROUP) {
                            var params = common.createNotificationMessage(msg.conversationType, msg.targetId, errMsg);
                            messageApi.insertMessage(params);
                            context.$emit('setInGroup', false);
                        }
                    } else {
                        var im = RongIM.instance;
                        im.$emit('messagechange');
                        im.$emit('sendMessage');
                    }
                });
            });
        });
    }

    function getFileExists(fileUrl) {
        var file = RongIM.file;
        var existed = false;
        if (!utils.isEmpty(fileUrl)) {
            existed = file.checkExist(fileUrl);
        }
        return existed;
    }

    function openFolder(message) {
        var file = RongIM.file;
        var localUrl = message.content.localPath;
        var url = localUrl || message.content.fileUrl;
        var localPath = file.checkExist(url);
        if (localPath) {
            file.openDir(localUrl);
        } else {
            common.toastError('file-404');
        }
    }
}(RongIM, {
    UploadClient: UploadClient,
    jQuery: jQuery
}, RongIM.components));