/* datamodel 的核心,其他 model 均会用到这里的接口 */
(function (RongIM, dependencies) {
'use strict';

var win = dependencies.win;
var utils = {};
var store = RongIM.utils.cache;
var $ = dependencies.jQuery;
var RongIMClient = dependencies.RongIMClient;
var RongIMLib = dependencies.RongIMLib;

var friendApi = null;
var starApi = null;
var pinApi = null;
var remoteCtrlApi = null;
var groupNtcApi = null;
var orgApi = null;
var statusApi = null;
var groupApi = null;
var userApi = null;
var conversationApi = null;
var messageApi = null;
var fileApi = null;
var contactApi = null;
var publicApi = null;

var apiList = [];
function loadApi() {
    var dataModel = RongIM.dataModel;
    userApi = dataModel.User;
    groupApi = dataModel.Group;
    conversationApi = dataModel.Conversation;
    statusApi = dataModel.Status;
    messageApi = dataModel.Message;
    orgApi = dataModel.Organization;
    friendApi = dataModel.Friend;
    starApi = dataModel.Star;
    pinApi = dataModel.Pin;
    groupNtcApi = dataModel.GroupNotice;
    fileApi = dataModel.File;
    contactApi = dataModel.Contact;
    remoteCtrlApi = dataModel.RemoteControl;
    publicApi = dataModel.Public;

    apiList = [userApi, groupApi, conversationApi, statusApi, messageApi, orgApi];
    apiList = apiList.concat([friendApi, starApi, pinApi, groupNtcApi]);
    apiList = apiList.concat([fileApi, contactApi, remoteCtrlApi, publicApi]);
}

var config = {
    dataModel: {
        getHistoryMessagesNumber: 10,
        // getUsers 接口合并数据的个数
        requestCount: 50,
        // getUsers 等待合并数据等待时长，单位：毫秒
        waitTime: 50
    }
};

$.extend(true, config, RongIM.config);

var checkIndexOutBound = function (index, bound) {
    return index > -1 && index < bound;
};

function ObserverList() {
    this.observerList = [];
}

ObserverList.prototype.add = function (observer, force) {
    if (force) {
        this.observerList.length = 0;
    }
    this.observerList.push(observer);
};

ObserverList.prototype.get = function (index) {
    if (checkIndexOutBound(index, this.observerList.length)) {
        return this.observerList[index];
    }
    return undefined;
};

ObserverList.prototype.count = function () {
    return this.observerList.length;
};

ObserverList.prototype.removeAt = function (index) {
    if (checkIndexOutBound(index, this.observerList.length)) this.observerList.splice(index, 1);
};

ObserverList.prototype.remove = function (observer) {
    if (!observer) {
        this.observerList.length = 0;
        return;
    }
    var observerList = Object.prototype.toString.call(observer) === '[object Function]' ? [observer] : observer;
    for (var i = 0, len = this.observerList.length; i < len; i += 1) {
        for (var j = 0; j < observerList.length; j += 1) {
            if (this.observerList[i] === observerList[j]) {
                this.removeAt(i);
                break;
            }
        }
    }
};

ObserverList.prototype.notify = function (val) {
    for (var i = 0, len = this.observerList.length; i < len; i += 1) {
        this.observerList[i](val);
    }
};

ObserverList.prototype.indexOf = function (observer, startIndex) {
    var i = startIndex || 0;
    var len = this.observerList.length;
    while (i < len) {
        if (this.observerList[i] === observer) {
            return i;
        }
        i += 1;
    }
    return -1;
};

var messageQueueInterval = 0;
var Cache = {
    cleanMessageQueue: function () {
        var CONNECTED = 0;
        clearInterval(messageQueueInterval);
        messageQueueInterval = setInterval(function () {
            var params = Cache.messageQueue.shift();
            var disconnect = statusApi.getCurrentConnectionStatus() !== CONNECTED;
            if (disconnect || RongIM.utils.isEmpty(params)) {
                clearInterval(messageQueueInterval);
                return;
            }
            messageApi.sendCommandMessage(params);
        }, 1000 / 2);
    },
    auth: {},
    clean: function () {
        Cache.ready._done = false;
        Cache.auth = {};
        apiList.forEach(function (api) {
            if (typeof api.cleanCache === 'function') {
                api.cleanCache();
            }
        });
    }
};

var util = {
    noop: function () {},
    sameConversation: function (item, other) {
        return item.targetId === other.targetId && item.conversationType === other.conversationType;
    },
    getPrototype: Object.prototype.toString,
    isArray: function (arr) {
        return this.getPrototype.call(arr) === '[object Array]';
    },
    isString: function (str) {
        return this.getPrototype.call(str) === '[object String]';
    },
    // 此方法业务使用场景： 校验 messageUId 是否在 messageUIds 数组中，过滤单个字符 '_' 请注意。
    isInArray: function (node, arr) {
        return arr.join('_').indexOf(node) > -1;
    },
    /*
        重置对象的属性值, 支持单个对象，多个对象需再封装
        此方法操作对象引用，不重新创建对象
        var obj1 = { a: 1, b: 2 };
        var obj2 = reset(obj1, {a: 345});
        // obj1 => { a: 345, b: 2}
        // obj1 === obj2 => true
    */
    reset: function (obj, newValues) {
        var objs = [obj];
        var keys = function (o) {
            var tmp = [];
            $.each(o, function (k) {
                tmp.push(k);
            });
            return tmp;
        };
        var func = function (memo) {
            keys(newValues).forEach(function (key) {
                var memoObj = memo[0];
                memoObj[key] = newValues[key];
            });
            return memo;
        };
        return objs.reduce(func, objs)[0];
    },
    generatorKey: function (keys) {
        return keys.join('_');
    },
    getStoreKey: function (key) {
        var userId = Cache.auth.id;
        var keys = ['rce_g', userId, key];
        return util.generatorKey(keys);
    },
    getLibErrorCode: function (errorCode) {
        var prefix = 'lib-';
        if (errorCode) {
            var existed = String(errorCode).indexOf(prefix) >= 0;
            if (!existed) {
                errorCode = prefix + errorCode;
            }
        }
        return errorCode;
    }
};

function init(_config) {
    $.extend(true, config, _config);
    utils = RongIM.utils;
    utils.console = {};
    if (config.debug) {
        utils.console = win.console;
    } else {
        Object.keys(win.console).forEach(function (fnName) {
            if ($.isFunction(win.console[fnName])) {
                utils.console[fnName] = $.noop;
            }
        });
    }
    var provider = RongIM.lib.getDataProvider();
    if (provider) {
        provider = new RongIMLib.VCDataProvider(provider);
    }
    config.sdk.dbPath = RongIM.system.dbPath;
    RongIM.dataModel.Status.initRongIMClient(config, provider, Cache);

    Cache.auth = store.get('auth');
    loadApi();
    apiList.forEach(function (api) {
        if (typeof api.loadApi === 'function') {
            api.loadApi();
        }
    });
    messageApi.registerMessage();
    messageApi.setMessageListener();
}

function updateAuth(value) {
    Cache.auth = value;
}

function httpRequest(method, url, data, callback) {
    callback = callback || $.noop;
    if (requireAuth(url)) {
        return Cache.ready().then(function () {
            return request(url, method, data, callback);
        });
    }
    return request(url, method, data, callback);
}

var Http = {};

Http.get = function (url, data, callback) {
    return httpRequest('GET', url, data, callback);
};

Http.post = function (url, data, callback) {
    return httpRequest('POST', url, data, callback);
};

Http.put = function (url, data, callback) {
    return request(url, 'PUT', data, callback);
};

Http.del = function (url, data, callback) {
    return request(url, 'DELETE', data, callback);
};

function requireAuth(url) {
    var whiteList = [/^\/user\//, /^\/configuration\/all/];
    var publicAccess = whiteList.filter(function (pattern) {
        return pattern.test(url);
    }).length === 0;
    return publicAccess;
}

function ajax(url, method, data, callback) {
    var options = {
        url: getFullURL(url),
        method: method,
        xhrFields: {
            withCredentials: true
        },
        dataType: 'json'
    };

    if (method.toLowerCase() === 'get') {
        options.data = $.isEmptyObject(data) ? null : RongIM.utils.getQuerystr(data);
        options.processData = false;
    } else {
        options.data = $.isEmptyObject(data) ? null : JSON.stringify(data);
        options.contentType = 'application/json;charset=UTF-8';
    }

    $.ajax(options).then(function (response) {
        var errorCode = getErrorCode(response.code);
        if (errorCode) {
            callback(errorCode, response);
        } else {
            var result = response.result;
            callback(null, result);
        }
    }, function () {
        callback('request-failed');
    });
}

function getFullURL(path) {
    return config.dataModel.server + path;
}

function request(url, method, data, callback, fromServer) {
    callback = callback || $.noop;
    /*
    data is optional

    request(url, method, data, callback)
    request(url, method, callback)
    */
    if ($.isFunction(data)) {
        callback = data;
        data = null;
    }
    var defer = $.Deferred();
    // 部分接口数据从本地数据库获取：拦截器根据 URL 匹配
    var Intercepter = RongIM.Intercepter;
    var intercept = Intercepter.find(method, url);
    if (intercept && !fromServer) {
        var params = Intercepter.parseUrl(intercept.path, url);
        if (method.toLowerCase() === 'get') {
            $.extend(params.query, data);
        } else {
            params.data = data;
        }
        intercept.handle(params, function (errorCode, result) {
            if (errorCode) {
                defer.reject(errorCode);
                callback(errorCode, result);
                return;
            }
            defer.resolve(result);
            callback(errorCode, result);
        });
        return defer.promise();
    }
    // 从 server 获取数据
    ajax(url, method, data, function (errorCode, result) {
        callback(errorCode, result);
        if (errorCode) {
            defer.reject(errorCode, result);
            return;
        }
        defer.resolve(result);
    });
    return defer.promise();
}

/*
说明： 获取并缓存 server 配置到 RongIM.serverConfig
*/
function getServerConfig(callback) {
    var url = '/configuration/all';
    var serverConfig = RongIM.serverConfig;
    if (serverConfig) {
        callback(serverConfig);
        return;
    }
    Http.get(url, function (errorCode, result) {
        if (errorCode || !result) {
            callback(errorCode || 'request_server_config_failed');
            return;
        }
        RongIM.serverConfig = result.features;
        callback(null, result.features);
    });
}

Cache.ready = function (callback) {
    callback = callback || $.noop;
    if (Cache.ready._busy || Cache.ready._done) {
        return $.Deferred().resolve().promise();
    }
    Cache.ready._busy = true;
    /*
    多部门使用 getDeptTree 代替 getTree
    getAllUser 接口失效，获取高管人员列表排除高管计算全选
    */
    return $.when(
        orgApi.getOrgTree(), userApi.getNewestAlias(),
        starApi.getStarList(), friendApi.getFriendList(), orgApi.getAllCompany()
    ).then(function (tree, alias, starList, friendList, companyList) {
        friendList = friendList.map(function (item) {
            return {
                id: item.id,
                name: item.name,
                avatar: item.portrait_url,
                tel: item.tel,
                user_type: item.user_type,
                bothFriend: item.is_both_friend,
                create_dt: item.create_dt
            };
        });
        Cache.orgTree = tree;
        Cache.alias = alias;
        Cache.starList = starList;
        Cache.friendList = friendList;
        companyList.forEach(function (company) {
            if (company.state !== 2) {
                Cache.company[company.id] = company;
            }
        });
        var users = {};
        users._defer = {};
        if (!Cache.user) {
            Cache.user = users;
        }
        Cache.ready._done = true;
        callback();
        /*
             缓存 root 级公司 在在搜索是需要判断是否有 独立子公司
            */
        orgApi.getRoot();
        orgApi.getRootCompanyMemberCount();
        return $.Deferred().resolve().promise();
    })
        .fail(function (error) {
            callback(error);
            return $.Deferred().reject(error).promise();
        })
        .always(function () {
            Cache.ready._busy = false;
        });
};

function getErrorCode(code) {
    var SUCCESS_CODE = 10000;
    return (code === SUCCESS_CODE) ? null : code;
}

function getUserSetting(callback) {
    callback = callback || $.noop;
    var url = '/usetting/multiclient/pc_lock_screen';
    Http.get(url, function (errorCode, result) {
        callback(result);
    });
}

function getServerTime() {
    return new Date().getTime() - RongIMClient.getInstance().getDeltaTime();
}

RongIM.dataModel = {
    getUserSetting: getUserSetting,
    getServerConfig: getServerConfig,
    init: init,
    updateAuth: updateAuth,
    _Cache: Cache,
    _Http: Http,
    _httpRequest: httpRequest,
    _ObserverList: ObserverList,
    _request: request,
    getServerTime: getServerTime,
    util: util,
    config: config
};
}(RongIM, {
    win: window,
    jQuery: jQuery,
    RongIMClient: RongIMClient,
    RongIMLib: RongIMLib
}));
