(function (RongIM, dependencies) {
'use strict';

var $ = dependencies.jQuery;
var RongIMLib = dependencies.RongIMLib;
var RongIMClient = dependencies.RongIMClient;
var Http = RongIM.dataModel._Http;
var Cache = RongIM.dataModel._Cache;
var ObserverList = RongIM.dataModel._ObserverList;
var utils = RongIM.utils;
var request = RongIM.dataModel._request;
var common = RongIM.common;

var messageApi = null;
var conversationApi = null;

var Friend = {
    observerList: new ObserverList()
};

var friendObserverList = Friend.observerList;

Cache.friendList = [];
Cache.friendRequest = [];

Friend.cleanCache = function () {
    Cache.friendList = [];
    Cache.friendRequest = [];
};

Friend.loadApi = function () {
    messageApi = RongIM.dataModel.Message;
    conversationApi = RongIM.dataModel.Conversation;
    utils = RongIM.utils;
};

Friend.registerMessage = function () {
    // 好友消息
    var messageName = 'ContactNotifyMessage';
    var objectName = 'RCE:ContactNtfy';
    var messageTag = new RongIMLib.MessageTag(false, false);
    var properties = ['actionType', 'operator', 'target', 'data'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    // 非好友需要验证
    messageName = 'RequestFriendVerificationMessage';
    objectName = 'RCE:RFVMsg';
    messageTag = new RongIMLib.MessageTag(false, true);
    properties = ['extra'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);
};

Friend.insertRFVMsg = function (targetId) {
    var rfvMsg = RongIM.common.buildMessage.RequestFriendVerificationMessage({});
    var arg = {
        conversationType: RongIMLib.ConversationType.PRIVATE,
        targetId: targetId,
        messageType: 'RequestFriendVerificationMessage',
        objectName: 'RCE:RFVMsg',
        sentStatus: RongIMLib.SentStatus.SENT,
        content: rfvMsg
    };
    messageApi.insertMessage(arg);
};

Friend.requestFriendVerification = function (message) {
    message.messageType = message.messageType || message.content.messageName;
    messageApi.insertMessage(message);
    Friend.insertRFVMsg(message.targetId);
};

function acceptFriend(message) {
    var conversation = {
        conversationType: message.conversationType,
        targetId: message.targetId,
        timestamp: 0,
        position: 1
    };
    messageApi.get(conversation, function (errorCode, messageList) {
        var addConversation = function (list, con) {
            con.latestMessage = list[list.length - 1];
            conversationApi.add(con);
        };
        addConversation(messageList, conversation);
    });
}

function updateUserstatus(message, action) {
    var operatorId = message.content.operator.userId;
    var operatorIsSelf = Cache.auth.id === operatorId;
    var targetId = message.targetId;
    var targetUser = Cache.user[targetId] || {};
    switch (action) {
        case 'Accept':
            addFriend(targetId);
            targetUser.isFriend = true;
            targetUser.bothFriend = true;
            break;
        case 'Delete':
            if (operatorIsSelf) {
                targetUser.isFriend = false;
                targetUser.bothFriend = false;
            } else {
                targetUser.bothFriend = false;
            }
            break;
        default:
            break;
    }
}

function addFriend(userId) {
    var friend = Friend.getCacheFriend(userId);
    if (friend) {
        friend.bothFriend = true;
        return;
    }
    var user = Cache.user[userId] || { id: userId };
    friend = {
        id: user.id,
        name: user.name,
        avatar: user.avatar,
        bothFriend: true
    };
    Cache.friendList.push(friend);
}

Friend.messageCtrol = {
    ContactNotifyMessage: function (message) {
        if (message.offLineMessage) {
            return;
        }
        var actionMap = {
            1: 'Add',
            2: 'Accept',
            3: 'Reject',
            4: 'Delete'
        };
        var actionType = message.content.actionType;
        var action = actionMap[actionType];
        updateUserstatus(message, action);
        switch (action) {
            case 'Add':
                Friend.notifyFriendRequest();
                break;
            case 'Accept':
                messageApi._push(message);
                messageApi.observerList.notify(message);
                Friend.notifyFriendRequest();
                Friend.notifyFriend();
                acceptFriend(message);
                var lconv = conversationApi.getLocalOne(message.conversationType, message.targetId);
                if (lconv) {
                    lconv.latestMessage = message;
                    var lconvList = conversationApi.getLocalList();
                    conversationApi.observerList.notify(lconvList);
                }
                break;
            case 'Delete':
                Friend.notifyFriendRequest();
                Friend.notifyFriend();
                break;
            default:
                // TODO: ContactNotifyMessage
                // utils.console.log('TODO ContactNotifyMessage:' + action);
                break;
        }
    }
};
// Friend 文档地址  http://gitlab.rongcloud.net/RCE/RCE-Doc/blob/master/docs/design/subsystem/contact_service.md
// 根据号码搜索联系人
Friend.search = function (mobile, callback) {
    callback = callback || $.noop;
    var deferred = $.Deferred();
    var params = {
        keywords: mobile
    };
    // 添加好友查询从 server 获取（本地无访客/员工全量数据）
    request('/user/search/mobile', 'post', params, function (errorCode, result) {
        if (errorCode) {
            callback(errorCode);
            deferred.reject(errorCode);
            return;
        }
        if (result && result[0]) {
            result[0].avatar = result[0].portrait_url;
        }
        callback(null, result);
        deferred.resolve(result);
    }, true);
    return deferred.promise();
};

// 邀请好友
Friend.invite = function (id, content, callback) {
    callback = callback || $.noop;
    var params = {
        id: id,
        content: content
    };
    Http.post('/friends/invite/', params)
        .then(function () {
            Friend.notifyFriendRequest();
            callback();
        }).fail(callback);
};

// 接受请求
Friend.accept = function (friendRequest, callback) {
    callback = callback || $.noop;
    Cache.friendList = Cache.friendList || [];
    Http.post('/friends/accept/' + friendRequest.requestId)
        .then(function (result) {
            Friend.notifyFriend();
            Friend.notifyFriendRequest();
            callback(result);
        }).fail(callback);
};

// 获取好友信息
Friend.getFriend = function (friendId, callback) {
    callback = callback || $.noop;
    // mock
    Http.get('/friends/' + friendId)
        .then(function (friend) {
            friend.avatar = friend.portrait_url;
            friend.mobile = friend.tel;
            callback(null, friend);
        }).fail(callback);
};

// 从缓存获取单个好友信息
Friend.getCacheFriend = function (userId) {
    var cacheList = Cache.friendList;
    if (!cacheList) {
        return null;
    }
    for (var i = 0, len = cacheList.length; i < len; i += 1) {
        if (cacheList[i].id === userId) {
            return cacheList[i];
        }
    }
    return null;
};

// 获取好友列表
Friend.getList = function (callback) {
    callback = callback || $.noop;
    Http.get('/friends')
        .then(function (result) {
            var friendList = result.data.map(function (item) {
                return {
                    id: item.id,
                    name: item.name,
                    avatar: item.portrait_url,
                    tel: item.tel,
                    user_type: item.user_type,
                    bothFriend: item.is_both_friend,
                    create_dt: item.create_dt
                };
            });
            Cache.friendList = friendList;
            return callback(null, friendList);
        }).fail(callback);
};

Friend.isFileHelper = function (id) {
    var file = RongIM.serverConfig.file;
    var fileHelperId = file.file_transfer_robot_id;
    var isFileHelper = id === fileHelperId;
    return isFileHelper;
};

// TODO: 代码没有异步 callback 的必要
Friend.getFileHelper = function (callback) {
    callback = callback || $.noop;
    var file = RongIM.serverConfig.file;
    var error;
    if (!file) {
        error = 'RongIM.serverConfig.file is undefined!';
        utils.console.error(error);
        callback(error, null);
        return undefined;
    }
    var fileHelperId = file.file_transfer_robot_id;
    if (!fileHelperId) {
        error = 'fileHelperId is undefined';
        utils.console.error(error);
        callback(error, null);
        return undefined;
    }
    var locale = (RongIM.instance || { config: {} }).config.locale || 'zh';
    var info = {
        id: fileHelperId,
        name: RongIM.locale[locale].components.getFileHelper.title,
        type: 3
    };
    callback(null, info);
    return info;
};

Friend.getCacheList = function () {
    var cacheList = Cache.friendList || [];
    return cacheList;
};

// 删除好友
Friend.delFriend = function (friendId, callback) {
    callback = callback || $.noop;
    Http.del('/friends/' + friendId).then(function (result) {
        var idList = Cache.friendList.map(function (item) {
            return item.id;
        });
        var index = idList.indexOf(friendId);
        if (index >= 0) {
            Cache.friendList.splice(index, 1);
        }
        var requestList = Cache.friendRequest;
        var friendRequest = requestList.filter(function (req) {
            return req.uid === friendId;
        })[0];
        if (friendRequest) Friend.delRequest(friendRequest.requestId);
        Friend.notifyFriend();
        // notifyFriendRequest();
        callback(null, result);
    }).fail(callback);
};

// 删除所有好友
Friend.delAllFriend = function (callback) {
    callback = callback || $.noop;
    Http.del('/friends/all')
        .then(function () {
            callback();
            Cache.friendList = [];
            Friend.notifyFriend();
        }).fail(callback);
};

// 获取请求列表
Friend.getRequestList = function (callback) {
    callback = callback || $.noop;
    // var cacheList = Cache.friendRequest;
    // if(!$.isEmptyObject(cacheList)){
    //     return callback(null, cacheList);
    // }
    // mock
    // $.getJSON('http://localhost:3000/friend-request-list')
    Http.get('/friends/request_list')
        .then(function (result) {
            var list = result.data;
            list = list.filter(function (item) {
                item.user = {
                    id: item.uid,
                    name: item.name,
                    avatar: item.portrait_url
                };
                return item.state > 0;
            });
            Cache.friendRequest = list;
            callback(null, list);
        }).fail(callback);
};

// 从缓存获取好友请求信息
Friend.getCacheRequest = function () {
    var cacheList = Cache.friendRequest || [];
    // for (var i = 0, len = cacheList.length; i < len; i++) {
    //     if (cacheList[i].id === userId) {
    //         return cacheList[i];
    //     }
    // }
    return cacheList;
};

// 删除请求记录
Friend.delRequest = function (requestId, callback) {
    callback = callback || $.noop;
    Http.del('/friends/request_list/' + requestId).then(function (result) {
        var idList = Cache.friendRequest.map(function (item) {
            return item.requestId;
        });
        var index = idList.indexOf(requestId);
        if (index >= 0) {
            Cache.friendRequest.splice(index, 1);
        }
        // requestObserverList.notify();
        callback(result);
    }).fail(callback);
};

// 删除所有请求记录
Friend.delAllRequest = function (callback) {
    callback = callback || $.noop;
    Http.del('/friends/request_list/all')
        .then(function (result) {
            callback(result);
            Cache.friendRequest = {};
            // requestObserverList.notify();
        }).fail(callback);
};

// 清除邀请记录未读数
Friend.clearUnread = function (callback) {
    callback = callback || $.noop;
    Http.put('/friends/clear_unread')
        .then(function () {
            Friend.notifyFriendRequest();
            callback();
        }).fail(callback);
};
// 获取用户申请记录
Friend.getRequest = function (id) {
    var requestInfo = null;
    var cacheList = Cache.friendRequest;
    for (var i = 0, len = cacheList.length; i < len; i += 1) {
        if (cacheList[i].uid === id) {
            requestInfo = cacheList[i];
            break;
        }
    }
    return requestInfo;
};

Friend.watch = function (listener) {
    friendObserverList.add(listener);
};

Friend.unwatch = function (listener) {
    friendObserverList.remove(listener);
};

// Friend.watchRequest = function (listener) {
//     requestObserverList.add(listener);
// };

// Friend.unwatchRequest = function (listener) {
//     requestObserverList.remove(listener);
// };

Friend.notifyFriend = function () {
    Friend.getList(function (errorCode, list) {
        if (errorCode) {
            return;
        }
        list = list || [];
        var result = {
            type: 'Friend',
            list: list
        };
        friendObserverList.notify(result);
    });
};

Friend.notifyFriendRequest = function () {
    Friend.getRequestList(function (errorCode, list) {
        if (errorCode) {
            return;
        }
        list = list || [];
        var result = {
            type: 'Request',
            list: list
        };
        friendObserverList.notify(result);
    });
};

Friend.getFriendList = function () {
    function callback(result) {
        return result.data;
    }

    return request('/friends', 'GET').then(callback);
};

Friend.addExternalFriend = function(context, createrPhone) {
    var friendApi = this;
    $.ajax({
        url: RongIM.config.dataModel.serverOA+ '/default/rongyu/tjhaoyou',
        method: 'post',
        data: {
            str1: context.mobile,
            str2: context.contactRemark,
            str3: createrPhone,
            str5: context.contactName
        },
        dataType: 'json'
    }).done(function (resp) {
        
        if (resp.code == 1) {
            friendApi.notifyFriend();
            context.errors = {
              message: "添加失败"
            }
            common.messageToast({
              message: resp.message,
              type: 'success'
            });
            context.close();
        } else {
            context.errors = {
                message: resp.message
            }
        }
    }).fail(function () {
        context.errors = {
            message: "添加失败"
        }
    });
};

RongIM.dataModel.Friend = Friend;
}(RongIM, {
    jQuery: jQuery,
    RongIMClient: RongIMClient,
    RongIMLib: RongIMLib
}));
