(function (RongIM, dependencies, components) {
'use strict';

var common = RongIM.common;
var utils = RongIM.utils;
var $ = dependencies.jQuery;

/*
说明：转发时选择最近聊天
功能：消息转发时，选择 最近聊天 模块
*/
function getRecent(resolve, reject) {
    var im = RongIM.instance;
    var dataModel = im.dataModel;
    var conversationApi = dataModel.Conversation;
    var contactApi = im.dataModel.Contact;
    var options = {
        name: 'recent',
        template: 'templates/group/recent.html',
        data: function () {
            return {
                keyword: '',
                searchResult: [],
                members: []
            };
        },
        props: ['selected', 'defaultSelected', 'disableExecutive'],
        computed: {
            defaultIdList: function () {
                return (this.defaultSelected || []).map(function (item) {
                    return item.id;
                });
            },
            checkedAll: {
                get: function () {
                    return getCheckedAll(this);
                },
                set: function (value) {
                    setCheckedAll(this, value);
                }
            },
            indeterminate: function () {
                if (this.isCardSelect) {
                    return false;
                }
                return (typeof this.checkedAll) !== 'boolean';
            },
            checked: {
                get: function () {
                    return this.selected.map(function (item) {
                        return item.id;
                    });
                },
                set: function (newMemberIds) {
                    var context = this;
                    contactApi.get(newMemberIds, function (errorCode, newMembers) {
                        newMembers = [].concat(newMembers);
                        if (errorCode) {
                            context.toastError(errorCode);
                            return;
                        }
                        contactApi.get(context.checked, function (UerrorCode, oldMembers) {
                            oldMembers = [].concat(oldMembers);
                            if (UerrorCode) {
                                context.toastError(UerrorCode);
                                return;
                            }
                            var addedList = common.without(newMembers, oldMembers);
                            if (addedList.length > 0) context.$emit('added', addedList);

                            var removedList = common.without(oldMembers, newMembers);
                            var listLen = removedList.length;
                            if (listLen > 0) context.$emit('removed', removedList);
                        });
                    });
                    if (context.keyword) context.clear();
                }
            },
            isCardSelect: function () {
                return this.$parent.$options.name === 'card';
            }
        },
        components: {
            avatar: components.getAvatar
        },
        watch: {
            keyword: function (keyword) {
                keywordChanged(this, keyword);
            }
        },
        created: function () {
            created(this, conversationApi);
        },
        methods: {
            toastError: function (errorCode) {
                var el = null;
                if (this.$parent) {
                    el = this.$parent.$el.firstChild;
                }
                common.toastError(errorCode, el);
            },
            getUsername: function (item) {
                var name = common.getUsernameHighlight(item);
                return common.filterMark(name);
            },
            executiveLimit: function (item) {
                if (item.isFriend || im.auth.isExecutive || this.disableExecutive) {
                    return false;
                }
                var isExecutive = !!item.isExecutive;
                return isExecutive;
            },
            isDefault: function (item) {
                return this.defaultIdList.indexOf(item.id) >= 0;
            },
            isbanned: function (item) {
                if (this.isGroup(item)) {
                    var group = item.group;
                    if (group) {
                        var members = group.members || [];
                        var theMember = members.filter(function (member) {
                            return member.id === im.auth.id;
                        })[0] || {};
                        if (group.is_all_mute && theMember.mute_status !== 2) {
                            return true;
                        }
                        if (theMember.mute_status === 1) {
                            return true;
                        }
                    }
                }
                return false;
            },
            isValidGroup: function (item) {
                if (this.isGroup(item)) {
                    return checkGroupValid(item, im.auth.id);
                }
                return true;
            },
            isDisabled: function (item) {
                return this.isDefault(item) || this.executiveLimit(item) || this.isbanned(item) || !this.isValidGroup(item);
            },
            isGroup: function (item) {
                // return item.conversationType === 3;
                return item.id && item.id.startsWith('group_');
            },
            clear: function () {
                this.keyword = '';
            },
            showGroupType: function (group) {
                return group && group.type > 0;
            },
            getGroupType: common.getGroupType
        }
    };
    utils.asyncComponent(options, resolve, reject);
}

function checkGroupValid(conversation, authId) {
    var group = conversation.group || {};
    var members = group.member_id_list || [];
    var inGroup = members.indexOf(authId) >= 0;
    var isDismiss = (group.group_status === 2);
    return inGroup && !isDismiss;
}

function getCheckedAll(context) {
    var starChecked = [];
    var checkedIdList = context.checked;
    context.searchResult.forEach(function (item) {
        var existed = checkedIdList.indexOf(item.id) >= 0;
        if (existed) starChecked.push(item);
    });
    var length = starChecked.length;
    var result;
    if (length > 0) {
        var allMember = context.searchResult.filter(function (item) {
            return !context.executiveLimit(item) && context.isValidGroup(item);
        });
        var isAll = length === allMember.length;
        result = isAll ? true : null;
    } else {
        result = false;
    }
    return result;
}

function setCheckedAll(context, value) {
    var members = context.searchResult;
    var memberIds = members.filter(function (item) {
        return !context.executiveLimit(item) && !context.isbanned(item) && context.isValidGroup(item);
    }).map(function (item) {
        return item.id;
    });
    if (value) {
        context.checked = [].concat(memberIds, context.checked);
    } else {
        context.checked = context.checked.filter(function (id) {
            return memberIds.indexOf(id) < 0;
        });
    }
}

/*
说明：转发消息，最近聊天搜索框 内容变化时，获取满足条件的群组列表
参数：
    @param {object}     context         我的群组信息
    @param {string}     keyword         搜索框关键字
*/
function keywordChanged(context, keyword) {
    if (keyword.length === 0) {
        context.searchResult = context.members;
        // context.searchResult = context.members;
    } else {
        context.searchResult = [];
        var searchResult = [];
        context.members.forEach(function (item) {
            var nameRange = utils.searchStrRange(item.name, context.keyword);
            var aliasRange = utils.searchStrRange(item.alias, context.keyword);
            if (nameRange || aliasRange) {
                var result = $.extend({
                    range: nameRange,
                    aliasRange: aliasRange
                }, item);
                searchResult.push(result);
            }
        });
        // common.sortUsers(searchResult);
        context.searchResult = searchResult;
    }
}

/*
说明：转发消息时，获取 最近聊天 列表
参数：
    @param {object}     context
    @param {object}     groupApi
*/
function created(context, conversationApi) {
    conversationApi.getList(null, function (errorCode, list) {
        if (errorCode) {
            context.toastError(errorCode);
            return;
        }
        var result = [];
        list.forEach(function (conversation) {
            var isGroup = conversation.conversationType === 3;
            var isPublic = conversation.conversationType === 7;
            var target = {};
            if (isGroup) {
                target = conversation.group;
            } else if (isPublic) {
                target = conversation.latestMessage.user;
            } else {
                target = conversation.user;
            }
            var isSysUser = conversation.user && common.isSysUser(conversation.user);
            if (conversation.conversationType !== 6 && !isSysUser) {
                result.push(formateContact(target, isGroup));
            }
        });
        context.members = result;
        context.searchResult = result;
    });
}

function formateContact(item, isGroup) {
    var obj = {
        id: isGroup ? 'group_' + item.id : item.id,
        name: item.name,
        avatar: item.avatar,
        member_avatars: isGroup ? item.member_avatars : '',
        member_names: isGroup ? item.member_names : '',
        member_id_list: isGroup ? item.member_id_list : '',
        isFriend: isGroup ? true : item.isFriend,
        type: item.type
    };
    if (isGroup) {
        obj.group = item;
    }
    return obj;
}

$.extend(true, components, {
    group: {
        getRecent: getRecent
    }
});
}(RongIM, {
    jQuery: jQuery
}, RongIM.components));
