
/*
该文件需渲染进程引入
*/
const path = require('path');
const { exec } = require('child_process');
const zmq = require('./zeromq');

const RemoteStatus = {
    0: 'IDLE', // 空闲
    IDLE: 0,
    1: 'CONTROL', // 控制另一端
    CONTROL: 1,
    2: 'CHAIN', // 被控制
    CHAIN: 2,
};

const RemoteType = {
    public: 'public',
    private: 'private',
};

let sock;
let remoteStatus = RemoteStatus.IDLE;
let isRemoteRunning = false;

let isPublic;

const init = (option) => {
    const type = option.type;
    isPublic = type === RemoteType.public;
};

const getRemoteClientName = () => {
    let remoteClientName = isPublic ? 'sunloginsdk-remoteclient-public.app' : 'sunloginsdk-remoteclient-private.app';
    remoteClientName += '/Contents/MacOS/sunloginsdk-remoteclient';
    return remoteClientName;
};

/*
杀死进程中的 sunloginsdk
使用 stop_sunloginsdk.sh 脚本, 程序进入和退出时执行
 */
const stopRemoteClient = (callback) => {
    const shellPath = path.join(__dirname, 'stop_sunloginsdk.sh');
    const addAuth = `chmod +x ${shellPath}`;
    exec(addAuth, (/* err */) => {
        if (sock) sock.close();
        sock = null;
        exec(shellPath, callback);
    });
};

/**
 * Mac 下, 运行 sunloginsdk-remoteclient 脚本
 */
const runRemoteClient = (callback) => {
    const clientName = getRemoteClientName();
    const clientPath = path.join(__dirname, clientName);
    stopRemoteClient(() => {
        console.log('sunloginsdk-remoteclient start');
        if (!isRemoteRunning) {
            isRemoteRunning = true;
            exec(clientPath, () => {
                console.log('sunloginsdk-remoteclient close');
                isRemoteRunning = false;
            });
        }
        callback();
    });
};

/**
 * 使用 sock.send 向 sunloginsdk-remoteclient 发送数据.
 * @param  {Object} params 发送的信息
 */
const send = (msg) => {
    msg = JSON.stringify(msg);
    console.log('send msg', msg);
    sock.send(msg);
};

const formatMsg = (msg) => {
    msg = msg.toString();
    return JSON.parse(msg);
};

/**
 * 开启主控端和被控端时, 都必须先 connect
 * 1. 运行 sunloginsdk-remoteclient 进程
 * 2. 运行 sock.connect. 作用: 使 node 程序与 sunloginsdk-remoteclient 通讯
 */
const connect = (callback) => {
    runRemoteClient(() => {
        sock = zmq.socket('pair');
        sock.connect('ipc:///tmp/test');
        setTimeout(callback, 500);
    });
};


/**
 * 处理主控端
 * @param  {Object}   msg     sock 返回的 msg
 * @param  {Object}   callback 回调
 *                    callback.onConnected  连接成功
 *                    callback.onDisconnected  断开连接
 *                    callback.onDisplayChanged
 *                    callback.onError address 或者 name 传输错误
 */
const handleJoinMsg = (msg, callback) => {
    console.log('主控端收到', msg);
    callback = callback || {};
    if (msg.error) {
        if (callback.onError) callback.onError(msg.error);
        return;
    }
    if (msg.status === 1000) {
        if (callback.onClose) callback.onClose();
        return;
    }
    switch (msg.status) {
    case 1:
        if (callback.onConnected) callback.onConnected();
        break;
    case 2:
        if (callback.onDisconnected) callback.onDisconnected();
        break;
    case 3:
        if (callback.onDisplayChanged) callback.onDisplayChanged();
        break;
    case 1000:
        if (remoteStatus === RemoteStatus.IDLE && callback.onClose) callback.onClose();
        break;
    default:
        break;
    }
};

/**
 * 处理被控端
 * @param  {Object}   msg  sock 返回的 msg
 * @param  {Object} callback 回调
 *                  callback.onError openKey 或 openId 错误
 *                  callback.onConnect  开始连接
 *                  callback.onDisconnected  断开连接
 *                  callback.onLogin  登录成功, 返回 address 和 sessionName
 *                  callback.onLoginFail  登录失败
 */
const handleAcceptMsg = (msg, callback) => {
    console.log('被控端收到', msg);
    callback = callback || {};
    if (msg.error) {
        if (callback.onError) callback.onError();
        return;
    }
    if (msg.sessionStutus === 2) {
        if (callback.onEnd) callback.onEnd();
        return;
    }
    if (msg.sessionStutus !== -1) {
        return;
    }
    switch (msg.clientStutus) {
    case 0:
        if (callback.onConnect) callback.onConnect();
        break;
    case 1:
        if (callback.onDisconnected) callback.onDisconnected();
        break;
    case 2:
        if (callback.onLogin) callback.onLogin(msg);
        break;
    case 3:
        if (callback.onLoginFail) callback.onLoginFail();
        break;
    case 1000:
        if (callback.onClose) callback.onClose();
        break;
    default:
        break;
    }
};

/**
 * 主控端开始远程控制
 * @param  {String} address 地址, 由 被控端提供
 * @param  {String} name    由 被控端提供
 * @params {Object} 回调
 */
const joinRemoteControl = (address, name, callback) => {
    connect(() => {
        sock.on('message', (msg) => {
            msg = formatMsg(msg);
            handleJoinMsg(msg, callback);
        });
        const launchMsg = {
            code: 1,
            address,
            name,
        };
        send(launchMsg);
        remoteStatus = RemoteStatus.CONTROL;
    });
};

/**
 * 被控端接受被控, 并返回 address 和 name
 * @param  {String} openId
 * @param  {String} openKey
 * @params {Object} 回调
 */
const acceptRemoteControl = (openId, openKey, callback) => {
    const watchMsg = () => {
        sock.on('message', (msg) => {
            msg = formatMsg(msg);
            handleAcceptMsg(msg, callback);
        });
    };
    connect(() => {
        watchMsg();
        const acceptMsg = isPublic ? {
            code: '2',
            openId,
            openKey,
        } : {
            code: '2',
            License: openId,
            szaddr: openKey,
        };
        send(acceptMsg);
        remoteStatus = RemoteStatus.CHAIN;
    });
};

/**
 * 关闭 sunloginsdk-remoteclient
 */
// const closeRemoteClient = () => {
//     const closeMsg = {
//         code: 5,
//     };
//     if (sock) send(closeMsg);
//     sock = null;
// };

/**
 * 关闭远程控制
 */
const hangupRemoteControl = (/* callback */) => {
    if (remoteStatus !== RemoteStatus.IDLE) {
        sock.on('message', (msg) => {
            console.log('hangup receive', msg.toString());
        });
        const isControl = remoteStatus === RemoteStatus.CONTROL;
        const hangupMsg = {
            code: isControl ? 3 : 4,
        };
        send(hangupMsg);
        remoteStatus = RemoteStatus.IDLE;
        // closeRemoteClient();
        stopRemoteClient();
    }
};

module.exports = {
    init,
    join: joinRemoteControl,
    accept: acceptRemoteControl,
    hangup: hangupRemoteControl,
};
