(function (RongIM, dependencies, components) {
'use strict';

var utils = RongIM.utils;
var common = RongIM.common;

function transUrl(url) {
    return common.trans2Localfile(url, 'icons');
}

function imgLoaded(src, callback, onerror) {
    if (!src) {
        return;
    }
    var img = new Image();
    img.onload = callback;
    img.onerror = function () {
        if (typeof onerror === 'function') {
            onerror();
        }
    };
    img.src = transUrl(src);
}

/*
说明：头像组件，用于显示头像，包括用户头像、群组头像、文件助手头像等
*/

components.getAvatar = function (resolve, reject) {
    var im = RongIM.instance;
    var dataModel = im.dataModel;
    var userApi = dataModel.User;
    var options = {
        name: 'avatar',
        template: '#rong-template-avatar',
        props: ['user', 'group', 'online-status'],
        data: function () {
            return {
                userData: {},
                memberAvatars: [],
                avatarLoaded: false,
                // 默认设置为 true 200ms 后设置为 false 防止默认蓝色头像闪一下
                avatarLoadSuccess: true,
                groupAvatarLoadSuccess: false,
                groupMemberavatarLoadSuccess: []
            };
        },
        computed: {
            memberNames: function () {
                // 获取群组中人员姓名列表
                var memberNames = this.group.member_names;
                return memberNames;
            },
            memberIdList: function () {
                // 获取群组中人员id列表
                var memberIdList = getReverseList(this.group.member_id_list);
                return memberIdList;
            },
            isOneMember: function () {
                // 群组是否只剩一个人
                return this.memberAvatars.length === 1;
            },
            validUser: function () {
                // 判断是否可以显示用户信息
                var user = this.user;
                if (!user) {
                    return false;
                }
                var isSysUser = common.isSysUser(user);
                return !isSysUser;
            },
            isFileHelper: function () {
                // 是否为文件助手
                var user = this.user || {};
                return user.type === 3;
            }
        },
        filters: {
            slice: function (name) {
                // 如果没有头像，中文下显示最后一个汉字，英文下显示第一个英文字母
                if (!name) {
                    return name;
                }
                var isChinese = /^[^\x20-\xff]+$/.test(name);
                return isChinese ? name.slice(-1) : name[0].toUpperCase();
            }
        },
        watch: {
            user: function (newValue) {
                this.setUserData(newValue);
            },
            'userData.avatar': function (newValue, oldValue) {
                var context = this;
                var avatarChanged = newValue && (utils.isEmpty(oldValue) || newValue !== oldValue);
                if (avatarChanged) {
                    context.avatarLoadSuccess = false;
                    imgLoaded(newValue, function () {
                        context.avatarLoadSuccess = true;
                    });
                }
            },
            'group.avatar': function (newValue, oldValue) {
                var context = this;
                if (utils.isEmpty(newValue)) {
                    return;
                }
                var avatarChanged = newValue !== oldValue;
                if (avatarChanged) {
                    context.groupAvatarLoadSuccess = false;
                    imgLoaded(newValue, function () {
                        context.groupAvatarLoadSuccess = true;
                    });
                }
            },
            'group.member_avatars': function (newValue) {
                var context = this;
                var oldMemberAvatars = context.memberAvatars || [];
                var memberAvatars = [];
                if (context.group) {
                    memberAvatars = newValue || [];
                }
                context.memberAvatars = memberAvatars.slice(0, 9);
                var newMemberAvatars = context.memberAvatars;
                if (newMemberAvatars.length === 0) {
                    return;
                }
                var memberAvatarChanged = oldMemberAvatars.toString() !== newMemberAvatars.toString();
                if (memberAvatarChanged) {
                    var loadSuccessArr = [];
                    for (var i = 0; i < newMemberAvatars.length; i += 1) {
                        loadSuccessArr.push(false);
                    }
                    context.groupMemberavatarLoadSuccess = loadSuccessArr;
                    newMemberAvatars.forEach(function (src, index) {
                        imgLoaded(src, function () {
                            loadSuccessArr.splice(index, 1, true);
                        });
                    });
                }
            }
        },
        mounted: function () {
            var context = this;
            this.setUserData(context.user);
            if (context.user) {
                if (context.userData.avatar) {
                    context.avatarLoadSuccess = false;
                    imgLoaded(context.userData.avatar, function () {
                        context.avatarLoaded = true;
                        context.avatarLoadSuccess = true;
                    }, function () {
                        context.avatarLoaded = true;
                        context.avatarLoadSuccess = false;
                    });
                    // avatarLoadSuccess 默认设置为 true 200ms 后设置为 false 防止默认蓝色头像闪一下
                    setTimeout(function () {
                        if (!context.avatarLoaded) {
                            context.avatarLoadSuccess = false;
                        }
                    }, 200);
                }
                // 当用户信息有修改时，需要更新头像信息
                context.userChanged = function (user) {
                    if (!context.user) {
                        return;
                    }
                    if (user.id === context.user.id) {
                        context.setUserData(user);
                    }
                };
                userApi.watch(context.userChanged);
                return;
            }
            if (context.group) {
                if (context.group.avatar) {
                    imgLoaded(context.group.avatar, function () {
                        context.groupAvatarLoadSuccess = true;
                    });
                }
                var memberAvatars = context.group.member_avatars || [];
                context.memberAvatars = memberAvatars.slice(0, 9);
                if (memberAvatars.length > 0) {
                    var loadSuccessArr = [];
                    for (var i = 0; i < memberAvatars.length; i += 1) {
                        loadSuccessArr.push(false);
                    }
                    context.groupMemberavatarLoadSuccess = loadSuccessArr;
                    memberAvatars.forEach(function (src, index) {
                        imgLoaded(src, function () {
                            loadSuccessArr.splice(index, 1, true);
                        });
                    });
                }
                // 群组拼接的头像，当群组成员改变时，需要更新头像信息
                context.memberChanged = function (user) {
                    if (context.group && context.group.member_id_list) {
                        // idList为倒序，获取最后9个人信息，判断当前用户是否在这9个人当中，是则更新头像
                        var memberIds = context.group.member_id_list;
                        var count = memberIds.length;
                        var members = memberIds.slice(count - 9);
                        members.reverse();
                        var index = members.indexOf(user.id);
                        var hasMembers = (index >= 0 && context.group);
                        if (!hasMembers) {
                            return;
                        }
                        var avatar = user.localAvatar || user.avatar;
                        var avatarChanged = context.group.member_avatars[index] !== avatar;
                        if (avatarChanged) {
                            context.group.member_avatars.splice(index, 1, avatar);
                        }
                    }
                };
                userApi.watch(context.memberChanged);
            }
        },
        methods: {
            userAlt(tips) {
                console.log(tips)
                this.$emit("userAlt",tips)
            },
            setUserData: function (val) {
                var temp = val;
                // 公众号默认头像显示加载失败时的头像而非名字首字母
                if (temp && temp.type === 2) {
                    if (!temp.avatar) {
                        var location = window.location;
                        var defautlUrl = location.pathname.replace(/\/[^/]+$/g, '/css/images/person-base.png');
                        temp.avatar = location.origin + defautlUrl;
                    }
                }
                this.userData = temp;
            },
            getThemeIndex: function (id) {
                // 根据id返回固定数字，用于显示头像背景色，共6种颜色
                var LENGTH = 6;
                return id ? (id.slice(-1).charCodeAt(0) % LENGTH) : 0;
            },
            getAvatar: transUrl
        },
        destroyed: function () {
            // 组件销毁后，取消监听
            if (this.userChanged) userApi.unwatch(this.userChanged);
            if (this.memberChanged) userApi.unwatch(this.memberChanged);
        }
    };
    utils.asyncComponent(options, resolve, reject);
};

// 因为server返回成员顺序是新加入的在前面，所以，需要反转数据
function getReverseList(list) {
    var reverseList = list.concat();
    return reverseList.reverse();
}
}(RongIM, null, RongIM.components));
