(function (RongIM, dependencies) {
'use strict';

var utils = RongIM.utils;
var RongIMEmoji = dependencies.RongIMLib.RongIMEmoji;
var emojiToHTML = RongIMEmoji.emojiToHTML;
var $ = dependencies.jQuery;
var Vue = dependencies.Vue;
var client = RongIM.client;
var system = RongIM.system;
var defaultSize = RongIM.config.emoji.sizePX;
var setupScrollHeight = function (el) {
    var $el = $(el).parent();
    Vue.nextTick(function () {
        $el.find('.rong-scroll-bar-y>i').css({
            display: 'block',
            height: el.scrollHeight - 5,
            width: '7px'
        });
    });
}

/*
纵向滚动条解决透明问题
update 数据变化时重新计算滚动条高度
*/
Vue.directive('rong-scroll-bar-y', {
    inserted: function (el) {
        var browserType = utils.getBrowser().type.toLowerCase();
        if (browserType !== 'chrome' && browserType !== 'safari') {
            return;
        }
        var $scrollContain = $(el);
        var $parent = $('<div class="rong-scroll-container"></div>');
        $parent.css({
            height: '100%',
            position: 'relative'
        });
        $scrollContain.wrap($parent);
        var $scrollbar = $('<div class="rong-scroll-bar-y"><i></i></div>');
        if (browserType === 'safari') {
            $scrollbar.addClass('rong-scroll-bar-safari');
        }
        // $parent.append($scrollbar);
        $scrollContain.parent().append($scrollbar);
        $scrollContain.addClass('rong-scroll-content');
        var scrollContain = true;
        var timeoutContain;
        var scrollBar = true;
        var timeoutBar;
        var timeoutShowScrollBar;

        var startScrollBar = function () {
            if (scrollBar) {
                setupScrollHeight(el);
                var top = $scrollbar.scrollTop();
                $scrollContain.scrollTop(top);
                scrollContain = false;
                clearTimeout(timeoutContain);
                timeoutContain = setTimeout(function () {
                    scrollContain = true;
                }, 500);
            }
        };
        var startScrollContain = function () {
            if (scrollContain) {
                setupScrollHeight(el);
                var top = $scrollContain.scrollTop();
                $scrollbar.scrollTop(top);
                scrollBar = false;
                clearTimeout(timeoutBar);
                timeoutBar = setTimeout(function () {
                    scrollBar = true;
                }, 500);
            }
        };
        var showScrollBar = function () {
            clearTimeout(timeoutShowScrollBar);
            $scrollbar.addClass('rong-scroll-bar-show');
            setupScrollHeight(el);
        };
        var hideScrollBar = function () {
            timeoutShowScrollBar = setTimeout(function () {
                $scrollbar.removeClass('rong-scroll-bar-show');
            }, 100);
        };

        $scrollbar.on('scroll', startScrollBar);
        $scrollContain.on('scroll', startScrollContain);
        $scrollbar.on('mouseenter', showScrollBar);
        $scrollContain.on('mouseenter', showScrollBar);
        $scrollbar.on('mouseleave', hideScrollBar);
        $scrollContain.on('mouseleave', hideScrollBar);
        // $scrollContain.hover(startScrollContain);
    },
    update: function (el) {
        var browserType = utils.getBrowser().type.toLowerCase();
        if (browserType == 'chrome' || browserType == 'safari') {
            setupScrollHeight(el);
        }
    }
});

Vue.directive('rong-succession-click', {
    bind: function (el, binding) {
        if (typeof binding.value !== 'function') {
            return;
        }
        var modifiers = binding.modifiers;
        var handle = binding.value;
        var count = 4;
        var timesArg = parseInt(binding.arg);
        if (timesArg > 1) {
            count = timesArg;
        }
        var clickTimes = 0;
        var timeout = null;
        el._successionclick = function (event) {
            if (modifiers.stop) {
                event.stopPropagation();
            }
            if (modifiers.prevent) {
                event.preventDefault();
            }
            clickTimes += 1;
            if (clickTimes >= count) {
                clickTimes = 0;
                handle(event);
            }
            clearTimeout(timeout);
            timeout = setTimeout(function () {
                clickTimes = 0;
            }, 350);
        };
        el.addEventListener('click', el._successionclick);
    },
    unbind: function (el) {
        el.removeEventListener('click', el._successionclick);
        el._successionclick = null;
    }
});
var EventNameWheel = ('onwheel' in document.createElement('div')) ? 'wheel' : 'mousewheel';
Vue.directive('rong-scroll-to-bottom', {
    bind: function (el, binding) {
        if (typeof binding.value !== 'function') {
            return;
        }
        var handle = binding.value;
        var locked = false;// 防止重复调用
        el._scrolltobottom = function (event) {
            var down = event.deltaY > 0;
            // 滑动到底部时有时会差零点几像素导致无法正确加载下一页
            var fill = 1;
            var bottom = el.scrollTop + el.clientHeight + fill >= el.scrollHeight;
            if (down && bottom && !locked) {
                handle();
                locked = true;
                setTimeout(function () {
                    locked = false;
                }, 500);
            }
        };
        el.addEventListener(EventNameWheel, el._scrolltobottom);
    },
    unbind: function (el) {
        el.addEventListener(EventNameWheel, el._scrolltobottom);
        el._scrolltobottom = null;
    }
});

var watermark = function (el, binding) {
    var im = RongIM.instance;
    var watermarkConfig = im.serverConfig.watermark;
    if (!watermarkConfig.enable) {
        return;
    }
    var $container = $(el);
    if (!binding.value) {
        if (!im.loginUser) {
            return;
        }
        var showText = im.loginUser.mobile || im.loginUser.staff_no || im.loginUser.id;
        showText = im.loginUser.name + showText.slice(-4);
        $container.watermark({
            texts: showText,
            // texts : '章颖',
            backgroundScroll: false
        });
    } else {
        $container.css({ 'background-image': 'none' });
        $container.css({ 'background-color': '#ffffff' });
    }
};
// 背景加水印
Vue.directive('watermark', {
    inserted: watermark,
    update: watermark
});

function textMessageFormat(content) {
    if (utils.isEmpty(content) || content.length === 0) {
        return '';
    }

    // 要处理的到底是message？还是message里的content？
    // str到处都是？
    // 传入依赖
    // if-else只处理差异化部分

    content = utils.encodeHtmlStr(content);

    content = utils.replaceUri(content, function (uri, protocol) {
        var link = uri;
        if (!protocol) {
            link = 'http://' + uri;
        }
        var loginInfo = utils.cache.get('loginInfo');
        if (loginInfo && loginInfo.zkxToken) {
            var token = decodeURIComponent(loginInfo.zkxToken) + ';path=/';
            if (link.match(/\/taskcenter\//i) != null && loginInfo.RCESESSIONID) {
                let urlParts = link.split('#');
                link = urlParts[0] + (
                    urlParts[0].indexOf('?') > 0 ? '&' : '?'
                ) + "sessionid=" + encodeURIComponent(loginInfo.RCESESSIONID) +
                "&uid=RY" + loginInfo.zkxEmpId + (
                    urlParts[1] ? ('#' + urlParts[1]) : ''
                );
            }
            link = RongIM.config.loginServer + '/sign-bridge.html' +
                '?target=' + encodeURIComponent(link.replace(/amp;/g, '')) +
                '&empId=' + (loginInfo.zkxEmpId || '') +
                '&token=' + encodeURIComponent(token);
        }
        return '<a class="rong-link-site" href="' + location.hash +
            '" onclick="openExternal(' +
            "'" + link + "'" + ',' + (system.platform.indexOf('win') > -1) +
            ')">' + uri + '</a>';
    });

    content = utils.replaceEmail(content, function (email) {
        return '<a class="rong-link-email" href="mailto:' + email + '">' + email + '</a>';
    });
    // content = RongIMEmoji.messageDecode(content);
    return convertEmoji(content, 18);
}

// 群公告格式
function groupNoticeFormat(content, locale) {
    var atPanel = locale.components.atPanel;
    var everyone = atPanel.everyone;
    var message = locale.message || {};
    return [message.noticeTitle, '\n@', everyone, ' ', content].join('');
}

// 会话列表最后一条群公告格式
function latestGroupNoticeFormat(content, locale) {
    var message = locale.message || {};
    return [message.noticeTitle, ':', content].join('');
}

// 转化消息，用于显示
function convertMessage(text) {
    if (utils.isEmpty(text)) {
        return text;
    }
    // var content = RongIMEmoji.messageDecode(text.trim());
    var content = text.trim();
    content = content.replace(/\n/g, '');
    content = utils.encodeHtmlStr(content);
    content = RongIMEmoji.symbolToEmoji(content);
    var SIZE_PX = defaultSize || 16;
    return convertEmoji(content, SIZE_PX);
}

var formatEmojiOverlap = (function () {
    if (utils.isEmojiOverlap()) {
        var tagReg = utils.emojiNativeReg.toString();
        tagReg = tagReg.substring(1, tagReg.length - 3);
        tagReg += '(?=[^ ]|$)';
        tagReg = new RegExp(tagReg, 'ig');
        return function (text) {
            return text.replace(tagReg, function (emoji) {
                return emoji + ' ';
            });
        };
    }
    return function (text) {
        return text;
    };
}());

/**
 * 转化emoji，不同端显示不同, 用于文本显示
 * 1, pc端并且支持本地emoji读取的版本中, 直接使用RCE本地图片
 * 2, web mac 中, 不使用任何图片, 直接全部使用原生emoji
 * 3, web win 中, 使用/modules/emoji中的图片
 * 4, 当 isForbidNative 传入 true时, mac下也显示图片
 * @param  {String} content 需要转化的内容
 * @param  {Int | Undefined} size    转化后emoji图片大小
 * @param  {Bool} isForbidNative  是否禁止使用原生emoji
 */
function convertEmoji(content, size, isForbidNative) {
    if (!utils.isString(content)) {
        return content;
    }
    var hasEmojiUrl = client.getEmojiUrl;
    var platform = system.platform;
    var isInMac = platform.indexOf('darwin') !== -1;
    if (hasEmojiUrl) {
        // 使用RCE本地图片
        content = emojiConvertHTML(content, size || defaultSize);
    } else if (isInMac && !isForbidNative) {
        // 使用symbolToEmoji, 作用是把unicode转化为可显示的emoji
        content = RongIMEmoji.symbolToEmoji(content);
    } else {
        // 网页版, 只显示modules/emoji/emoji-48.png内包含的图片
        content = emojiToHTML(content, size);
    }
    content = formatEmojiOverlap(content);
    return content;
}

// 构建消息体，发送消息时的消息体
var buildMessage = {
    TextMessage: function (context) {
        return new RongIMLib.TextMessage(context);
    },
    ImageMessage: function (context) {
        return new RongIMLib.ImageMessage(context);
    },
    FileMessage: function (context) {
        return new RongIMLib.FileMessage(context);
    },
    VoiceMessage: function (context) {
        return new RongIMLib.VoiceMessage(context);
    },
    LocationMessage: function (context) {
        return new RongIMLib.LocationMessage(context);
    },
    CardMessage: function (context) {
        return new RongIMClient.RegisterMessage.CardMessage(context);
    },
    SightMessage: function (context) {
        return new RongIMClient.RegisterMessage.SightMessage(context);
    },
    RichContentMessage: function (content) {
        return new RongIMLib.RichContentMessage(content);
    },
    ReferenceMessage: function (content) {
        return new RongIMClient.RegisterMessage.ReferenceMessage(content);
    },
    RequestFriendVerificationMessage: function (content) {
        return new RongIMLib.RongIMClient.RegisterMessage.RequestFriendVerificationMessage(content);
    }
};

/*
说明：统一弹窗格式
参数：
    @param {object}      params
    @param {string}      params.type          - 'confirm':有2个按钮的弹窗，'alert':有1个按钮的弹窗，默认为'alert'
    @param {string}      params.title         - 弹窗title，不传则为空
    @param {string}      params.message       - 弹窗内容，不传则为空
    @param {string}      params.submitText    - 确定按钮的文案
    @param {boolean}     params.isAlignLeft   - 弹窗内容不都左对齐
    @param {function}    params.closeCallback - 取消按钮回调函数
    @param {function}    params.callback      - 确认按钮回调函数
    @param {boolean}     params.hashchangeClose - default true
*/
function messagebox(params) {
    if (utils.isEmpty(params.hashchangeClose)) {
        params.hashchangeClose = true;
    }
    var options = {
        name: 'messagebox',
        template: $('#rong-messagebox').html(),
        data: function () {
            return {
                type: params.type || 'alert',
                title: params.title,
                message: params.message,
                submitText: params.submitText,
                show: true,
                isAlignLeft: params.isAlignLeft
            };
        },
        created: function () {
            var context = this;
            var im = RongIM.instance;
            context.title = params.title || context.locale.tips.msgboxTitle;
            context.submitText = params.submitText || context.locale.tips.msgboxSubmitText;
            im.$on('imlogined', context.close);
            im.$on('messageboxHide', context.close);
            if (params.hashchangeClose) {
                window.addEventListener('hashchange', context.close);
            }
        },
        methods: {
            close: function () {
                if (params.closeCallback) params.closeCallback();
                this.show = false;
                this.$destroy();
                $(this.$el).remove();
            },
            confirm: function () {
                if (params.callback) params.callback();
                this.show = false;
                this.$destroy();
                $(this.$el).remove();
            }
        },
        directives: {
            autofocus: {
                inserted: function (el) {
                    Vue.nextTick(function () {
                        el.focus();
                    });
                }
            }
        },
        destroyed: function () {
            var im = RongIM.instance;
            im.$off('imlogined', this.close);
            im.$on('messageboxHide', this.close);
            window.removeEventListener('hashchange', this.close);
        }
    };
    var localeMix = {
        computed: {
            locale: function () {
                var locale = RongIM.instance.locale;
                var name = utils.kebabToCamel(options.name);
                return $.extend(true, {}, locale, locale.components[name]);
            }
        }
    };
    options.mixins = [localeMix];
    var Messagebox = Vue.extend(options);
    var instance = new Messagebox({
        el: document.createElement('div')
    });
    var wrap = RongIM.instance.$el.firstChild;
    $(wrap).append(instance.$el);
    return instance;
}

/*
说明：统一弹窗格式
参数：
    @param {string}      params.message       - toast内容
    @param {string}      params.type          - toast类型，'success'绿色，'error'红色，默认为'success'
*/
function messageToast(params) {
    var options = {
        name: 'messagetoast',
        template: $('#rong-messagetoast').html(),
        data: function () {
            return {
                type: params.type || 'success',
                message: params.message,
                show: true,
                position:params.position||'middle'
            };
        },
        created: function () {
            var context = this;
            setTimeout(function () {
                context.show = false;
                if (typeof params.callback === 'function') {
                    params.callback();
                }
            }, 1500);
        }
    };
    var messagetoast = Vue.extend(options);
    // eslint-disable-next-line new-cap
    var instance = new messagetoast({
        el: document.createElement('div')
    });
    var wrap = RongIM.instance.$el.firstChild;
    if (params.el) {
        wrap = params.el;
    }
    $(wrap).append(instance.$el);
}

// 根据错误码获取错误信息
function getErrorMessage(errorCode, defaultMessage) {
    var locale = RongIM.instance.locale;
    if (utils.isEmpty(defaultMessage)) {
        // defaultMessage = locale.errorCode['unknown-error'];
        defaultMessage = '错误码：' + errorCode;
    }
    // 如果readyState为0，即请求没发送出去，获取对应的错误码
    if (errorCode && errorCode.readyState === 0) {
        defaultMessage = locale.errorCode['request-data-failed'];
    }
    var message = locale.errorCode[errorCode] || defaultMessage;
    utils.console.warn(message + '错误码：', errorCode);
    return message;
}

function toastError(errorCode, el) {
    var message = getErrorMessage(errorCode);
    var noLoginCodeList = [10102, 10108];
    // 登录信息验证失败，则返回登录页面
    if (noLoginCodeList.indexOf(errorCode) >= 0) {
        var im = RongIM.instance;
        im.logout();
        if (message) {
            messagebox({
                hashchangeClose: false,
                message: message
            });
        }
    } else if (message) {
        messageToast({
            el: el,
            type: 'error',
            message: message
        });
    }
}

function handleError(errorCode, defaultMessage) {
    var im = RongIM.instance;
    var message = getErrorMessage(errorCode, defaultMessage);
    var noLoginCodeList = [10102, 10108];
    // 登录信息验证失败，则返回登录页面
    if (noLoginCodeList.indexOf(errorCode) >= 0) {
        im.logout();
        if (message) {
            messagebox({
                hashchangeClose: false,
                message: message
            });
        }
    } else if (message) {
        im.$emit('messageboxHide');
        messagebox({ message: message });
    }
}

// 获取群组的类型，部门群、公司群、全员群
function getGroupType(type) {
    var locale = RongIM.instance.locale;
    var map = {
        // '0': '自建群',
        1: locale.tips.departmentGroup,
        2: locale.tips.companyGroup,
        4: locale.tips.allMemberGroup
    };
    return map[type];
}

// 获取userName，type等于3为文件助手
function getUsername(user) {
    if (!user) {
        return '';
    }
    if (user.type === 3) {
        return RongIM.instance.locale.components.getFileHelper.title;
    }
    return user.alias || user.name;
}

// 获取 username 的 html, 包含emoji转化为html
function getHtmlUsername(user, size, convertText) {
    var userName = getUsername(user) || convertText;
    if (!userName) {
        return '';
    }
    userName = utils.encodeHtmlStr(userName);
    return convertEmoji(userName, size, true);
}

// TODO: 性能考虑，删除该方法，避免运行时遍历调用
// 获取群组中人员的名称显示
function getGroupUsername(user, groupId) {
    var _user = {};
    if (user) {
        _user = RongIM.dataModel._Cache.user[user.id] || user;
    }
    _user.alias = _user.alias || RongIM.dataModel._Cache.alias[_user.id];
    var group = RongIM.dataModel._Cache.group[groupId] || {};
    var members = group.members || [];
    var alias = _user.alias || _user.name;
    for (var i = 0, len = members.length; i < len; i += 1) {
        var item = members[i];
        if (item.id === _user.id) {
            alias = _user.alias || item.groupAlias || item.alias || alias;
            return alias;
        }
    }
    return alias;
}

function getHtmlGroupUsername(user, groupId, size, replaceText) {
    var name = getGroupUsername(user, groupId);
    return getHtmlGroupUsername2(name, size, replaceText);
}

function getHtmlGroupUsername2(name, size, replaceText) {
    if (!name) {
        return name;
    }
    name = utils.encodeHtmlStr(name);
    if (!name && replaceText) {
        name = replaceText;
    }
    return convertEmoji(name, size, true);
}

function unifyUser(user) {
    var keys = ['alias', 'avatar', 'createDt', 'deptId', 'dutyName', 'id', 'name', 'path', 'star', 'orgsInfo'];
    var result = {};
    keys.forEach(function (key) {
        result[key] = getUserKey(user, key);
    });
    return result;
}

function getUserKey(user, key) {
    var result = {};
    var cacheUser = RongIM.dataModel._Cache.user[user.id] || {};
    if (key === 'star') {
        result = user[key] || cacheUser[key] || false;
    } else {
        result = user[key] || cacheUser[key] || '';
    }
    return result;
}

function getSearchUsername(user) {
    user = user || {};
    return getUsernameFormat(user.name, user.alias);
}

// 获取群组的名称，如果没有设置，则获取前十个用户的名字组成群名称
function getGroupName(group) {
    if (utils.isEmpty(group)) {
        return '';
    }
    if (!utils.isEmpty(group.name)) {
        return group.name;
    }
    var limit = 10;
    var list = [];
    $.each(group.member_names, function (i, item) {
        var length = utils.getLength(list.concat(item).join(','));
        if (length > limit) {
            return;
        }
        list.push(item);
    });
    return list.join('、');
}
function getHtmlGroupName(group, size, convertText) {
    var groupName = getGroupName(group) || convertText;
    groupName = utils.encodeHtmlStr(groupName);
    return convertEmoji(groupName, size, true);
}

// 高亮显示搜索结果群组中的成员
function getMatchedMembers(keyword, group) {
    if (utils.isEmpty(keyword)) {
        return '';
    }
    var memberNames = group.member_names || [];
    var members = [];
    memberNames.forEach(function (name) {
        var range = utils.searchStrRange(name, keyword);
        if (range) {
            members.push(highlight(name, range));
        }
    });

    if (members.length > 0) {
        return members.join('，');
    }
    return '';
}

// 获取汉字的拼音
function getLetters(name) {
    var letters = utils.convertToABC(name).pinyin.toLocaleLowerCase();
    return letters;
}

function sortHandle(one, another) {
    var compareResult = one.name.localeCompare(another.name);
    if (compareResult === 0) {
        return one.id.localeCompare(another.id);
    }
    return compareResult;
}

// 用户排序，先按拼音顺序排，如果拼音一样，则按id排
function sortUsers(users) {
    return users.sort(sortHandle);
}

// 群组排序，先按拼音顺序排，如果拼音一样，则按id排
function sortGroups(groups) {
    return groups.sort(function (one, another) {
        // return getFirstCharCode(one) - getFirstCharCode(another);
        var onePinyin = getLetters(one.name);
        var anotherPinyin = getLetters(another.name);
        if (onePinyin === anotherPinyin) {
            return one.id.localeCompare(another.id);
        }
        return onePinyin.localeCompare(anotherPinyin);
    });
}

/* function getFirstCharCode(user) {
    var letters = utils.convertToABC(getUsername(user)).pinyin.toLocaleLowerCase();
    return letters.charCodeAt(0);
} */

// 返回一个删除所有others值后的members副本
function without(members, others) {
    var otherIds = others.map(function (item) {
        return item.id;
    });
    return members.filter(function (item) {
        return otherIds.indexOf(item.id) < 0;
    });
}

// 群通知的相关文案
function groupNotificationFormat(operation) {
    var locale = RongIM.instance.locale;
    var groupNotification = {
        Create: locale.message.create,
        Created: locale.message.created,
        Join: locale.message.join,
        JoinByQRCode: locale.message.joinByQRCode,
        Invite: locale.message.invite,
        Invited: locale.message.invited,
        InviteMe: locale.message.inviteMe,
        Kick: locale.message.kick,
        Kicked: locale.message.kicked,
        Rename: locale.message.rename,
        Renamed: locale.message.renamed,
        Quit: locale.message.quit,
        Dismiss: locale.message.dismiss,
        Notice: locale.message.notice,
        OpenMute: locale.message.OpenMute,
        CloseMute: locale.message.CloseMute,
        addMute: locale.message.addMute,
        removeMute: locale.message.removeMute,
        updateManager: locale.message.updateManager,
        updateManagerSelf: locale.message.updateManagerSelf,
        departmentInvite: locale.message.departmentInvite,
        departmentInviteSelf: locale.message.departmentInviteSelf,
        joinDepartment: locale.message.joinDepartment,
        joinDepartmentSelf: locale.message.joinDepartmentSelf,
        KickDepartment: locale.message.KickDepartment,
        KickedDepartment: locale.message.KickedDepartment,
        createDepartment: locale.message.createDepartment,
        dismissDepartment: locale.message.dismissDepartment,
        renameDepartment: locale.message.renameDepartment,
        updateDepartmentManager: locale.message.updateDepartmentManager,
        updateDepartmentManagerSelf: locale.message.updateDepartmentManagerSelf
    };
    var template = groupNotification[operation];

    if (!template) {
        template = locale.message.unSupport;
        utils.console.log('不支持操作类型' + operation);
    }
    return template;
}
/*
二维码扫描入群,content.extra
{
    type: 0 //扫码入群
    operatorId //分享二维码的用户id
}
 */
function getGroupNotificationByQRCode(joinInfo) {
    var action = 'JoinByQRCode';
    var format = groupNotificationFormat(action);
    var operatorName = joinInfo.operatorName;
    var QRCodeSharerName = joinInfo.QRCodeSharerName;
    if (!QRCodeSharerName) {
        QRCodeSharerName = RongIM.dataModel.Group.getMember(joinInfo.operatorId);
    }
    QRCodeSharerName = QRCodeSharerName || joinInfo.operatorId;

    return utils.templateFormat(format, operatorName, QRCodeSharerName);
}

// 根据后台传递的数据，获取群通知具体信息
function getGroupNotification(message, authId) {
    var locale = RongIM.instance.locale;
    var self = locale.message.self;
    var actionMap = {
        GroupMemChangedNotifyMessage: {
            1: 'Invite',
            2: 'Join',
            3: 'Kick',
            4: 'Quit'
        },
        GroupNotifyMessage: {
            1: 'Create',
            2: 'Dismiss',
            4: 'Rename',
            5: 'Notice',
            21: 'OpenMute',
            22: 'CloseMute',
            23: 'addMute',
            24: 'removeMute',
            10: 'updateManager'
        }
    };
    var departmentActionMap = {
        GroupMemChangedNotifyMessage: {
            1: 'departmentInvite',
            2: 'joinDepartment',
            3: 'KickDepartment',
            4: 'Quit'
        },
        GroupNotifyMessage: {
            1: 'createDepartment',
            2: 'dismissDepartment',
            4: 'renameDepartment',
            5: 'Notice',
            21: 'OpenMute',
            22: 'CloseMute',
            23: 'addMute',
            24: 'removeMute',
            10: 'updateDepartmentManager'
        }
    };
    var content = message.content;
    var actionName;

    // 根据 targetGroup 中 type 属性判断群组是普通群/部门群
    if (content.targetGroup.type === 0) {
        actionName = actionMap[message.messageType][content.action];
    } else {
        actionName = departmentActionMap[message.messageType][content.action];
    }

    var operator = content.operatorUser.name || '';
    if (content.operatorUser.id === authId) {
        operator = self;
    }

    // 二维码进群提示
    var extra = content.extra;
    if (extra) {
        extra = JSON.parse(extra);
    }
    var isScanCode = actionName === 'Join' || actionName === 'Invite';
    isScanCode = extra && +extra.type === 0 && isScanCode;
    if (isScanCode) {
        extra.operatorName = operator;
        extra.groupId = content.targetGroup.id;
        if (extra.operatorId === authId) {
            extra.QRCodeSharerName = self;
        }
        return getGroupNotificationByQRCode(extra);
    }

    var targetUsers = content.targetUsers || [];
    var targetIncludeMe = targetUsers.some(function (item) {
        return item.id === authId;
    });
    var operatorUserIncludeMe = content.operatorUser.id === authId;
    var includeMe = targetIncludeMe || operatorUserIncludeMe;
    var separator = RongIM.instance.locale.punctuation.separator;
    // 被禁言、解除禁言 小灰条提示显示人名
    if (actionName === 'addMute' || actionName === 'removeMute') {
        targetUsers = targetUsers.map(function (item) {
            return item.name;
        }).join(separator);
    } else if (targetIncludeMe && (actionName === 'Kick' || actionName === 'KickDepartment')) {
        // 移除成员只提示自己
        targetUsers = [self];
    } else {
        targetUsers = targetUsers.map(function (item) {
            return item.id === authId ? self : item.name;
        }).join(separator);
    }
    actionName = getAction(actionName, includeMe, targetIncludeMe, operatorUserIncludeMe);
    var groupTypeName = getGroupType(content.targetGroup.type);
    var targetGroupName = content.targetGroup.name;
    var format = groupNotificationFormat(actionName);
    return utils.templateFormat(format, operator, targetUsers, targetGroupName, groupTypeName);
}

function getJrmfRedPacketOpened(message, authId) {
    var locale = RongIM.instance.locale;
    var localeFormat = utils.templateFormat;
    // PC 与 Web 消息结构不一致
    var content = message.content.message || message.content;
    var isSelfSend = content.sendPacketId === authId;
    var isSelfOpen = content.openPacketId === authId;
    if (isSelfSend) {
        if (isSelfOpen) {
            return locale.message.youOpenedRedpackOfYouSent;
        }
        return localeFormat(locale.message.redPacketOpened, content.openNickName);
    }
    return localeFormat(locale.message.youOpenedRedpack, content.sendNickName);
}

function getJrmfRedPacket(message) {
    var locale = RongIM.instance.locale;
    var SEND = utils.messageDirection.SEND;
    if (message.messageDirection === SEND) {
        return locale.message.redPacketSent;
    }
    return locale.message.redPacketReceived;
}

// 根据操作者和被操作者是否包括当前登录者来判断action
function getAction(action, includeMe, targetIncludeMe, operatorUserIncludeMe) {
    if (action === 'Create' && includeMe) {
        action = 'Created';
    }
    if (action === 'Rename' && includeMe) {
        action = 'Renamed';
    }
    if (action === 'Invite' && targetIncludeMe) {
        action = 'InviteMe';
    }
    if (action === 'Invite' && operatorUserIncludeMe) {
        action = 'Invited';
    }
    if (action === 'Kick' && targetIncludeMe) {
        action = 'Kicked';
    }
    if (action === 'updateManager' && targetIncludeMe) {
        action = 'updateManagerSelf';
    }
    if (action === 'KickDepartment' && targetIncludeMe) {
        action = 'KickedDepartment';
    }
    if (action === 'joinDepartment' && targetIncludeMe) {
        action = 'joinDepartmentSelf';
    }
    if (action === 'departmentInvite' && targetIncludeMe) {
        action = 'departmentInviteSelf';
    }
    if (action === 'updateDepartmentManager' && targetIncludeMe) {
        action = 'updateDepartmentManagerSelf';
    }
    return action;
}

// 获取会话中的操作通知
function getContactNotification(content, authId) {
    var locale = RongIM.instance.locale;
    var actionMap = {
        1: 'Add',
        2: 'Accept',
        3: 'Reject',
        4: 'Delete'
    };
    var action = actionMap[content.actionType];
    if (action !== 'Accept') {
        return '';
    }
    var targetId = content.operator.userId;
    var targetName = content.operator.name;
    var notificaiton = locale.message.passed;
    // 如果操作者是当前登录者
    if (content.operator.userId === authId) {
        targetId = content.target.userId;
        targetName = content.target.name;
        notificaiton = locale.message.pass;
    }
    notificaiton = utils.templateFormat(notificaiton, targetName || targetId);
    return notificaiton;
}

function playSound() {
    if (!playSound.audio) {
        playSound.audio = document.createElement('audio');
        playSound.audio.src = 'data:audio/mp3;base64,SUQzAwAAAAAAI1RTU0UAAAAPAAAATGF2ZjU1LjE5LjEwMAAAAAAAAAAAAAAA//NgwAAAAAAAAAAAAEluZm8AAAAHAAAAIAAADawAExMTGhoaIiIiKSkpMTExOTk5QEBASEhIUFBQUFdXV19fX2dnZ25ubnZ2dn5+foWFhY2NjY2UlJScnJykpKSrq6uzs7O7u7vCwsLKysrK0tLS2dnZ4eHh6enp8PDw+Pj4////TGF2ZjU1LjE5LjEwMAAAAAAAAAAAJAAAAAAA//NAxAAMqAKxf0AQANJEtGaZAWqBCt4gOBjDAkdBA4CGCHKO8o7hjlHJwQ85B9+sH+XeD/+H/wQccq/9GULggvq3bO9O+9oIIy8HHaRGqKZnb2h9HsQEkS5TUXAig2jQO0mJAmzw5Rz/80LEKB2DcswBmGgAw8EScZpVJJTY0M3TW9T0eiZkuJmVHUZ0uE418nBwoKqXp7v//+zvUZD0PM7Lf8xq/9/0L1qrrXTc1Ez/27Ljt//6hPySNUVf/+PYW/4apAcjDkTtlkDCf4dXle3/80DEDhc7MuJfz2gCRbUFBvp7nUdli1pfzUxFrojgNndNFSZbQZlrRWtmU9TUv/0mJoTY1PXffpV30G0rP/r/3/66RnOtq6ra0Pa31t9bf//6+qupS1GKRw171YAYChUiiAppXFNwBf/zQsQMF2MSrZ7bWrkKUwUlF5RR5ZFe5bpr3nVtJyCSKnc0ALYeBOOpJqOjwWYMTEE0GmrpLZS1E1//Ty8NAWBapKjbu6/bVqdv+/ZX/9kZp//9v/11///+32mNMDnFaAdtUttottDYW//zQMQKFose5l5bRL9W2gFkaHcQwms1F0hHAwGhOIZzWJKtaJq6LEigTlmxqeZtaBtrmz9errcwcrDyS6RtXS10//////0ocUKVGbdv/v//9Py//3ezvMICh0HGxSqwmW11yWyi0ezn//NCxAoXMybyXnoKvoeQIYcLPe+GSG5w5t2LvASCg8R0/DgT7CYUH2LEd/wji9kGU6JTu7wjwic27iACLvmRK/sd/Pp/Sc7odyEZT+8hGump1/////////+6yEUxgt4y/8t26rVBiUpt//NAxAkWoc6wANPWlJsRL/Jq42t5xB1I+jVrv43WFiM5RoMazzNc6k2JwPF6YD8kqaneTU1dpHpSNrls36BbDnc1dRzX8V9/z/FtqHQQIbVKJ2ApLdH////9odOioCrRr7a27/7j0eX/80LECRXLGwJeetS7m826jHTWJC2j+OyBF1m3H2FEPpn6FW1kyx/xqOHoIa1f/0SHtayuKtKboC8CTFjzVc+636fSi/1NGIhW/5reZQnKN7/////////2rQli1cBfA/PPMcKvaOpGDOj/80DEDRPptrGVWVAA2TwWsObbjDIEvV8/x/XIl27ercjMZiOjl5uYgrgkhK6nq5qR6/OnNqepxCR1R7t///99hmgWV+XFf////8kFgsGQEvpPp/mSFgCxglECGmEIjY4uREHgRFbdDf/zQsQYGePGtAGZOABSujmuymjpFmnDYJUlRkwfPOZ0fq8bkz0bonzeGAXiWJYlng//y3+eTjc+N3b9f/5v757nvIFSYrB3/////uDsuYJbsTz889P//+3//+Th+vj5zg6wcB4l2MolaP/zQMQMFsJi0AGPUACg6BJAaoDIfp6mEJ9Xj0x0tepKUHu/nIkoOdQPdQgECw+UgImIjtEleuROdOnH///zH0orkpMDaLX+slAOf/scJLDYmDoiAz2MDtH9X4UpsZhVttltstGM28tW//NCxAwTwk72X89QAtHbDz0ark8LkJf/X37Ut9rYSC5djnsjNpCeSIY9nfY+RAPAHj0mLneqqTTRYJnvdK///b0Lf///+9TTWiIFp////5rwA/k//zuWvj27iWki34ebOgLAzBp/ajFx//NAxBkTulK1lMKOuMadqe6nuexKddke80HRq3TTMzgLnc1Nb1RXEva5t3P//9fR+yf//+eeqznOB2SNNxTR7v/600H35/9+h7VQHQJUsgqzYlZfQ8ylfkvL5cI77PWtIkGdjLdFb7H/80LEJRSqbqTK0064oGE1rXu2ydRWFwRUgyFNJVNq4w71pdP///vt/1//zy83wMGlWU2R7aWCP+38mv/f1NaEQnKdyoOao1FQuEHGhK/rJgxLnTldiqQnu7O+612Bl2K6JpTUFkw4nNP/80DELhPaTpwA2pS4WedOMOiCLZx3c7//fv2////6OaiTSgT7Rc+G+7Rf9v1ilRomnrtpttdR5LtCaH6xeRUI8KxarnqC2r4ljqgf9E2qXqCkTmEXxpHr9hf7mftal7kMm7Qg8kKz9v/zQsQ5EqE+8l5rUHIiKJF53/+GXUmHFHvtu/9PZoX/39jswX6ilSCgifYtlzD1M1mRcnmA6rQVRQUs8FyLFW199RMPoo7qtqaH0TYzRPI00HIVksAi7NZ3vX9f3/1r///4kjKuwGHkU//zQMRKFApSlADTSrgEQzIf//WqtaYdltltttGqfUdDZ0w8OsSYRb1Let7JXhbV1K0qTrmiJONalOpnXZG54NRqkmn01Y02obvOPtp//+2j1/9//9J6NaeMD6kAlvfLoSp9+5nqllUV//NCxFQUqlbeXmtOuqSVdtttstHl1FQgj4cdQuglE+1V12EZ82i9tBLLMqmqpzLPqDwBqEBEN1R3V1aoOTndUY5NkVP/fts//1//6GtO1GjWD3fr2+zypZSVKpY6ZrtZttth86W0creG//NAxF0Tik7aXmqOupS4agm1up9Vnase55J361tSUeFVrK9ts4L+3qRGNoHias6fVv7/Vm753///STmtpNBfDQJNL3VvvLX/Y8Y99ppUrlR9/6n3cLQWCz92sGF6kyQgAkMvbaGpnef/80LEaRRqTuJeW1S6MDaedT3PfYMXe6Nt6C8AIY5IruxGjs9hHtY5aW//3X53////qaxy6gOjVjhAb1ff/v2j5VUWOiWWWW220ecRUDAUqIcwxODsKHmsj5vFz99H9hHLtNnob0xgAir/80DEcxLSToAA0pS4hx77ZyXMJtsYm///7bLp/r//oqyyM6BUMKImTzaL1zPXXnLbn1NQ5Cn/q93ngr9s2E0gfLI6ksbsrLtpokf9LJdH/ZthHkNnSstn9EdwGOYGKSabLS1WacV7u//zQsSCE/JKyl5qjrp//+n3X////RbPzgJhLF1BxibHf9exUerrqV39/3P19ClTL+3VwVZQOgnHM8dIYeJjaqdqKrtd8zerBNBPWqo6rVBLAJD9zTd26XITX0Z9//+m2/T/+/b6XllNw//zQMSOEvJSdADLTrgz4xAFd7yNFe6vE2i91KJBDSgVddktskHvUMxtkuaFobgPqFataKzM01FT3U/Q9aA63QqmCDV4xQhSgqXTR3ST4JH6UKYvVe+XVO4CynM32tL3XLbS7C0LQw5l//NCxJ0Uckp0KsqUuBf3pvclH82pb19xyIgsAjACjc92zaH/z+fSprtmwrM1yqpvHOjIubS6+/rpo3nb4//9GQoCX8TEHf0kwOAKmzQThvCLzZFEKlAmRzEQrmOKEUysje52qyKjizKl//NAxKcWOk6qXmtEul8qPSxCm5JZmiGKLzj1uMbPfkVfWSCbi/rkahpRuIgc+trLXVbbveuVTwAKlzd1GXywuG4vN78upa36uimIYbmspKm2g7E4KdFlpFH2i/oUuRVSkDpW7V55uNr/80LEqRoSTnpeysq4tZ+U5if+G+P6mbSnlebVlNF/mhUNuEgSHFxHGNKDzS5iQSQePN6SuSXJrVUkgNqX7ayEa66huG1FE8Ww9UAMKN4+mcf77k59D9Ozj8bbuyln0t0BrAmRsZLJQor/80DEnBraUmA+01C4nNVtxh2uLIXK6sV0/ROx0VWqqKVqdN3L9W0mR9QgDDCtaEl2N2IFTzRo56zLqCulJuQOfUyy/kXsctdrPCOoDvp11VXHohh3257b/20f9X7TQEWzumnX/R9Naf/zQsSLF/JOcb6jSrj6M9yju9X9Kb9ffT2ZHww9tucrbL4qqhdf7/llK+UGmTUAVqKpkwGzgaNUM2XEEyI7/o8v/3/X8YDPFh1SbKPkeZZsUtVvrtqCeoevsvlauvsRzzN1yEl0KhQRc//zQMSHEaJKVBTShLhb6wMnsAyJPAUUgdJqHAlczLBbak/0s4a3y96pyr9BX54tu8rjeLZ776kGiy0Ddkk/6g64cpi3FdQSdYZjmoEyVPQQDAVWGpNXGzuNMFSGMNcuLMnYm5WoXmMD//NCxJsPCPpUFC0KcBIYzUeExU6ye195UrbU//KqHjCy867zp6dOqPFUdx0O9MCgEJHjv2laSRUsSEQmEqZMQU1FMy45OS41qqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqq//NAxLoRgP44CjUEcKqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqr/80LEzxIwkhAAG8RMqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqo=';
    }
    playSound.audio.play();
}

// 增加一个弹出框
function mountDialog(options, callback) {
    var templateSrc = options.template;
    if (mountDialog[templateSrc] === 'busy') {
        return;
    }
    mountDialog[templateSrc] = 'busy';
    var promise = utils.loadTemplate(options.template);
    promise.then(function (html) {
        $.extend(options, { template: html });
        options.mixins = options.mixins || [];
        var localeMix = {
            mounted: function () {
                // 当页面有跳转的时候，关闭弹层
                var im = RongIM.instance;
                im.$on('imLogouted', this.close);
                window.addEventListener('hashchange', this.close);
            },
            computed: {
                // 获取对应的相关文案
                locale: function () {
                    var locale = RongIM.instance.locale;
                    var name = utils.kebabToCamel(options.name);
                    return $.extend(true, {}, locale, locale.components[name]);
                }
            },
            methods: {
                close: function () {
                    this.$destroy();
                    $(this.$el).remove();
                },
                localeFormat: utils.templateFormat
            },
            destroyed: function () {
                var im = RongIM.instance;
                im.$off('imLogouted', this.close);
                window.removeEventListener('hashchange', this.close);
            }
        };
        options.mixins.push(localeMix);
        options.parent = RongIM.instance;
        var Dialog = Vue.extend(options);
        var instance = new Dialog({
            el: document.createElement('div')
        });
        var wrap = RongIM.instance.$el.firstChild;
        $(wrap).append(instance.$el);
        if ($.isFunction(callback)) callback(instance);
    }).fail(function () {
        var locale = RongIM.instance.locale;
        var im = RongIM.instance;
        var message = locale.errorCode['request-data-failed'];
        im.$emit('messageboxHide');
        if (message) messageToast({ message: message });
    }).always(function () {
        delete mountDialog[templateSrc];
    });
}

// 根据server传过来的range，来高亮对应的名字
function getUsernameHighlight(user) {
    var str = highlight(user.name, user.range, true);
    if (user.alias) {
        str = getUsernameFormat(str, highlight(user.alias, user.aliasRange, true));
    }
    return str;
}

// 根据range高亮对应名字
function getHighlight(duty, notfilterLabel) {
    var str = highlight(duty.name, duty.range, notfilterLabel);
    return str;
}

// 高亮对应的字符
function highlight(string, keyword, notfilterLabel) {
    if (utils.isEmpty(keyword)) {
        return string;
    }
    if ($.isArray(keyword)) {
        var range = keyword;
        var start = range[0];
        var length = range[1];
        keyword = string.substr(start, length);
    }
    keyword = keyword.replace(/([\\^$*+?()[\]])/g, '\\$1');
    var pattern;
    if (notfilterLabel) {
        pattern = new RegExp('(' + keyword + ')', 'ig');
        return string.replace(pattern, '<mark>$1</mark>');
    }
    var randomStr = Date.now() + '_' + Math.floor(Math.random() * 1000);
    pattern = /<[^>]+>/ig;
    var contents = [];
    // 使用随机字符串替换不可被replace的部分内容
    string = string.replace(pattern, function (words) {
        contents.push(words);
        return randomStr;
    });
    // 匹配内容，添加mark标记，使用随机字符串分割目标，以免随机字符串匹配keyword
    pattern = new RegExp('(' + keyword + ')');
    string = string.split(randomStr).map(function (temp) {
        return temp.replace(pattern, '<mark>$1</mark>');
    }).join(randomStr);
    // 将随机字符串恢复
    return string.replace(new RegExp(randomStr, 'g'), function () {
        return contents.shift();
    });
}

function getUsernameFormat(name, alias) {
    var str = name;
    if (alias) {
        str = alias + '(' + str + ')';
    }
    return str;
}

// 判断是否是同一条消息
function equalMessage(messageA, messageB) {
    var result;
    messageA = messageA || {};
    if (messageA.messageId) {
        result = messageA.messageId === messageB.messageId;
    } else {
        result = messageA.messageUId === messageB.messageUId;
    }
    return result;
}

/*
创建一个通知消息
参数：
conversationType: 会话类型
targetId: 发送对象的targetId
context: 消息内容
*/
function createNotificationMessage(conversationType, targetId, content) {
    var msg = new RongIMLib.InformationNotificationMessage({ message: content });
    var params = {
        conversationType: conversationType,
        targetId: targetId,
        objectName: 'RC:InfoNtf',
        content: msg,
        sentStatus: utils.sentStatus.SENT
    };
    return params;
}

// 根据keyword搜索出对应昵称的用户
function searchAlias(users, keyword, userApi) {
    var searchedIdList = users.map(function (item) {
        return item.id;
    });
    var aliasList = userApi.getAlias();

    $.each(aliasList, function (key, value) {
        var user = {
            id: key,
            alias: value
        };
        if (!user.alias) {
            return;
        }
        var range = utils.searchStrRange(user.alias, keyword);
        if (!range) {
            return;
        }
        var existed = searchedIdList.indexOf(user.id) >= 0;
        if (existed) {
            return;
        }
        userApi.get(user.id, function (errorCode, userInfo) {
            if (errorCode) {
                utils.console.log('userApi.get failed:' + errorCode + ', userid:' + userInfo.id);
                return;
            }
            users.push($.extend({ aliasRange: range }, userInfo));
        });
    });
}

var CallType = {
    MEDIA_AUDIO: 1,
    MEDIA_VEDIO: 2
};

var RCCallStatus = {
    /*!
      初始状态
      */
    // RCCallIdle      =  0,
    /*!
      正在呼出
      */
    RCCallDialing: 1,
    /*!
      正在呼入
      */
    RCCallIncoming: 2,
    /*!
      收到一个通话呼入后，正在振铃
      */
    RCCallRinging: 3,
    /*!
      正在通话
      */
    RCCallActive: 4,
    /*!
      已经挂断
      */
    RCCallHangup: 5
};

var RemoteControlStatus = {
    /*
    请求主控另一端中
     */
    Request: 1,
    1: 'Request',
    /*
    邀请另一端主控中
     */
    Invite: 2,
    2: 'Invite',
    /*
    收到另一端请求主控
     */
    ReceiveRequest: 3,
    3: 'ReceiveRequest',
    /*
    收到另一端邀请主控
     */
    ReceiveInvite: 4,
    4: 'ReceiveInvite',
    /*
    通过请求, 建立中
     */
    Building: 5,
    5: 'Building',
    /*
    建立成功, 本端为主控
     */
    RequestBuilded: 6,
    6: 'RequestBuilded',
    /*
    建立成功, 本端为被控
     */
    InviteBuilded: 7,
    7: 'InviteBuilded',
    /*
    空闲
     */
    Idle: 0,
    0: 'Idle'
};

var RemotePlatform = {
    win32: 0,
    darwin: 1,
    linux: 2
};

var RemoteControlHangupReason = {
    // 己方取消已发出的请求
    Cancel: 100,
    100: 'Cancel',
    // 己方拒绝收到的请求
    Reject: 101,
    101: 'Reject',
    // 己方挂断
    Hangup: 102,
    102: 'Hangup',
    // 己方忙碌
    BusyLine: 103,
    103: 'BusyLine',
    // 己方未接听
    NoResponse: 104,
    104: 'NoResponse',
    // 己方当前引擎不支持
    EngineUnsupported: 105,
    105: 'EngineUnsupported',
    // 己方网络出错
    NetworkError: 106,
    106: 'NetworkError',
    // 其他端已处理
    OtherDeviceHadAccepted: 107,
    107: 'OtherDeviceHadAccepted',
    // 对方取消已发出的请求
    RemoteCancel: 200,
    200: 'RemoteCancel',
    // 对方拒绝收到的请求
    RemoteReject: 201,
    201: 'RemoteReject',
    // 对方挂断
    RemoteHangup: 202,
    202: 'RemoteHangup',
    // 对方忙碌
    RemoteBusyLine: 203,
    203: 'RemoteBusyLine',
    // 对方未接听
    RemoteNoResponse: 204,
    204: 'RemoteNoResponse',
    // 对方当前引擎不支持
    RemoteEngineUnsupported: 205,
    205: 'RemoteEngineUnsupported',
    // 对方网络错误
    RemoteNetworkError: 206,
    206: 'RemoteNetworkError',
    // 远程桌面连接失败
    DesktopConnectFailed: 300,
    300: 'DesktopConnectFailed'
};

function changeHangupReceiveReason(direction, reason) {
    var isReceive = direction === RongIMLib.MessageDirection.RECEIVE;
    if (isReceive && reason === RemoteControlHangupReason.RemoteNoResponse) {
        reason = RemoteControlHangupReason.NoResponse;
        direction = RongIMLib.MessageDirection.SEND;
        isReceive = false;
    }
    if (isReceive) {
        // 转化 对方发过来的挂断原因
        // 如: 已取消(101) 转化为 对方已取消(201)
        reason = RemoteControlHangupReason[reason];
        reason = RemoteControlHangupReason['Remote' + reason];
    }
    return reason;
}

function getRemoteControlSummary(message) {
    var reason = message.content.reason;
    var im = RongIM.instance;
    var locale = im.locale;
    locale = locale.components.remoteHangup;
    var content = locale[reason] || locale[RemoteControlHangupReason.RemoteNetworkError];
    var direction = message.messageDirection;
    if (direction === RongIMLib.MessageDirection.SEND) {
        content += '<span class="rong-remote-summary-sent"></span>';
    } else {
        content = '<span class="rong-remote-summary-receive"></span>' + content;
    }
    return content;
}

// 文件消息是否取消上传
function isCanceled(message) {
    var isFileMessage = message.messageType === 'LocalFileMessage' || message.messageType === 'FileMessage';
    var isFlieCancelStatus = isFileMessage ? message.content.status === 0 : false;
    return isFileMessage && isFlieCancelStatus;
}

function sameConversaton(one, another) {
    var oneConversationType = +one.conversationType;
    var anotherConversationType = +another.conversationType;
    var sameConversationType = oneConversationType === anotherConversationType;
    var sameTargetId = one.targetId === another.targetId;
    return sameConversationType && sameTargetId;
}

var UserType = {
    STAFF: 0,
    VISITOR: 1,
    OTHERAPP: 2,
    ROBOT: 3
};

var UserState = {
    NORMAL: 0,
    VISITOR: 1,
    DELETED: 2
};

var FriendState = {
    // 邀请 0
    INVITE: 0,
    // 受邀 1
    INVITEE: 1,
    // 接受 2
    ACCEPT: 2,
    // 被接受 3
    ACCEPTEE: 3
};

var ApproveState = {
    // 请及时审批 0 非系统状态,用于控制显示
    0: 'REMIND',
    // 审批中 1
    1: 'PENDING',
    // 审批通过 2
    2: 'PASSED',
    // 审批拒绝 3
    3: 'REFUSED'
};

var OrgType = {
    // 员工
    STAFF: 0,
    // 部门
    DEPT: 1,
    // 公司
    COMPANY: 2
};

var CompanyType = {
    // 默认
    DEFAULT: 0,
    // 独立子公司
    AUTONOMY: 1
};

var ErrorCode = {
    QR_INVALID_TOKEN: 10122,
    ACCOUNT_IS_LOCKED: 10128,
    INVALID_TOKEN: 10123,
    RCEC_FRIEND_REQUEST_TIMEOUT: 10403,
    DEPART_NOT_EXIST: 10200,
    SENSITIVE_WORDS_INTERCEPT: 21501,
    SENSITIVE_WORDS_REPLACED: 21502
};
// 获取私有云下载链接
function getDownloadUrl(config, data) {
    var url = '';
    var uploadType = config.upload.type;
    if (uploadType === 'RongCloud') {
        url = data.rc_url.path;
        var urlType = data.rc_url.type.toString();
        if (urlType === '0') {
            url = config.upload.file.domain + data.rc_url.path;
        }
    }
    return url;
}

// 新消息提示，消息条数提示在头像右上角显示，消息数最多显示99，
// 超过两位数（100-999）显示：99+，超过三位数（1000以上）显示： ⋯。会话列表、会话消息、PIN消息显示逻辑一致。
function formateUnreadCount(count) {
    if (count === 0) {
        return '';
    }
    var display = count < 100 ? count.toString() : '99+';
    display = count < 1000 ? display : '⋯';
    return display;
}

function getEmojiHTML(emoji, unicode, size) {
    var getUrl = client.getEmojiUrl;
    var url = getUrl(unicode);
    if (!url) {
        return emojiToHTML(emoji, size);
    }
    var symbol = 'other';
    var list = RongIMEmoji.list;
    list.forEach(function (item) {
        if (item.emoji === emoji) {
            var name = item.symbol;
            symbol = name.substring(1, name.length - 1);
        }
    });
    var emojiItem = {
        en: symbol, zh: symbol, tag: emoji, position: '0px, 0px', background: url
    };
    return RongIMEmoji.createNodeHTML(emojiItem, size);
}

/**
 * 原生emoji 转化为 unicode码
 * TODO 待优化, 当前转化不完善, 部分emoji转化错误. 当前仅把已知错误emoji直接判断转化.
 * @param  {String} emoji 原生emoji
 */
function emojiToUnicode(emoji) {
    /* emoji: 方块1 */
    if (emoji === '1\uFE0F\u20E3' || emoji === '1\u20E3') {
        return '0031-fe0f-20e3';
    }
    var unicodeList = [];
    for (var i = 0; i < emoji.length; i += 2) {
        unicodeList.push(emoji.codePointAt(i).toString(16));
    }
    return unicodeList.join('-');
}

function convertUnicode(unic) {
    var unicodes = escape(unic).split('%u');
    unicodes = unicodes.filter(function (code) {
        return code !== '';
    });
    return unicodes.map(function (code) {
        if (code.indexOf('f') !== -1 || code.indexOf('F') !== -1) {
            return '1' + code;
        }
        return code;
    }).join('-');
}

// emoji的 html 使用模块内的图片
function emojiConvertHTML(content, size) {
    var emojiNativeReg = utils.emojiNativeReg;
    var emojiUnicodeReg = utils.emojiUnicodeReg;
    content = content.replace(emojiNativeReg, function (emoji) {
        var unicode = emojiToUnicode(emoji);
        return getEmojiHTML(emoji, unicode, size);
    });
    content = content.replace(emojiUnicodeReg, function (unic) {
        var unicode = convertUnicode(unic);
        return getEmojiHTML(unic, unicode, size);
    });
    return content;
}

function isSysUser(user) {
    return user.type === UserType.OTHERAPP;
}

/*
    params.container
    params.node
    params.hander
*/
function resizeNode(params) {
    var isString = function (str) {
        return $.type(str) === 'string';
    };
    var getNode = function (node) {
        if (isString(node)) {
            node = $(node);
        }
        return node;
    };

    var getContainer = function () {
        var im = RongIM.instance.$el;
        return '#' + im.getAttribute('id');
    };
    var im = RongIM.instance;
    var container = params.container || getContainer();
    container = getNode(container);

    var node = params.node;
    node = getNode(node);

    var hander = params.hander || '.rong-resize';
    var getHander = function () {
        return isString(hander) ? node.find(hander) : hander;
    };
    hander = getHander();

    var onbefore = params.onbefore || $.noop;
    var onresize = params.onresize || $.noop;
    var onended = params.onended || $.noop;

    var direction = params.direction;
    var directionMap = {
        top: {
            page: 'pageY',
            attr: 'height',
            getRange: function (parent, resize) {
                return parent - resize;
            }
        },
        right: {
            page: 'pageX',
            attr: 'width',
            getRange: function (parent, resize) {
                var config = RongIM.config;
                var layout = config.layout;
                var navBar = layout.navBar;
                var navBarWidth = navBar.width.min;
                return resize - navBarWidth;
            }
        }
        // ... 按需扩展
    };

    direction = directionMap[direction];

    var page = direction.page;
    var attr = direction.attr;

    var parentRange = parseInt(container.css(attr));

    onbefore();

    var getBound = function (type) {
        var propName = [type, attr].join('-');
        var bound = node.css(propName);
        return parseInt(bound);
    };

    var min = getBound('min');
    var max = getBound('max');

    var resumeSelect = function () {
        // Chrome
        document.onselectstart = function () {
            return true;
        };
        // FireFox Opera12.6+
        var bodyEl = $('body');
        bodyEl.removeClass('rong-disable-select');
        // Opera12.5 及以下
        bodyEl.attr('unselectable', 'off');
    };

    var pauseSelect = function () {
        // Chrome IE6-9
        document.onselectstart = function () {
            return false;
        };
        // FireFox Opera
        var bodyEl = $('body');
        bodyEl.addClass('rong-disable-select');
        // Opera12.5 及以下
        bodyEl.attr('unselectable', 'on');
    };

    var resizeP = 0;
    var isMouseDown = false;

    hander.mousedown(function () {
        isMouseDown = true;
        pauseSelect();
    });

    var resizePause = function () {
        if (isMouseDown) {
            isMouseDown = false;
            onended();
            resumeSelect();
        }
    };

    container.bind('click mouseup', resizePause).mousemove(function (e) {
        if (isMouseDown) {
            resizeP = e[page];
            parentRange = parseInt(container.css(attr));
            var range = direction.getRange(parentRange, resizeP);

            if (range > max) {
                range = max;
            }
            if (range < min) {
                range = min;
            }
            var isLegal = (range <= max && range >= min);
            if (isLegal) {
                onresize({
                    range: range
                });
            } else {
                isMouseDown = false;
            }
        }
    });

    im.$el.onmouseleave = function () {
        resizePause();
    };
}

function getResizeDirection(params) {
    // 实时距离
    var range = params.range;
    var bound = params.bound;
    var directions = params.directions;

    var min = parseInt(bound.min);
    var max = parseInt(bound.max);

    var pre = (directions[0] === 'left') ? 'x' : 'y';
    var prop = 'normal';
    if ((range + 5) >= max) {
        prop = directions[0];
    } else if (range <= min) {
        prop = directions[1];
    } else {
        prop = 'middle';
    }
    return [pre, prop].join('-');
}

function resizeNavNode(context, im) {
    var config = RongIM.config;
    var layout = config.layout;
    var navBar = layout.navBar;
    var navBarWidth = navBar.width.min;

    var rongList = $(context.$el);

    var resizeDirection = im.resizeDirection;

    resizeNode({
        el: im.$el,
        node: rongList,
        direction: 'right',
        onbefore: function () {
            im.resizeRange += 1;
            im.resizeRange -= 1;
        },
        onresize: function (result) {
            context.isResizing = true;
            var range = result.range;
            var node = im.resizeNode;
            node.main.marginLeft = range + navBarWidth;
            node.rongList.width = range;
            resizeDirection.use = resizeDirection.temp;
        },
        onended: function () {
            context.isResizing = false;
            resizeDirection.use = 'normal';
        }
    });
    var getBound = function (name) {
        return rongList.css(name);
    };
    context.bound.width = {
        min: getBound('min-width'),
        max: getBound('max-width')
    };
}

var cacheTextRanderArr = [];
var cacheTextRanderMax = 200;
function getTextRenderWidth(str, fontSize) {
    var key = str + '~;' + fontSize;
    // 查找缓存是否存在
    for (var i = cacheTextRanderArr.length - 1; i >= 0; i -= 1) {
        var item = cacheTextRanderArr[i];
        if (item.key === key) {
            return item.value;
        }
    }
    var width = 0;
    var $span = $('<span style="position:absolute;left: -999999px;">' + str + '</span>');
    if (fontSize) {
        $span.css('fontSize', fontSize);
    }
    var $contain = RongIM.instance ? RongIM.instance.$el : document.body;
    $($contain).append($span);
    width = parseInt($span.css('width'));
    $span.remove();

    // 缓存不存在加入到缓存，并判断缓存是否超限超限则清除
    cacheTextRanderArr.push({ key: key, value: width });
    if (cacheTextRanderArr.length > cacheTextRanderMax) {
        cacheTextRanderArr.shift();
    }
    return width;
}

function isPrivate(config) {
    var uploadType = config.upload.type;
    return uploadType === 'RongCloud';
}
/*
说明：根据对象中 id 去除重复对象
参数：arr 对象数组对象要包含 id 属性
*/
function removeDuplicatesById(arr) {
    if (!$.isArray(arr)) {
        return [];
    }
    var cacheKey = [];
    var result = [];
    arr.forEach(function (item) {
        var key = item.id;
        if (cacheKey.indexOf(key) === -1) {
            result.push(item);
            cacheKey.push(key);
        }
    });
    cacheKey = null;
    return result;
}

/*
说明： 根据 server 返回获取详细组织结构信息 path 如果有独立子公司从独立子公司开始显示 path
*/
function getFullOrgInfo(orgsInfo) {
    orgsInfo = orgsInfo || [];
    return orgsInfo.map(function (item) {
        /*
        path 部门路径总公司信息在第一级
        独立子公司在第二级
        */
        var company = item.path[0];
        var autonomy = item.path[1];
        var isAutonomyCompany = RongIM.dataModel.Organization.isAutonomyCompany((autonomy || {}).id);
        if (autonomy && isAutonomyCompany) {
            company = autonomy;
        }
        return {
            companyId: company.id,
            companyName: company.name,
            isAutonomyCompany: isAutonomyCompany,
            deptId: item.id,
            deptName: item.name,
            deptType: item.type,
            path: item.path
        };
    });
}
var downloadersObj = {};
var downloaders = {
    get: function (key) {
        return downloadersObj[key];
    },
    add: function (key, value) {
        downloadersObj[key] = value;
    },
    delete: function (key) {
        delete downloadersObj[key];
    },
    clear: function () {
        Object.keys(downloadersObj).forEach(function (key) {
            if (downloadersObj.contains(key)) {
                delete downloadersObj[key];
            }
        });
    },
    getAll: function () {
        return downloadersObj;
    }
};

function groupSummaryFormat(content, locale) {
    var isVideo = content.mediaType === CallType.MEDIA_VEDIO;
    var str = '';
    if (isVideo) {
        str = locale.voip.video;
    } else {
        str = locale.voip.audio;
    }
    var selfUnread = [5, 11].indexOf(content.status) !== -1;
    if (content.status === 8) {
        str = locale.voip.summaryCodeMap[8];
    } else if (selfUnread) {
        str += ' ' + locale.voip.unAccept;
    } else {
        str += ' ' + locale.voip.end;
    }
    return str;
}

function getPathName(item, orgApi) {
    var boundSymbol = ' > ';
    var pathList = [];
    if (item.org_path) {
        pathList = item.org_path;
    }
    if (item.path) {
        pathList = item.path;
    }
    /* 多公司搜索员工使用 orgs_info */
    if (item.orgs_info) {
        // 取主公司信息
        var mainOrgInfo = item.orgs_info[0] || {};
        item.orgs_info.forEach(function (orgInfo) {
            var company = orgInfo.path[0] || {};
            var subcompany = orgInfo.path[1];
            var isAutomy = subcompany && orgApi.isAutonomyCompany(subcompany.id);
            if (isAutomy) {
                company = subcompany;
            }
            var mainCompanyId = item.companyId || item.company_id;
            if (company.id === mainCompanyId) {
                mainOrgInfo = orgInfo;
            }
        });
        pathList = mainOrgInfo.path || [];
    }
    // 增加日志用于排查错误
    if (!(pathList instanceof Array)) {
        console.error('pathList error!\npathList => ', pathList, '\nitem => ', item);
    }
    // 排除无效信息
    pathList = pathList.filter(function (pathItem) {
        return !utils.isEmpty(pathItem.id);
    });

    var pathNameList = [];
    pathNameList = pathList.map(function (dept) {
        return dept.name;
    });
    // 第二级是独立子公司则从独立子公司开始显示路径
    var subcompany = pathList[1];
    var isAutonomy = subcompany && orgApi.isAutonomyCompany(subcompany.id);
    if (isAutonomy) {
        pathNameList.shift();
    }
    var pathName = pathNameList.join(boundSymbol);
    return pathName;
}

function getPathNameEllipsis(item, orgApi) {
    var pathName = item.pathName || getPathName(item, orgApi);
    var length = utils.getLength(pathName);
    pathName = utils.slice(pathName, Math.max(0, length - 5));
    return pathName;
}

function getPathNameSuffix(item, orgApi) {
    var pathName = item.pathName || getPathName(item, orgApi);
    var length = getPathNameEllipsis(item, orgApi).length;
    var suffix = pathName.substring(length);
    return suffix;
}

/*
是否显示 RemoteControl
1. pc下
2. setting.js 中的设置项
3. 个人聊天
 */
function isSupportRemoteControl(conversation) {
    var isPc = RongIM.config.support.remoteControl;
    // var isSupport = RongIM.serverConfig.remote_control.enabled || RongIM.serverConfig.remote_control.enable;
    var remoteControlConfig = RongIM.serverConfig.remote_control;
    var isSupport = remoteControlConfig ? RongIM.serverConfig.remote_control.enabled || RongIM.serverConfig.remote_control.enable : false;
    var isPrivateConversation = conversation.conversationType === RongIMLib.ConversationType.PRIVATE;
    return isPc && isPrivateConversation && isSupport;
}

/*
过滤除了mark标签以外的html
*/
function filterMark(str) {
    var randomStr1 = Date.now() + '_' + Math.floor(Math.random() * 1000);
    var randomStr2 = Date.now() + '_' + Math.floor(Math.random() * 1000);
    if (!str) {
        return '';
    }
    str = str.replace(/<mark>/g, randomStr1);
    str = str.replace(/<\/mark>/g, randomStr2);
    str = utils.encodeHtmlStr(str);
    // 将随机字符串恢复
    str = str.replace(new RegExp(randomStr1, 'g'), function () {
        return '<mark>';
    }).replace(new RegExp(randomStr2, 'g'), function () {
        return '</mark>';
    });
    return str;
}

function encodeUrl(url) {
    var reg = /^(https|http|ftp|file|data)/;
    if (!reg.test(url)) {
        return '';
    }
    var this_url = url.split('?');
    if (this_url.length < 2) {
        this_url = url;
    } else {
        this_url = this_url.map(function (item, index) {
            if (index !== 0) {
                item = utils.encodeHtmlStr(item);
            }
            return item;
        }).join('?');
    }
    return this_url;
}

function getDeviceId() {
    var deviceId = utils.cache.get('deviceId');
    if (utils.isEmpty(deviceId)) {
        deviceId = createUid();
        utils.cache.set('deviceId', deviceId);
    }
    return deviceId;
}

function createUid() {
    var deviceId = RongIM.system.getDeviceId();
    var config = RongIM.instance.config;
    if (deviceId) {
        deviceId = deviceId + config.appkey + config.product.name.en;
        deviceId = RongIMLib.RongUtil.MD5(deviceId);
        if (deviceId.length === 32) {
            deviceId = deviceId.substr(5, 22);
            utils.console.log('deviceId', deviceId);
            return deviceId;
        }
        utils.console.log('md5加密异常');
    }
    return '';
}

function getFileName(url) {
    return url.split('?')[0].split('/').pop();
}

var origin = location.protocol + '//' + location.host;
/**
 * 将远程地址转为 PC 主进程提供的文件缓存服务请求地址
 * @param {string} url 目标文件远程地址
 * @param {string} dir 缓存服务指定的缓存目录
 * @returns {string} 转换后的请求地址
 */
function trans2Localfile(url, dir) {
    if (RongIM.system.platform.startsWith('web-')) {
        return url;
    }
    return url ? (
        origin + '/localfile/' + getFileName(url) + '?url=' + encodeURIComponent(url) + '&dir=' + (dir || '')
    ) : url;
}

/**
 * 确定缓存服务是否已缓存指定网络资源
 * @param {string} url 网路资源路径
 * @param {*} dir 所缓存的目标目录
 * @param {*} callback 结果回调，若存在则回传本地缓存路径，否则回传 null
 */
function localfileInDesk(url, dir, callback) {
    if (RongIM.system.platform.startsWith('web-')) {
        callback(null);
        return;
    }
    var requestUrl = origin + '/localpath?url=' + encodeURIComponent(url) + '&dir=' + (dir || '');
    $.ajax({
        url: requestUrl,
        dataType: 'json',
        method: 'GET'
    }).then(function (response) {
        callback(response.localPath);
    }, function () {
        callback(null);
    });
}


function showImage(messageList, messageUId, locale) {
    RongIM.dataModel.File.getImageDownloadToken(function (token) {
        var dataSource = [];
        var defaultIndex = 0;
        for (var j = 0; j < messageList.length; j += 1) {
            var msg = messageList[j];
            var content = msg.content;
            if (msg.messageType === 'ReferenceMessage') {
                content = msg.content.content;
            }
            var url = content.imageUri || content.sightUrl;
            if (url) {
                // 网络资源内容，将资源路径转为本地文件代理服务路径
                if (url.indexOf('?') !== -1) {
                    url += '&token=' + token;
                } else {
                    url += '?token=' + token;
                }
                url = trans2Localfile(url, 'media');
            } else {
                // 本地磁盘文件，使用文件代理服务读取该文件
                url = '/file2http/' + getFileName(content.localPath) + '?url=' + encodeURIComponent(content.localPath);
            }
            var source = {
                thumbnail: content.content,
                url: url,
                type: content.messageName,
                name: content.name || '',
                uid: msg.messageUId
            };
            dataSource.push(source);
            if (msg.messageUId === messageUId) {
                defaultIndex = j;
            }
        }
        var options = {
            dataSource: dataSource,
            defaultIndex: defaultIndex,
            locale: locale
        };
        RongIM.imageViewer.openWin(options);
    });
}

function sortByMydeptAndMajorCompany(list, majorCompanyId) {
    list.sort(function (a, b) {
        if (utils.isEmpty(a.myDept) && !utils.isEmpty(b.myDept)) {
            return 1;
        }
        if (!utils.isEmpty(a.myDept) && utils.isEmpty(b.myDept)) {
            return -1;
        }
        if (!utils.isEmpty(a.myDept) && !utils.isEmpty(b.myDept)) {
            return a.order - b.order;
        }
        return a.order - b.order;
    });
    var index = 0;
    list.forEach(function (item, i) {
        if (item.id === majorCompanyId) {
            index = i;
        }
    });
    var majorCompany = list.splice(index, 1);
    list.unshift(majorCompany[0]);
}

function timestampToDisplayTime(timestamp) {
    var locale = RongIM.instance.locale;
    var minute = Math.floor(timestamp / 60 / 1000);
    if (minute > 0) {
        return minute + ' ' + locale.time.minutes;
    }
    var second = Math.ceil(timestamp / 1000);
    return second + ' ' + locale.time.seconds;
}

// 密码加密传输
function encryptPassword(value) {
    // 加密公钥由 server 在 configuration_all 下发配置
    var jsencrypt = new JSEncrypt();
    var publicKey = RongIM.serverConfig.security.password_public_key_rsa;
    jsencrypt.setPublicKey(publicKey);
    value = jsencrypt.encrypt(value);
    return value;
}

RongIM.common = {
    timestampToDisplayTime: timestampToDisplayTime,
    sortByMydeptAndMajorCompany: sortByMydeptAndMajorCompany,
    showImage: showImage,
    trans2Localfile: trans2Localfile,
    localfileInDesk: localfileInDesk,
    getPathName: getPathName,
    getPathNameEllipsis: getPathNameEllipsis,
    getPathNameSuffix: getPathNameSuffix,
    getFullOrgInfo: getFullOrgInfo,
    removeDuplicatesById: removeDuplicatesById,
    getTextRenderWidth: getTextRenderWidth,
    CallType: CallType,
    RCCallStatus: RCCallStatus,
    resizeNode: resizeNode,
    getResizeDirection: getResizeDirection,
    resizeNavNode: resizeNavNode,
    sameConversaton: sameConversaton,
    textMessageFormat: textMessageFormat,
    groupNoticeFormat: groupNoticeFormat,
    groupSummaryFormat: groupSummaryFormat,
    latestGroupNoticeFormat: latestGroupNoticeFormat,
    convertMessage: convertMessage,
    buildMessage: buildMessage,
    messagebox: messagebox,
    messageToast: messageToast,
    getErrorMessage: getErrorMessage,
    handleError: handleError,
    toastError: toastError,
    getGroupType: getGroupType,
    getUsername: getUsername,
    getHtmlUsername: getHtmlUsername,
    getGroupUsername: getGroupUsername,
    getHtmlGroupUsername: getHtmlGroupUsername,
    getHtmlGroupUsername2: getHtmlGroupUsername2,
    unifyUser: unifyUser,
    getSearchUsername: getSearchUsername,
    getGroupName: getGroupName,
    getHtmlGroupName: getHtmlGroupName,
    getMatchedMembers: getMatchedMembers,
    sortUsers: sortUsers,
    without: without,
    // showGroupNotification: showGroupNotification,
    getGroupNotification: getGroupNotification,
    getJrmfRedPacket: getJrmfRedPacket,
    getJrmfRedPacketOpened: getJrmfRedPacketOpened,
    playSound: playSound,
    mountDialog: mountDialog,
    highlight: highlight,
    getHighlight: getHighlight,
    getUsernameHighlight: getUsernameHighlight,
    filterMark: filterMark,
    equalMessage: equalMessage,
    createNotificationMessage: createNotificationMessage,
    searchAlias: searchAlias,
    isCanceled: isCanceled,
    UserType: UserType,
    FriendState: FriendState,
    OrgType: OrgType,
    CompanyType: CompanyType,
    getContactNotification: getContactNotification,
    UserState: UserState,
    getDownloadUrl: getDownloadUrl,
    formateUnreadCount: formateUnreadCount,
    isSysUser: isSysUser,
    ApproveState: ApproveState,
    ErrorCode: ErrorCode,
    sortGroups: sortGroups,
    emojiConvertHTML: emojiConvertHTML,
    isPrivate: isPrivate,
    downloaders: downloaders,
    RemoteControlStatus: RemoteControlStatus,
    RemoteControlHangupReason: RemoteControlHangupReason,
    getRemoteControlSummary: getRemoteControlSummary,
    isSupportRemoteControl: isSupportRemoteControl,
    changeHangupReceiveReason: changeHangupReceiveReason,
    RemotePlatform: RemotePlatform,
    encodeUrl: encodeUrl,
    getDeviceId: getDeviceId,
    encryptPassword: encryptPassword
};
}(RongIM, {
    RongIMLib: RongIMLib,
    jQuery: jQuery,
    Vue: Vue
}));
