(function (RongIM, dependencies, components) {
'use strict';

var common = RongIM.common;
var utils = RongIM.utils;
var dialog = RongIM.dialog;
var $ = dependencies.jQuery;
var CallType = common.CallType;
var RongIMLib = dependencies.RongIMLib;
var win = dependencies.win;
var RemoteControlStatus = common.RemoteControlStatus;
var RemoteControlHangupReason = common.RemoteControlHangupReason;
var RemoteCtrlOpt = RongIM.remoteControl;
var browserWindow = RongIM.browserWindow;
var remoteWindowHandler = browserWindow.remoteWindowHandler;
var remoteApi;

/*
说明： 会话界面，包含消息列表，输入框，历史消息，会话设置，群设置等组件
    两种状态，1. 搜索消息跳转进入会话 query 参数携带 messageUId 2. 正常会话
*/
components.getConversation = function (resolve, reject) {
    var im = RongIM.instance;
    var support = im.config.support;
    var dataModel = im.dataModel;
    remoteApi = dataModel.RemoteControl;
    var userApi = dataModel.User;
    var screenshot = RongIM.screenshot;
    var options = {
        name: 'conversation',
        template: 'templates/conversation/conversation.html',
        data: function () {
            return {
                busy: false,

                /*
                panel可选值：
                conversation-setting: 单聊设置
                group-setting: 群聊设置
                history: 历史消息
                */
                panel: null,
                conversation: {
                    conversationType: this.conversationType,
                    targetId: this.targetId
                },
                newMessage: null,
                // 单聊显示用户在线状态
                userStatus: 'offline',
                // 输入框是否自动获取焦点
                autoFocus: true,
                // 群会话 - 自己是否在当前群组中
                inGroup: true,
                disableExecutive: false,
                // 群会话 - 是否被禁言
                isBanned: false,
                // 群会话 - 是否有效群组
                isInvalidGroup: true,
                // 群会话 - 群成员
                members: [],
                // 公众号 - 菜单
                menuEnabled: false,
                // 公众号 - 輸入框
                inputEnabled: true,
                // 公众号菜单信息
                menuInfo: {},
                remoteControl: {},
                remoteControlEnable: RongIM.serverConfig.remote_control.enable,
                // 引用消息
                quote: null,
            };
        },
        computed: getComputed(im),
        components: {
            status: components.getStatus,
            avatar: components.getAvatar,
            'message-list': components.getMessageList,
            'message-input': components.getMessageInput,
            history: components.getHistory,
            'conversation-setting': components.getConversationSetting,
            'group-setting': components.getGroupSetting,
            'group-notice': components.getGroupNotice,
            'public-menu': components.getPublicMenu,
            'public-detail': components.getPublicInfo,
            'remote-control': components.getRemoteControl
        },
        mounted: function () {
            var context = this;
            var api = {
                status: dataModel.Status,
                conversation: dataModel.Conversation,
                group: dataModel.Group,
                user: dataModel.User,
                message: dataModel.Message,
                remoteControl: dataModel.RemoteControl
            };
            mounted(context, api, im);
            // 监听用户状态改变
            userChanged(dataModel.User, context.$route.params);
            // 监听远程控制消息
            watchRemoteControl(context);
            // 监听其他窗口发送的远程控制命令
            watchRemoteControlCommand(context);
            watchWindowFocus(function () {
                closeRepeatRemotePrompt(context.remoteControl);
            });
            initRemoteControl(context);
        },
        created: function () {
            var context = this;
            im.$on('conversationchange', function (conversation) {
                context.conversation = conversation;
            });

            // 支持截屏则添加截屏回调
            if (support.screenshot) {
                // 判断当前 APP 是否隐藏,如果隐藏,则不执行下述操作
                var isExcuteCallback = function () {
                    //
                    // 点击触发
                    // if (RongIM.screenshot.isClick) {
                    //     RongIM.screenshot.isClick = false;
                    //     return true;
                    // }
                    // RongIM.browserWindow.show();
                    // 快捷键触发
                    var isVisible = RongIM.browserWindow.isVisible();
                    var isFocused = RongIM.browserWindow.isFocused();
                    var winShow = isVisible && isFocused;
                    return winShow;
                };
                var callback = function (data) {
                    
                    //
                    if (!isExcuteCallback()) {
                        return;
                    }
                    if (context.$refs.editor && data !== 'exit') {
                        context.$refs.editor.focus();
                        Vue.nextTick(function () {
                            document.execCommand('Paste');
                        });
                    }
                };
                // screenshot.setCallback(callback);
                if (!RongIM.screenshot.onComplet) {
                    
                    RongIM.screenshot.onComplete = callback;
                }

                if (!RongIM.screenshot.onCancel) {
                    RongIM.screenshot.onCancel = function () {
                        isExcuteCallback();
                    };
                }
            }

            im.$on('messageinputfocus', function () {
                if (context.$refs.editor) {
                    context.$refs.editor.focus();
                }
            });
        },
        destroyed: function () {
            im.$off('messageinputfocus');
            im.$off('conversationchange');
            cleanup(dataModel.Group, this.groupChange);
            remoteApi.unwatch(this.receiveRemoteControl);
            remoteWindowHandler.setDefaultMinWidth();
            remoteWindowHandler.restore();
            if (support.screenshot) {
                screenshot.onComplete = null;
                RongIM.screenshot.onCancel = null;
            }
        },
        watch: {
            $route: function (newValue, oldValue) {
                // 路由 query messageUId 不为空则为搜索消息跳转，输入框不自动获取焦点
                var context = this;
                if (newValue.query.messageUId && !newValue.params.focus) {
                    context.autoFocus = false;
                } else {
                    context.autoFocus = true;
                }
                context.conversationChanged();
                if (context.isGroup) {
                    context.inGroup = true;
                } else {
                    context.isBanned = false;
                }
                userChanged(userApi, newValue.params, oldValue.params);
                // 初始化 remoteControl
                initRemoteControl(context);
                closeRepeatRemotePrompt(context.remoteControl);
            },
            'conversation.group': function (group) {
                var context = this;
                console.log(111,context.conversation.group);
                if (this.validGroup) {
                    // var userApi = dataModel.User;
                    var authId = im.auth.id;
                    userApi.get(group.admin_id, function (errorCode, user) {
                        if (errorCode) {
                            common.toastError(errorCode);
                            return;
                        }
                        var thisGroup = context.conversation.group;
                        context.$set(thisGroup, 'creator_name', common.getUsername(user));
                        context.$set(thisGroup, 'is_creator', group.admin_id === authId);
                    });
                }
            },
            inputEnabled: function (newValue) {
                if (newValue === false) {
                    this.removeQuote();
                }
            }
        },
        methods: getMethods(im)
    };
    utils.asyncComponent(options, resolve, reject);
};

function getComputed(im) {
    return {
        isFileHelper: function () {
            var conversation = this.conversation || {};
            var user = conversation.user || {};
            return user.type === common.UserType.ROBOT;
        },
        isOtherApp: function () {
            var conversation = this.conversation || {};
            if (conversation.conversationType === utils.conversationType.PRIVATE) {
                var user = conversation.user || {};
                return user.type === common.UserType.OTHERAPP;
            }
            return false;
        },
        // 判断是否为公众号
        isPublicMsg: function () {
            var conversation = this.conversation || {};
            return conversation.conversationType === 7;
        },
        status: function () {
            return im.status;
        },
        isConversationView: function () {
            return !utils.isEmpty(this.conversationType) && !utils.isEmpty(this.targetId);
        },
        conversationType: function () {
            return +this.$route.params.conversationType;
        },
        targetId: function () {
            return this.$route.params.targetId;
        },
        isGroup: function () {
            return this.conversationType === utils.conversationType.GROUP;
        },
        validGroup: function () {
            return checkGroupValid(this.conversation, im.auth.id);
        },
        validUser: function () {
            if (!this.isPrivate) {
                if (this.isGroup && !this.validGroup) {
                    return false;
                }
                return true;
            }
            var user = this.conversation.user;
            if (!user) {
                return false;
            }
            // return true;
            // update: 同步需求 可以发送消息给被删除用户
            // update: 产品确认需求 见 bug 23899 备注
            return user.state !== common.UserState.DELETED;
        },
        // 系统用户（特殊用户，如审批助手）
        sysUser: function () {
            var user = this.conversation.user;
            if (!user) {
                return false;
            }
            var isSysUser = common.isSysUser(user);
            return isSysUser;
        },
        isPrivate: function () {
            return this.conversationType === utils.conversationType.PRIVATE;
        },
        disabled: function () {
            return !this.inGroup;
        },
        // 输入框拖拽，绑定会话显示大小
        style: function () {
            var node = im.resizeNode.messageInput;
            var height = node.height;
            var style = null;
            if (height && !this.menuEnabled) {
                style = {
                    'padding-bottom': height + 'px'
                };
            }
            return style;
        },
        inputHeight: function () {
            var node = im.resizeNode.messageInput;
            return node.height || 155;
        },
        voipTip: function () {
            var voipTip = {};
            voipTip[CallType.MEDIA_VEDIO] = this.locale.voip.videoTip;
            voipTip[CallType.MEDIA_AUDIO] = this.locale.voip.audioTip;
            return voipTip;
        },
        // 是否支持远程控制
        isSupportRemoteControl: function () {
            var UserType = common.UserType || {};
            var conversation = this.conversation || {};
            var user = conversation.user || {};
            var isRobot = user.type === UserType.ROBOT;
            var isOtherApp = user.type === UserType.OTHERAPP;
            var isConverSupport = common.isSupportRemoteControl(conversation);
            return isConverSupport && !isRobot && !isOtherApp;
        }
    };
}

/*
说明： 会话组件初始化时判断连接状态，连接成功获取会话信息，否则等待连接成功再获取
*/
function mounted(context, api, im) {
    // 检查是否需要显示科协一家弹窗（每次登录都显示，但同一会话只显示一次）
    var hasShownKexieDialogInSession = sessionStorage.getItem('hasShownKexieDialog');
    var serverConfig = RongIM.serverConfig;
    if (!hasShownKexieDialogInSession) {
        console.log(RongIM.serverConfig, 'RongIM.serverConfig')
        if (serverConfig.zhkx.download_tips == 1) {
            // 标记本次会话已显示弹窗，避免重复显示
            sessionStorage.setItem('hasShownKexieDialog', 'true');
            // 显示科协一家弹窗
            showKexieInfoDialog();
        } else {
            
        }
        
    }
    
    if (context.status === utils.status.CONNECTED) {
        context.conversationChanged();
    } else {
        var unwatch = context.$watch('status', function (newValue) {
            if (newValue === utils.status.CONNECTED) {
                context.conversationChanged();
                unwatch();
            }
        });
    }

    api.conversation.watch(function () {
        var params = context.$route.params;
        if (params.conversationType && params.targetId) {
            context.conversation.unreadMessageCount = api.conversation.getUnreadCount(params);
        }
    });

    api.message.watch(function (message) {
        if (message.content && message.messageType === 'RecallCommandMessage' && context.quote) {
            if (context.quote.messageUId === message.content.messageUId) {
                // 引用消息已被撤回
                context.removeQuote();
            }
        }
    });

    // 监听群信息改变更新群信息
    context.groupChange = function (group) {
        /**
         * 路由设计不合理导致退出群时返回默认界面（/conversation?force=1）组件未销毁监听未取消
         * context.conversation 为之前缓存的值 context.targetId 为空后续代码执行报错
         */
        var currentGroup = context.conversation.group;
        if (currentGroup && group.id === context.targetId) {
            context.conversation.group = group;
            var params = {
                group: context.conversation.group,
                setIsBand: function (isBanned) {
                    context.isBanned = isBanned;
                },
                authId: im.auth.id
            };

            checkGroupBanned(params);
            context.isInvalidGroup = !checkGroupValid(context.conversation, im.auth.id);
            getAllMembers(context, api.group, true);
        }
    };
    api.group.watch(context.groupChange);

    // 监听用户信息改变
    api.user.watch(function (user) {
        var currentUser = context.conversation.user;
        if (currentUser && currentUser.id === user.id) {
            $.extend(context.conversation.user, user);
            context.userStatus = getUserOnlineStatus(context.conversation.user);
        }
    });

    var query = context.$route.query;
    if (!utils.isEmpty(query.keyword)) {
        context.panel = 'history';
    }
}

// 显示科协一家信息弹窗
function showKexieInfoDialog() {
    var options = {
        name: 'kexie-info-dialog',
        template: 'templates/conversation/kexie-info-dialog.html',
        data: function() {
            return {};
        },
        mounted: function() {
            // 生成二维码
            var qrCodeElement = this.$el.querySelector('#qrcode');
            if (qrCodeElement && window.QRCode) {
                var qr = new QRCode(qrCodeElement, {
                    text: 'https://kx-base.cast.org.cn/tygzt/h5-c-new/#/share-app',
                    width: 120,
                    height: 120,
                    colorDark: '#000000',
                    colorLight: '#ffffff',
                    correctLevel: QRCode.CorrectLevel.M
                });
            }
        },
        methods: {
            close: function() {
                this.$destroy();
                $(this.$el).remove();
            }
        }
    };
    common.mountDialog(options);
}

// 获取群组全部成员
function getAllMembers(context, groupApi, isRefresh) {
    var groupId = context.targetId;
    groupApi.getAllMembers(groupId, function (errorCode, members, group) {
        if (errorCode) {
            // 成员信息获取失败无法进行后续操作
            return;
        }
        var cacheAlias = RongIM.dataModel._Cache.alias;
        var tmpMembers = members.map(function (item, i) {
            var member = {
                id: item.id,
                // 备注名显示规则：我对其增加的备注名称 > 其自己添加的组内备注名称 > 名字
                alias: cacheAlias[item.id] || group.members[i].alias,
                avatar: item.avatar,
                mute_status: item.mute_status,
                name: item.name,
                portrait_big_url: item.portrait_big_url,
                state: item.state,
                type: item.type
            };
            var tmpName = member.alias || member.name;
            member.htmlAlias = common.getHtmlGroupUsername2(tmpName, 16);
            return member;
        });
        context.members = tmpMembers;
    }, isRefresh);
}

function initRemoteControl(context) {
    context.remoteControl = context.conversation.remoteControl || {};
    setRemoteControlStyle(context);
}

function sameConversaton(one, another) {
    var sameConversationType = one.conversationType === another.conversationType;
    var sameTargetId = one.targetId === another.targetId;
    return sameConversationType && sameTargetId;
}

function cleanup(groupApi, groupChange) {
    /* 为什么要清空所有group监听? */
    // groupApi.unwatch();
    groupApi.unwatch(groupChange);
}

/*
说明： 获取用户状态字符串
    如果是移动端在线则展示 "手机在线"
    否则展示 "在线" "忙碌" "离线"
*/
function getUserOnlineStatus(user) {
    var time = 0;
    var status = 'offline';
    if (utils.isEmpty(user.onlineStatus)) {
        return status;
    }
    Object.keys(user.onlineStatus).forEach(function (title) {
        var item = user.onlineStatus[title];
        if (time < item.updateDt && item.value !== 'offline') {
            time = item.updateDt;
            var isMobile = (item.title === 'Login_Status_Mobile');
            status = isMobile ? 'mobile' : item.value;
        }
    });
    return status;
}

/*
说明： 会话切换时注册和取消注册用户状态监听
*/
function userChanged(userApi, newValue, oldValue) {
    var newUserId = getUserId(newValue);
    var oldUserId = getUserId(oldValue);
    watchUserStatus(newUserId, userApi);
    unwatchUserStatus(oldUserId, userApi);
}

function getUserId(params) {
    params = params || {};
    var userId = '';
    if (+params.conversationType === utils.conversationType.PRIVATE) {
        userId = params.targetId;
    }
    return userId;
}

/*
说明：设置用户状态监听
*/
var userStatusInterval = null;
function watchUserStatus(userId, userApi) {
    clearInterval(userStatusInterval);
    if (utils.isEmpty(userId)) {
        return;
    }
    var duration = 1800;
    userApi.subscribe(userId, duration);
    userStatusInterval = setInterval(function () {
        userApi.subscribe(userId, duration);
    }, duration * 1000);
}
/*
说明： 取消用户状态监听
*/
function unwatchUserStatus(userId, userApi) {
    if (utils.isEmpty(userId)) {
        return;
    }
    userApi.unsubscribe(userId);
}

function getMethods(im) {
    var dataModel = im.dataModel;
    var orgApi = dataModel.Organization;
    return {
        showHistory: function(){
          this.panel= "history";
        },
        // 右键头像@成员 bus 中央通讯 接收：edit-box.js  
        setSendVal(val) {
            eventBus.$emit('altUse', val);   
        },
        getGroupCompany: function (org) {
            if (utils.isEmpty(org)) {
                return '';
            }
            var name = '';
            if (org.type === common.OrgType.DEPT) {
                for (var i = org.pathList.length - 1; i >= 0; i -= 1) {
                    var item = org.pathList[i];
                    if (item.type === common.OrgType.COMPANY) {
                        name = orgApi.getLocalCompany(item.id).fullName;
                        break;
                    }
                }
            } else if (org.type === common.OrgType.COMPANY) {
                name = org.fullName || org.name;
            }
            return name;
        },
        getUsername: common.getUsername,
        getHtmlUsername: common.getHtmlUsername,
        getGroupName: common.getGroupName,
        getHtmlGroupName: function (group) {
            if (group) {
                return common.getHtmlGroupName(group, 18);
            }
            return this.targetId;
        },
        getGroupType: common.getGroupType,
        userProfile: function (user) {
            var isRobot = user.type === common.UserType.ROBOT;
            if (isRobot) {
                return;
            }
            var userId = user.id;
            dialog.user(userId);
        },
        // 用户在线状态文字提示
        getStatusText: function () {
            var localeStatus = this.locale.components.onlineStatus;
            return localeStatus[this.userStatus];
        },
        reset: function () {
            this.conversation = {
                conversationType: this.conversationType,
                targetId: this.targetId
            };
            this.messageList = [];
            this.panel = null;
        },
        conversationChanged: function () {
            if (utils.isEmpty(this.conversationType) || utils.isEmpty(this.targetId)) {
                return;
            }
            var params = {
                conversationApi: dataModel.Conversation,
                groupApi: dataModel.Group,
                userApi: dataModel.User,
                publicApi: dataModel.Public,
                authId: im.auth.id,
                emit: function (method) {
                    im.$emit(method);
                }
            };
            conversationChanged(this, params);
        },
        setProperty: function (key, value) {
            // key可能是个路径，例如'group.save'
            var keys = key.split('.');
            var lastKey = keys.slice(-1)[0];
            keys = keys.slice(0, -1);
            var obj = this.conversation;
            keys.forEach(function (item) {
                obj = obj[item];
            });
            this.$set(obj, lastKey, value);
        },
        clearUnReadCount: function () {
            dataModel.Conversation.clearUnReadCount(this.conversationType, this.targetId);
        },
        // appendToMessageList: function (list) {
        //     this.messageList.unshift.apply(this.messageList, list);
        // },
        sendTextMessage: function (message) {
            var context = this;
            var params = {
                clearUnReadCount: this.clearUnReadCount,
                setInGroup: function (inGroup) {
                    context.inGroup = inGroup;
                },
                conversationType: this.conversationType,
                targetId: this.targetId,
                messageApi: dataModel.Message,
                conversationApi: dataModel.Conversation,
                quote: context.quote,
                message: message
            };
            sendTextMessage(params);
            context.quote = null;
        },
        hidePanel: function () {
            this.panel = null;
        },
        append: function (message) {
            this.newMessage = message;
        },
        // 发送复制的消息 复制消息直接发送原消息体
        sendCopyMessage: function (message) {
            var context = this;
            var conversationType = context.conversationType;
            var targetId = context.conversation.targetId;
            dataModel.Message.send({
                conversationType: conversationType,
                targetId: targetId,
                content: message
            });
        },
        // 发送消息前如果是由搜索消息跳转来，则先重置会话为正常会话
        prepareinput: function () {
            var context = this;
            var query = context.$route.query;
            if (query.messageUId) {
                context.$router.push({
                    name: 'conversation',
                    params: {
                        conversationType: context.conversationType,
                        targetId: context.targetId
                    }
                });
            }
        },
        setInGroup: function (boolean) {
            this.inGroup = boolean;
        },
        sendVoip: function (isVideo) {
            var params = {
                conversation: this.conversation,
                type: isVideo ? CallType.MEDIA_VEDIO : CallType.MEDIA_AUDIO
            };
            var userApi = RongIM.dataModel.User;
            var friendApi = RongIM.dataModel.Friend;
            var isPrivate = Number(this.conversation.conversationType) === 1;
            if (isPrivate) {
                var canNotChat = !userApi.validateCanChat(this.conversation.targetId);
                if (canNotChat) {
                    friendApi.insertRFVMsg(this.conversation.targetId);
                    return;
                }
            }
            var voipTip = this.voipTip;
            RCCall.start(params, function (errorCode, data) {
                if (errorCode) {
                    common.messageToast({
                        type: 'error',
                        message: voipTip[data.type]
                    });
                }
            });
        },
        memberCount: function (item) {
            var format = this.locale.contact.person;
            if (item.member_count === 1) {
                format = this.locale.contact.personSingle;
            }
            return this.localeFormat(format, item.member_count || 0);
        },
        inputMenuChanged: function (value) {
            // 公众号 键盘状态切换 输入框或菜单的显示/隐藏
            this.menuEnabled = value;
            this.inputEnabled = (!value);
            if (!value) {
                var messageList = this.$refs.list;
                messageList.scrollWhenInputResize();
            }
        },
        addQuote: function (message) {
            if (!this.inputEnabled && !this.menuEnabled) {
                return;
            }
            this.inputEnabled = true;
            this.menuEnabled = false;
            var objName = RongIMClient.MessageParams[message.messageType].objectName;
            this.quote = {
                content: message.content,
                objName: objName,
                userId: message.senderUserId,
                user: message.user,
                messageUId: message.messageUId
            };
            this.$refs.editor.focus();
        },
        removeQuote: function () {
            this.quote = null;
        },
        getTextContent: function (content) {
            content = utils.htmlLang.check(content);
            content = common.textMessageFormat(content);
            return content;
        },
        getGroupUsername: function (user) {
            var groupId = this.conversation.targetId;
            return common.getGroupUsername(user, groupId);
        },
        sendRichContentMessage:function(message){
            sendRichContentMessage(this,message,dataModel.Message)
        }
    };
}

var delayUpdateGroupInfo = null;
var delayUpdateOfficalAccountInfo = null;
/*
说明： 会话聊天对象改变，初始化会话信息，获取会话历史消息
*/
function conversationChanged(context, params) {
    var conversationApi = params.conversationApi;
    var publicApi = params.publicApi;
    var groupApi = params.groupApi;
    var userApi = params.userApi;
    context.quote = null;
    var conversationType = context.conversationType;
    var targetId = context.targetId;
    if (context.$refs.list) {
        context.$refs.list.reset();
    }
    // 设置群成员为空
    context.members = [];

    // 如果 conversation 是应用公众号信息，则获取公众号菜单详情
    var isPublic = conversationType === utils.conversationType.APP_PUBLIC_SERVICE;
    var cache = RongIM.dataModel._Cache || {};
    var publicCache = cache.public;
    if (isPublic) {
        var publicLocale = publicCache[targetId];
        if (publicLocale) {
            showMenu(publicLocale, context);
            publicApi.getPublicMenu(targetId, function (result) {
                showMenu(result, context);
            });
        } else {
            publicApi.getPublicMenu(targetId, function (result) {
                showMenu(result, context);
            });
        }
    } else {
        context.menuEnabled = false;
        context.inputEnabled = true;
    }

    var currentConversation = {
        conversationType: context.conversationType,
        targetId: context.targetId
    };
    conversationApi.setActive(currentConversation);
    // 获取会话信息
    conversationApi.getOne(conversationType, targetId, function (errorCode, conversation) {
        var isGroup = conversationType === utils.conversationType.GROUP;
        if (errorCode && !isGroup) {
            common.toastError(errorCode);
            return;
        }
        context.reset();
        conversation = conversation || {
            conversationType: conversationType,
            targetId: targetId
        };
        if (!sameConversaton(conversation, currentConversation)) {
            return;
        }
        if (context.$refs.editor) {
            context.$refs.editor.focus();
        }
        context.conversation = conversation;

        if (context.isPrivate) {
            context.userStatus = getUserOnlineStatus(context.conversation.user);
        }

        // 群会话 - 获取群成员信息
        context.isInvalidGroup = false;

        clearTimeout(delayUpdateGroupInfo);
        clearTimeout(delayUpdateOfficalAccountInfo);
        if (isGroup) {
            // 1.判断自己是否被禁言 2.群组是否有效群组（解散或被踢出）
            var checkGroupInfo = function () {
                var chkGroupParams = {
                    group: context.conversation.group,
                    setIsBand: function (isBanned) {
                        context.isBanned = isBanned;
                    },
                    authId: params.authId
                };
                checkGroupBanned(chkGroupParams);
                context.isInvalidGroup = !checkGroupValid(context.conversation, params.authId);
            };
            checkGroupInfo();
            // 获取并设置群信息
            var getGroupInfo = function () {
                RongIM.syncdata.groupById(context.targetId, function () {
                    groupApi.getNewGroup(context.targetId, function (error, group) {
                        if (error || utils.isEmpty(group) || context.targetId !== group.id) {
                            return;
                        }
                        context.conversation.group = group;
                        checkGroupInfo();
                        getAllMembers(context, groupApi);
                    });
                });
            };
            // 如果群组信息未获取到则立即重新获取
            if (!context.conversation.group || !context.conversation.group.member_id_list || context.conversation.group.member_id_list.length === 0) {
                getGroupInfo();
            } else {
                // 每次进入会话更新群组信息。增加延时获取防止频繁切换重复获取
                delayUpdateGroupInfo = setTimeout(function () {
                    getGroupInfo();
                }, RongIM.config.syncDelayTime);
            }
            // 获取群成员信息
            getAllMembers(context, groupApi);
        } else {
            // 获取并设置用户信息
            var getUserInfo = function () {
                RongIM.syncdata.userById([context.targetId], function () {
                    userApi.getNewUser(context.targetId, function (error, user) {
                        if (error || utils.isEmpty(user) || context.targetId !== user.id) {
                            return;
                        }
                        context.conversation.user = user;
                    });
                });
            };
            if ($.isEmptyObject(context.conversation.user)) {
                getUserInfo();
            } else {
                // 更新公众号（staff 表中数据）、外部联系人信息，无同步机制需每次进入会话同步
                var isOfficialAccount = conversationType === RongIMLib.ConversationType.APP_PUBLIC_SERVICE;
                var notStaff = context.conversation.user.type !== 0;
                if (isOfficialAccount || notStaff) {
                    delayUpdateOfficalAccountInfo = setTimeout(function () {
                        getUserInfo();
                    }, RongIM.config.syncDelayTime);
                }
            }
        }
    });
}

function showMenu(publicMenu, context) {
    var isPublic = context.conversationType === utils.conversationType.APP_PUBLIC_SERVICE;
    if (!isPublic) {
        // 弱网情况下，快递切换会话列表，菜单显示错误问题
        return;
    }
    var cache = RongIM.dataModel._Cache || {};
    var publicCache = cache.public;

    var menu_content = publicMenu.menu_content || [];
    // 如果菜单设置为 true ，并且有菜单内容，则显示公众号菜单
    var menuEnabled = publicMenu.menu_enabled && menu_content.length > 0;
    var inputEnabled = publicMenu.input_enabled;
    if (menuEnabled && inputEnabled) {
        // 菜单和输入框同时都开启时，优先显示菜单
        context.menuEnabled = true;
        context.inputEnabled = false;
    } else {
        context.menuEnabled = menuEnabled;
        context.inputEnabled = inputEnabled;
    }
    context.menuInfo = publicMenu;
    publicCache[publicMenu.app_id] = publicMenu;
}

function checkGroupValid(conversation, authId) {
    var group = conversation.group || {};
    var members = group.member_id_list || [];
    var inGroup = members.indexOf(authId) >= 0;
    var isDismiss = (group.group_status === 2);
    return inGroup && !isDismiss;
}

/*
说明： 判断自己在群组中的禁言状态
*/
function checkGroupBanned(params) {
    var group = params.group;
    if (group) {
        var members = group.members || [];
        var selfInfo = members.filter(function (member) {
            return member.id === params.authId;
        })[0] || {};
        var isBanned = false;
        if (group.is_all_mute && selfInfo.mute_status !== 2) {
            isBanned = true;
        }
        if (selfInfo.mute_status === 1) {
            isBanned = true;
        }
        if (selfInfo.role === 1) {
            isBanned = false;
        }
        params.setIsBand(isBanned);
    }
}

/*
说明: 发送文本消息
*/
function sendTextMessage(params) {
    var message = params.message;
    if (!message.text) {
        return;
    }
    var conversationType = params.conversationType;
    var targetId = params.targetId;
    message.text = RongIMLib.RongIMEmoji.symbolToEmoji(message.text);

    var msg = params.messageApi.TextMessage.obtain(message.text);
    if (params.quote) {
        var content = $.extend({}, params.quote, {
            text: message.text
        });
        msg = common.buildMessage.ReferenceMessage(content);
    }

    //  获取 '@' 人员信息
    var mentiondMsg = false;
    if (message.at && message.at.length > 0) {
        var mentioneds = new RongIMLib.MentionedInfo();
        var userIdList = message.at.map(function (item) {
            return item.id;
        });
        // '@所有人' 的 userId 为 0
        var isAtAll = userIdList.indexOf(0) !== -1;
        if (isAtAll) {
            mentioneds.type = utils.mentionedType.ALL;
            mentioneds.userIdList = [];
        } else {
            mentioneds.type = utils.mentionedType.PART;
            mentioneds.userIdList = userIdList;
        }

        msg.mentionedInfo = mentioneds;
        mentiondMsg = true;
    }

    params.messageApi.send({
        conversationType: conversationType,
        targetId: targetId,
        content: msg,
        mentiondMsg: mentiondMsg
    }, function (errorCode) {
        if (errorCode) {
            var ignoreErrorCodeMap = [
                'lib--1',
                'lib-30001',
                'lib-30003'
                // 'lib-21502'
            ];
            var existed = ignoreErrorCodeMap.indexOf(errorCode) >= 0;
            if (existed) {
                return;
            }
            var errMsg = common.getErrorMessage(errorCode);
            if (errorCode !== 'lib-' + RongIMLib.ErrorCode.NOT_IN_GROUP) {
                common.toastError(errorCode);
                return;
            }
            var messageParams = common.createNotificationMessage(conversationType, targetId, errMsg);
            params.messageApi.insertMessage(messageParams);
            params.setInGroup(false);
        }
    });
    params.conversationApi.clearDraft(conversationType, targetId);
    params.clearUnReadCount();
}
// 发送富文本消息
function sendRichContentMessage(context,message,messageApi){
    if (!message) {
        return;
    }
    let content = message.content;
    let title = message.title;
    let imageUri = "http://111.203.146.82/package/icon-jielong.png"
    let msg = new RongIMLib.RichContentMessage({ title: title, content: content,imageUri:imageUri,extra: message.extra});
    let conversationType = RongIMLib.ConversationType.GROUP; // 单聊, 其他会话选择相应的会话类型即可
    var targetId = context.targetId; // 目标 Id
    messageApi.send({
        conversationType: conversationType,
        targetId: targetId,
        content: msg
    }, function (errorCode) {
    })

}
/**
 * 监听 remoteControl 消息, 赋值给 conversation, 用来做 UI 展示
 * @param  {object} context   vue 组件(conversation)
 * @param  {object} remoteApi  RongIM.dataModel.RemoteControl
 */
function watchRemoteControl(context) {
    var im = RongIM.instance;
    // 手动修改时, 设置 remoteControl
    im.$on('setRemoteControl', function (message) {
        remoteApi.setLocalRemote(message, function (remoteControl) {
            var params = im.$route.params;
            var currentTargetId = params.targetId;
            if (currentTargetId === message.targetId) {
                context.remoteControl = remoteControl;
                setRemoteControlStyle(context);
            }
        });
    });
    // 监听到消息时, 设置 remoteControl
    context.receiveRemoteControl = function (message) {
        switch (message.messageType) {
            case 'RemoteControlRequestMessage':
                handleRemoteRequestMessage(message);
                break;
            case 'RemoteControlInviteMessage':
                handleRemoteRequestMessage(message);
                break;
            case 'RemoteControlAcceptMessage':
                handleRemoteAcceptMessage(message);
                break;
            case 'RemoteControlCofigMessage':
                handleRemoteConfigMessage(message);
                break;
            case 'RemoteControlHangupMessage':
                handleRemoteHangupMessage(message);
                break;
            default:
                break;
        }
    };
    remoteApi.watch(context.receiveRemoteControl);
}

function setRemoteControlStyle(context) {
    // 产品临时改需求, 暂时的解决方案
    var remoteStatus = context.remoteControl.status;
    if (context.isSupportRemoteControl && remoteStatus) {
        // 100 延迟: 先等待窗口放大后, 再弹出右侧滑框
        setTimeout(function () {
            remoteStatus = context.remoteControl.status;
            if (remoteStatus) $(context.$el).addClass('rong-conversation-remote');
        }, 100);
    } else {
        // 50 延迟: 避免窗口缩小与右侧划框两个动作同时进行, 造成页面闪动
        setTimeout(function () {
            $(context.$el).removeClass('rong-conversation-remote');
        }, 100);
    }
}

// 处理 request 和 invite 消息, 暂时放在一起处理
function handleRemoteRequestMessage(message) {
    var im = RongIM.instance;
    var isBusy = remoteApi.isBusyline();
    // 如果是同帐号不同端发送的 request 和 invite, 不处理
    if (message.isOtherPlatSend) {
        return;
    }
    // 忙碌时, 给对方发送挂断消息
    if (isBusy) {
        var sessionId = message.content.sessionId;
        var reason = RemoteControlHangupReason.BusyLine;
        remoteApi.hangup(message, sessionId, reason);
        return;
    }
    im.$emit('setRemoteControl', message);
    openRemoteControlPrompt(message);
}

function handleRemoteAcceptMessage(message) {
    var im = RongIM.instance;
    var currentRemoteStatus = remoteApi.get(message.targetId).status;
    var isIdle = currentRemoteStatus === RemoteControlStatus.Idle;
    // 当前状态如果为空闲
    if (!currentRemoteStatus || isIdle) {
        return;
    }
    var isInviteCurrent = currentRemoteStatus === RemoteControlStatus.Invite;
    var isReceiveRequestCurrent = currentRemoteStatus === RemoteControlStatus.ReceiveRequest;
    if (message.isOtherPlatSend) {
        // 其他端已经处理
        dealRemoteWithOtherPlat(message);
    } else if (isInviteCurrent || isReceiveRequestCurrent) {
        setRemoteControlConfig(message);
        im.$emit('setRemoteControl', message);
    }
}

function dealRemoteWithOtherPlat(message) {
    var im = RongIM.instance;
    var content = {
        reason: RemoteControlHangupReason.OtherDeviceHadAccepted,
        direction: RongIMLib.MessageDirection.SEND,
        duration: 0,
        isControl: true
    };
    remoteApi.summary(message, content, function (err, msg) {
        im.$emit('setRemoteControl', msg);
        var locale = im.locale.components.remoteHangup;
        var title = locale[RemoteControlHangupReason.OtherDeviceHadAccepted];
        browserWindow.changeRemoteControlTitle(title);
        browserWindow.closeRemoteControl();
    });
}

/**
 * 处理 hangupMessage. 获取挂断原因, 插入小灰条提示
 * @param  {object} message
 */
function handleRemoteHangupMessage(message) {
    var im = RongIM.instance;
    var currentRemote = remoteApi.get(message.targetId);
    var currentRemoteStatus = currentRemote.status;
    var isIdle = currentRemoteStatus === RemoteControlStatus.Idle;
    var isRequest = currentRemoteStatus === RemoteControlStatus.ReceiveRequest || RemoteControlStatus.ReceiveInvite;
    var currentMessage = currentRemote.message || {};
    var currentContent = currentMessage.content || {};
    var isSameSession = currentContent.sessionId === message.content.sessionId;
    if (!currentRemoteStatus || isIdle || !isSameSession) {
        return;
    }
    var reason = message.content.reason;
    var isBusylineMessage = reason === RemoteControlHangupReason.BusyLine;
    // 同帐号不同端处理, 且当前为 reqeust 消息时, 说明是其他端挂断了该控制. 本端也应该挂断
    // 产品要求, 如果是 自己发的 busy 消息, 正常提示 对方忙
    if (message.isOtherPlatSend && isRequest && !isBusylineMessage) {
        dealRemoteWithOtherPlat(message);
        return;
    }
    RemoteCtrlOpt.hangup();
    remoteSummary(message);
    im.$emit('setRemoteControl', message);
}

function remoteSummary(message) {
    var currentRemote = remoteApi.get(message.targetId);
    if (currentRemote.status === RemoteControlStatus.Idle) {
        return;
    }
    var reason = message.content.reason;
    var direction = message.messageDirection;
    reason = common.changeHangupReceiveReason(direction, reason);
    var remoteStatus = currentRemote.status;
    var sessionId = message.content.sessionId;
    var summaryParams = getSummaryParams(reason, remoteStatus, null, sessionId);
    remoteApi.summary(message, summaryParams, function () {
        browserWindow.closeRemoteControl();
    });
}

function getSummaryParams(reason, status, direction, sessionId) {
    var MessageDirection = RongIMLib.MessageDirection;
    direction = direction || MessageDirection.RECEIVE;
    var isControl = false;
    var isHangup = reason === RemoteControlHangupReason.Hangup || reason === RemoteControlHangupReason.RemoteHangup;
    var isNoResponse = reason === RemoteControlHangupReason.NoResponse || reason === RemoteControlHangupReason.RemoteNoResponse;
    // 己方主动发出的挂断, 只有连接后的挂断气泡在右侧, 其他都在左侧
    if (!isHangup && !isNoResponse) {
        direction = direction === MessageDirection.SEND ? MessageDirection.RECEIVE : MessageDirection.SEND;
    }
    if (isHangup) {
        // 己方为发起者时, 气泡在左边
        var isInitiator = remoteApi.isInitiator(sessionId);
        direction = isInitiator ? MessageDirection.SEND : MessageDirection.RECEIVE;
    }
    // 挂断前, 为这些状态时, 表示自己为主控端, 此时提示 远程控制, 否则提示 远程协助
    var controlStatusList = [
        'Request', 'ReceiveInvite', 'Building', 'RequestBuilded'
    ];
    status = RemoteControlStatus[status];
    if (controlStatusList.indexOf(status) !== -1) {
        isControl = true;
    }
    return {
        reason: reason,
        direction: direction,
        isControl: isControl
    };
}

// 处理 configMessage
function handleRemoteConfigMessage(message) {
    var im = RongIM.instance;
    var isSelfSend = message.messageDirection === RongIMLib.MessageDirection.SEND;
    var currentRemoteStatus = remoteApi.get(message.targetId).status;
    var isIdle = currentRemoteStatus === RemoteControlStatus.Idle;
    if (isSelfSend || isIdle || !currentRemoteStatus) {
        return;
    }
    var content = message.content;
    var address = content.mediaAddress;
    var sessionName = content.mediaId;
    // 开启 主控端 远程控制
    RemoteCtrlOpt.join(address, sessionName, {
        onConnected: $.noop,
        onDisconnected: function (msg) {
            utils.console.log('远程控制, join disconnected: ', msg);
        },
        onDisplayChanged: function (msg) {
            utils.console.log('远程控制, on display changed: ', msg);
        },
        onClose: function () {
            var sessionId = message.content.sessionId;
            var reason = RemoteControlHangupReason.Hangup;
            hangupRemoteControl(message, sessionId, reason);
        }
    });
    im.$emit('setRemoteControl', message);
}

function setRemoteControlConfig(message) {
    var im = RongIM.instance;
    var sessionId = message.content.sessionId;
    remoteApi.setConfig(message, sessionId, function (err, msg) {
        var remote = remoteApi.get(msg.targetId);
        if (remote.status !== RemoteControlStatus.Idle) {
            im.$emit('setRemoteControl', msg);
        }
    }, function () {
        // 挂断的回调, 对方杀进程也会执行
        var remote = remoteApi.get(message.targetId);
        if (remote.status !== RemoteControlStatus.Idle) {
            var reason = RemoteControlHangupReason.Hangup;
            var direction = RongIMLib.MessageDirection.RECEIVE;
            var summaryParams = getSummaryParams(reason, remote.status, direction, sessionId);
            remoteApi.summary(message, summaryParams, function (err, summaryMsg) {
                im.$emit('setRemoteControl', summaryMsg);
            });
        }
    });
}

/* 打开远程控制的提示弹框 */
function openRemoteControlPrompt(message) {
    var im = RongIM.instance;
    var conversation = im.$route.params;
    var isRequest = message.messageType === 'RemoteControlRequestMessage';
    var isInvite = message.messageType === 'RemoteControlInviteMessage';
    var isInConversation = message.targetId === conversation.targetId;
    var isWindowFocus = browserWindow.isFocused();
    if ((!isRequest && !isInvite) || (isInConversation && isWindowFocus)) {
        return;
    }
    var templateFormat = utils.templateFormat;
    var userApi = im.dataModel.User;
    var targetId = message.targetId;
    userApi.get(targetId, function (err, result) {
        if (err) {
            result.name = targetId;
        }
        var locale = im.locale.components.remoteControl;
        var title = isRequest ? locale.receiveRequest : locale.receiveInvite;
        var sessionId = message.content.sessionId;
        title = templateFormat(title, result.name);
        RongIM.browserWindow.openRemoteControl(title, targetId, sessionId);
    });
}

/**
 * 当前窗口处于远程控制请求中时, 右侧滑出窗口与屏幕右下角弹框重复, 此时关闭右下角远程控制提示弹框
 * @param  {object} remoteControl conversation.remoteControl
 */
function closeRepeatRemotePrompt(remoteControl) {
    var im = RongIM.instance;
    var params = im.$route.params;
    var status = remoteControl.status;
    if (params.targetId && status && status !== RemoteControlStatus.Idle) {
        RongIM.browserWindow.closeRemoteControl();
    }
}

/**
 * 当窗口focus时, 调用 closeRemoteControlPrompt 关闭窗口
 */
function watchWindowFocus(callback) {
    win.onfocus = function () {
        callback();
    };
}

/* 监听从提示弹框中发过来的 accept 和 reject */
function watchRemoteControlCommand() {
    watchRemoteAcceptCommand();
    watchRemoteRejectCommand();
}

function watchRemoteAcceptCommand() {
    var im = RongIM.instance;
    var setRemoteControl = function (err, message, callback) {
        if (!err) {
            im.$emit('setRemoteControl', message);
            callback();
        }
    };
    browserWindow.onRemoteControlAccept(function (targetId, sessionId) {
        browserWindow.closeRemoteControl();
        var remoteStatus = remoteApi.get(targetId).status;
        var params = {
            targetId: targetId,
            conversationType: RongIMLib.ConversationType.PRIVATE
        };
        remoteApi.accept(params, sessionId, function (err, message) {
            showConversation(params);
            setRemoteControl(err, message, function () {
                if (remoteStatus === RemoteControlStatus.ReceiveRequest) {
                    setRemoteControlConfig(message);
                }
                browserWindow.show();
            });
        });
    });
}

function watchRemoteRejectCommand() {
    browserWindow.onRemoteControlReject(function (targetId, sessionId) {
        browserWindow.closeRemoteControl();
        var reason = RemoteControlHangupReason.Reject;
        var params = {
            targetId: targetId,
            conversationType: RongIMLib.ConversationType.PRIVATE
        };
        showConversation(params);
        hangupRemoteControl(params, sessionId, reason);
        browserWindow.show();
    });
}

function hangupRemoteControl(params, sessionId, reason) {
    var im = RongIM.instance;
    var status = remoteApi.get(params.targetId).status;
    remoteApi.hangup(params, sessionId, reason, function (err, message) {
        if (!err) {
            im.$emit('setRemoteControl', message);
        }
    });
    var summaryParams = getSummaryParams(reason, status);
    summaryParams.direction = RongIMLib.MessageDirection.RECEIVE;
    remoteApi.summary(params, summaryParams);
}

function showConversation(conversation) {
    var im = RongIM.instance;
    var params = im.$route.params;
    var conversationType = conversation.conversationType;
    var targetId = conversation.targetId;
    var isSameConversation = params.conversationType === conversationType && params.targetId === targetId;
    if (isSameConversation) {
        return;
    }
    im.$router.push({
        name: 'conversation',
        params: {
            targetId: targetId,
            conversationType: conversationType
        }
    });
}
}(RongIM, {
    win: window,
    RongIMLib: RongIMLib,
    RCCall: RCCall,
    jQuery: jQuery
}, RongIM.components));
