/*
说明： 消息输入
功能： 发送文字，发送表情，发送文件，发送名片，群@成员
    桌面端还包括 截图，发起音视频
*/
(function (RongIM, dependencies, components) {
    'use strict';

    var common = RongIM.common;
    var utils = RongIM.utils;
    var Base64Util = utils.Base64;
    var dialog = RongIM.dialog;
    var CallType = common.CallType;
    var $ = dependencies.jQuery;
    var UploadClient = dependencies.UploadClient;
    var dataModel = RongIM.dataModel;
    var cache = utils.cache;
    var saveKey = 'screenshotOption';
    var remoteOverTime = RongIM.config.modules.remoteControl.overtime;

    components.getMessageInput = function (resolve, reject) {
        var im = RongIM.instance;
        var options = {
            name: 'message-input',
            template: '#rong-template-message-input',
            props: {
                autoFocus: {
                    type: Boolean,
                    required: false,
                    default: true
                },
                targetId: {
                    type: String,
                    required: true
                },
                atMembers: {
                    type: Array,
                    required: true
                },
                inGroup: false,
                isRobot: false,
                disabled: false,
                // 被禁言
                isBanned: false,
                isInvalidGroup: false,
                isGroup: {
                    type: Boolean,
                    required: false,
                    default: true
                },
            },
            data: function () {
                return {
                    sendBtnAvailable: false,
                    at: {},
                    atFilterMembers: [],
                    atPanelStyle: {},
                    atSelectedMembers: [],
                    showEmojiPanel: false,
                    support: RongIM.config.support,
                    bound: {
                        height: {
                            min: 0,
                            max: 0
                        }
                    },
                    isResizing: false,
                    isShowWindow: !!cache.get(saveKey),
                    isShowScreenOption: false,
                    isShowRemoteCtrlPanel: false,
                    isShowCollect: RongIM.config.modules.collect,
                    fileMessageList: [],
                    fileSending: false,
                    isPC: !RongIM.system.platform.startsWith('web'),
                    isMac: !RongIM.system.platform.startsWith('darwin'),
                    isWindows: RongIM.system.platform === 'win32'
                };
            },
            computed: {
                // 是否支持截图（只有桌面版支持）
                screenshotSupported: function () {
                    return im.config.support.screenshot;
                },
                // 当前连接状态
                status: function () {
                    return im.status;
                },
                height: function () {
                    var node = im.resizeNode.messageInput;
                    return node.height;
                },
                resizeDirection: function () {
                    return im.resizeDirection.use;
                },
                // 根据 server 配置决定显示不显示音视频按钮
                voipConf: function () {
                    return im.serverConfig.voip;
                },
                // 判断是否显示公众号服务切换菜单按钮
                isPublicConversation: function () {
                    return this.$parent.conversationType === 7;
                },
                isShowMenuSwitch: function () {
                    var parent = this.$parent;
                    if (parent.menuInfo.menu_content && parent.conversationType === 7) {
                        return this.$parent.menuInfo.menu_content.length > 0;
                    }
                    return false;
                },
                isSupportRemoteControl: function () {
                    return this.$parent.isSupportRemoteControl;
                },
                showVideo: function () {
                    var available = this.support.voip && this.voipConf.video_enable;
                    return available && !this.isPublicConversation && !this.isRobot;
                },
                showAudio: function () {
                    var available = this.support.voip && this.voipConf.audio_enable;
                    return available && !this.isPublicConversation && !this.isRobot;
                },
                voipTip: function () {
                    var voipTip = {};
                    voipTip[CallType.MEDIA_VEDIO] = this.locale.voip.videoTip;
                    voipTip[CallType.MEDIA_AUDIO] = this.locale.voip.audioTip;
                    return voipTip;
                }
            },
            mounted: function () {
                var context = this;
                document.onkeydown = function (event) {
                    var e = event || window.event;
                    // var keyCode = e.keyCode || e.which;
                    if (e.ctrlKey && e.keyCode == 13) {
                        context.sendMessage();
                    }
                    // if (e && e.preventDefault) {
                    //   e.preventDefault();
                    // } else {
                    //   window.event.returnValue = false;
                    // }           
                     }
                inputBoxResize(context, im);
                im.$on('imclick', function (event) {
                    context.hideScreenOption();
                    context.hideRemoteCtrlPanel(event);
                });
                context.editrecalled = function (value) {
                    var at = [];
                    if (value.atIdList && value.atIdList.length > 0 && context.atMembers) {
                        var tmpMap = {};
                        context.atMembers.forEach(function (item) {
                            tmpMap[item.id] = item.name;
                        });
                        value.atIdList.forEach(function (id) {
                            var name = tmpMap[id];
                            if (id === 0) {
                                name = context.locale.components.atPanel.everyone;
                            }
                            if (name) {
                                at.push({
                                    id: id,
                                    name: name
                                });
                            }
                        });
                    }
                    context.$refs.editor.appendValue({
                        text: value.text,
                        at: at
                    });
                };
                im.$on('editrecalled', context.editrecalled);
                // 首先设置一下pc端是否隐藏窗口的参数
                RongIM.screenshot.setHideWindow(!!cache.get(saveKey));
            },
            destroyed: function () {
                im.$off('editrecalled', this.editrecalled);
            },
            components: {
                'edit-box': components.getEditBox,
                'emoji-panel': components.getEmojiPanel
            },
            methods: getMethods(im)
        };
        utils.asyncComponent(options, resolve, reject);
    };

    /*
    说明： 文本框拖拽改变大小
    */
    function inputBoxResize(context, im) {
        var el = im.$el;
        var editorNode = $(context.$el);
        var getBound = function (name) {
            return editorNode.css(name);
        };
        /* messageList组件滚动到底部 */
        var scrollToBottom = function () {
            /* messageList组件 */
            var messageList = context.$parent.$refs.list;
            messageList.scrollWhenInputResize();
        };
        context.bound = {
            height: {
                min: getBound('min-height'),
                max: getBound('max-height')
            }
        };
        var resizeDirection = im.resizeDirection;
        common.resizeNode({
            el: el,
            node: editorNode,
            direction: 'top',
            onresize: function (result) {
                scrollToBottom();
                context.isResizing = true;
                var range = result.range;
                var resizeNode = im.resizeNode.messageInput;
                resizeNode.height = range;
                resizeDirection.use = resizeDirection.temp;
            },
            onended: function () {
                context.isResizing = false;
                resizeDirection.use = 'normal';
            }
        });
    }

    function getMethods(im) {
        var remoteApi = dataModel.RemoteControl;
        var methods = {
            showHistory: function () {
                this.$emit("showHistory")
            },
            reset: function () {
                if (this.$refs.editor) {
                    this.$refs.editor.reset();
                }
            },
            // 使输入框获得焦点
            focus: function () {
                if (this.$refs.editor) {
                    this.$refs.editor.focus();
                }
            },
            // 获取输入框的内容
            getValue: function () {
                return this.$refs.editor.getValue();
            },
            // 触发发送消息
            sendMessage: function () {
                sendMessage(this, im);
            },
            // 发送收藏消息
            sendCollectMessage: function (message) {
                sendCollectMessage(this, im, message);
            },
            messageInputChanged: function (value) {
                messageInputChanged(this, value);
            },
            toggleEmoji: function () {
                toggleEmoji(this);
                this.$emit('prepareinput');
            },
            prepareinput: function () {
                this.$emit('prepareinput');
            },
            selectedEmoji: function (emoji) {
                this.$refs.editor.insertText(emoji);
            },
            hideEmojiPanel: function () {
                this.showEmojiPanel = false;
            },
            takeScreenshot: function () {
                this.$emit('removeQuote');
                this.$emit('prepareinput');
                if (this.isShowWindow) {
                    RongIM.screenshot.start(true);
                    return;
                }
                RongIM.screenshot.start();


            },
            selectFile: function () {
                this.$emit('removeQuote');
                this.$emit('prepareinput');
            },
            fileChanged: function (event) {
                var context = this;
                Vue.nextTick(function () {
                    var fileList = event.target.files;
                    var uploadFiles = [];
                    for (var i = fileList.length - 1; i >= 0; i -= 1) {
                        var item = fileList[i];
                        // 选中非 0 字节文件
                        if (item.size > 0) {
                            uploadFiles.unshift(item);
                        }
                    }
                    // 上传所选中文件
                    if (uploadFiles.length) {
                        uploadFileList(uploadFiles, context, im);
                    }
                    // 提示所选中的文件中有 0 字节文件
                    if (uploadFiles.length < fileList.length) {
                        common.messageToast({
                            type: 'error',
                            message: context.locale.zeroSize
                        });
                    }
                    // 重置 input file 的 value 使可以连续多次上传同一文件
                    resetInputFileValue(event.target, methods.fileChanged, context);
                });
            },
            dragover: function (event) {
                event.preventDefault();
                event.stopPropagation();
            },
            // 拖拽上传文件
            drop: function (event) {
                event.preventDefault();
                event.stopPropagation();
                var file = RongIM.file;
                var context = this;
                var items = event.dataTransfer.items;
                var fileList = [];
                if (items) {
                    // chrome firefox 过滤文件夹
                    $.each(items, function (index, item) {
                        var entry = item.webkitGetAsEntry();
                        if (!entry) {
                            // 拖动 a 标签等内容时 entry 值会为空
                            return;
                        }
                        if (entry.isFile) {
                            fileList.push(item.getAsFile());
                        } else if (entry.isDirectory) {
                            // web和 desktop 分开处理
                            // item.getAsFile().path
                            /* name: "css"
                            path: "/Users/zy/Downloads/desktop-client.git/css"
                            size: 578 */

                            var dirPath = item.getAsFile().path;
                            file.zipFolders([dirPath], function (zipFileList) {
                                // var _fileList =[fileList[0].zipFile];
                                uploadFolderByPath(zipFileList, context, im);
                            });
                        }
                    });
                } else {
                    // IE 不包含文件夹
                    fileList = event.dataTransfer.files;
                }
                var filterZero = fileList.filter(function (item) {
                    return item.size > 0;
                });
                if (filterZero.length < fileList.length) {
                    common.messageToast({
                        type: 'error',
                        message: this.locale.zeroSize
                    });
                }
                if (filterZero.length === 0) {
                    return;
                }
                uploadFileList(filterZero, this, im);
            },
            paste: function (event) {

                paste(this, event, im);
            },
            clearUnReadCount: function () {
                var params = this.$route.params;
                var conversationType = params.conversationType;
                var targetId = params.targetId;
                dataModel.Conversation.clearUnReadCount(conversationType, targetId);
            },
            sendVideo: function () {
                if (remoteApi.isBusyline()) {
                    showRemoteBusyBox();
                    return;
                }
                this.$emit('removeQuote');
                this.$emit('sendVideo');
            },
            sendAudio: function () {
                if (remoteApi.isBusyline()) {
                    showRemoteBusyBox();
                    return;
                }
                this.$emit('removeQuote');
                this.$emit('sendAudio');
            },
            sendCard: function () {
                this.$emit('removeQuote');
                var userId = '';
                if (+this.$route.params.conversationType === 1) {
                    userId = this.$route.params.targetId;
                }
                // dialog.card(userId);
                // dialog.newCard(userId);
                dialog.cardDialog(userId)
            },
            getResizeDirection: function () {
                var direction = common.getResizeDirection({
                    range: this.height,
                    bound: this.bound.height,
                    directions: ['bottom', 'top']
                });
                if (this.isResizing) {
                    im.resizeDirection.temp = direction;
                }
                return direction;
            },
            getTextareaHeight: function () {
                return this.height - 50;
            },
            // 公众号点击 切换输入框和菜单
            inputMenuChanged: function () {
                this.$emit('inputMenuChanged', true);
            },
            toggleScreenOption: function () {
                this.isShowScreenOption = !this.isShowScreenOption;
            },
            hideScreenOption: function () {
                this.isShowScreenOption = false;
            },
            // 展示 远程控制 发送面板
            showRemoteCtrlPanel: function () {
                if (remoteApi.isBusyline()) {
                    showRemoteBusyBox();
                    return;
                }
                var voipStatus = RCCall.getStatus();
                if (voipStatus.isBusying) {
                    var voipTip = this.voipTip;
                    var data = voipStatus.current;
                    common.messageToast({
                        type: 'error',
                        message: voipTip[data.type]
                    });
                    return;
                }
                this.isShowRemoteCtrlPanel = true;
            },
            // 隐藏 远程控制 发送面板
            hideRemoteCtrlPanel: function (event) {
                var context = this;
                var $target = $(event.target);
                var wrap = '.rong-pannel-remote, .rong-toolbar-remote';
                var inBody = $target.closest('body').length > 0;
                var inWrap = $target.closest(wrap).length < 1;
                var isOuter = inBody && inWrap;
                if (isOuter) {
                    context.isShowRemoteCtrlPanel = false;
                }
            },
            // 发送远程控制, 控制对方
            requestRemoteControl: function () {
                sendRemoteControl(this, remoteApi, true);
            },
            // 邀请对方进行远程控制 `45
            inviteRemoteControl: function () {
                sendRemoteControl(this, remoteApi);
            },
            // 显示收藏列表
            showCollect: function () {
                dialog.collect(this);
            },
            //接龙
            buildChain: function () {
                let that = this
                dialog.buildChain({}, function (message) {
                    let obj = message
                    obj.content = utils.deepClone(message.title)
                    obj.title = '【接龙】由' + message.operate_name + '发起'
                    that.$emit("sendRichContentMessage", obj)

                });
            }
        };
        return methods;
    }

    function showRemoteBusyBox() {
        var im = RongIM.instance;
        var locale = im.locale.components.remoteControl;
        var busyLocale = locale.busy;
        return common.messageToast({
            type: 'error',
            message: busyLocale
        });
    }

    // 表情面板隐藏展示切换
    function toggleEmoji(context) {
        var hideEmoji = function () {
            context.showEmojiPanel = false;
            $(document).off('click', hideEmoji);
        };
        context.showEmojiPanel = !context.showEmojiPanel;
        if (context.showEmojiPanel) {
            $(document).on('click', hideEmoji);
        }
    }

    /*
    说明： 根据文件路径获取文件并上传
    */
    function uploadFileByPath(filePaths, context, im) {
        var fileApi = RongIM.file;
        filePaths.forEach(function (path) {
            fileApi.getBlob(path).then(function (file) {
                uploadFileList([file], context, im);
            }, function (err) {
                console.warn('get local file err', err);
            });
        });
    }

    /*
    说明： 上传文件夹
    */
    function uploadFolderByPath(fileObjs, context, im) {
        var file = RongIM.file;
        var filePaths = [];
        fileObjs.forEach(function (item) {
            filePaths.push(item.zipFile);
        });
        var fileList = file.getBlobs(filePaths);

        uploadFileList(fileList, context, im);
    }

    function getPasteData(clipboardData) {
        var data = {};
        var items = clipboardData.items;
        if (utils.isEmpty(items)) {
            var string = clipboardData.getData('text');
            data.str = {
                getAsString: function (callback) {
                    callback(string);
                }
            };
        } else {
            for (var i = items.length - 1; i >= 0; i -= 1) {
                var item = items[i];
                if (item.kind === 'file') {
                    var file = item.getAsFile();
                    if (file.size > 0) {
                        data.file = file;
                    }
                }
                if (item.kind === 'string' && item.type === 'text/plain') {
                    data.str = item;
                }
            }
        }
        return data;
    }

    function decodeMessageJSON(str) {
        try {
            var obj = JSON.parse(str);
            return obj;
        } catch (error) {
            return null;
        }
    }

    function decodeMessage(msgStr) {
        var messageApi = dataModel.Message;
        var message = decodeMessageJSON(msgStr);
        if (utils.isEmpty(message) || utils.isEmpty(message.messageName)) {
            return undefined;
        }
        var msg = {
            messageType: message.messageName,
            content: message
        };
        message = messageApi.create(msg);
        // 文件消息有本地存储地址
        if (message.localPath) {
            message.localPath = message.localPath;
        }
        return message;
    }

    /*
    说明： 重置 input file 使可以重复上传同一个文件。
    */
    function resetInputFileValue(inputFile, changed, context) {
        var $inputFile = $(inputFile);
        var $newInputFile = $inputFile.clone();
        $newInputFile.val('');
        $inputFile.replaceWith($newInputFile);
        $newInputFile.change(function (event) {
            changed.call(context || this, event);
        });
    }

    function getBase64(blob, callback) {
        var fr = new FileReader();
        fr.onload = function (event) {
            var base64Str = event.target.result;
            base64Str = Base64Util.replace(base64Str);

            callback(base64Str);
        };
        fr.readAsDataURL(blob);
    }

    function getConversationInfo(conversation) {
        return {
            targetId: conversation.targetId,
            conversationType: conversation.conversationType
        };
    }

    /*
    说明： 上传图片 base64 字符串
    */
    function uploadBase64(base64Str, context, im, base64Type, isEmojiImage) {
        var fileApi = dataModel.File;
        var messageApi = dataModel.Message;
        var params = getConversationInfo(context.$route.params);
        params.data = base64Str;
        var uploadMessage = fileApi.createUploadMessage(params);

        var base64Config = $.extend({}, im.config.upload.base64);
        var size = utils.getBase64Size(base64Str);
        var base64Size = base64Config.size;
        if (size > base64Size) {
            var message = utils.templateFormat(context.locale.screenshotMaxSize, parseInt(base64Size / 1024) + 'KB');
            common.messageToast({
                type: 'error',
                message: message
            });
            return;
        }
        base64Config.ext = base64Type;
        base64Config.data = UploadClient.dataType.data;
        var api = {
            file: fileApi,
            message: messageApi
        };
        // 消息体最大 128K 限制 base64 字串在 100K
        var maxLength = 100 * 1024;
        if (isEmojiImage && uploadMessage.data.length < maxLength) {
            base64Config.isEmoji = true;
            uploadMessage.content.content = uploadMessage.data;
        }
        upload(uploadMessage, base64Config, context, api);
    }

    /*
    说明： 上传多个文件
    */
    function uploadFileList(fileList, context, im) {
        var fileApi = dataModel.File;
        var messageApi = dataModel.Message;
        var params = getConversationInfo(im.$route.params);
        // var chunk_size = RongIM.config.upload.file.fileSize;
        var fileSize = RongIM.serverConfig.media.max_file_size * 1024 * 1024;
        var fileMaxSize = fileSize ? utils.formatFileSize(fileSize) : '';
        var message = fileMaxSize ? utils.templateFormat(context.locale.overSize, fileMaxSize) : '';

        for (var i = fileList.length - 1; i >= 0; i -= 1) {
            if (fileSize && fileList[i].size > fileSize) {
                common.messageToast({
                    type: 'error',
                    message: message
                });
            } else {
                var file = fileList[i];
                params.data = file;

                // localPath 为了兼容复制的本地文件,File 的 path 属性只读
                params.localPath = file.path || file.localPath;
                params.isFolder = !!file.folder;
                var uploadMessage = fileApi.createUploadMessage(params);
                var api = {
                    file: fileApi,
                    message: messageApi
                };
                upload(uploadMessage, im.config.upload.file, context, api);
            }
        }
    }

    /*
    说明： 文件上传
    */
    function upload(uploadMessage, config, context, api) {
        var userApi = RongIM.dataModel.User;
        var friendApi = RongIM.dataModel.Friend;
        var isPrivate = Number(uploadMessage.conversationType) === 1;
        if (isPrivate) {
            var canNotChat = !userApi.validateCanChat(uploadMessage.targetId);
            if (canNotChat) {
                friendApi.insertRFVMsg(uploadMessage.targetId);
                return;
            }
        }
        api.file.upload(uploadMessage, config, function (errorCode, _uploadMessage, data) {
            if (errorCode) {
                common.messageToast({
                    type: 'error',
                    message: context.locale.components.addAttachment.uploadFaild
                });
                return;
            }
            // api.file.delZip(uploadMessage.localPath);
            api.file.addFileUrl(_uploadMessage, data, function (error, message) {
                sendFileMessage(message, context, api);
            });
        });
    }

    function sendFileMessage(uploadMessage, context, api) {
        // 加入预发队列
        context.fileMessageList.push(uploadMessage);
        if (context.fileSending) {
            return;
        }
        context.fileSending = true;

        function onComplete() {
            // 有待发消息
            if (context.fileMessageList.length > 0) {
                // 间隔 500 毫秒发送，避免发送过快发送导致消息被 RCX 拒绝
                setTimeout(
                    sendFileMessageHandle, 500, context.fileMessageList.shift(),
                    context, api, onComplete
                );
                return;
            }
            context.fileSending = false;
        }
        // 处理一条消息
        sendFileMessageHandle(context.fileMessageList.shift(), context, api, onComplete);
    }

    function sendFileMessageHandle(message, context, api, done) {
        api.file.send(message, function (errorCode, uploadMessage) {
            if (errorCode) {
                var errMsg = common.getErrorMessage('lib-' + errorCode);
                if (errorCode === RongIMLib.ErrorCode.NOT_IN_GROUP) {
                    var targetId = uploadMessage.targetId;
                    var conversationType = uploadMessage.conversationType;
                    var params = common.createNotificationMessage(conversationType, targetId, errMsg);
                    api.message.insertMessage(params);
                    context.$emit('setInGroup', false);
                }
            } else {
                var im = RongIM.instance;
                im.$emit('messagechange');
                im.$emit('sendMessage');
            }
            done();
        });
    }

    function sendCollectMessage(context, im, message) {
        var connected = context.status === utils.status.CONNECTED;
        if (!connected) {
            return;
        }
        var Message = {};
        Message.text = message;
        context.panel = null;
        context.showEmojiPanel = false;
        context.$emit('sendMessage', Message);
        im.$emit('sendMessage');
    }

    /*
    说明： 发送消息 使用 vue 事件 sendMessage 通知发送消息
    功能： 1. 获取输入框内容
           2. 清空输入框，重置显示界面（关闭表情选择面板）
           3. 发送消息
    */
    function sendMessage(context, im) {
        var connected = context.status === utils.status.CONNECTED;
        if (!connected) {
            return;
        }
        var message = context.$refs.editor.getValue();
        if (utils.isEmpty(message.text.trim())) {
            return;
        }
        // props 只能在子组件中只读
        // context.draft = { content: '', atMembers: [] };
        context.$refs.editor.clear();
        context.panel = null;
        context.showEmojiPanel = false;
        if (message.at && message.at.length > 0) {
            // 过滤已删除的 @ 人员
            message.at = message.at.filter(function (item) {
                console.log(item)
                return message.text.indexOf('@' + item.name) > -1 || message.text.indexOf('@' + item.groupAlias) > -1 || message.text.indexOf('@' + item.groupAlias) > -1;
            });
        }
        message.text = convertSendMessage(message.text);
        context.$emit('sendMessage', message);
        im.$emit('sendMessage');
    }

    /*
    说明： Mac 非高分屏 emoji 字符展示会重叠需要在后面拼接空格，发送消息前删除 emoji 字符后的空格。
    */
    function convertSendMessage(text) {
        if (utils.isEmojiOverlap()) {
            var nativeReg = utils.emojiNativeReg.toString();
            var tagReg = nativeReg.substring(1, nativeReg.length - 3);
            // 出现emoji重叠时, 匹配后面有一个空格的emoji, 替换
            tagReg += '([ ])';
            tagReg = new RegExp(tagReg, 'ig');
            text = text.replace(tagReg, function (emoji) {
                return emoji.split(' ')[0];
            });
        }
        return text;
    }

    function messageInputChanged(context, value) {
        var connected = context.status === utils.status.CONNECTED;
        context.sendBtnAvailable = !utils.isEmpty((value || '').trim()) && connected;
        console.log(context.sendBtnAvailable)
    }

    /*
    说明： 粘贴文字，图片
    功能： Ctrl + V 粘贴时获取剪切板内容
           文字则插入到输入框，图片则上传发送
    */
    var inputPaste = {
        pasteString: function (event, data, context) {
            data.str.getAsString(function (str) {
                var msg = decodeMessage(str);
                if (msg) {
                    context.$emit('sendCopyMessage', msg);
                } else {
                    context.$refs.editor.insertText(str);
                }
            });
            event.preventDefault();
        },
        pasteImage: function (event, data, context, im) {
            var hasString = !utils.isEmpty(data.str);
            var dataString = '';
            if (hasString) {
                data.str.getAsString(function (str) {
                    dataString = str;
                });
            }
            getBase64(data.file, function (base64Str) {
                var base64 = Base64Util.concat(base64Str);
                RongIM.dialog.previewImage(base64, function (isConvertStr) {
                    if (!context.$refs.editor) {
                        return;
                    }
                    if (isConvertStr) {
                        context.$refs.editor.insertText(dataString);
                        return;
                    }
                    context.$refs.editor.focus();
                    uploadBase64(base64Str, context, im);
                });
            });
            event.preventDefault();
        },
        empty: function () { }
    };

    function getFileExt(fileName) {
        fileName = fileName || '';
        var reg = /\.([0-9a-z]+)$/i;
        var result = reg.exec(fileName);
        if (result) {
            return result[1];
        }
        return '';
    }

    /*
    说明： 粘贴事件处理
    功能： 上传单张图片先预览，多张图片直接发送（大于 5 兆图片按文件处理）
    */
    function paste(context, event, im) {

        var file = RongIM.file;
        // var clipFile = file.getPathsFromClip();
        var clipFile = file.getPaths();
        var isReturn = false;
        if (clipFile && clipFile.fileList.length > 0) {
            // 判断如果是粘贴的单张图片则需要预览图片然后发送，多文件则直接上传发送
            var clipboardImg = file.getImgByPath();
            var imagePath = clipFile.fileList[0];
            var hasSogoFace = imagePath.indexOf('SGPicFaceTpBq') !== -1;
            var hasBaiduFace = imagePath.indexOf('BaiduPinyin') !== -1;
            var isEmojiImage = hasSogoFace || hasBaiduFace;
            if (clipboardImg) {
                var base64Type = getFileExt(clipboardImg.name);
                var reader = new FileReader();
                reader.addEventListener('load', function () {
                    var base64 = reader.result;
                    var base64Str = Base64Util.replace(base64);
                    RongIM.dialog.previewImage(base64, function () {
                        if (context.$refs.editor) {
                            context.$refs.editor.focus();
                            uploadBase64(base64Str, context, im, base64Type, isEmojiImage);
                        }
                    });
                }, false);

                reader.onerror = function (err) {
                    console.warn('paste image failed', err);
                };
                reader.readAsDataURL(clipboardImg);
            } else {
                uploadFileByPath(clipFile.fileList, context, im);
            }
            isReturn = true;
        }

        function isValidDir(dirList) {
            return dirList.length > 0 && dirList[0] !== '/';
        }
        if (clipFile && isValidDir(clipFile.dirList)) {
            // 上传目录  压缩目录,上传压缩结果
            file.zipFolders(clipFile.dirList, function (fileList) {
                uploadFolderByPath(fileList, context, im);
            });
            isReturn = true;
        }
        if (isReturn) {
            event.preventDefault();
            return;
        }
        handleClipboard(context, event, im);
    }

    /*
        chrome
        --------------- clipboardData
        属性
            dropEffect 拖拽相关属性
            effectAllowed 拖拽相关属性

        粘贴文件以下属性都为空，通过 C++ 处理粘贴文件
            files
            types
            items 与 types 一一对应
            getData('text')

        粘贴内容：
        files 属性都为空

            word:
                windows types ["text/plain", "text/html", "text/rtf"]
                        types ["text/plain", "text/html", "text/rtf"] 仅复制表格
                        types ["text/html", "Files"] 仅复制图片
                macOS   types ["text/plain", "text/html", "text/rtf", "Files"]
            pages: types ["text/plain", "text/html", "text/rtf"]
                   types ["text/plain", "text/html", "text/rtf", "Files"] 仅复制表格
                   types ["Files"] 仅复制图片
            excel: types ["text/plain", "text/html", "text/rtf", "Files"]
            ppt:   types ["Files"]
            pdf:   types ["text/plain", "text/html", "text/rtf"]
            text:  types ["text/plain"]
            md:    types ["text/plain"]
            截图:
                   types ["Files"]
            网页图片:
                windows types: ["text/html", "Files"]
                macOS   types: ["Files"] mac
                        types: ["text/plain", "Files"] 图片包含 alt 属性 (仅 mac )

        items 内容说明
            text/plain
                纯文本不包含图片 file 表格
            text/html
                包含图片 <img src = ""file:///C:\Users\***\AppData\Local\Temp\ksohtml\wps87D8.tmp.jpg">

        --------------- Electron 内置 clipboard
            clipboard.readText
            clipboard.readHTML
            clipboard.readRTF
    */
    function handleClipboard(context, event, im) {

        var clipboardData = event.clipboardData || window.clipboardData;
        var data = getPasteData(clipboardData);
        var hasImage = !utils.isEmpty(data.file);
        var hasString = !utils.isEmpty(data.str);
        var pasteHandler = 'empty';
        if (hasImage) {
            var hasHtml = clipboardData.getData('text/html') !== '';
            if (hasHtml && hasString) {
                // "text/html" is Table
                var table = isTable(clipboardData);
                if (table) {
                    pasteHandler = 'pasteImage';
                } else {
                    pasteHandler = 'pasteString';
                }
            } else {
                pasteHandler = 'pasteImage';
            }
        } else if (hasString) {
            pasteHandler = 'pasteString';
        }
        inputPaste[pasteHandler](event, data, context, im);
    }

    function isTable(clipboardData) {
        // https://msdn.microsoft.com/zh-cn/library/windows/desktop/ms649015(v=vs.85).aspx
        var result = false;
        if (utils.isEmpty(clipboardData.items)) {
            return result;
        }
        var html = clipboardData.getData('text/html');

        var match = html.match(/<body[^>]*>([\s\S]*)<\/body>/);
        if (match) {
            html = match[1];
        }
        html = html.replace('<!--StartFragment-->', '');
        html = html.replace('<!--EndFragment-->', '');
        html = html.trim();

        var $hmlt = $(html);
        var onlyOne = $hmlt.length === 1;
        var tableEle = Object.prototype.toString.apply($hmlt.get(0)) === '[object HTMLTableElement]';
        if (onlyOne && tableEle) {
            result = true;
        }
        return result;
    }

    /**
     * 发送远程控制
     * @param  {object} context  message-input 的 Vue 示例
     * @param  {object} remoteApi
     * @param  {bool} isControl true: 请求主控  false: 请求被主控
     */
    function sendRemoteControl(context, remoteApi, isControl) {
        var im = RongIM.instance;
        var params = context.$route.params;
        // 隐藏 远程控制 面板
        context.isShowRemoteCtrlPanel = false;
        var userApi = RongIM.dataModel.User;
        var friendApi = RongIM.dataModel.Friend;
        var isPrivate = Number(params.conversationType) === 1;
        if (isPrivate) {
            var canNotChat = !userApi.validateCanChat(params.targetId);
            if (canNotChat) {
                friendApi.insertRFVMsg(params.targetId);
                return;
            }
        }
        var remoteFuc = isControl ? remoteApi.request : remoteApi.invite;
        remoteFuc(params, function (err, message) {
            if (!err) {
                im.$emit('setRemoteControl', message);
                setRemoteTimeout(message, remoteApi);
            }
        });
    }

    function setRemoteTimeout(message, remoteApi) {
        var RemoteControlHangupReason = common.RemoteControlHangupReason;
        var targetId = message.targetId;
        var messageType = message.messageType;
        var sessionId = message.content.sessionId;
        setTimeout(function () {
            var remote = remoteApi.get(targetId);
            var currentMsg = remote.message || {};
            var isSameType = currentMsg.messageType === messageType;
            var isSameSession = currentMsg.content.sessionId === sessionId;
            if (currentMsg && isSameType && isSameSession) {
                var reason = RemoteControlHangupReason.RemoteNoResponse;
                hangupRemoteControl(message, sessionId, reason);
            }
        }, remoteOverTime);
    }

    function hangupRemoteControl(params, sessionId, reason) {
        var remoteApi = RongIM.dataModel.RemoteControl;
        var im = RongIM.instance;
        remoteApi.hangup(params, sessionId, reason, function (err, message) {
            if (!err) {
                im.$emit('setRemoteControl', message);
            }
        });
        remoteApi.summary(params, {
            reason: reason,
            direction: RongIMLib.MessageDirection.SEND
        });
    }
}(RongIM, {
    UploadClient: UploadClient,
    jQuery: jQuery,
    RCCall: RCCall
}, RongIM.components));