(function (RongIM, dependencies) {
'use strict';

var $ = dependencies.jQuery;
var Http = RongIM.dataModel._Http;
var Cache = RongIM.dataModel._Cache;
var ObserverList = RongIM.dataModel._ObserverList;
var util = RongIM.dataModel.util;
var store = RongIM.utils.cache;
var generatorKey = util.generatorKey;
var getStoreKey = util.getStoreKey;
var getLibErrorCode = util.getLibErrorCode;
var utils = RongIM.utils;

var getCurrentConnectionStatus;

var groupApi = null;
var userApi = null;
var friendApi = null;
var messageApi = null;
var statusApi = null;

var Conversation = {
    isPullMessageFinished: true,
    draft: {},
    observerList: new ObserverList(),
    active: {
        conversationType: 0,
        targetId: ''
    }
};

var converObserverList = Conversation.observerList;

Cache.conversation = {
    searchTempList: []
};

Conversation.cleanCache = function () {
    Cache.conversation = {
        searchTempList: []
    };
    Conversation.draft = {};
};

Conversation.loadApi = function () {
    var dataModel = RongIM.dataModel;
    groupApi = dataModel.Group;
    userApi = dataModel.User;
    friendApi = dataModel.Friend;
    messageApi = dataModel.Message;
    statusApi = dataModel.Status;
    getCurrentConnectionStatus = statusApi.getCurrentConnectionStatus;
};

function isGroup(conversationType) {
    return +conversationType === 3;
}

function debounceConversation() {
    converObserverList.notify();
}

/*
    获取置顶的会话
 */
var getTopListFromRemote = function (callback) {
    // 
    var data = Cache.conversation;
    if (data.isSetting) {
        var settings = data.settings;
        callback(null, settings);
        return;
    }
    if (data._defer) {
        data._defer.done(function (result) {
            callback(null, result);
        }).fail(callback);
        return;
    }
    data._defer = $.Deferred();
    Http.get('/conversation').done(function (result) {
        var conversationSettings = result.settings;

        data.settings = conversationSettings;
        data.isSetting = true;

        callback(null, conversationSettings);
        data._defer.resolve(conversationSettings);
    }).fail(function (error) {
        callback(error);
        data._defer.reject(error);
    });
};

// 判断 source 与 target 是否是同一会话
var checkConversation = function (source, target) {
    return +source.conversation_type === +target.conversationType && source.target_id === target.targetId;
};

var delDuplicate = function (arr) {
    return arr.reduce(function (p, c) {
        // create an identifying id from the object values
        var id = [c.conversation_type, c.target_id].join('|');

        // if the id is not found in the temp array
        // add the object to the output array
        // and add the key to the temp array
        if (p.temp.indexOf(id) === -1) {
            p.out.push(c);
            p.temp.push(id);
        }
        return p;

    // return the deduped array
    }, { temp: [], out: [] }).out;
};

/*
    1. 取出置顶会话
    2. 将置顶会话合并到会话中
    3. 会话排序
*/
function splitTopConversation(list, callback) {
    // 1、获取置顶数组 √
    // 2、匹配会话列表 List √
    // 3、按 sentTime 排序置顶 √
    // 4、追加到 list 中，保证操作的是同一个对象 √
    var tempList = [];

    var mergeConversation = function (topList) {
        if (util.isArray(topList)) {
            var idxs = [];
            for (var i = 0; i < topList.length; i += 1) {
                var topConversation = topList[i];
                for (var j = 0; j < list.length; j += 1) {
                    var conversation = list[j];
                    if (checkConversation(topConversation, conversation)) {
                        conversation.isTop = topConversation.top;
                        conversation.notificationStatus = topConversation.not_disturb;
                        if (conversation.isTop) {
                            idxs.push(j);
                            tempList.push(conversation);
                        }
                    }
                }
            }

            idxs.sort(function (a, b) {
                return b - a;
            });
            for (var k = 0; k < idxs.length; k += 1) {
                list.splice(idxs[k], 1);
            }
        }
    };

    var buildConversationList = function (result) {
        mergeConversation(result);
        tempList.sort(function (a, b) {
            return b.sentTime - a.sentTime;
        });
        tempList.reverse();
        // TODO 暂时做法，后期会话列表统一优化
        list.sort(function (a, b) {
            return b.sentTime - a.sentTime;
        });
        for (var i = 0, len = tempList.length; i < len; i += 1) {
            list.unshift(tempList[i]);
        }
        callback([].concat(list));
    };

    getTopListFromRemote(function (errorCode, result) {
        if (errorCode) {
            utils.console.warn(errorCode);
            return;
        }
        // utils.console.log(result);
        // 服务端 toplist 有重复,需加入排重
        var _list = delDuplicate(result);
        if (!Cache.conversation.topList) {
            Cache.conversation.topList = _list;
        }
        buildConversationList(Cache.conversation.topList);
    });
}
Conversation.splitTopConversation = splitTopConversation;

// 会话列表分页，置顶会话缺失，获取缺失的会话
function getOtherTopList(list, callback) {
    var _list = [];
    var tempIdList = [];
    tempIdList = list.map(function (con) {
        return con.targetId + con.conversationType;
    });
    getTopListFromRemote(function (errorCode, result) {
        if (errorCode) {
            utils.console.warn(errorCode);
            return;
        }
        // utils.console.log(result);
        // 服务端 toplist 有重复,需加入排重
        var _result = delDuplicate(result);

        var allTopList = _result.filter(function (con) {
            return con.top;
        });
        var otherTopList = allTopList.filter(function (con) {
            return tempIdList.indexOf(con.target_id + con.conversation_type) < 0;
        });
        if (otherTopList.length === 0) {
            callback(null, []);
            return;
        }
        var promiseList = [];
        var deferList = [];
        for (var i = 0; i < otherTopList.length; i += 1) {
            var defer = $.Deferred();
            var promise = defer.promise();
            deferList.push(defer);
            promiseList.push(promise);
        }
        otherTopList.forEach(function (con, index) {
            Conversation.getOneWithoutInfo(con.conversation_type, con.target_id, function (error, conversation) {
                if (error) {
                    utils.console.warn(error);
                    deferList[index].reject(error);
                    return;
                }
                if (conversation.latestMessage) {
                    _list.push(conversation);
                }
                deferList[index].resolve();
            });
        });
        $.when.apply(null, promiseList).done(function () {
            callback(null, _list);
        }).fail(function (error) {
            callback(error);
        });
    });
}

function addConversationUserInfo(list, callback) {
    callback = callback || $.noop;
    var ConversationType = RongIMLib.ConversationType;
    var isGroupConversation = function (item) {
        return item.conversationType === ConversationType.GROUP;
    };
    var isUser = function (item) {
        return item.conversationType === ConversationType.PRIVATE || item.conversationType === ConversationType.SYSTEM;
    };
    var isPublic = function (item) {
        return item.conversationType === ConversationType.APP_PUBLIC_SERVICE;
    };

    var bindInfo = function () {
        var addMessageUserInfo = function (latestMessage, groupAlias) {
            messageApi.fixSendMessageBug(latestMessage);
            var senderUserId = latestMessage.senderUserId;
            userApi.get(senderUserId, function (errorCode, user) {
                if (errorCode) {
                    utils.console.warn('获取用户信息失败（错误码：' + errorCode + '）');
                    latestMessage.user = {};
                } else if (groupAlias) {
                    var _user = $.extend({}, user);
                    _user.groupAlias = groupAlias;
                    latestMessage.user = _user;
                } else {
                    latestMessage.user = user;
                }
            });
        };

        list.forEach(function (item) {
            if (isGroupConversation(item)) {
                item.group = {};
                groupApi.getOne(item.targetId, function (errorCode, group) {
                    if (errorCode) {
                        utils.console.warn('获取群组信息失败, groupId=' + item.targetId + '（错误码：' + errorCode + '）');
                        groupApi.getOne(item.targetId, function (_errorCode, _group) {
                            if (_errorCode) {
                                utils.console.warn('获取群组信息失败, groupId=' + item.targetId + '（错误码：' + _errorCode + '）');
                            } else {
                                item.group = _group;
                                var groupAlias = '';
                                if (_group.members && item.latestMessage) {
                                    groupAlias = groupApi.getGroupAlias(item.latestMessage.senderUserId, _group.members);
                                }
                                addMessageUserInfo(item.latestMessage, groupAlias);
                            }
                        });
                    } else {
                        item.group = group;
                        var groupAlias = '';
                        if (group && group.members && item.latestMessage) {
                            groupAlias = groupApi.getGroupAlias(item.latestMessage.senderUserId, group.members);
                        }
                        addMessageUserInfo(item.latestMessage, groupAlias);
                    }
                });
            } else if (isUser(item) || isPublic(item)) {
                if (friendApi.isFileHelper(item.targetId)) {
                    item.user = friendApi.getFileHelper();
                } else {
                    item.user = {};
                }
                var userId = item.targetId;
                userApi.get(userId, function (errorCode, user) {
                    if (errorCode) {
                        utils.console.warn('获取用户信息失败, userId=' + userId + '(错误码：' + errorCode + ')');
                        userApi.get(userId, function (_errorCode, _user) {
                            if (_errorCode) {
                                utils.console.warn('获取用户信息失败, userId=' + userId + '(错误码：' + _errorCode + ')');
                            } else {
                                item.user = _user;
                            }
                            if (!_user) {
                                Conversation.remove(item.conversationType, item.targetId);
                                // TODO: 确定是否是要做过滤，若是需修改算法，不要 forEach
                                list.forEach(function (conv, index) {
                                    if (util.sameConversation(item, conv)) {
                                        list.splice(index, 1);
                                    }
                                });
                            }
                            addMessageUserInfo(item.latestMessage);
                        });
                    } else {
                        item.user = user;
                        if (!user) {
                            Conversation.remove(item.conversationType, item.targetId);
                            // TODO: 确认是否是要过滤，如果是要过滤，不要用 forEach
                            list.forEach(function (conv, index) {
                                if (util.sameConversation(item, conv)) {
                                    list.splice(index, 1);
                                }
                            });
                        }
                        addMessageUserInfo(item.latestMessage);
                    }
                });
            }
        });
        var error = null;
        callback(error, list);
    };

    var getIds = function (items) {
        return items.map(function (item) {
            return item.targetId;
        });
    };

    var groups = list.filter(isGroupConversation);
    var groupIds = getIds(groups);

    var users = list.filter(isUser);
    var userIds = getIds(users);

    var senderIds = list.map(function (item) {
        return item.latestMessage.senderUserId;
    });
    userIds = userIds.concat(senderIds);
    var unique = function (arr) {
        var tmp = {};
        return arr.filter(function (item) {
            if (tmp[item]) {
                return false;
            }
            tmp[item] = true;
            return true;
        });
    };
    userIds = unique(userIds);

    $.when(groupApi.getBatchGroups(groupIds), userApi.getUsers(userIds)).then(function () {
        bindInfo();
    });
}

/**
 * @params.count 返回 count 个会话;如果会话总数 total < count 则返回 total 个会话
 */
var getInitConversationList = function (count, callback, callbackHasUserinfo) {
    callback = callback || $.noop;
    if (!Conversation.isPullMessageFinished) {
        return;
    }
    if (!RongIM.isAvailableData()) {
        var status = getCurrentConnectionStatus();
        var errorCode = 'status-' + status;
        callback(errorCode);
        return;
    }
    var ConversationType = RongIMLib.ConversationType;
    // var isGroup = function (type) {
    //     return type === ConversationType.GROUP;
    // };
    var isUser = function (type) {
        return type === ConversationType.PRIVATE;
    };
    var isPublic = function (type) {
        return type === ConversationType.APP_PUBLIC_SERVICE;
    };
    var isSystem = function (type) {
        return type === ConversationType.SYSTEM;
    };
    // callback(null, []);
    // RongIMClient.getInstance().getConversationList({
    var conversation = {
        types: [1, 2, 3, 4, 5, 7, 8],
        sentTime: 0,
        count: count
    };
    RongIMClient.getInstance().getConversationsByPage(conversation, {
        onSuccess: function (list) {
            getOtherTopList(list, function (error, _otherList) {
                // 排除系统会话，会话列表不显示系统会话(朋友圈的消息)
                list = list.filter(function (item) {
                    return !isSystem(item.conversationType);
                });
                if (count > 0) {
                    list.length = list.length < count ? list.length : count;
                }
                list = list.concat(_otherList);
                var searchList = Cache.conversation.searchTempList;
                var searchObj = {};
                var keys = [];

                searchList.forEach(function (item) {
                    var key = generatorKey([item.conversationType, item.targetId]);
                    searchObj[key] = 1;
                });

                list.forEach(function (item) {
                    messageApi.hackWebSDKMessage(item.latestMessage);
                    var type = item.conversationType;
                    var targetId = item.targetId;
                    var key = generatorKey([type, targetId]);
                    var has = (key in searchObj);
                    if (has) {
                        keys.push(key);
                    }
                    // 置顶和免打扰置为默认值 false, 然后从缓存中匹配置顶和免打扰设置
                    item.isTop = false;
                    item.notificationStatus = false;
                    if (isUser(type) || isPublic(type)) {
                        if (friendApi.isFileHelper(targetId)) {
                            item.user = friendApi.getFileHelper();
                        } else {
                            item.user = Cache.user[targetId] || {};
                        }
                    }
                    // if(isPublic(type)) {
                    //     item.user = item.latestMessage;
                    // }
                    var message = item.latestMessage;
                    var senderUserId = message.senderUserId;
                    var msgUser = Cache.user[senderUserId] || {};
                    if (isGroup(type)) {
                        item.group = Cache.group[targetId] || {};
                        var groupAlias = '';
                        if (item.group.members) {
                            groupAlias = groupApi.getGroupAlias(senderUserId, item.group.members);
                        }
                        var _user = $.extend({}, msgUser);
                        _user.groupAlias = groupAlias;
                        message.user = _user;
                    } else {
                        message.user = msgUser;
                    }
                    item.draft = Conversation.getDraft(item.conversationType, item.targetId);
                });
                searchList.forEach(function (item, index) {
                    item.isTop = false;
                    item.notificationStatus = false;
                    var type = item.conversationType;
                    var targetId = item.targetId;
                    keys.forEach(function (key) {
                        if (generatorKey([type, targetId]) === key) {
                            searchList.splice(index, 1);
                        }
                    });
                    item.draft = Conversation.getDraft(item.conversationType, item.targetId);
                });
                list = searchList.concat(list);
                splitTopConversation(list, function (result) {
                    bindRemoteControl(result);
                    Cache.conversation.list = result;
                    bindConversationRequestMsgs(result);
                    addConversationUserInfo(result, callbackHasUserinfo);
                    callback(null, result);
                });
            });
        },
        onError: function (error) {
            callback(getLibErrorCode(error));
        }
    }, null);
};

// 返回全部会话
var getConversationList = function (callback, callbackHasUserinfo) {
    callback = callback || $.noop;
    getInitConversationList(0, callback, callbackHasUserinfo);
};

// conversation list 获取 Cache.conversation.list 中的 remoteControl
function bindRemoteControl(list) {
    var cacheList = Cache.conversation.list || [];
    var platform = RongIM.utils.getPlatform();
    var isPc = platform.indexOf('web') === -1;
    if (!isPc || !cacheList) {
        return undefined;
    }
    list.forEach(function (conversation) {
        var targetId = conversation.targetId;
        var conversationType = conversation.conversationType;
        var cacheCon = cacheList.filter(function (item) {
            return item.targetId === targetId && item.conversationType === conversationType;
        });
        cacheCon = cacheCon.length ? cacheCon[0] : {};
        conversation.remoteControl = cacheCon.remoteControl;
    });
    return list;
}

// todo: bindConversationRequestMsgs 备注
function bindConversationRequestMsgs(list) {
    var storeKey = getStoreKey('req');
    var data = store.get(storeKey) || {};
    list.forEach(function (item) {
        var conversationKey = generatorKey([item.conversationType, item.targetId]);
        item.requestMsgs = data[conversationKey];
    });
}

// 获取会话详细信息
function getConversationInfo(conversation, callback) {
    var conversationType = conversation.conversationType;
    var targetId = conversation.targetId;
    var getInfo = isGroup(conversationType) ? groupApi.getOne : userApi.get;
    getInfo(targetId, done);
    function done(errorCode, result) {
        if (errorCode) {
            callback(errorCode);
            return;
        }
        var type = isGroup(conversationType) ? 'group' : 'user';
        conversation[type] = result || {};
        bindConversationRequestMsgs([conversation]);
        Conversation.getExpandInfo(conversation, function (error, topMute) {
            conversation.isTop = topMute.top;
            conversation.notificationStatus = topMute.not_disturb;
            callback(null, conversation);
        });
    }
}

Conversation.notifyConversation = RongIM.utils.throttle(debounceConversation, 600);

// 创建一条新会话
var createConversation = function (params) {
    var reset = util.reset;
    var sentTime = +new Date();
    var data = {
        conversationType: 0,
        isTop: false,
        latestMessage: {
            sentTime: sentTime,
            messageType: 'TextMessage',
            content: ''
        },
        notificationStatus: false,
        receivedStatus: 0,
        targetId: '',
        sentTime: sentTime,
        unreadMessageCount: 0
    };
    reset(data, params);
    var conversation = new RongIMLib.Conversation();
    reset(conversation, data);
    messageApi.get({
        targetId: conversation.targetId,
        conversationType: conversation.conversationType
    }, function (error, list) {
        if (list && list.length > 0) {
            conversation.latestMessage = list[list.length - 1];
        }
    });
    return conversation;
};

// 插入一条新会话
Conversation.add = function (params, callback) {
    callback = callback || $.noop;
    var type = params.conversationType;
    var targetId = params.targetId;
    var isSame = function (item) {
        return item.conversationType === type && item.targetId === targetId;
    };
    var filter = function (item) {
        return isSame(item);
    };
    var cacheList = Cache.conversation.list || [];
    var list = cacheList.filter(filter);
    var isExist = (list.length > 0);
    if (!isExist) {
        var conversation = createConversation(params);
        var typeKey = isGroup(type) ? 'group' : 'user';
        conversation[typeKey] = {};
        conversation.latestMessage = params.latestMessage || conversation.latestMessage;
        var getIndex = function () {
            var topList = Cache.conversation.topList || [];
            topList.forEach(function (item) {
                if (checkConversation(item, conversation)) {
                    conversation.isTop = item.top;
                    conversation.notificationStatus = item.not_disturb;
                }
            });
            topList = topList.filter(function (item) {
                return item.top;
            });
            var getKey = function (item) {
                return generatorKey([item.conversationType, item.targetId]);
            };
            var has = function (key, obj) {
                return (key in obj);
            };
            var index = 0;
            if (!conversation.isTop) {
                var topObj = {};
                topList.forEach(function (item) {
                    // var key = getKey(item);
                    var key = generatorKey([item.conversation_type, item.target_id]);
                    topObj[key] = 1;
                });
                cacheList.forEach(function (item) {
                    var key = getKey(item);
                    if (has(key, topObj)) index += 1;
                });
            }
            return index;
        };
        var insert = function () {
            var index = getIndex();
            cacheList.splice(index, 0, conversation);
        };
        Cache.conversation.searchTempList.unshift(conversation);
        insert();
        RongIMClient.getInstance().addConversation(conversation);
        getConversationInfo(conversation, function () {
            converObserverList.notify(cacheList);
            callback(conversation);
        });
    }
};

// 增加文件助手
Conversation.addFileHelper = function () {
    var key = 'phoneLogined-' + Cache.auth.id;
    var isLogined = store.get(key);
    if (!isLogined) {
        friendApi.getFileHelper(function (err, fileHelper) {
            if (!err) {
                var params = {
                    targetId: fileHelper.id,
                    conversationType: RongIMLib.ConversationType.PRIVATE,
                    user: fileHelper
                };
                Conversation.add(params, function () {
                    var arg = {
                        targetId: params.targetId,
                        conversationType: params.conversationType,
                        objectName: 'RC:InfoNtf',
                        sentStatus: RongIMLib.SentStatus.SENT,
                        content: {
                            message: RongIM.instance.locale.components.getFileHelper.desc
                        }
                    };
                    messageApi.insertMessage(arg);
                });
                var list = Cache.conversation.list || [];
                converObserverList.notify(list);
                store.set(key, true);
            }
        });
    }
};

Conversation.getInitList = function (count, callback) {
    getInitConversationList(count, callback);
};

Conversation.getList = function (callback, callbackHasUserinfo) {
    getConversationList(callback, callbackHasUserinfo);
};

// 返回缓存中的会话
Conversation.getLocalList = function () {
    return Cache.conversation.list || [];
};

/*
    获取原始会话数据
    主要用于计算未读消息数; 搜索时判断会话是否存在以保证能正确定位
*/
Conversation.getNativeList = function (callback) {
    callback = callback || $.noop;
    if (!RongIM.isAvailableData()) {
        var status = getCurrentConnectionStatus();
        var errorCode = 'status-' + status;
        callback(errorCode);
        return;
    }
    var ConversationType = RongIMLib.ConversationType;
    var isSystem = function (type) {
        return type === ConversationType.SYSTEM;
    };
    RongIMClient.getInstance().getConversationList({
        onSuccess: function (list) {
            // 排除系统会话，会话列表不显示系统会话(朋友圈的消息)
            list = list.filter(function (item) {
                return !isSystem(item.conversationType);
            });
            callback(null, list);
        },
        onError: function (error) {
            callback(getLibErrorCode(error));
        }
    }, null);
};

// 根据 conversationType, targetId 获取缓存会话
Conversation.getLocalOne = function (conversationType, targetId) {
    var conversation = null;
    var list = Conversation.getLocalList();
    list.some(function (item) {
        var sameConversation = +item.conversationType === +conversationType && item.targetId === targetId;
        if (sameConversation) {
            conversation = item;
        }
        return sameConversation;
    });
    return conversation;
};

// 根据 conversationType, targetId 从接口获取会话
Conversation.getOne = function (conversationType, targetId, callback) {
    callback = callback || $.noop;
    var lconv = Conversation.getLocalOne(conversationType, targetId);
    if (lconv) {
        callback(null, lconv);
        return;
    }
    RongIMClient.getInstance().getConversation(+conversationType, targetId, {
        onSuccess: function (conversation) {
            if (!conversation) {
                // 会话不存在新建一个
                conversation = {
                    targetId: targetId,
                    conversationType: conversationType,
                    unreadMessageCount: 0
                };
            }

            getConversationInfo(conversation, callback);
        },
        onError: function (errorCode) {
            callback(errorCode);
        }
    });
};

Conversation.getOneWithoutInfo = function (conversationType, targetId, callback) {
    callback = callback || $.noop;
    var lconv = Conversation.getLocalOne(conversationType, targetId);
    if (lconv) {
        callback(null, lconv);
        return;
    }
    RongIMClient.getInstance().getConversation(+conversationType, targetId, {
        onSuccess: function (conversation) {
            if (!conversation) {
                // 会话不存在新建一个
                conversation = {
                    targetId: targetId,
                    conversationType: conversationType,
                    unreadMessageCount: 0
                };
            }
            callback(null, conversation);
        },
        onError: function (errorCode) {
            callback(errorCode);
        }
    });
};

// 获取单条会话未读数
Conversation.getUnreadCount = function (conversation) {
    var callbacks = {
        onSuccess: function () {},
        onError: function () {}
    };
    var type = conversation.conversationType || conversation.conversation_type;
    var targetId = conversation.targetId || conversation.target_id;
    return RongIMClient.getInstance().getUnreadCount(Number(type), targetId, callbacks);
};

// 获取总未读数
Conversation.getTotalUnreadCount = function (list, callback) {
    callback = callback || $.noop;
    var conversationTypes = [
        utils.conversationType.PRIVATE,
        utils.conversationType.GROUP,
        utils.conversationType.APP_PUBLIC_SERVICE
    ];
    var total = 0;
    list.forEach(function (item) {
        if (!item.notificationStatus && conversationTypes.indexOf(item.conversationType) !== -1) {
            total += item.unreadMessageCount;
        }
    });
    if (list.length > 0) {
        callback(null, total);
        return;
    }
    var callbacks = {
        onSuccess: function () {},
        onError: function () {}
    };

    total = RongIMClient.getInstance().getTotalUnreadCount(callbacks, conversationTypes);
    getTopListFromRemote(function (error, settingList) {
        if (error) {
            callback(error, settingList);
            return;
        }
        var muteCount = 0;
        settingList.forEach(function (conversation) {
            if (conversation.not_disturb) {
                muteCount += Conversation.getUnreadCount(conversation);
            }
        });
        total -= muteCount;
        if (total < 0) {
            total = 0;
        }
        callback(error, total);
    });
};

Conversation.clearUnReadCount = function (conversationType, targetId, callback) {
    callback = callback || $.noop;
    // 类型转换 SDK 要求 number 类型
    conversationType = +conversationType;
    RongIMClient.getInstance().getConversation(conversationType, targetId, {
        onSuccess: function (conversation) {
            if (!conversation) {
                return;
            }
            bindConversationRequestMsgs([conversation]);
            if (conversation.unreadMessageCount > 0) {
                clearUnreadCount(conversationType, targetId, callback);
                if (conversationType === RongIMLib.ConversationType.PRIVATE) {
                    messageApi.sendReadStatus(conversation.latestMessage);
                } else {
                    messageApi.sendSyncReadStatusMessage(conversation.latestMessage);
                }
                if (isGroup(conversationType) && conversation.requestMsgs) {
                    messageApi.sendGroupResponse(conversationType, targetId, conversation.requestMsgs);
                }
            }
        },
        onError: function (errorCode) {
            callback(errorCode);
        }
    });
};

function clearUnreadCount(conversationType, targetId, callback) {
    callback = callback || $.noop;
    RongIMClient.getInstance().clearUnreadCount(conversationType, targetId, {
        onSuccess: function () {
            var list = Conversation.getLocalList();
            var conversation = Conversation.getLocalOne(conversationType, targetId);
            if (conversation) {
                conversation.unreadMessageCount = 0;
                converObserverList.notify(list);
            }
            callback();
        },
        onError: function (errorCode) {
            callback(getLibErrorCode(errorCode));
        }
    });
}

Conversation.clearUnreadCount = clearUnreadCount;

Conversation.getDraft = function (conversationType, targetId) {
    var path = conversationType + '/' + targetId;
    return Conversation.draft[path] || { content: '', atMembers: [] };
};

Conversation.setDraft = function (conversationType, targetId, draft) {
    var path = conversationType + '/' + targetId;
    Conversation.draft[path] = draft;
    var conversation = Conversation.getLocalOne(conversationType, targetId);
    if (conversation) {
        conversation.draft = draft;
        var list = Conversation.getLocalList();
        converObserverList.notify(list);
    }
};

Conversation.clearDraft = function (conversationType, targetId) {
    Conversation.setDraft(conversationType, targetId, '');
    var conversation = Conversation.getLocalOne(conversationType, targetId);
    if (conversation) {
        conversation.draft = {
            atMembers: [],
            content: ''
        };
        var list = Conversation.getLocalList();
        converObserverList.notify(list);
    }
};

Conversation.remove = function (conversationType, targetId, callback) {
    callback = callback || $.noop;
    if (!RongIM.isAvailableData()) {
        var status = getCurrentConnectionStatus();
        var errorCode = 'status-' + status;
        callback(errorCode);
        return;
    }
    var isSame = function (item) {
        return item.conversationType === conversationType && item.targetId === targetId;
    };
    var searchList = Cache.conversation.searchTempList;
    searchList.forEach(function (item, index) {
        if (isSame(item)) {
            searchList.splice(index, 1);
        }
    });
    RongIMClient.getInstance().removeConversation(conversationType, targetId, {
        onSuccess: function (bool) {
            if (bool) {
                var list = Conversation.getLocalList();
                var item = Conversation.getLocalOne(conversationType, targetId);
                var index = list.indexOf(item);
                if (index > -1) {
                    list.splice(index, 1);
                }
                converObserverList.notify(list);
                callback();
            } else {
                callback(bool);
            }
        },
        onError: function (error) {
            callback(getLibErrorCode(error));
        }
    });
};

/*
正在会话的会话，切换会话时调用
var active = {
    conversationType: 1,
    targetId: ''
};
*/
Conversation.setActive = function (active) {
    Conversation.active = active;
};

var toggleTop = function (conversationType, targetId, isTop, callback) {
    callback = callback || $.noop;
    var item = {
        1: function () {
            var topObj = {
                conversation_type: conversationType,
                target_id: targetId,
                top: true,
                not_disturb: false
            };
            var notHave = true;
            var topList = Cache.conversation.topList || [];
            for (var i = 0; i < topList.length; i += 1) {
                var temp = topList[i];
                if (+temp.conversation_type === +conversationType && temp.target_id === targetId) {
                    temp.top = true;
                    notHave = false;
                }
            }
            if (notHave) topList.push(topObj);
        },
        0: function () {
            var topList = Cache.conversation.topList || [];
            for (var i = 0; i < topList.length; i += 1) {
                var temp = topList[i];
                if (+temp.conversation_type === +conversationType && temp.target_id === targetId) {
                    if (temp.not_disturb) {
                        temp.top = false;
                    } else {
                        topList.splice(i, 1);
                    }
                }
            }
        }
    };
    Http.put('/conversation/top', {
        conversation_type: conversationType,
        target_id: targetId,
        top: isTop
    }).done(function (result) {
        // if(isTop === 1){
        //     Cache.conversation.isSetting = false;
        //     Cache.conversation._defer = null;
        // }
        item[isTop]();
        callback(null, result);
        // Conversation.getList(function (errorCode, list) {
        //     if (errorCode) {
        //         return callback( errorCode);
        //     }
        //     converObserverList.notify(list);
        // });
    }).fail(callback);
};
// 置顶
Conversation.top = function (conversationType, targetId, callback) {
    toggleTop(conversationType, targetId, 1, callback);
};
// 取消置顶
Conversation.untop = function (conversationType, targetId, callback) {
    toggleTop(conversationType, targetId, 0, callback);
};

Conversation.search = function (keyword, callback) {
    callback = callback || $.noop;
    if (!RongIM.isAvailableData()) {
        var status = getCurrentConnectionStatus();
        var errorCode = 'status-' + status;
        callback(errorCode);
        return;
    }

    RongIMClient.getInstance().searchConversationByContent(keyword, {
        onSuccess: function (conversationList) {
            addConversationUserInfo(conversationList, function (error, list) {
                // 
                // 补充搜索到几条消息。TODO: 希望 SDK 可以支持
                if (error) {
                    callback(getLibErrorCode(error));
                    return;
                }
                callback(null, list);
            });
        },
        onError: function (code) {
            // 
            callback(code);
        }
    }, undefined, ['RCE:GrpNoticeNtfy']);
};

var toggleMute = function (conversationType, targetId, isMute, callback) {
    callback = callback || $.noop;
    var item = {
        1: function () {
            var topObj = {
                conversation_type: conversationType,
                target_id: targetId,
                top: false,
                not_disturb: true
            };
            var notHave = true;
            var topList = Cache.conversation.topList || [];
            for (var i = 0; i < topList.length; i += 1) {
                var temp = topList[i];
                if (+temp.conversation_type === +conversationType && temp.target_id === targetId) {
                    temp.not_disturb = true;
                    notHave = false;
                }
            }
            if (notHave) topList.push(topObj);
        },
        0: function () {
            var topList = Cache.conversation.topList || [];
            for (var i = 0; i < topList.length; i += 1) {
                var temp = topList[i];
                if (+temp.conversation_type === +conversationType && temp.target_id === targetId) {
                    if (temp.top || temp.not_disturb) {
                        temp.not_disturb = false;
                    } else {
                        topList.splice(i, 1);
                    }
                }
            }
        }
    };
    Http.put('/conversation/notdisturb', {
        conversation_type: conversationType,
        target_id: targetId,
        not_disturb: isMute
    }).done(function (result) {
        // if(isMute === 1){
        //     Cache.conversation.isSetting = false;
        //     Cache.conversation._defer = null;
        // }
        item[isMute]();
        callback(null, result);

        var list = Conversation.getLocalList();
        var conversation = Conversation.getLocalOne(conversationType, targetId);
        conversation.notificationStatus = isMute === 1;
        converObserverList.notify(list);
    }).fail(callback);
};

function getExpandInfo(type, id) {
    var result = {};
    Cache.conversation.topList.forEach(function (item) {
        if (+item.conversation_type === +type && item.target_id === id) {
            result = item;
        }
    });
    return result;
}

/*
会话相关详细信息
params.conversationType 会话类型
params.targetId 会话 Id
*/
Conversation.getExpandInfo = function (params, callback) {
    var type = params.conversationType;
    var id = params.targetId;
    callback = callback || $.noop;
    if (Cache.conversation.topList) {
        var expandInfo = getExpandInfo(type, id);
        callback(null, expandInfo);
    } else {
        getTopListFromRemote(function (errorCode, list) {
            Cache.conversation.topList = list || [];
            var result = getExpandInfo(type, id);
            callback(null, result);
        });
    }
};

// 免打扰
Conversation.mute = function (conversationType, targetId, callback) {
    toggleMute(conversationType, targetId, 1, callback);
};
// 取消免打扰
Conversation.unmute = function (conversationType, targetId, callback) {
    toggleMute(conversationType, targetId, 0, callback);
};

Conversation.watch = function (handle) {
    converObserverList.add(handle);
};

Conversation.unwatch = function (handle) {
    converObserverList.remove(handle);
};
Conversation.createJieLong =function (params, callback) {
    var url = '/apps/JieLong/createJieLong';
    Http.post(url, params)
        .then(function (result) {
            callback(null, result);
        });
};
Conversation.getJieLong =function (uid, callback) {
    var url = '/apps/JieLong/'+uid+'/getJieLong';
    Http.get(url)
        .then(function (result) {
            callback(null, result);
        });
};
Conversation.updateJieLong =function (uid,params,callback) {
    var url = '/apps/JieLong/'+uid+'/updateJieLong';
    Http.put(url, params)
        .then(function (result) {
            callback(null, result);
        });
};
Conversation.deleteJieLong =function (uid,params,callback) {
    var url = '/apps/JieLong/'+uid+'/deleteJieLong';
    Http.del(url, params)
        .then(function (result) {
            callback(null, result);
        });
};
RongIM.dataModel.Conversation = Conversation;
}(RongIM, {
    jQuery: jQuery
}));
