(function (RongIM, dependencies) {
'use strict';

var $ = dependencies.jQuery;
var RongIMClient = dependencies.RongIMClient;
var Cache = RongIM.dataModel._Cache;
var ObserverList = RongIM.dataModel._ObserverList;
var getLibErrorCode = RongIM.dataModel.util.getLibErrorCode;

var Status = {
    observerList: new ObserverList()
};

var statusObserver = Status.observerList;

/**
 * 连接 IM Token 失效的回调监听，包含主动调用 connect 及 C++ 断线重连时的状态码抛出
 * 单独监听是因 SDK 只在 connect 方法中可传参用于监听 TokenIncorret 状态的回调方法，
 * 但该回调方法会在连接状态变化时也被调用
 * 为了确保调用栈信息统一且不会出现非计划的回调，故业务层封装处理
 */
function onTokenIncorrectListeners() {
    // 派发变更
    Status.observerList.notify(31004);
}

Status.connect = function (result, token, deviceId, callback) {
    callback = callback || $.noop;
    if (deviceId) {
        var instance = RongIMClient.getInstance();
        instance.setDeviceInfo({ id: deviceId });
    }
    RongIM.system.appLogger('info',
        new Error('RongIMClient start to connect!').stack.replace('Error: ', ''));
    // 记录 IM Token 失效回调
    onTokenIncorrectListeners.whileconnect = callback;
    RongIMClient.connect(token, {
        onSuccess: function (userId) {
            // 清理本次调用上下文中的回调，避免重复回调
            delete onTokenIncorrectListeners.whileconnect;
            RongIM.system.appLogger('info', 'RongIMClient connect succeed!');
            callback(null, userId);
        },
        onTokenIncorrect: function () {
            // 正常 connect 调用时回调
            if (onTokenIncorrectListeners.whileconnect) {
                RongIM.system.appLogger('error', 'RongIMClient connect failed!\nerrorCode: invalid-token');
                onTokenIncorrectListeners.whileconnect('invalid-token');
                delete onTokenIncorrectListeners.whileconnect;
                return;
            }
            // C++ 重连时派发 status 变更
            onTokenIncorrectListeners('invalid-token');
        },
        onError: function (error) {
            // 清理本次调用上下文中的回调，避免重复回调
            delete onTokenIncorrectListeners.whileconnect;
            RongIM.system.appLogger('error', 'RongIMClient connect failed!\nerrorCode: ' + error);
            callback(error);
        }
    }, Cache.auth.id, result);
};

Status.reconnect = function (callback) {
    callback = callback || $.noop;
    // 日志记录调用栈
    RongIM.system.appLogger('info',
        new Error('RongIMClient start to reconnect!').stack.replace('Error: ', ''));
    RongIMClient.reconnect({
        onSuccess: function () {
            RongIM.system.appLogger('info', 'RongIMClient reconnect succeed!');
            // 重连成功
            callback(null);
        },
        onError: function (errorCode) {
            RongIM.system.appLogger('warn', 'RongIMClient reconnect failed!\nerrorCode: ' + errorCode);
            // 重连失败
            callback(getLibErrorCode(errorCode));
        }
    });
};

Status.disconnect = function () {
     
    RongIMClient.getInstance().logout();
    // 日志记录调用栈
    RongIM.system.appLogger('info',
        new Error('RongIMClient logout!').stack.replace('Error: ', ''));
};

Status.watch = function (listener) {
    statusObserver.add(listener);
};

Status.unwatch = function (listener) {
    statusObserver.remove(listener);
};

Status.getCurrentConnectionStatus = function () {
    var instance = RongIMClient.getInstance();
    return instance.getCurrentConnectionStatus();
};

Status.initRongIMClient = function (config, provider, cache) {
    RongIMClient.init(config.appkey, provider, config.sdk);
    RongIMClient.setConnectionStatusListener({
        onChanged: function (status) {
            if (status === 0) {
                cache.cleanMessageQueue();
            }
            RongIM.system.appLogger('info', 'RongIMClient conenction status changed: ' + status);
            Status.observerList.notify(status);
        }
    });
};

RongIM.dataModel.Status = Status;
}(RongIM, {
    jQuery: jQuery,
    RongIMClient: RongIMClient
}));
