(function (RongIM, components) {
'use strict';

var utils = RongIM.utils;
var common = RongIM.common;
var messageApi = RongIM.dataModel.Message;
var fileApi = RongIM.dataModel.File;

components.getQuoteFile = function (resolve, reject) {
    var im = RongIM.instance;
    var options = {
        name: 'quote-file-message',
        props: ['message'],
        template: '#rong-template-quote-file',
        data: function () {
            return {
                filename: '',
                basename: '',
                extname: '',
                extnameWidth: 0,
                downloadStatus: '',
                downloadProgress: 0
            };
        },
        mounted: function () {
            var context = this;
            var message = context.message;
            var downloadStatus = '';
            var url = message.content.content.localPath;
            if (!utils.isEmpty(url)) {
                var fileExists = getFileExists(url);
                if (fileExists) {
                    downloadStatus = 'DOWNLOADED';
                }
            }
            context.downloadStatus = downloadStatus;
            var fileState = RongIM.downloader.getProgress(message.messageUId);
            if (!$.isEmptyObject(fileState)) {
                context.downloadStatus = 'CANCELLED';
                context.downloadProgress = (fileState.offset / message.content.content.size) * 100;
                var downloader = fileApi.downloadManage.get(message.messageUId);
                if (downloader) {
                    context.downloadProgress = (downloader.stats.offset / message.content.size) * 100;
                    attachDownloadEvents(downloader, context);
                    context.downloader = downloader;
                    if (!downloader.isPause) {
                        context.downloadStatus = 'DOWNLOADING';
                    }
                }
            }

            var fileMessage = context.message.content;
            context.filename = fileMessage.content.name;

            var index = context.filename.lastIndexOf('.');
            var extname = '';
            if (index > -1) {
                // 扩展名前多显示2个字
                var prefix = 2;
                var chPatrn = /[\u4E00-\u9FA5]|[\uFE30-\uFFA0]|[\u0391-\uFFE5]/gi;
                extname = context.filename.slice(Math.max(0, index - prefix));
                extname = chPatrn.exec(extname) ? extname.substring(1) : extname;
            }
            context.extname = utils.encodeHtmlStr(extname);

            var end = 0 - context.extname.length;
            if (end === 0) {
                end = context.filename.length;
            }
            var basename = context.filename.slice(0, end);
            context.basename = utils.encodeHtmlStr(basename);

            context.extnameWidth = common.getTextRenderWidth(context.extname, 12);
        },
        computed: {
            support: function () {
                return im.config.support;
            }
        },
        methods: {
            download: function (message/* , isResume */) {
                if (this.downloadStatus === 'DOWNLOADING') {
                    return;
                }
                if (this.downloader) {
                    this.downloader.continue();
                    return;
                }
                var context = this;
                var content = message.content.content;

                var file = {
                    uId: message.messageUId,
                    url: content.fileUrl,
                    name: content.name,
                    size: content.size
                };
                var downloader = RongIM.downloader.load(file);
                attachDownloadEvents(downloader, context);
                downloader.continue();
            },
            openFile: function () {
                if (this.downloadStatus !== 'DOWNLOADED') {
                    return;
                }
                var localPath = this.message.content.content.localPath;
                if (!getFileExists(localPath)) {
                    this.download(this.message);
                    return;
                }
                RongIM.file.open(localPath);
            },
            isShowProgress: function () {
                var isDownloading = this.downloadStatus === 'DOWNLOADING';
                var isCanceled = this.downloadStatus === 'CANCELLED';
                return isDownloading || isCanceled;
            }
        }
    };
    utils.asyncComponent(options, resolve, reject);
};

function getFileExists(fileUrl) {
    var file = RongIM.file;
    var existed = false;
    if (!utils.isEmpty(fileUrl)) {
        existed = file.checkExist(fileUrl);
    }
    return existed;
}

function attachDownloadEvents(downloader, context) {
    // 下载中,返回下载进度
    var message = context.message;
    // 下载准备完毕, 此时可知文件大小
    downloader.onReady = function (data) {
        console.debug('full size:' + data.total);
        fileApi.downloadManage.add(message.messageUId, downloader);
        context.downloader = downloader;
    };

    downloader.onProgress = function (data) {
        context.downloadStatus = 'DOWNLOADING';
        context.downloadProgress = (data.loaded / data.total) * 100;
    };

    // 下载完成
    downloader.onComplete = function (data) {
        fileApi.downloadManage.remove(message.messageUId);
        context.downloader = null;
        context.downloadStatus = 'DOWNLOADED';
        if (!data.path) {
            // 被取消
            return;
        }
        message.content.content.localPath = data.path;
        messageApi.setContent(message);
        context.downloadProgress = 0;
    };

    downloader.onError = function (error) {
        fileApi.downloadManage.remove(message.messageUId);
        context.downloader = null;
        var errorCode = ['ebusy', 'eperm'].indexOf(error) > -1 ? error : 'error';
        common.toastError('download-' + errorCode);
        if (context.downloadProgress === 0) {
            context.downloadStatus = 'READY';
        } else {
            context.downloadStatus = 'CANCELLED';
        }
    };

    // 下载取消
    downloader.onCancel = function () {
        context.downloadStatus = 'READY';
    };
}
}(RongIM, RongIM.components));
