(function (RongIM) {
'use strict';

/**
 * 依赖 RongIM config server 地址 ajax 请求单独实现
 */

var Database = RongIM.Database;
var noop = function () {};
var SUCCESS_CODE = 10000;

function getFullURL(path) {
    return RongIM.config.dataModel.server + path;
}

function ajax(options) {
    options = $.extend({
        method: 'get',
        onanswer: noop,
        onsuccess: noop,
        onerror: noop
    }, options);

    var onerror = options.onerror;
    var url = getFullURL(options.url);
    var xhr = new XMLHttpRequest();
    xhr.onreadystatechange = function () {
        if (xhr.readyState === 2) {
            options.onanswer();
        }
        if (xhr.readyState === 4) {
            if (xhr.status === 200) {
                var response = xhr.response;
                try {
                    // console.time('[JSON.parse][' + options.url + ']');
                    response = JSON.parse(response);
                    // console.timeEnd('[JSON.parse][' + options.url + ']');
                } catch (e) {
                    onerror('response-body-parsed-error', e);
                    return;
                }
                if (response.code === SUCCESS_CODE) {
                    options.onsuccess(response.result);
                } else {
                    onerror(response.code || 'network-error', response.result);
                }
            } else {
                // onerror('not ok', xhr.status);
                // workaround: 忽略此错误
                options.onsuccess({
                    count: 0,
                    data: [],
                    timestamp: Date.now()
                });
            }
        }
    };
    xhr.ontimeout = function () {
        onerror('timeout');
    };
    xhr.open(options.method, url);
    if (options.method.toLowerCase() !== 'get') {
        xhr.setRequestHeader('Content-Type', 'application/json;charset=UTF-8');
    }
    var data = options.data;
    data = JSON.stringify(data);
    xhr.withCredentials = true;
    xhr.send(data);
}

function request(url, method, data, callback) {
    if (typeof data === 'function') {
        callback = data;
        data = null;
    }
    ajax({
        url: url,
        method: method,
        data: data,
        onsuccess: function (result) {
            callback(null, result);
        },
        onerror: function (error, msg) {
            callback(error, msg);
        }
    });
}

var sliceSize = 10000;
/**
 * 分批处理数据，仅返回成功或错误信息并不返回具体结果
 * this 被操作的数据库
 * @param {Function} handle 处理数据的函数
 * @param {Array} dataset 需要分批处理的数据
 * @param {Function} callback 处理完成回调
 */
function sliceHandle(handle, dataset, callback) {
    var db = this;
    if (handle.call == null) {
        handle = db[handle];
    }
    var arr = [];
    var length = dataset.length;
    while (length > 0) {
        arr.push(dataset.splice(0, sliceSize));
        length = dataset.length;
    }
    var promiseList = [];
    arr.forEach(function (item) {
        var defer = $.Deferred();
        handle.call(db, item, function (error, result) {
            if (error) {
                defer.reject(error);
            } else {
                defer.resolve(result);
            }
        });
        promiseList.push(defer.promise());
    });
    $.when.apply(null, promiseList).done(noop).fail(function (error) {
        console.warn(error);
    }).always(callback);
}

function forceSync(tag, dbPath, db, callback, onprogress) {
    onprogress = onprogress || noop;
    callback = callback || noop;
    var sys = RongIM.system;
    var im = RongIM.instance;

    db.getVersion(function (error, version) {
        if (error) {
            sys.appLogger('error', tag + '[获取]' + JSON.stringify(error));
            callback(error);
            return;
        }

        var startTime = new Date();
        console.time(tag + '[获取]');
        // version -= 3 * 24 * 3600 * 1000; // fake: 触发本地数据库更新
        ajax({
            url: dbPath + '/diff/' + version,
            onanswer: function () {
                onprogress(0.1);
            },
            onsuccess: function (result) {
                var endTime = new Date();
                sys.appLogger('info', tag + '[获取]' + (endTime - startTime) + 'ms');
                console.timeEnd(tag + '[获取]');

                var dataset = result.data || [];
                if (dataset.length === 0) {
                    onprogress(1);
                    callback(null);
                    return;
                }

                var insertList = dataset.filter(function (item) {
                    return (item.state || item.member_state || item.user_state) != 2;
                });
                console.info(tag + '[已获取 ' + insertList.length + ' 条]');
                if (im.syncdataLoad.state == 'loading' && insertList.length > 100) {
                    im.syncdataLoad.show = true;
                }

                function removeMethod(cb) {
                    if (version == 0 && db.removeAll) {
                        db.removeAll(cb);
                    } else if (db.removeBatch) {
                        var removeList = [];
                        dataset.forEach(function (item) {
                            if ((item.state || item.member_state || item.user_state) == 2) {
                                removeList.push(item.id);
                            }
                        });
                        if (removeList.length > 0) {
                            sliceHandle.call(db, db.removeBatch, removeList, cb);
                        } else {
                            cb();
                        }
                    } else if (db.remove) {
                        sliceHandle.call(db, db.remove, dataset, cb);
                    } else {
                        cb();
                    }
                };

                onprogress(0.6);
                var startTime1 = new Date();
                console.time(tag + '[清理]');
                removeMethod(function () {
                    console.timeEnd(tag + '[清理]');
                    console.time(tag + '[插入]');
                    sliceHandle.call(db, db.insertOrReplace || db.insert, insertList, function (err) {
                        console.timeEnd(tag + '[插入]');
                        var endTime1 = new Date();
                        sys.appLogger('info', tag + '[插入]' + (endTime1 - startTime1) + 'ms');
                        if (err) {
                            sys.appLogger('error', tag + '[插入]' + JSON.stringify(err));
                            callback(err);
                            return;
                        }
                        db.updateVersion(result.timestamp);
                        onprogress(1);
                        callback(null);
                    });
                });
            },
            onerror: function (errorCode) {
                sys.appLogger('error', tag + '[获取]' + JSON.stringify(errorCode));
                callback(errorCode);
            }
        });
    });
}

function syncCompany(callback, onprogress) {
    forceSync('[公司信息]', '/companies', Database.company, callback, onprogress);
}

function syncDepartment(callback, onprogress) {
    forceSync('[部门信息]', '/departments', Database.department, callback, onprogress);
}

function syncOrganization(callback, onprogress) {
    forceSync('[组织机构关系]', '/organization', Database.organization, callback, onprogress);
}

function syncStaff(callback, onprogress) {
    forceSync('[员工信息]', '/staffs', Database.staff, callback, onprogress);
}

function syncStaffById(id, callback) {
    callback = callback || noop;
    request('/staffs/' + id, 'get', function (error, result) {
        if (error) {
            callback(error, result);
            return;
        }
        Database.staff.update([result], function (err) {
            if (err) {
                callback(err);
                return;
            }
            callback(null, result);
        });
    });
}

function syncUserById(idList, callback) {
    callback = callback || noop;
    request('/staffs/batch', 'post', {
        ids: idList
    }, function (error, result) {
        if (error) {
            callback(error, result);
            return;
        }
        var userList = result.data;
        // 员工不更新 state 状态，通过 diff 接口同步
        userList.forEach(function (item) {
            var isStaff = item.user_type === 0;
            if (isStaff) {
                delete item.state;
            }
        });
        Database.staff.update(userList, function (err) {
            if (err) {
                callback(err);
                return;
            }
            callback(null, userList);
        });
    });
}

function syncOfficialaccount(callback, onprogress) {
    onprogress = onprogress || noop;
    callback = callback || noop;

    var db = Database.officialAccount;
    var url = '/apps/subscriptions/apps?menu=1&update_dt=';
    db.getVersion(function (error, version) {
        if (error) {
            callback(error);
            return;
        }
        ajax({
            url: url + version,
            onanswer: function () {
                onprogress(0.1);
            },
            onsuccess: function (result) {
                onprogress(0.6);
                var newVersion = result.update_dt;
                var dataset = result.apps || [];
                db.insertOrReplace(dataset, function (err) {
                    if (err) {
                        callback(err);
                        return;
                    }
                    db.updateVersion(newVersion);
                    onprogress(1);
                    callback(null);
                });
            },
            onerror: function (errorCode) {
                // 无数据更新时返回 11714
                if (errorCode === 11714) {
                    callback(null);
                    return;
                }
                callback(errorCode);
            }
        });
    });
}

function syncAllOrgs(callback) {
    // fake: disable sync data
    callback(null);
    // syncSerialHandle([syncCompany, syncDepartment, syncOrganization], callback, $.noop);
}

/**
 * 并行同步
 * @param {Array} syncHandleList 
 * @param {Function} callback 
 * @param {Function} onchange 
 */
function syncParallelHandle(syncHandleList, callback, onchange) {
    var promiseList = [];
    syncHandleList.forEach(function (handle, index) {
        var defer = $.Deferred();
        handle(function (error) {
            if (error) {
                defer.reject(error);
                return;
            }
            defer.resolve();
        }, function (p) {
            onchange(index, p);
        });
        promiseList.push(defer.promise());
    });
    $.when.apply(null, promiseList).then(function () {
        callback(null);
    }, function (error) {
        callback(error);
    });
}

/**
 * 串行同步
 * @param {Array} syncHandleList 
 * @param {Function} callback 
 * @param {Function} onchange 
 */
function syncSerialHandle(syncHandleList, callback, onchange, index) {
    if (syncHandleList.length === 0) {
        callback(null);
        return;
    }
    index = index || 0;
    var handle = syncHandleList.shift();
    handle(function (error) {
        if (error) {
            callback(error);
            return;
        }
        syncSerialHandle(syncHandleList, callback, onchange, index + 1);
    }, function (p) {
        onchange(index, p);
    });
}

/**
 * 数据同步
 * @param {boolean} isStaff 是否为内部联系人
 * @param {function} callback 同步成功 or 错误回调
 * @param {function} onprogress 同步进度回调
 */
function syncAll(isStaff, callback, onprogress) {
    onprogress = onprogress || noop;
    callback = callback || noop;

    // fake: disable sync data
    onprogress(1);
    callback(null);
    return;

    if (!isStaff) {
        onprogress(1);
        callback(null);
        return;
    }
    var syncHandleList = [
        syncCompany,
        syncDepartment,
        syncOrganization,
        syncStaff
    ];
    var progressList = new Array(syncHandleList.length);
    var onchange = function (index, progress) {
        progressList[index] = progress;
        var total = progressList.reduce(function (a, b) {
            return a + b;
        });
        onprogress(total / progressList.length);
    };
    syncSerialHandle(syncHandleList, callback, onchange);
}

function syncGroupById(groupIdList, callback) {
    callback = callback || noop;
    ajax({
        url: '/groups/batch',
        method: 'post',
        data: {
            ids: groupIdList
        },
        onsuccess: function (groupList) {
            Database.group.insertOrReplace(groupList, function (error) {
                if (error) {
                    console.warn(error);
                }
                callback(null, groupList);
            });
        },
        onerror: function (error) {
            callback(error);
        }
    });
}

RongIM.syncdata = {
    all: syncAll,
    allOrgs: syncAllOrgs,
    company: syncCompany,
    department: syncDepartment,
    organization: syncOrganization,
    staff: syncStaff,
    staffById: syncStaffById,
    userById: syncUserById,
    groupById: syncGroupById,
    officialAccount: syncOfficialaccount
};

RongIM.ajax = ajax;
}(RongIM));
