(function (RongIM, dependencies, components) {
'use strict';

var $ = dependencies.jQuery;
var Vue = dependencies.Vue;

/*
表单验证
说明：
    对form中表单元素进行验证，可自定义规则
    初始化执行init方法，对input, textarea, select标签进行验证
    绑定失去焦点、键盘等事件，当事件触发时进行表单验证
*/

var methods = {
    required: function (value/* , elem, param */) {
        // 不能为空
        return value.length > 0;
    },
    mobile: function (value, elem/* , param */) {
        // 手机号验证
        return optional(elem) || /^1\d{10}$/.test(value);
    },
    equalto: function (value, elem, param) {
        // 和表单中的另一个元素的value相同
        return optional(elem) || value === $(elem).closest('form').find(param).val();
    },
    mix: function (value, elem, param) {
        // 最小值
        return optional(elem) || value >= param;
    },
    max: function (value, elem, param) {
        // 最大值
        return optional(elem) || value <= param;
    },
    minlength: function (value, elem, param) {
        // 最小长度
        var length = $.isArray(value) ? value.length : getLength(value, elem);
        return optional(elem) || length >= param;
    },
    maxlength: function (value, elem, param) {
        // 最大长度
        var length = $.isArray(value) ? value.length : getLength(value, elem);
        return optional(elem) || length <= param;
    },
    range: function (value, elem, param) {
        // 范围
        var range = getRange(param);
        if (!range) {
            return true;
        }
        return optional(elem) || (value >= range[0] && value <= range[1]);
    },
    rangelength: function (value, elem, param) {
        // 范围长度
        var range = getRange(param);
        if (!range) {
            return true;
        }
        var length = getLength(value, elem);
        return optional(elem) || (length >= range[0] && length <= range[1]);
    },
    specialsymbol: function (value, elem, param) {
        // 特殊字符验证
        var specialsymbol = param.split(',');
        var hasSpecialSymbol = false;
        specialsymbol.forEach(function (symbol) {
            if (value.indexOf(symbol) !== -1) {
                hasSpecialSymbol = true;
            }
        });
        return optional(elem) || !hasSpecialSymbol;
    },
    password: function (value) {
        // 密码验证
        var num = /[0-9]+/;
        var letter = /[a-zA-Z]+/;
        var containNum = num.test(value);
        var containLetter = letter.test(value);
        // var invalidSign = /[^0-9a-zA-Z~`@#$%^&*\-_=+\|\\?/()<>\[\]{}",.;'!]+/;
        // var validSign = !invalidSign.test(value);
        // return containNum && containLetter && validSign;
        return containNum && containLetter;
    }
};

function getValidate(options) {
    options = options || {};
    $.extend(methods, options.methods);
    return {
        name: 'validate',
        data: function () {
            return {
                errors: {}
            };
        },
        computed: {
            isValid: function () {
                var errors = JSON.parse(JSON.stringify(this.errors));
                return $.isEmptyObject(errors);
            }
        },
        mounted: init,
        methods: {
            valid: function (selector) {
                var self = this;
                selector = selector || 'input, textarea, select';
                var $fieldList = $(self.$el).find(selector).trigger('check');
                var errors = {};
                $fieldList.each(function () {
                    var name = $(this).attr('name');
                    if (self.errors[name]) {
                        errors[name] = self.errors[name];
                    }
                });
                return $.isEmptyObject(errors);
            }
        }
    };
}

function getLength(value, elem) {
    switch (elem.nodeName.toLowerCase()) {
        case 'select':
            return $('option:selected', elem).length;
        case 'input':
            if (checkable(elem)) {
                var $field = $(elem).closest('form').find('[name="' + elem.name + '"]');
                return $field.filter(':checked').length;
            }
            break;
        default:
    }
    return value.length;
}

function checkable(elem) {
    return (/radio|checkbox/i).test(elem.type);
}

function getRange(param) {
    var range = param.match(/(\d+)\s*,\s*(\d+)/);
    if (range) {
        return [Number(range[1]), Number(range[2])];
    }
    return range;
}

function optional(elem) {
    return !$(elem).val();
}

function init() {
    var context = this;
    var selector = 'input, textarea, select';
    $(context.$el).on('blur check', selector, function (event) {
        validateField(context, event);
    }).on('focus keyup', selector, function (event) {
        var isTab = (event.key || '').toLowerCase() === 'tab' || event.keyCode === 9;
        var isEnter = (event.key || '').toLowerCase() === 'enter' || event.keyCode === 13;
        if (isTab || isEnter) {
            return;
        }
        Vue.delete(context.errors, event.target.name);
    });
}

function validateField(context, event) {
    var field = event.target;
    var tabKey = (event.key || '').toLowerCase() === 'tab';
    var tabKeyCode = event.keyCode === 9;
    var isTab = tabKey || tabKeyCode;
    var enterKey = (event.key || '').toLowerCase() === 'enter';
    var enterKeyCode = event.keyCode === 13;
    var isEnter = enterKey || enterKeyCode;
    if (isTab || isEnter) {
        return;
    }
    Vue.delete(context.errors, field.name);
    $.each(getRules(field), function (key, value) {
        var valid = methods[key]($(field).val(), field, value);
        if (!valid) {
            var message = valid ? null : $(field).data('message-' + key);
            Vue.set(context.errors, field.name, message);
        }
        return valid;
    });
}

function getRules(field) {
    var rules = {};
    $.each($(field).data(), function (key, value) {
        if (key.slice(0, 4) === 'rule') {
            var name = key.slice(4).toLowerCase();
            rules[name] = value;
        }
    });
    return rules;
}

components.getValidate = getValidate;
}(RongIM, {
    jQuery: jQuery,
    Vue: Vue
}, RongIM.components));
