export interface StackFrame {
    url: string;
    func: string;
    args: string[];
    line: number;
    column: number;
    context: string[];
}
export interface StackTrace {
    /**
     * Known modes: callers, failed, multiline, onerror, stack, stacktrace
     */
    mode: string;
    mechanism: string;
    name: string;
    message: string;
    url: string;
    stack: StackFrame[];
    useragent: string;
    original?: string;
}
interface ComputeStackTrace {
    /**
     * Computes a stack trace for an exception.
     * @param {Error} ex
     * @param {(string|number)=} depth
     */
    (ex: Error, depth?: string | number): StackTrace;
    /**
     * Adds information about the first frame to incomplete stack traces.
     * Safari and IE require this to get complete data on the first frame.
     * @param {Object.<string, *>} stackInfo Stack trace information from
     * one of the compute* methods.
     * @param {string} url The URL of the script that caused an error.
     * @param {(number|string)} lineNo The line number of the script that
     * caused an error.
     * @param {string=} message The error generated by the browser, which
     * hopefully contains the name of the object that caused the error.
     * @return {boolean} Whether or not the stack information was
     * augmented.
     */
    augmentStackTraceWithInitialElement: (stackInfo: string, url: string, lineNo: string | number, message?: string) => boolean;
    /**
     * Tries to use an externally loaded copy of source code to determine
     * the name of a function by looking at the name of the variable it was
     * assigned to, if any.
     * @param {string} url URL of source code.
     * @param {(string|number)} lineNo Line number in source code.
     * @return {string} The function name, if discoverable.
     */
    guessFunctionName: (url: string, lineNo: string | number) => string;
    /**
     * Retrieves the surrounding lines from where an exception occurred.
     * @param {string} url URL of source code.
     * @param {(string|number)} line Line number in source code to centre
     * around for context.
     * @return {?Array.<string>} Lines of source code.
     */
    gatherContext: (url: string, line: string | number) => string[];
    /**
     * Logs a stacktrace starting from the previous call and working down.
     * @param {(number|string)=} depth How many frames deep to trace.
     * @return {Object.<string, *>} Stack trace information.
     */
    ofCaller: (depth?: string | number) => StackTrace;
    /**
     * Retrieves source code from the source code cache.
     * @param {string} url URL of source code.
     * @return {Array.<string>} Source contents.
     */
    getSource: (url: string) => string[];
}
interface ReportSubscriber {
    (stackTrace: StackTrace, isWindowError: boolean, error: any): void;
}
interface Report {
    /**
     * Reports an unhandled Error to TraceKit.
     * @param {Error} ex
     */
    (ex: Error): void;
    /**
     * Add a crash handler.
     * @param {Function} handler
     */
    subscribe(handler: ReportSubscriber): void;
    /**
     * Remove a crash handler.
     * @param {Function} handler
     */
    unsubscribe(handler: ReportSubscriber): void;
}
declare const report: Report;
declare const subscribe: any;
declare const installGlobalHandler: any;
declare const installGlobalUnhandledRejectionHandler: any;
declare const computeStackTrace: ComputeStackTrace;
export { report, subscribe, installGlobalHandler, installGlobalUnhandledRejectionHandler, computeStackTrace };
