import { addGlobalEventProcessor, getCurrentHub } from '@sentry/hub/esm';
import { isError, isString } from '@sentry/utils/esm/is';
import { logger } from '@sentry/utils/esm/logger';
import { safeNormalize } from '@sentry/utils/esm/object';
/** Patch toString calls to return proper name for wrapped functions */
export class ExtraErrorData {
    /**
     * @inheritDoc
     */
    constructor(options = { depth: 3 }) {
        this.options = options;
        /**
         * @inheritDoc
         */
        this.name = ExtraErrorData.id;
    }
    /**
     * @inheritDoc
     */
    setupOnce() {
        addGlobalEventProcessor(async (event, hint) => {
            const self = getCurrentHub().getIntegration(ExtraErrorData);
            if (!self) {
                return event;
            }
            return self.enhanceEventWithErrorData(event, hint);
        });
    }
    /**
     * Attaches extracted information from the Error object to extra field in the SentryEvent
     */
    enhanceEventWithErrorData(event, hint) {
        if (!hint || !hint.originalException || !isError(hint.originalException)) {
            return event;
        }
        const errorData = this.extractErrorData(hint.originalException);
        if (errorData) {
            let extra = {
                ...event.extra,
            };
            const normalizedErrorData = safeNormalize(errorData, this.options.depth);
            if (!isString(normalizedErrorData)) {
                extra = {
                    ...event.extra,
                    ...normalizedErrorData,
                };
            }
            return {
                ...event,
                extra,
            };
        }
        return event;
    }
    /**
     * Extract extra information from the Error object
     */
    extractErrorData(error) {
        let result = null;
        // We are trying to enhance already existing event, so no harm done if it won't succeed
        try {
            const nativeKeys = ['name', 'message', 'stack', 'line', 'column', 'fileName', 'lineNumber', 'columnNumber'];
            const name = error.name || error.constructor.name;
            const errorKeys = Object.getOwnPropertyNames(error).filter(key => nativeKeys.indexOf(key) === -1);
            if (errorKeys.length) {
                const extraErrorInfo = {};
                for (const key of errorKeys) {
                    let value = error[key];
                    if (isError(value)) {
                        value = value.toString();
                    }
                    extraErrorInfo[key] = value;
                }
                result = {
                    [name]: extraErrorInfo,
                };
            }
        }
        catch (oO) {
            logger.error('Unable to extract extra data from the Error object:', oO);
        }
        return result;
    }
}
/**
 * @inheritDoc
 */
ExtraErrorData.id = 'ExtraErrorData';
//# sourceMappingURL=extraerrordata.js.map