const electron = require('electron');

const {
    app,
    BrowserWindow,
    ipcMain,
} = electron;
// const path = require('path')
// const fs = require('fs')
const i18n = require('i18n');
const Config = require('./config');
const Utils = require('./utils');
const startReport = require('./common/reporter');

const { platform } = Utils;
const AppConfig = require('./configuration');
const createLocalServer = require('./modules/proxy');
// const AutoLaunch = require('auto-launch');

const { processLogger, appLogger } = require('./common/logger');
const dialog = require('./common/dialog');

const RCEWindow = require('./windows/rce');
const AppTray = require('./windows/app_tray');
const ImageViewerModule = require('./modules/image_viewer/image_viewer.main');

// 远程控制相关. 关闭sunloginsdk进程, before-quit时使用
let stopRemoteClient;

const initSize = {
    width: Config.WIN.INIT_WIDTH,
    height: Config.WIN.INIT_HEIGHT,
};

const argvs = process.argv.slice(2);
[global.processENV] = argvs;

let rceWindow = null;
let tray = null;

let workAreaSize = null;
const winState = [];

const loadWindowBounds = () => {
    const x = AppConfig.readSettings('x');
    const y = AppConfig.readSettings('y');
    const width = AppConfig.readSettings('width');
    const height = AppConfig.readSettings('height');
    if (!x || !y || !width || !height) {
        return null;
    }
    return {
        x,
        y,
        width,
        height,
    };
};

const checkScreenInner = (position) => {
    if (!position) {
        return false;
    }
    const { screen } = electron;
    const screenList = screen.getAllDisplays();
    const { x } = position;
    const { y } = position;
    for (let i = 0, { length } = screenList; i < length; i += 1) {
        const { bounds } = screenList[i];
        const xInner = x > bounds.x && x < bounds.x + bounds.width;
        const yInner = y > bounds.y && y < bounds.y + bounds.height;
        if (xInner && yInner) {
            return true;
        }
    }
    return false;
};

// Load window bounds info from setting file. create an empty file when not exist
const saveWindowBounds = (bounds) => {
    AppConfig.saveSettings('x', bounds.x);
    AppConfig.saveSettings('y', bounds.y);
    AppConfig.saveSettings('width', bounds.width);
    AppConfig.saveSettings('height', bounds.height);
};

const getWindowBounds = () => {
    const lastWindowBounds = loadWindowBounds();
    let bounds = {
        x: (workAreaSize.width - initSize.width) / 2,
        y: (workAreaSize.height - initSize.height) / 2,
        width: initSize.width,
        height: initSize.height,
    };
    const useLastBounds = !Config.ALWAYS_SHOW_IN_PRIMARY &&
        lastWindowBounds !== null &&
        checkScreenInner(lastWindowBounds);
    if (useLastBounds) {
        bounds = lastWindowBounds;
    }
    saveWindowBounds(bounds);
    return bounds;
};

const intScreen = () => {
    const {
        screen,
    } = electron;
    workAreaSize = screen.getPrimaryDisplay().workAreaSize;
    screen.on('display-metrics-changed', (event, display) => {
        workAreaSize = display.workAreaSize;
    });
};
const createRCEWindow = () => {
    const winBounds = getWindowBounds();
    rceWindow = new RCEWindow(winBounds, winBounds);
    BrowserWindow.mainWindow = rceWindow;
};
const createTray = () => {
    tray = new AppTray(rceWindow, global.locale);
    BrowserWindow.tray = tray;
};
const initIpcEvents = () => {
    require('./modules/system/system.main');
    require('./modules/window/window.main');
    ipcMain.on('notification-click', () => {
        if (rceWindow) {
            rceWindow.show();
        }
    });
    ipcMain.on('reload', () => {
        if (rceWindow) {
            rceWindow.reload();
        }
    });
    ipcMain.on('open-settings', () => {
        if (rceWindow) {
            rceWindow.show();
            rceWindow.setting();
        }
    });
    // Focus on search input element.
    ipcMain.on('search', () => {
        if (rceWindow) {
            rceWindow.search();
        }
    });
    // ipcMain.on('bring-front', (event, isFront) => {
    ipcMain.on('bring-front', () => {
        if (rceWindow) {
            // rceWindow.setAlwaysOnTop(isFront);
            rceWindow.focus();
            rceWindow.show();
        }
    });
    ipcMain.on('set-locale', (event, lang) => {
        i18n.setLocale(lang);
    });
    ipcMain.on('openConversation', (event, params) => {
        BrowserWindow.mainWindow.sendCommand('openConversation', params);
    });
    ipcMain.on('start_main', () => {
        rceWindow.start();
    });
};

const appReady = () => {
    // qt会影响 menu 的显示,所以需放在前面初始化
    // 如放在后面初始化,页面加载完成后需要重新初始化菜单
    if (Config.SUPPORT.SCREENSHOT) {
        require('./modules/screenshot/screenshot.main.js');
    }
    if (Config.SUPPORT.RemoteControl && Utils.platform.darwin) {
        const RemoteControl = require('./modules/remote_control/darwin/remote_control.main.js');
        stopRemoteClient = RemoteControl.stopRemoteClient;
    }

    require('./modules/browser_window_new/browser_window.main.js');
    require('./modules/browser_window/browser.main');

    intScreen();
    createRCEWindow();
    // 图片查看器初始化
    ImageViewerModule.init();
    createTray();
    initIpcEvents();

    const voipMain = require('./modules/voip/voip.main');
    if (typeof voipMain !== 'undefined' && voipMain) {
        voipMain.init(rceWindow);
    }
};

const initApp = () => {
    app.commandLine.appendSwitch('autoplay-policy', 'no-user-gesture-required');
    if (Config.IGNORE_CERTIFICATE_ERRORS) {
        app.commandLine.appendSwitch('ignore-certificate-errors');
    }
    if (Config.NO_PROXY) {
        app.commandLine.appendSwitch('no-proxy-server');
    }
    // 禁用本地缓存
    // app.commandLine.appendSwitch("--disable-http-cache");
    require('./modules/crash_report/crashreport.main.js');
    if (platform.win32) {
        app.setAppUserModelId(Config.APP_ID);
    }
    app.on('ready', () => {
        appReady();
        app.on('activate', () => {
            if (!rceWindow.isVisible()) {
                rceWindow.show();
                winState.length = 0;
                return;
            }
            let isWinHide = false;
            if (winState[0] === 'focus' && winState[1] === 'blur') {
                isWinHide = true;
                winState.unshift('focus');
                winState.length = 2;
            }
            if (isWinHide) {
                rceWindow.show();
            } else {
                rceWindow.dockClick();
            }
        });
    });
    app.on('before-quit', () => {
        // 远程控制相关, 关闭 sunloginsdk 进程, 避免遗留
        if (stopRemoteClient) stopRemoteClient();
        if (rceWindow) {
            rceWindow.enableForceQuit();
        }
        tray.reset();
    });
    app.on('window-all-closed', () => {
        app.quit();
    });
    // app.on('menu.view.bringFront', (checked) => {
    app.on('menu.view.bringFront', () => {
        if (rceWindow) {
            // rceWindow.setAlwaysOnTop(checked);
            rceWindow.focus();
            rceWindow.show();
        }
    });
    app.on('browser-window-blur', () => {
        winState.unshift('blur');
    });

    app.on('browser-window-focus', () => {
        winState.unshift('focus');
    });
};

(async() => {
    const isPrimary = app.requestSingleInstanceLock();
    if (!isPrimary) {
        app.exit();
        return;
    }
    appLogger.info('------------------ 应用已启动 ------------------');
    appLogger.info({
        electronVersion: process.versions.electron,
        appVersion: Config.APP_VERSION,
        versionCode: Config.APP_VERSION_CODE,
    });

    // 错误上报
    startReport('main');

    app.on('second-instance', ( /* event, argv, workingDirectory */ ) => {
        if (rceWindow) {
            rceWindow.show();
        }
    });
    global.locale = {};
    i18n.configure({
        locales: ['en', 'zh-CN'],
        directory: `${__dirname}/locales`,
        objectNotation: true,
        register: global.locale,
    });
    const userData = app.getPath('userData');
    let port = parseInt(Config.DEFAULT_PORT, 10);
    try {
        appLogger.info('开始启动代理服务！');
        // 启动 Web 代理服务
        port = await createLocalServer(port, userData);
    } catch (error) {
        appLogger.error(`启动失败，应用退出 => ${error.stack}`);
        app.exit();
        return;
    }
    appLogger.info(`代理服务已启动，监听端口：${port}`);
    Config.DEFAULT_PORT = port;
    initApp();
})();

process.on('error', (err) => {
    processLogger.error(err.toString());
});

process.on('uncaughtException', (error) => {
    processLogger.error(error);
    if (rceWindow) {
        rceWindow.sendCommand('logError', error.stack);
    } else {
        dialog.showError(error);
    }
});