
/*
窗口级独立模块, 不依赖任何逻辑
Electron BrowserWindow 文档: https://electronjs.org/docs/api/browser-window
 */
const {
    ipcMain,
    BrowserWindow,
    screen,
} = require('electron');
const path = require('path');
const Utils = require('../../utils.js');

let configInfo;
try {
    configInfo = require('../../config.js');
} catch (err) {
    configInfo = null;
}

/**
 * this.window
 */
class Window {
    /**
     * 初始化
     * @param  {object} option 设置项
     *                  option.id      选填, 作为该窗口的标识, 不填则以 path 为标识
     *                  option.path    必填, 窗口所展示的 url
     *                  option.xxx     选填, 详见 createWindow
     */
    constructor(option) {
        const { path: url } = option;
        this.option = option;
        this.createWindow(option);
        this.loadURL(url);
        return this.window;
    }

    /*
     * option.backgroundColor  选填, 窗口的背景颜色
     * option.width   选填, 默认项见 createWindow 方法的 defaultOpt
     * ...            选填, 详见 createWindow 方法的 defaultOpt
     */
    createWindow(option) {
        const defaultOpt = {
            width: 1000,
            height: 640,
            minWidth: 890,
            minHeight: 640,
            titleBarStyle: 'hidden',
            show: false,
            frame: false,
            webPreferences: {
                preload: path.join(__dirname, 'preload.js'),
                nodeIntegration: false,
                allowDisplayingInsecureContent: true,
                webSecurity: false,
                plugins: true,
                webviewTag: true,
            },
        };
        const position = this.getPosition(option);
        option = Utils.extend(option, position);
        option = Utils.extend(defaultOpt, option);
        const newWindow = new BrowserWindow(option);
        // newWindow.maximize();
        const isShow = option.show;
        // 避免显示窗口时有视觉闪烁
        if (!isShow) {
            newWindow.once('ready-to-show', () => {
                newWindow.show();
            });
        }
        this.window = newWindow;
        // this.window.webContents.openDevTools();
    }

    loadURL(url) {
        const { option } = this;
        const appHost = option.isLocal ? configInfo.getAppHost() : '';
        url = appHost + url;
        this.window.loadURL(url);
    }

    getPosition(option) {
        const screenSize = screen.getPrimaryDisplay()
            .size;
        let x = (screenSize.width - (option.width || 890)) / 2;
        let y = (screenSize.height - (option.height || 640)) / 2;
        if (option.bottom) {
            y = screenSize.height - option.bottom;
        }
        if (option.right) {
            x = screenSize.width - option.right;
        }
        const left = option.x || option.left;
        if (left) {
            x = left;
        }
        const top = option.y || option.top;
        if (top) {
            y = top;
        }
        return {
            x,
            y,
        };
    }

    // webContents 事件, 例: dom-ready
    setWindowWebContents() {

    }

    // 设置 window 事件, 例: window.on('focus')
    setWindowEvents() {

    }
}

const WindowHandler = {
    saveId: 'OpenedBrowserWindow',
    get: (id) => {
        const OpenedWindow = BrowserWindow[WindowHandler.saveId] || {};
        return OpenedWindow[id];
    },
    getAll: () => BrowserWindow[WindowHandler.saveId],
    remove: (id) => {
        const windows = BrowserWindow[WindowHandler.saveId];
        if (windows && windows[id]) {
            delete BrowserWindow[WindowHandler.saveId][id];
        }
    },
    save(id, browserWin) {
        const { saveId } = WindowHandler;
        if (!BrowserWindow[saveId]) {
            BrowserWindow[saveId] = {};
        }
        BrowserWindow[saveId][id] = browserWin;
    },
};

/*
创建窗口, option 见 Window 的 constructor
 */
const create = (option) => {
    const id = option.id || option.path;
    // 如果该弹框已存在, 直接 focus
    const browserWin = WindowHandler.get(id);
    if (browserWin) {
        browserWin.focus();
        return;
    }
    let newWindow = new Window(option);
    newWindow.on('closed', () => {
        newWindow = null;
        WindowHandler.remove(id);
    });
    WindowHandler.save(id, newWindow);
};

const closeAll = () => {
    const openedWindow = WindowHandler.getAll() || {};
    Object.keys(openedWindow).forEach((id) => {
        const browserWin = WindowHandler.get(id);
        browserWin.close();
        WindowHandler.remove(id);
    });
};

const close = (id) => {
    if (id) {
        const browserWin = WindowHandler.get(id);
        if (browserWin) {
            browserWin.close();
        }
        WindowHandler.remove(id);
    } else {
        closeAll();
    }
};

const sendMessage = (id, key, ...args) => {
    if (id) {
        WindowHandler.get(id).webContents.send(key, ...args);
        return;
    }
    const openedWindow = WindowHandler.getAll() || {};
    Object.keys(openedWindow).forEach((winId) => {
        const browserWin = WindowHandler.get(winId);
        browserWin.webContents.send(key, ...args);
    });
    // 为匹配现有代码而加入mainWindow. 之后将删除
    const mainWindow = BrowserWindow.mainWindow.rceWindow;
    mainWindow.webContents.send(key, ...args);
};

const setIpcMain = () => {
    const mark = 'browser_window';
    const createMark = `${mark}_create`;
    const closeMark = `${mark}_close`;
    const messageMark = `${mark}_message`;
    // 监听 browser_window.render.js 中的 ipcRenderer.send, 使该方法可以用于渲染进程
    ipcMain.on(createMark, (event, option) => {
        create(option);
    });
    ipcMain.on(closeMark, (event, id) => {
        close(id);
    });

    // 监听窗口间的通讯
    ipcMain.on(messageMark, (event, opt, ...args) => {
        let key = opt;
        let windowId;
        if (Object.prototype.toString.call(opt) === '[object Object]') {
            key = opt.key;
            windowId = opt.windowId;
        }
        sendMessage(windowId, key, ...args);
    });
};

setIpcMain();

module.exports = {
    create,
    close,
};
