const path = require('path');
const fs = require('fs');
const mime = require('mime');
const { promisify } = require('util');
const { appLogger } = require('../../../common/logger');

const readFile = promisify(fs.readFile);
const exists = promisify(fs.exists);

const matchReg = /<!--#include virtual=[^>]+ -->/g;
const clearReg = /(<!--#include virtual=['"])|(['"]\s*-->)/g;

function generate(baseUrl) {
    return async function SSITransform(source) {
        let content = source;
        const arr = content.match(matchReg);
        if (!arr || arr.length === 0) {
            return content;
        }
        // 替换所有 SSI 内容
        await Promise.all(arr.map(item => new Promise(async (resolve) => {
            // 获取路径
            const resUrl = item.replace(clearReg, '');
            const resPath = path.join(baseUrl, resUrl);
            // 读取文件，无法正确读取的文件做替换标记处理
            let fileData;
            try {
                fileData = await readFile(resPath);
                fileData = fileData.toString();
            } catch (err) {
                fileData = item.replace('#include', '#miss');
                appLogger.error(err);
            }
            // 对读取到文件内容再次进行 SSI 转化，以当前文件父目录为基础目录
            const str = await generate(path.join(resPath, '../'))(fileData);
            // 替换文件内容
            content = content.replace(item, str);
            resolve();
        })));
        return Promise.resolve(content);
    };
}

/**
 * express ssi 功能中间件
 */
module.exports = root => async (req, res, next) => {
    // 非指定文件跳过检测
    if (!/\.s?html$/.test(req.path)) {
        next();
        return;
    }
    // 资源文件路径
    const filePath = path.join(root, req.path);
    const bool = await exists(filePath);
    if (!bool) {
        next();
        return;
    }
    let fileData;
    try {
        fileData = await readFile(filePath);
    } catch (err) {
        res.status(404).send(err);
        return;
    }
    const SSITransform = generate(path.join(filePath, '../'));
    const tmpData = await SSITransform(fileData.toString());
    res.set('Content-Type', mime.types[req.path.split('.').pop()]);
    res.send(tmpData);
};
