(function (RongIM, dependencies, components) {
'use strict';

var common = RongIM.common;
var utils = RongIM.utils;
var $ = dependencies.jQuery;
var UploadClient = dependencies.UploadClient;
var counter = 0;

/*
说明： 转发消息。
       可从最近联系人（会话列表）、组织机构、好友、我的群组中选择。
       客户端每秒最多发送 5 条需要限制发送频率最大转发条数。
*/
RongIM.dialog.forward = function (message) {
    var im = RongIM.instance;
    var dataModel = im.dataModel;

    // 模块化屏蔽入口
    // 好友
    var modules = im.config.modules;
    var enabledFriend = modules.friend;

    var options = {
        name: 'forward',
        template: 'templates/conversation/forward.html',
        data: function () {
            return {
                maxCount: 10,
                show: true,
                enabledFriend: enabledFriend,
                groupName: '',
                tip: '',
                // 'recent' or 'star' or 'org'
                tab: 'recent',
                defaultSelected: [],
                busy: false,
                selected: [], /* 已选择的会话（包括高管） */
                isStaff: im.auth.isStaff,
                disableExecutive: false,
                tipState: 'error',
                // 转发二维码图片上传后的下载链接
                downloadUrl: '',
                thumbnail: ''
            };
        },
        components: {
            avatar: components.getAvatar,
            org: components.group.getOrg,
            friend: components.group.getFriend,
            recent: components.group.getRecent,
            group: components.group.getGroup
        },
        computed: {
            /* 说明： 已选会话个数 */
            selectedLen: function () {
                var context = this;
                // 选择时包含高管显示时要排除高管
                return context.selected.filter(function (item) {
                    return !context.executiveLimit(item);
                }).length;
            }
        },
        directives: {
            autoScroll: function (el) {
                counter += 1;
                if (counter < 4) {
                    return;
                }
                Vue.nextTick(function () {
                    var $el = $(el);
                    var height = $el.children().outerHeight();
                    $el.scrollTop(height);
                });
            }
        },
        mounted: function () {
            var context = this;
            var messageApi = dataModel.Message;
            messageApi.watch(function (newMessage) {
                if (newMessage.content && newMessage.messageType === 'RecallCommandMessage') {
                    if (message.messageUId === newMessage.content.messageUId) {
                        // 转发的消息被撤回
                        var tip = context.locale.forwardedFailed;
                        context.showMessage({
                            type: 'error',
                            message: tip,
                            callback: function () {
                                context.busy = false;
                                context.close();
                            }
                        });
                    }
                }
            });
        },
        methods: getMethods(dataModel, im, message)
    };
    common.mountDialog(options, function (instance) {
        RongIM._forword = instance;
    });
};

function getMethods(dataModel, im, message) {
    return {
        reset: function () {
            this.selected.push({});
            this.selected.pop();
        },
        toast: function (params) {
            params.el = this.$el.firstChild;
            common.messageToast(params);
        },
        close: function () {
            this.show = false;
        },
        setTab: function (tab) {
            this.tab = tab;
        },
        setTip: function (tip) {
            this.tip = tip;
        },
        setTipState: function (state) {
            this.tipState = state;
        },
        showMessage: function (params) {
            var context = this;
            params.el = context.$el.firstChild;
            common.messageToast(params);
        },
        /* 说明： 移除已选择的会话 右侧已选列表的事件绑定 */
        removeMembers: function (members) {
            members = [].concat(members);
            var idList = members.map(function (item) {
                return item.id;
            });
            this.selected = this.selected.filter(function (item) {
                return idList.indexOf(item.id) < 0;
            });
        },
        added: function (members) {
            var context = this;
            var selectedIdList = context.selected.map(function (item) {
                return item.id;
            });
            var addedList = members.filter(function (item) {
                return selectedIdList.indexOf(item.id) < 0;
            });
            var totalCount = selectedIdList.length + addedList.length;
            context.selected = context.selected.concat(addedList);
            if (totalCount > context.maxCount) {
                var tip = common.getErrorMessage('forward-limit');
                context.showMessage({
                    type: 'error',
                    message: tip
                });
                context.removed(members);
            }
        },
        /* 说明： 移除已选择的会话 组件事件触发绑定 */
        removed: function (members) {
            var context = this;
            var idList = members.map(function (item) {
                return item.id;
            });
            var reservedIdList = context.defaultSelected.map(function (item) {
                return item.id;
            });
            context.selected = context.selected.filter(function (item) {
                var reserved = reservedIdList.indexOf(item.id) >= 0;
                return reserved || idList.indexOf(item.id) < 0;
            });
        },
        getUsername: common.getUsername,
        submit: function () {
            var context = this;
            var params = {
                context: {
                    selected: this.selected,
                    isGroup: this.isGroup,
                    locale: this.locale,
                    close: this.close,
                    showMessage: context.showMessage,
                    busy: this.busy,
                    downloadUrl: this.downloadUrl,
                    thumbnail: this.thumbnail
                },
                api: {
                    message: dataModel.Message
                }
            };
            console.log(params)
            submit(params, message.content, context);
        },
        isGroup: function (item) {
            return item.id.startsWith('group_');
        },
        executiveLimit: function (item) {
            if (item.isFriend || im.auth.isExecutive || this.disableExecutive) {
                return false;
            }
            var isExecutive = !!item.isExecutive;
            return isExecutive;
        },
        maxCountLimit: function () {
            var tip = common.getErrorMessage('forward-limit');
            this.showMessage({
                type: 'error',
                message: tip
            });
            this.reset();
        }
    };
}

/*
说明： 根据选择的会话转发消息， 客户端限制每秒最多发送 5 条消息，需要限制发送频率。
*/
function submit(params, content, forwardComponent) {
    var context = params.context;
    var messageApi = params.api.message;
    if (forwardComponent.busy) {
        return;
    }
    forwardComponent.busy = true;
    if (content.isForwaed) {
        // 转发群组二维码图片，需要上传文件服务器
        var base64Str = content.content || '';
        forwardGroupQRcode(context, base64Str, function (forwordContent) {
            sendForwardMsg(forwordContent, messageApi, context, forwardComponent);
        });
        return;
    }
    sendForwardMsg(content, messageApi, context, forwardComponent);
}

// 转发之前，上传群二维码
function forwardGroupQRcode(context, base64Str, callback) {
    var getContent = function (thumbnail, url) {
        var content = {
            content: thumbnail,
            imageUri: url,
            messageName: utils.messageType.ImageMessage,
            thumbnailPath: ''
        };
        return content;
    };
    if (context.downloadUrl && context.thumbnail) {
        var forwordContent = getContent(context.thumbnail, context.downloadUrl);
        callback(forwordContent);
        return;
    }
    upload('base64', base64Str, function (errorCode, result) {
        if (errorCode) {
            console.warn('upload failed', errorCode);
            return;
        }
        context.downloadUrl = result.imageUrl;
        context.thumbnail = result.thumbnail;
        callback(getContent(result.thumbnail, result.imageUrl));
    });
}

// 发送转发的消息
function sendForwardMsg(content, messageApi, context, forwardComponent) {
    var message = messageApi.create({
        messageType: content.messageName,
        content: content
    });
    if (message.messageName === utils.messageType.FileMessage) {
        message.localPath = content.localPath;
    }
    var paramList = context.selected.map(function (item) {
        var conversationType = utils.conversationType.PRIVATE;
        var targetId = item.id;
        if (context.isGroup(item)) {
            conversationType = utils.conversationType.GROUP;
            targetId = targetId.replace('group_', '');
        }
        return {
            conversationType: conversationType,
            targetId: targetId,
            content: message
        };
    });
    var index = 0;
    var timer = setInterval(function () {
        if (index < paramList.length) {
            try {
                messageApi.send(paramList[index], function (errorCode, msg) {
                    // 群组被禁言
                    if (errorCode === 'lib-22408') {
                        messageApi.addForwardFaildMessage(msg);
                    }
                });
            } catch (error) {
                clearInterval(timer);
            }
            index += 1;
        } else {
            clearInterval(timer);
        }
    }, 1000 / 5);
    context.showMessage({
        message: context.locale.forwarded,
        callback: function () {
            forwardComponent.busy = false;
            context.close();
        }
    });
}

/**
 * 上传图片到文件服务器
 * @param type - 'file' or 'image' or 'base64'
 * @param fileData
 * @param callback
 */
function upload(type, fileData, callback) {
    var config = RongIM.config.upload[type] || RongIM.config.upload.file;
    var dataType = RongIMLib.FileType.IMAGE;
    var thumbnail = '';
    config.data = UploadClient.dataType.data;
    config.getToken = function (done) {
        var dataModel = RongIM.instance.dataModel;
        dataModel.File.getFileToken(function (token) {
            done(token);
        }, dataType);
    };
    var uploadCallback = {
        onBeforeUpload: function (data) {
            thumbnail = data;
        },
        onProgress: function () {
        },
        onCompleted: function (data) {
            var url = common.getDownloadUrl(RongIM.config, data);

            if (url) {
                callback(null, { imageUrl: url, thumbnail: thumbnail });
                return;
            }
            RongIMClient.getInstance().getFileUrl(dataType, data.filename, data.name, {
                onSuccess: function (result) {
                    var imageUrl = result.downloadUrl;
                    callback(null, { imageUrl: imageUrl, thumbnail: thumbnail });
                },
                onError: function () {
                    utils.console.log('获取URL失败');
                }
            });

            // url = url || location.protocol + '//' + domain + '/' + data.key;
            // callback(null, url);
        },
        onError: callback
    };
    UploadClient.initImgBase64(config, function (uploadFile) {
        uploadFile.upload(fileData, uploadCallback);
    });
}
}(RongIM, {
    UploadClient: UploadClient,
    jQuery: jQuery
}, RongIM.components));
