(function (RongIM, dependencies, components) {
'use strict';

var common = RongIM.common;
var utils = RongIM.utils;
var $ = dependencies.jQuery;
var pageNum = RongIM.config.profile.pageNum;

// 禁言类型 0 正常状态 1 禁言黑名单 2 禁言白名单（全员禁言时可发言）
var BannedType = {
    NONE: 0,
    BANNED: 1,
    UNBANNED: 2
};

/*
说明： 群组禁言
*/
RongIM.dialog.groupBanned = function (group, _members) {
    var im = RongIM.instance;
    var dataModel = im.dataModel;

    // 群组成员禁言排除自己（自己为群组的管理者）
    group = $.extend(true, {}, group);
    var members = _members.filter(function (member) {
        member._muteStatus = member.mute_status;
        return member.id !== im.auth.id;
    });
    // 获取禁言黑名单和白名单列表。
    var statusList = getSpearkersAndSilenters(group, members);
    var options = {
        name: 'group-banned',
        template: 'templates/conversation/group-banned.html',
        data: function () {
            return {
                keyword: '',
                show: true,
                members: members,
                bannedList: [],
                // 禁言黑名单
                silenters: statusList.silenters,
                // 禁言白名单
                speakers: statusList.speakers,
                editList: [],
                group: group,
                defaultGroup: $.extend(true, {}, group),
                searchResult: members,
                isBannedAll: group.is_all_mute,
                busy: false,
                pageNum: pageNum,
                currentPage: 1,
                loadingNextPage: false
            };
        },
        components: {
            avatar: components.getAvatar
        },
        computed: {
            bannedAll: {
                get: function () {
                    return this.isBannedAll;
                },
                set: function (isBan) {
                    this.isBannedAll = isBan;
                }
            },
            /* 绑定成员列表的选中状态 */
            checked: {
                get: function () {
                    return getChecked(this);
                },
                set: function (memberIds) {
                    var isAdd = true;
                    var id;
                    var oldIds = this.checked;
                    if (memberIds.length < oldIds.length) {
                        isAdd = false;
                        id = without(oldIds, memberIds)[0];
                    } else {
                        id = without(memberIds, oldIds)[0];
                    }
                    setChecked(this, id, isAdd);
                }
            },
            // 显示禁言黑名单列表
            showBannedList: function () {
                return !this.isBannedAll;
            },
            pageList: function () {
                var context = this;
                var end = context.currentPage * context.pageNum;
                return this.searchResult.slice(0, end);
            }
        },
        watch: {
            keyword: function (keyword) {
                keywordChanged(this, keyword, members);
                $(this.$refs.list).scrollTop(0);
                this.currentPage = 1;
            }
        },
        mounted: function () {
            mounted(this, dataModel, im);
        },
        destroyed: function () {
            dataModel.Group.unwatch(this.groupChangeInBanned);
        },
        methods: getMethods(dataModel, im)
    };
    common.mountDialog(options, function (instance) {
        RongIM._groupInstance = instance;
    });
};

function without(members, others) {
    return members.filter(function (item) {
        return others.indexOf(item) < 0;
    });
}

function mounted(context, dataModel, im) {
    var groupApi = dataModel.Group;
    var auth = im.auth || {};
    var authId = auth.id;
    // 群的管理员被转让时自己已非管理员提示并关闭编辑窗口
    context.groupChangeInBanned = function (group) {
        var isSelected = group.id === context.group.id;
        var hasGroup = group && group.admin_id;
        var isAdmin = hasGroup && group.admin_id === authId;
        var message = context.locale.components.forward.success;
        if (!isAdmin && isSelected) {
            groupApi.unwatch(context.groupChangeInBanned);
            if (context.show) {
                common.messagebox({
                    message: message,
                    callback: function () {
                        im.$emit('messageboxHide');
                        context.show = false;
                    }
                });
            }
        }
    };
    groupApi.watch(context.groupChangeInBanned);
}

/*
说明： 成员选择状态绑定 - 获取选择状态
*/
function getChecked(context) {
    var dataType = context.showBannedList ? 'silenters' : 'speakers';
    return context[dataType].map(function (member) {
        return member.id;
    });
}

/*
说明： 成员选择状态绑定 - 改变选择状态
*/
function setChecked(context, memberId, isAdd) {
    var showBannedList = context.showBannedList;
    var dataType = showBannedList ? 'silenters' : 'speakers';
    var status = showBannedList ? BannedType.BANNED : BannedType.UNBANNED;

    var members = context.members;
    var member = members.filter(function (item) {
        return item.id === memberId;
    })[0];

    var addFunction = {
        silenters: function () {
            context.silenters.push(member);
            if (context.speakers.indexOf(member) > -1) {
                var index = context.speakers.indexOf(member);
                context.speakers.splice(index, 1);
            }
        },
        speakers: function () {
            context.speakers.push(member);
            if (context.silenters.indexOf(member) > -1) {
                var index = context.silenters.indexOf(member);
                context.silenters.splice(index, 1);
            }
        }
    };
    var removeFuntion = {
        silenters: function () {
            context.silenters = context.silenters.filter(function (item) {
                return item.id !== memberId;
            });
        },
        speakers: function () {
            context.speakers = context.speakers.filter(function (item) {
                return item.id !== memberId;
            });
        }
    };

    if (isAdd) {
        member._muteStatus = status;
        addFunction[dataType]();
    } else {
        member._muteStatus = BannedType.NONE;
        removeFuntion[dataType]();
    }
    pushEditList(context, member);
}

/*
说明： 成员禁言状态已改变加入到 editList 中确定时提交服务器
*/
function pushEditList(context, theMember) {
    var isSameMember = function (_member) {
        return _member.id === theMember.id;
    };
    var editList = context.editList;
    var isExist = editList.some(function (member) {
        return isSameMember(member);
    });
    if (isExist) {
        editList.forEach(function (member, index) {
            if (isSameMember(member)) {
                editList.splice(index, 1);
            }
        });
    }
    editList.push(theMember);
}

/*
说明： 设置全员禁言状态
*/
function setBannedAll(context, isBan, dataModel, callback) {
    var group = context.group;
    group.is_all_mute = isBan;
    var groupId = context.group.id;
    var status = Number(context.bannedAll);
    var params = {
        id: groupId,
        status: status
    };
    dataModel.Group.bannedAll(params, function (errorCode) {
        if (errorCode) {
            context.toastError(errorCode);
            return;
        }
        callback();
    });
}
function getMethods(dataModel) {
    return {
        toastError: function (errorCode) {
            var el = null;
            if (this.$el) {
                el = this.$el.firstChild;
            }
            common.toastError(errorCode, el);
        },
        close: function () {
            this.show = false;
        },
        clear: function () {
            this.keyword = '';
        },
        getUsername: function (item) {
            var name = common.getUsernameHighlight(item);
            return common.filterMark(name);
        },
        getMemberBanned: function (member) {
            var isMuteAll = this.isBannedAll;
            var muteStatus = member._muteStatus;
            var isBanned = muteStatus === BannedType.BANNED;
            var isNone = muteStatus === BannedType.NONE;
            var isMute = isMuteAll && isNone;
            return isMute || isBanned;
        },
        /*
        说明： 从 speakers （禁言白名单）中移除
        */
        clearSpeakMember: function (member) {
            clearMember(this, {
                member: member,
                members: this.speakers,
                status: BannedType.NONE
            });
        },
        /*
        说明： 从 silenters （禁言黑名单）中移除
        */
        clearSilentMember: function (member) {
            clearMember(this, {
                member: member,
                members: this.silenters,
                status: BannedType.NONE
            });
            if (this.searchResult.indexOf(member) === -1) {
                for (var i = this.searchResult.length - 1; i >= 0; i -= 1) {
                    var tempMember = this.searchResult[i];
                    if (tempMember.id === member.id) {
                        tempMember._muteStatus = BannedType.NONE;
                    }
                }
            }
        },
        submit: function () {
            var context = this;
            var group = this.defaultGroup;
            // 比对群组全员禁言状态与之前一致则不做修改，只修改成员的状态。
            var isBannedAll = this.isBannedAll;
            var isSelectBannedAll = isBannedAll !== group.is_all_mute;
            var close = context.close;
            if (context.busy === true) {
                close();
                return;
            }
            context.busy = true;
            if (!isSelectBannedAll) {
                context.banMember();
            } else {
                setBannedAll(context, isBannedAll, dataModel, function () {
                    context.banMember();
                });
            }
        },
        /* 说明： 更新群成员的禁言状态 */
        banMember: function () {
            var context = this;
            var members = context.editList;
            var close = context.close;
            if (members.length === 0) {
                context.busy = false;
                close();
                return;
            }
            var _members = members.map(function (member) {
                return { memberId: member.id, status: member._muteStatus };
            });
            var groupId = context.group.id;
            var params = {
                id: groupId,
                members: _members
            };
            dataModel.Group.banned(params, function (errorCode) {
                if (errorCode) {
                    common.toastError(errorCode);
                    return;
                }
                context.busy = false;
                close();
                members.forEach(function (item) {
                    item.mute_status = item._muteStatus;
                });
            });
        },
        loadMore: function () {
            loadMore(this);
        }
    };
}

function loadMore(context) {
    if (!context.loadingNextPage) {
        var end = context.currentPage * context.pageNum;
        var list = context.searchResult;
        if (list && list.length > end) {
            context.loadingNextPage = true;
            setTimeout(function () {
                context.currentPage += 1;
                context.loadingNextPage = false;
            }, 200);
        }
    }
}

/*
说明： 从禁言黑名单白名单移除成员时 更新 editList
*/
function clearMember(context, params) {
    var member = params.member;
    var members = params.members;
    var status = params.status;
    member._muteStatus = status;
    var index = members.indexOf(member);
    members.splice(index, 1);
    pushEditList(context, member);
}

/*
说明： 根据关键字查找成员
*/
function keywordChanged(context, keyword, members) {
    if (keyword.length === 0) {
        context.searchResult = members;
    } else {
        var searchResult = [];
        members.forEach(function (item) {
            var nameRange = utils.searchStrRange(item.name, context.keyword);
            var aliasRange = utils.searchStrRange(item.alias, context.keyword);
            if (nameRange || aliasRange) {
                var result = $.extend({
                    range: nameRange,
                    aliasRange: aliasRange
                }, item);
                searchResult.push(result);
            }
        });
        // common.sortUsers(searchResult);
        context.searchResult = searchResult;
    }
}

/*
说明： 获取禁言黑名单和白名单列表。
*/
function getSpearkersAndSilenters(group, members) {
    var speakers = [];
    var silenters = [];
    members.forEach(function (member) {
        var status = member._muteStatus;
        if (status === BannedType.BANNED) {
            silenters.push(member);
        }
        if (status === BannedType.UNBANNED) {
            speakers.push(member);
        }
    });
    return {
        silenters: silenters,
        speakers: speakers
    };
}
}(RongIM, {
    jQuery: jQuery
}, RongIM.components));
