(function (RongIM, dependencies, components) {
'use strict';

var common = RongIM.common;
var utils = RongIM.utils;
var $ = dependencies.jQuery;
var pageNum = RongIM.config.profile.pageNum;

/*
说明： 群组管理员转让
参数： @param {object} group 需要转让管理员的群组
*/
RongIM.dialog.groupTransfer = function (group, _members) {
    var im = RongIM.instance;
    // 成员列表排除自己，自己不可将管理员转让给自己
    group = $.extend(true, {}, group);
    var members = _members.filter(function (member) {
        var isMemberRole = (member.id !== im.auth.id);
        Vue.set(member, 'role', 0);
        return isMemberRole;
    });
    var dataModel = im.dataModel;
    var options = {
        name: 'group-transfer',
        template: 'templates/conversation/group-transfer.html',
        data: function () {
            return {
                show: true,
                keyword: '',
                members: members,
                groupId: group.id,
                searchResult: members,
                pageNum: pageNum,
                currentPage: 1,
                loadingNextPage: false
            };
        },
        components: {
            avatar: components.getAvatar
        },
        computed: {
            // 双向绑定成员选择状态
            groupOwner: {
                get: function () {
                    return getChecked(this.members);
                },
                set: function (memberIds) {
                    setChecked(this.members, memberIds);
                }
            },
            pageList: function () {
                var context = this;
                var end = context.currentPage * context.pageNum;
                return this.searchResult.slice(0, end);
            }
        },
        watch: {
            keyword: function (keyword) {
                this.searchResult = searchMember(keyword, members);
                $(this.$refs.list).scrollTop(0);
                this.currentPage = 1;
            }
        },
        mounted: function () {
            mounted(this, dataModel, im);
        },
        destroyed: function () {
            dataModel.Group.unwatch(this.groupChangeInTransfer);
        },
        methods: getMethods(dataModel, im)
    };
    common.mountDialog(options, function (instance) {
        RongIM._groupInstance = instance;
    });
};

function mounted(context, dataModel, im) {
    var groupApi = dataModel.Group;
    var auth = im.auth || {};
    var authId = auth.id;
    // 监听群组信息改变，自己不是管理员时关闭转让面板并提示
    context.groupChangeInTransfer = function (group) {
        var isSelected = group.id === context.groupId;
        var hasGroup = group && group.admin_id;
        var isAdmin = hasGroup && group.admin_id === authId;
        var message = context.locale.components.forward.success;
        if (!isAdmin && isSelected) {
            groupApi.unwatch(context.groupChangeInTransfer);
            if (context.show) {
                common.messagebox({
                    message: message,
                    callback: function () {
                        im.$emit('messageboxHide');
                        context.show = false;
                    }
                });
            }
        }
    };
    groupApi.watch(context.groupChangeInTransfer);
}

// 获取成员选择状态
function getChecked(members) {
    return members.filter(function (member) {
        var isOwner = (member.role === 1);
        return isOwner;
    }).map(function (member) {
        return member.id;
    });
}

// 设置成员选择状态
function setChecked(members, memberIds) {
    members.forEach(function (member) {
        var isOwner = (member.role === 1);
        member.role = 0;
        memberIds.forEach(function (memberId) {
            var isSelected = (member.id === memberId);
            if (!isOwner && isSelected) {
                member.role = 1;
            }
        });
    });
}

function getMethods(dataModel) {
    var group = dataModel.Group;
    return {
        toastError: function (errorCode) {
            var el = null;
            if (this.$el) {
                el = this.$el.firstChild;
            }
            common.toastError(errorCode, el);
        },
        close: function () {
            this.show = false;
        },
        clear: function () {
            this.keyword = '';
        },
        getUsername: function (item) {
            var name = common.getUsernameHighlight(item);
            return common.filterMark(name);
        },
        /*
        说明： 转让群管理
        */
        transfer: function () {
            var context = this;
            var locale = context.locale;
            var groupId = context.groupId;
            var members = context.members;
            var theMember = members.filter(function (member) {
                return member.role === 1;
            })[0];
            var message = utils.templateFormat(locale.message, theMember.name);
            common.messagebox({
                type: 'confirm',
                message: message,
                submitText: locale.tips.confirm,
                isAlignLeft: true,
                callback: function () {
                    var parmas = {
                        id: groupId,
                        manager: theMember.id
                    };
                    group.transfer(parmas, function (errorCode) {
                        if (errorCode) {
                            context.toastError(errorCode);
                            return;
                        }
                        context.show = false;
                        common.messagebox({
                            message: context.locale.components.forward.success
                        });
                    });
                }
            });
        },
        loadMore: function () {
            loadMore(this);
        }
    };
}

function loadMore(context) {
    if (!context.loadingNextPage) {
        var end = context.currentPage * context.pageNum;
        var list = context.searchResult;
        if (list && list.length > end) {
            context.loadingNextPage = true;
            setTimeout(function () {
                context.currentPage += 1;
                context.loadingNextPage = false;
            }, 200);
        }
    }
}

/*
说明： 根据关键字查找群成员
参数：
    @param {string} keyword 搜索关键字
    @param {Array}  members 从改成员集合中查找
*/
function searchMember(keyword, members) {
    var searchResult = [];
    if (keyword.length === 0) {
        searchResult = members;
    } else {
        members.forEach(function (item) {
            var nameRange = utils.searchStrRange(item.name, keyword);
            var aliasRange = utils.searchStrRange(item.alias, keyword);
            if (nameRange || aliasRange) {
                var result = $.extend({
                    range: nameRange,
                    aliasRange: aliasRange
                }, item);
                searchResult.push(result);
            }
        });
        // common.sortUsers(searchResult);
    }
    return searchResult;
}
}(RongIM, {
    jQuery: jQuery
}, RongIM.components));
