/*
说明： 最后一条消息(会话列表显示的最后一条消息)
功能： 对各类消息进行格式化，展示为一行文本。
       如： 图片消息 => [图片]
            文件消息 => [文件] 文件名称
*/
(function (RongIM, dependencies, components) {
'use strict';

var common = RongIM.common;
var utils = RongIM.utils;
var system = RongIM.system;
var $ = dependencies.jQuery;
var userAPI = RongIM.dataModel.User;

components.getLatestMessage = function (resolve, reject) {
    var im = RongIM.instance;
    var imParams = {
        locale: im.locale,
        auth: im.auth,
        templateFormat: utils.templateFormat
    };
    var convertLatestMessage = getConvertLatestMessage(imParams);
    var options = {
        name: 'latest-message',
        template: '#rong-template-latest-message',
        props: ['message', 'conversation'],
        computed: {
            authId: function () {
                return im.auth.id;
            },
            // 是否是无效消息,比如历史消息全部删除后会有一条 messageId=-1 的消息
            isInvalid: function () {
                return this.message.messageId === -1;
            },
            // 是否支持显示此类型消息
            isSupport: function () {
                var message = this.message;
                var content = !utils.isEmpty(convertLatestMessage[message.messageType]);
                var prefix = !utils.isEmpty(this.prefix);
                return prefix || content;
            },
            /*
            消息前缀
                文本消息 => ''
                图片消息 => '[图片]'
                文件消息 => '[文件]'
                图文消息 => '[图文]'
                ...
            */
            prefix: function () {
                var message = this.message;
                var _prefix = this.locale.message.prefix[message.messageType] || '';
                if (message.messageType === 'ApprovalMessage' && _prefix) {
                    _prefix = '<em>' + _prefix + '</em>';
                }
                // 接龙
                if(message.messageType === "RichContentMessage" && message.content.extra){
                    let extra = JSON.parse(message.content.extra)
                    if(extra.jielongId){
                        _prefix =  this.locale.message.prefix.JieLongMessage;
                    }

                }
                return _prefix;
            },
            /*
            最终展示内容
                消息前缀 + ' ' +
            */
            content: function () {
                var message = this.message;
                var convert = convertLatestMessage[message.messageType] || $.noop;
                var content = convert(message, this.authId, this.conversation) || '';
                return this.prefix + ' ' + content;
            },
            platform: function () {
                return system.platform;
            }
        }
    };
    utils.asyncComponent(options, resolve, reject);
};

function getHtmlUsername(user, groupId) {
    return common.getHtmlGroupUsername(user, groupId, 16);
}

/*
说明：  消息内容 (“最后一条消息”消息内容转换)
            文本消息 => '文本内容'
            图片消息 => ''
            文件消息 => '文件名称'
            图文消息 => '图文标题'
            ...
*/
function getConvertLatestMessage(params) {
    // var locale = params.locale;
    var localeFormat = params.templateFormat;
    var auth = params.auth;
    return {
        TextMessage: function (message) {
            var textMessage = message.content;
            var content = textMessage.content;
            content = utils.htmlLang.check(content);
            return common.convertMessage(content);
        },
        FileMessage: function (message) {
            return message.content.name;
        },
        LocalFileMessage: function (message) {
            return message.content.name;
        },
        CardMessage: function (message) {
            var fromMe = message.messageDirection === 1;
            var name = message.content.name;
            var locale = RongIM.instance.locale;
            var text = locale.message.cardOther;
            if (fromMe) {
                text = locale.message.cardSelf;
            }
            return localeFormat(text, name);
        },
        GroupMemChangedNotifyMessage: function (message, authId, conversation) {
            var isMe = RongIM.instance.auth.id;
            var group = conversation.group;
            // 操作行为类型
            var messageAction = message.content.action;
            // 群组减员通知：3，被移除；4，主动退出
            if (messageAction === 3 || messageAction === 4) {
                // 非操作者，后台操作时，操作者 id 为群主
                var isNotOperatorUser = message.content.operatorUser.id !== isMe;
                // 非被操作者
                var isNotTargetUsers = !message.content.targetUsers.some(function (item) {
                    return item.id === isMe;
                });
                var isManager = group && (group.admin_id === isMe);
                // 无关消息，不显示
                if (isNotOperatorUser && isNotTargetUsers && !isManager) {
                    return '';
                }
                // 群组减员消息，只通知群主且群必须为自建群，部门群等不通知
                if (!isManager || !group || group.type !== 0) {
                    return '';
                }
            }
            return common.getGroupNotification(message, authId);
        },
        GroupNotifyMessage: function (message, authId) {
            return common.getGroupNotification(message, authId);
        },
        GroupCmdMessage: function (message, authId) {
            return common.getGroupNotification(message, authId);
        },
        InformationNotificationMessage: function (message) {
            var content = message.content.message;
            if (content.messageName === 'ContactNotifyMessage') {
                return common.getContactNotification(content, auth.id);
            }
            var enDesc = RongIM.locale.en.components.getFileHelper.desc;
            var zhDesc = RongIM.locale.zh.components.getFileHelper.desc;
            var isFhMessage = content === enDesc || content === zhDesc;
            if (isFhMessage) {
                return RongIM.instance.locale.components.getFileHelper.desc;
            }
            return content;
        },
        RecallCommandMessage: function (message, authId, conversation) {
            var isMe = message.senderUserId === authId;
            var name = conversation.user && conversation.user.name;
            if (conversation.conversationType === utils.conversationType.GROUP) {
                if (conversation.latestMessage.user && conversation.group) {
                    name = getHtmlUsername(conversation.latestMessage.user, conversation.group.id);
                } else {
                    // 解决 web 端收到消息撤回时，最后一条消息显示错误问题
                    userAPI.get(conversation.latestMessage.senderUserId, function (errorCode, user) {
                        if (errorCode) {
                            console.warn(errorCode);
                            return;
                        }
                        name = user.name;
                        if (conversation.group) {
                            name = getHtmlUsername(user, conversation.group.id);
                        }
                    });
                }
            }
            var locale = RongIM.instance.locale;
            var result = localeFormat(locale.message.recallOther, name);
            if (message.content && message.content.isAdmin) {
                // return locale.message.recallAdmin;
                return;
            }
            if (isMe) {
                result = locale.message.recallSelf;
            }
            return result;
        },
        ContactNotifyMessage: function (message) {
            return common.getContactNotification(message.content, auth.id);
        },
        ApprovalMessage: function (message) {
            return message.content.content;
        },
        GroupNoticeNotifyMessage: function (message) {
            var content = message.content.content;
            var locale = RongIM.instance.locale;
            return common.latestGroupNoticeFormat(content, locale);
        },
        JrmfRedPacketMessage: function (message) {
            return common.getJrmfRedPacket(message);
        },
        JrmfRedPacketOpenedMessage: function (message) {
            return common.getJrmfRedPacketOpened(message, auth.id);
        },
        PublicServiceRichContentMessage: function (message) {
            var articles = [];
            var content = message.content;
            // Web SDK 与 C++ SDK 定义消息结构不一致导致
            if (content.richContentMessage) {
                articles = content.richContentMessage.articles;
            } else {
                articles = content.articles;
            }
            return articles[0].title;
        },
        PublicServiceMultiRichContentMessage: function (message) {
            var articles = [];
            var content = message.content;
            // Web SDK 与 C++ SDK 定义消息结构不一致导致
            if (content.richContentMessages) {
                articles = content.richContentMessages.articles || [];
            } else {
                articles = content.articles || [];
            }
            return articles[0].title;
        },
        VideoSummaryMessage: function (message) {
            var locale = RongIM.instance.locale;
            var isGroup = message.conversationType === utils.conversationType.GROUP;
            if (isGroup) {
                return common.groupSummaryFormat(message.content, locale);
            }
            var isVideo = message.content.mediaType === 2;
            if (isVideo) {
                return locale.message.prefix.VideoMessage;
            }
            return locale.message.prefix.AudioMessage;
        },
        RemoteControlSummaryMessage: function (message) {
            var im = RongIM.instance;
            var locale = im.locale;
            locale = locale.components.remoteControl;
            var content = message.content;
            var isControl = content.isControl;
            if (isControl) {
                return '[' + locale.controlTitle + ']';
            }
            return '[' + locale.controledTitle + ']';
        },
        ReferenceMessage: function (message) {
            var content = message.content.text;
            content = utils.htmlLang.check(content);
            return common.convertMessage(content);
        },
        RealTimeLocationStartMessage: function () {
            return '';
        },
        RequestFriendVerificationMessage: function () {
            var locale = RongIM.instance.locale;
            return locale.message.requestFriendVerification;
        }
    };
}
}(RongIM, {
    jQuery: jQuery,
    RongIMLib: RongIMLib
}, RongIM.components));
