/* 文件上传/下载 */
(function (RongIM, dependencies) {
'use strict';

var $ = dependencies.jQuery;
var RongIMClient = dependencies.RongIMClient;
var RongIMLib = dependencies.RongIMLib;
var UploadClient = dependencies.UploadClient;
var Cache = RongIM.dataModel._Cache;
var config = RongIM.dataModel.config;

var utils = RongIM.utils;
var common = RongIM.common;

var messageApi = null;
var conversationApi = null;

var File = {};

// token 缓存并维持，避免频繁获取
var TokenCache = {};

File.loadApi = function () {
    messageApi = RongIM.dataModel.Message;
    conversationApi = RongIM.dataModel.Conversation;
    config = RongIM.dataModel.config;
    utils = RongIM.utils;
};

function getTokenHandle(callback, type, isDownload) {
    // 上传与下载场景下 token 不可共用，需分开缓存
    var key = type + '_' + Boolean(isDownload);
    var cache = TokenCache[key];
    // 优先使用缓存，维系 50 分钟
    if (cache && Date.now() - cache.timestamp < 50 * 60 * 1000) {
        callback(null, cache.token);
        return;
    }
    // 缓存强制失效
    TokenCache[key] = null;

    RongIMClient.getInstance().getFileToken(type, {
        onSuccess: function (data) {
            var token = data.token;
            if (isDownload) {
                token = utils.base64Encode(token);
            }
            // 缓存 token 数据
            TokenCache[key] = {
                timestamp: Date.now(),
                token: token
            };
            callback(null, token);
        },
        onError: function (error) {
            callback(error);
        }
    });
}
function getToken(callback, type, isDownload) {
    getTokenHandle(function (error, token) {
        if (error) {
            common.toastError('network-error');
            return;
        }
        callback(token);
    }, type, isDownload);
}
File.getFileToken = getToken;

function getFileDownloadToken(callback) {
    getTokenHandle(callback, RongIMLib.FileType.FILE, true);
}
File.getFileDownloadToken = getFileDownloadToken;

function getImageDownloadToken(callback) {
    getToken(callback, RongIMLib.FileType.IMAGE, true);
}
File.getImageDownloadToken = getImageDownloadToken;

function expendUploadMessage(uploadMessage, uploadFile, uploadCallback) {
    var uploadId = uploadMessage.content.uploadId;
    File.uploadManage.add(uploadId, uploadMessage);
    // 标识上传id,用于断点续传
    var isFile = uploadMessage.dataType === RongIMLib.FileType.FILE;
    if (isFile) {
        uploadMessage.data.uploadId = uploadId;
    }
    uploadMessage.suspend = function (callback) {
        callback = callback || $.noop;
        uploadFile.cancel();
        callback();
    };
    uploadMessage.cancel = function (callback) {
        File.uploadManage.remove(uploadId, uploadMessage);
        callback = callback || $.noop;
        uploadMessage.uploadStatus = RongIM.utils.uploadStatus.CANCELLED;
        uploadFile.cancel();
        callback();
    };
    uploadMessage.upload = function () {
        // TODO: expendUploadMessage 中判断需优化
        // console.log('todo: expendUploadMessage 中判断需优化');
        if (uploadMessage.uploadStatus === RongIM.utils.uploadStatus.READY || uploadMessage.uploadStatus === RongIM.utils.uploadStatus.FAIL || uploadMessage.uploadStatus === RongIM.utils.uploadStatus.CANCELLED || uploadMessage.isSuspend) {
            uploadMessage.isSuspend = false;
            uploadMessage.uploadStatus = RongIM.utils.uploadStatus.UPLOADING;
            uploadFile.upload(uploadMessage.data, uploadCallback);
        }
    };
}

function getDataType(data) {
    var fileType = RongIMLib.FileType.FILE;
    var isBase64 = (typeof data === 'string');
    // var isImage =  (/^image\/(png|jpg|jpeg|gif|webp|x-icon)/i.test(data.type));
    var isImage = (/^image\/(png|jpg|jpeg|gif)/i.test(data.type));
    var configSize = config.upload.file.imageSize / 1024 / 1024 * 1000 * 1000;
    var isNormalSize = (data.size < configSize);
    if ((isImage && isNormalSize) || isBase64) {
        fileType = RongIMLib.FileType.IMAGE;
    }
    return fileType;
}

/*
params.targetId
params.conversationType
params.data 上传的数据
params.data.localPath 为了兼容复制的本地文件,File 的 path 属性只读
*/
File.createUploadMessage = function (params) {
    // console.log('File.createUploadMessage => params: ' + JSON.stringify(params, null, '\t'));
    var message = {
        senderUserId: Cache.auth.id,
        targetId: params.targetId,
        conversationType: params.conversationType,
        messageDirection: RongIM.utils.messageDirection.SEND,
        uploadStatus: RongIM.utils.uploadStatus.READY,
        sentStatus: RongIM.utils.sentStatus.SENDING,
        sentTime: new Date().getTime(),
        messageType: '',

        progress: 0,
        dataType: getDataType(params.data),
        localPath: params.localPath,
        data: params.data
    };

    if (message.dataType === RongIMLib.FileType.IMAGE) {
        message.messageType = RongIMClient.MessageType.ImageMessage;
        if (typeof params.data === 'string') {
            // base64 图片
            message.content = {
                content: params.data,
                imageUri: '',
                messageName: message.messageType,
                localPath: ''
            };
        } else {
            // 图片文件上传
            message.content = {
                content: '',
                imageUri: '',
                messageName: message.messageType,
                localPath: params.localPath
            };
        }
    } else {
        message.messageType = RongIMClient.MessageType.FileMessage;
        var type = RongIM.utils.getFilenameExtension(params.data.name);
        message.content = {
            name: params.data.name,
            size: params.data.size,
            type: type,
            fileUrl: '',
            localPath: params.data.path || params.data.localPath,
            uploadId: getUploadId(),
            messageName: message.messageType
        };
        console.debug('createUploadMessage uploadId', message.content.uploadId);
    }

    return message;
};

function getUploadId() {
    return Date.now();
}

function getChunkSize(filetype) {
    var platform = RongIM.utils.getPlatform();
    var isWeb = false;
    if (platform.startsWith('web')) {
        isWeb = true;
    }
    var fileConfig = config.upload.file;
    if (filetype === RongIMLib.FileType.FILE && !isWeb) {
        return fileConfig.chunkSize;
    }
    return fileConfig.fileSize;
}

function checkVideo(content) {
    // web 处理逻辑有问题暂屏蔽
    var platform = RongIM.utils.getPlatform();
    var isWeb = false;
    if (platform.startsWith('web')) {
        isWeb = true;
    }
    return content.type === 'mp4' && content.size <= 20 * 1024 * 1024 && !isWeb;
}

/*
    获取小视频的 时长duration, 缩略图content
 */
function getVideoInfo(localPath, callback) {
    callback = callback || $.noop;
    var video = document.createElement('video');
    video.setAttribute('src', localPath);
    video.setAttribute('controls', 'controls');
    video.currentTime = 1;
    video.onloadeddata = function () {
        var maxWidth = 220;
        var scale = 1;
        var canvas = document.createElement('canvas');
        if (video.videoHeight === 0) {
            callback();
            return;
        }
        if (video.videoWidth > maxWidth) {
            scale = maxWidth / video.videoWidth;
        }
        if (video.videoHeight > maxWidth) {
            scale = Math.min(maxWidth / video.videoHeight, scale);
        }
        canvas.width = Math.ceil(video.videoWidth * scale);
        canvas.height = Math.ceil(video.videoHeight * scale);
        canvas.getContext('2d').drawImage(video, 0, 0, canvas.width, canvas.height);
        var poster = canvas.toDataURL('image/png', 0.4);
        utils.console.log('video Base64 length', poster.length);
        video.setAttribute('poster', poster);
        var info = {
            content: utils.Base64.replace(poster),
            duration: Math.round(video.duration)
        };
        document.body.removeChild(video);
        callback(info);
    };
    document.body.appendChild(video);
}

File.upload = function (uploadMessage, uploadConfig, callback) {
    callback = callback || $.noop;
    var platform = RongIM.utils.getPlatform();
    var isWeb = false;
    if (platform.startsWith('web')) {
        isWeb = true;
    }
    var isVideo = checkVideo(uploadMessage.content);
    var insertMessage = function (insertUploadMessage, _callback) {
        var im = RongIM.instance;
        var content = insertUploadMessage.content;
        _callback = _callback || $.noop;
        var params = {
            conversationType: insertUploadMessage.conversationType,
            targetId: insertUploadMessage.targetId,
            sendUserId: im.auth.id,
            messageDirection: 1,
            progress: insertUploadMessage.progress,
            uploadStatus: utils.uploadStatus.UPLOADING,
            sentStatus: RongIM.utils.sentStatus.SENDING,
            content: {
                messageName: 'FileMessage',
                type: content.type,
                name: content.name,
                size: content.size,
                localPath: content.localPath,
                uploadId: content.uploadId,
                status: 0
            },
            objectName: 'LRC:fileMsg'
        };
        if (isVideo && uploadMessage.isVideo) {
            params.content.content = content.content;
            params.content.duration = content.duration;
            params.content.sightUrl = content.sightUrl;
            params.objectName = 'RC:SightMsg';
            params.content.messageName = RongIMClient.MessageType.SightMessage;
        }
        messageApi.insertMessage(params, function (errorCode, message) {
            if (errorCode) {
                console.warn('insertMessage', errorCode);
                _callback(errorCode);
                return;
            }
            var setParams = {
                messageId: message.messageId,
                status: message.sentStatus
            };

            messageApi.setMessageSentStatus(setParams, null);
            _callback(null, message);
        });
    };
    var uploadCallback = {
        onBeforeUpload: function (data) {
            if (uploadMessage.dataType === RongIMLib.FileType.IMAGE && !uploadConfig.isEmoji) {
                uploadMessage.content.content = data;
            }
            var key = messageApi.getCacheKey(uploadMessage);
            var cacheList = messageApi._cache[key] = messageApi._cache[key] || [];
            messageApi.addSendUserInfo(uploadMessage, function (errorCode, msg) {
                if (errorCode) {
                    callback(errorCode);
                    return;
                }
                if (uploadMessage.dataType === RongIMLib.FileType.FILE && !isWeb) {
                    insertMessage(msg, function (errCode, _message) {
                        if (errCode) {
                            console.warn(errCode);
                            return;
                        }
                        _message.dataType = uploadMessage.dataType;
                        _message.data = uploadMessage.data;
                        _message.cancel = uploadMessage.cancel;
                        _message.suspend = uploadMessage.suspend;
                        _message.upload = uploadMessage.upload;
                        _message.isVideo = uploadMessage.isVideo;
                        uploadMessage = _message;
                        var uploadId = uploadMessage.content.uploadId;
                        File.uploadManage.add(uploadId, uploadMessage);
                    });
                } else {
                    var uploadId = uploadMessage.content.uploadId;
                    File.uploadManage.add(uploadId, uploadMessage);
                    // 增加一个自定义的 messageId 用于删除消息，发送消息时使用真实 messageId 覆盖此次生成 id
                    msg.messageId = RongIM.utils.createUid();
                    cacheList.push(msg);
                }
            });
        },
        onProgress: function (loaded, total) {
            var percent = Math.floor(loaded / total * 100);
            uploadMessage.progress = percent;
        },
        onCompleted: function (data) {
            var uploadId = uploadMessage.content.uploadId;
            File.uploadManage.remove(uploadId);
            // name 非空表示上传成功（取消上传为空）
            var undef;
            var condition = data.name;
            if (config.upload.type === 'RongCloud') {
                condition = data.rc_url;
            }
            if (condition !== undef) {
                if (uploadMessage.messageType === RongIMClient.MessageType.LocalFileMessage) {
                    uploadMessage.messageType = RongIMClient.MessageType.FileMessage;
                    uploadMessage.content.messageName = RongIMClient.MessageType.FileMessage;
                }
                uploadMessage.uploadStatus = RongIM.utils.uploadStatus.SUCCESS;
                callback(null, uploadMessage, data);
            }
        },
        onError: function (errorCode) {
            var uploadId = uploadMessage.content.uploadId;
            File.uploadManage.remove(uploadId);
            uploadMessage.uploadStatus = RongIM.utils.uploadStatus.FAIL;
            // 上传失败同发送失败显示效果
            uploadMessage.sentStatus = RongIM.utils.sentStatus.FAILED;
            callback('upload-' + errorCode, uploadMessage);
            // insertMessage(uploadMessage);
        }
    };

    uploadConfig.getToken = function (tokenCallback) {
        getToken(tokenCallback, uploadMessage.dataType);
    };

    uploadConfig.timeout = config.upload.timeout;
    uploadConfig.chunk_size = getChunkSize(uploadMessage.dataType);
    if (uploadMessage.dataType === RongIMLib.FileType.FILE) {
        UploadClient.initFile(uploadConfig, function (uploadFile) {
            if (isVideo) {
                getVideoInfo('file://' + uploadMessage.content.localPath, function (info) {
                    if (info) {
                        uploadMessage.content.content = info.content;
                        uploadMessage.content.duration = info.duration;
                        uploadMessage.content.sightUrl = 'file://' + uploadMessage.content.localPath;
                        uploadMessage.isVideo = true;
                    } else {
                        uploadMessage.isVideo = false;
                    }
                    expendUploadMessage(uploadMessage, uploadFile, uploadCallback);
                    uploadMessage.upload();
                });
                return;
            }
            expendUploadMessage(uploadMessage, uploadFile, uploadCallback);
            uploadMessage.upload();
        });
    } else if (typeof uploadMessage.data === 'string') {
        uploadConfig.chunk_size = config.upload.file.fileSize;
        UploadClient.initImgBase64(uploadConfig, function (uploadFile) {
            expendUploadMessage(uploadMessage, uploadFile, uploadCallback);
            uploadMessage.upload();
        });
    } else {
        UploadClient.initImage(uploadConfig, function (uploadFile) {
            expendUploadMessage(uploadMessage, uploadFile, uploadCallback);
            uploadMessage.upload();
        });
    }
};

File.resumeUpload = function (uploadMessage, uploadConfig, callback) {
    callback = callback || $.noop;
    var uploadCallback = {
        onBeforeUpload: function () {
            uploadMessage.sentStatus = RongIM.utils.sentStatus.SENDING;
        },
        onProgress: function (loaded, total) {
            var percent = Math.floor(loaded / total * 100);
            uploadMessage.progress = percent;
            uploadMessage.uploadStatus = utils.uploadStatus.UPLOADING;
        },
        onCompleted: function (data) {
            // name 非空表示上传成功（取消上传为空）
            var undef;
            var condition = data.name;
            if (config.upload.type === 'RongCloud') {
                condition = data.rc_url;
            }
            if (condition !== undef) {
                if (uploadMessage.messageType === RongIMClient.MessageType.LocalFileMessage) {
                    uploadMessage.messageType = RongIMClient.MessageType.FileMessage;
                }
                uploadMessage.uploadStatus = RongIM.utils.uploadStatus.SUCCESS;
                callback(null, uploadMessage, data);
            }
        },
        onError: function (errorCode) {
            uploadMessage.uploadStatus = RongIM.utils.uploadStatus.FAIL;
            // 上传失败同发送失败显示效果
            uploadMessage.sentStatus = RongIM.utils.sentStatus.FAILED;
            callback(errorCode, uploadMessage);
        }
    };

    uploadConfig.getToken = function (tokenCallback) {
        getToken(tokenCallback, uploadMessage.dataType);
    };
    uploadConfig.chunk_size = getChunkSize(uploadMessage.dataType);

    if (uploadMessage.dataType === RongIMLib.FileType.FILE) {
        UploadClient.initFile(uploadConfig, function (uploadFile) {
            expendUploadMessage(uploadMessage, uploadFile, uploadCallback);
            uploadMessage.upload();
        });
    }
};

File.addFileUrl = function (uploadMessage, data, callback) {
    // 获取下载路径
    var url = common.getDownloadUrl(RongIM.config, data);
    if (url) {
        dealFileUrl(url, uploadMessage, callback);
        return;
    }
    RongIMClient.getInstance().getFileUrl(uploadMessage.dataType, data.filename, data.name, {
        onSuccess: function (result) {
            var fileUrl = result.downloadUrl;
            dealFileUrl(fileUrl, uploadMessage, callback);
        },
        onError: function () {
            uploadMessage.uploadStatus = RongIM.utils.uploadStatus.FAIL;
            utils.console.log('获取URL失败');
        }
    });
};

function dealFileUrl(url, uploadMessage, callback) {
    uploadMessage.sentStatus = RongIM.utils.sentStatus.SENDING;
    var content = uploadMessage.content;
    if (uploadMessage.dataType === RongIMLib.FileType.IMAGE) {
        content.imageUri = url;
    } else {
        content.fileUrl = url;
    }
    if (content.sightUrl) {
        content.sightUrl = url;
    }
    callback(null, uploadMessage);
}

File.send = function (uploadMessage, callback) {
    callback = callback || $.noop;
    var conversationType = Number(uploadMessage.conversationType);
    var targetId = uploadMessage.targetId;
    var message;
    if (uploadMessage.dataType === RongIMLib.FileType.IMAGE) {
        message = new RongIMLib.ImageMessage(uploadMessage.content);
    } else {
        message = new RongIMLib.FileMessage(uploadMessage.content);
    }
    var oldMessageId = uploadMessage.messageId;
    var isFile = uploadMessage.dataType === RongIMLib.FileType.FILE;

    // 判断是否视频文件且小于 20M, 是则发小视频消息
    var isVideo = checkVideo(uploadMessage.content);

    if (isVideo && uploadMessage.isVideo) {
        var msgContent = $.extend({}, uploadMessage.content);
        msgContent.MessageName = 'SightMessage';
        message = common.buildMessage.SightMessage(msgContent);
    }

    RongIMClient.getInstance().sendMessage(conversationType, targetId, message, {
        onBefore: function (messageId) {
            uploadMessage.messageId = messageId;
        },
        onSuccess: function (serverMessage) {
            uploadMessage.sentStatus = RongIM.utils.sentStatus.SENT;
            uploadMessage.sentTime = serverMessage.sentTime;
            uploadMessage.messageUId = serverMessage.messageUId;
            uploadMessage.messageId = serverMessage.messageId;
            var im = RongIM.instance;
            im.$emit('messagechange');
            callback(null, uploadMessage);
            if (serverMessage.content) {
                serverMessage.content.localPath = uploadMessage.content.localPath;
            }
            RongIMClient.getInstance().setMessageContent(serverMessage.messageId, serverMessage.content, '');

            // 文件消息需删除原消息
            if (oldMessageId !== serverMessage.messageId && isFile) {
                // 删除
                var params = {
                    conversation: uploadMessage.conversationType,
                    targetId: uploadMessage.targetId,
                    messageIds: [oldMessageId],
                    notNotify: true
                };
                messageApi.removeLocal(params, function (errorCode) {
                    if (errorCode) {
                        return;
                    }
                    messageApi.saveRemovedEarliestMessageTime(message);
                });
            }
        },
        onError: function (errorCode) {
            // 文件消息需删除原消息
            if (oldMessageId !== uploadMessage.messageId && isFile) {
                // 删除
                var params = {
                    conversation: uploadMessage.conversationType,
                    targetId: uploadMessage.targetId,
                    messageIds: [oldMessageId],
                    notNotify: true
                };
                messageApi.removeLocal(params, function (error) {
                    if (error) {
                        return;
                    }
                    messageApi.saveRemovedEarliestMessageTime(message);
                });
            }
            uploadMessage.sentStatus = RongIM.utils.sentStatus.FAILED;
            // fix SDK 发送失败 3016 消息状态为发送中
            messageApi.setMessageSentStatus({
                messageId: uploadMessage.messageId,
                status: RongIM.utils.sentStatus.FAILED
            });

            callback(errorCode, uploadMessage);
            var conversation = conversationApi.getLocalOne(uploadMessage.conversationType, uploadMessage.targetId);
            if (conversation) {
                conversation.latestMessage = uploadMessage;
                var list = conversationApi.getLocalList();
                conversationApi.observerList.notify(list);
            }
        }
    });
};

function Manage() {
    this._cache = {};
}
Manage.prototype = {
    add: function (id, downloader) {
        this._cache[id] = downloader;
    },
    get: function (id) {
        return this._cache[id];
    },
    remove: function (id) {
        delete this._cache[id];
    }
};
// 存放下载中的文件
File.downloadManage = new Manage();
File.downloadManage.abortAll = function () {
    var self = this;
    var keys = Object.keys(this._cache);
    keys.forEach(function (key) {
        var item = self._cache[key];
        item.abort();
    });
    self._cache = {};
};

// 存放上传中的文件
File.uploadManage = new Manage();
File.uploadManage.abortAll = function () {
    var self = this;
    var keys = Object.keys(this._cache);
    keys.forEach(function (key) {
        var item = self._cache[key];
        item.cancel();
    });
    self._cache = {};
};

File.getFileType = function (type) {
    switch (type) {
        case 0:
            type = 'qiniu';
            break;
        case 1:
            type = 'RongCloud';
            break;
        case 2:
            type = 'RongCloud';
            break;
        default:
            type = 'qiniu';
            break;
    }
    return type;
};

RongIM.dataModel.File = File;
}(RongIM, {
    jQuery: jQuery,
    UploadClient: UploadClient,
    RongIMClient: RongIMClient,
    RongIMLib: RongIMLib
}));
