(function (RongIM, dependencies) {
'use strict';

var $ = dependencies.jQuery;
var RongIMLib = dependencies.RongIMLib;
var RongIMClient = dependencies.RongIMClient;
var Cache = RongIM.dataModel._Cache;
var ObserverList = RongIM.dataModel._ObserverList;
var util = RongIM.dataModel.util;
var generatorKey = util.generatorKey;
var getStoreKey = util.getStoreKey;
var store = RongIM.utils.cache;
var getServerTime = RongIM.dataModel.getServerTime;
var config = RongIM.dataModel.config;
var getLibErrorCode = util.getLibErrorCode;

var utils = RongIM.utils;
var common = RongIM.common;
var getCurrentConnectionStatus = null;
var orgApi = null;
var userApi = null;
var conversationApi = null;
var pinApi = null;
var bindApi = null;
var groupApi = null;
var remoteCtrlApi = null;
var groupNtcApi = null;
var statusApi = null;
var friendApi = null;
var starApi = null;
var accountApi = null;

RongIM.messageTotal = 0;
var PullMessageStatus = {
    isFinished: true,
    set: function (isFinished) {
        this.isFinished = isFinished;
        conversationApi.isPullMessageFinished = isFinished;
    },
    get: function () {
        return this.isFinished;
    }
};

var UpdateStatusType = {
    FAV_GROUP_UPDATED: 1,
    FAV_CONTACT_UPDATED: 2,
    CONVERSATION_SETTING_UPDATED: 3,
    COMPANY_UPDATED: 4,
    DEPARTMENT_UPDATED: 5,
    DUTY_UPDATED: 6,
    USER_SETTING_UPDATED: 7,
    STAFF_UPDATED: 8,
    PASSWORD_UPDATED: 9,
    DEPART_MEMBER_UPDATED: 10,
    USER_UPDATED: 11,
    CONFIGURATION_UPDATED: 12,
    USERNAME_UPDATED: 13,
    FAV_CONTENT_UPDATED: 14, // 收藏内容更新
    VISIBLE_SCOPE_UPDATED: 15, // 通讯录可见范围更新
    DEL_FRIEND: 1000
};

var delaySync = RongIM.utils.debounce(function (callback) {
    if (typeof callback === 'function') {
        callback();
    }
}, RongIM.config.syncDelayTime);

function updateStatusMessageHandle(message) {
    RongIM.system.appLogger('debug', '[ReceiveMessage][RCEUpdateStatusMessage]' + JSON.stringify(message));
    // 数据同步不需要处理离线消息
    if (message.offLineMessage) {
        return;
    }
    switch (message.content.updateType) {
        case UpdateStatusType.FAV_GROUP_UPDATED:
            break;
        case UpdateStatusType.FAV_CONTACT_UPDATED:
            starApi.getStarList().done(function (idList) {
                Cache.starList = idList;
            });
            starApi.observerList.notify();
            break;
        case UpdateStatusType.CONVERSATION_SETTING_UPDATED:
            Cache.conversation.isSetting = false;
            Cache.conversation.topList = null;
            Cache.conversation._defer = null;
            conversationApi.observerList.notify();
            break;
        case UpdateStatusType.COMPANY_UPDATED:
            delaySync(function () {
                RongIM.syncdata.company();
            });
            break;
        case UpdateStatusType.DEPARTMENT_UPDATED:
            // 更新部门信息
            delaySync(function () {
                RongIM.syncdata.department();
            });
            break;
        case UpdateStatusType.DUTY_UPDATED:
            break;
        case UpdateStatusType.USER_SETTING_UPDATED:
            break;
        case UpdateStatusType.STAFF_UPDATED:
            delaySync(function () {
                RongIM.syncdata.staff(function () {
                // 目前只同步自己的用户信息
                    delete Cache.user[Cache.auth.id];
                    userApi.get(Cache.auth.id, function (errorCode, user) {
                        if (errorCode) {
                            return;
                        }
                        $.extend(RongIM.instance.auth, user);
                        $.extend(Cache.auth, user);
                        // 清除缓存消息更新消息中用户信息。
                        userApi.observerList.notify(user);
                    });
                });
            });
            break;
        case UpdateStatusType.PASSWORD_UPDATED:
            accountApi.observerList.notify('password-changed');
            break;
        case UpdateStatusType.DEPART_MEMBER_UPDATED:
            // 更新所有组织机构信息 使用随机 1-300 秒 timer
            delaySync(function () {
                var delay = Math.round(Math.random() * 300) * 1000;
                console.info('%cDEPART_MEMBER_UPDATED', 'color:green', delay);
                RongIM.system.appLogger('debug', '[DEPART_MEMBER_UPDATED]' + delay);
                setTimeout(function () {
                    var auth = RongIM.instance.auth;
                    if (auth) {
                        RongIM.syncdata.all(auth.isStaff);
                    }
                }, delay);
                // delete Organization.getCompany.cache;
                delete orgApi.getRoot.cache;
                orgApi.observerList.notify();
            });
            break;
        case UpdateStatusType.USER_UPDATED:
            break;
        case UpdateStatusType.CONFIGURATION_UPDATED:
            break;
        case UpdateStatusType.USERNAME_UPDATED:
            break;
        case UpdateStatusType.DEL_FRIEND:
            var targetId = message.content.uid;
            var ConversationType = RongIMLib.ConversationType;
            friendApi.delFriend(targetId, function () {});
            friendApi.delRequest(targetId, function () {});
            conversationApi.remove(ConversationType.PRIVATE, targetId, function () {});
            break;
        case UpdateStatusType.FAV_CONTENT_UPDATED:
            break;
        case UpdateStatusType.VISIBLE_SCOPE_UPDATED:
            // 清理所有用户缓存，因为可见性范围变化可能影响多个用户
            Cache.user = { _defer: {} };
            Cache.alias = {};
            // 通知所有用户观察者更新
            userApi.observerList.notify();
            break;
        default:
            utils.console.warn('Unknown update type ==> ', message.content.updateType);
            break;
    }
}

var isSameConversation = function (message) {
    var active = conversationApi.active;
    var activeType = active.conversationType;
    var activeId = active.targetId;
    return message.targetId === activeId && +activeType === +message.conversationType;
};

var Message = {
    TextMessage: RongIMLib.TextMessage,
    ImageMessage: RongIMLib.ImageMessage,
    FileMessage: RongIMLib.FileMessage,
    observerList: new ObserverList(),
    _cache: {},
    _push: function (message, callback) {
       
        callback = callback || $.noop;
        var key = getCacheKey(message);
        var cacheList = Message._cache[key] = Message._cache[key] || [];

        if (!messageUnexist(message, cacheList)) {
            return;
        }
        // 先设置一个 user 属性防止 Vue 监测不到用户信息变化
        message.user = {};

        if (message.offLineMessage) {
            if (cacheList.length > 60) {
                cacheList.splice(0, 50);
            }
            // server 下发离线消息有时顺序不对，需要重新排序
            var position = 0;
            var cacheLength = cacheList.length;
            for (var i = cacheLength - 1; i >= 0; i -= 1) {
                if (cacheList[i].sentTime < message.sentTime) {
                    position = i + 1;
                    break;
                }
            }
            // 顺序不对时打印日志
            if (position !== cacheLength) {
                utils.console.warn('离线消息顺序不正确 messageUId:'
                    + message.messageUId
                    + ' targetId:' + message.targetId
                    + ' conversationType:' + message.conversationType);
            }
            cacheList.splice(position, 0, message);
        } else {
            cacheList.push(message);
        }
        // 如果是当前会话则立即添加用户信息，否则切换会话时再获取
        if (isSameConversation(message)) {
            Message.addSendUserInfo(message, function (errorCode, msg) {
                if (errorCode) {
                    callback(errorCode);
                    return;
                }
                callback(null, msg);
            });
        }
    },
    _sendPush: function (message, callback) {
        callback = callback || $.noop;
        var key = getCacheKey(message);
        var cacheList = Message._cache[key] = Message._cache[key] || [];

        var cacheMessage = getCacheMessageById(cacheList, message.messageId);
        if (cacheMessage) {
            cacheMessage.sentStatus = message.sentStatus;
            $.extend(cacheMessage, message);
            callback(null, cacheMessage);
        } else {
            Message.addSendUserInfo(message, function (errorCode, msg) {
                if (errorCode) {
                    callback(errorCode);
                    return;
                }
                cacheList.push(msg);
                callback(null, msg);
            });
        }
    }
};
var msgObserverList = Message.observerList;

Cache.messageQueue = [];
Cache.messageRecallEdit = {};

Message.cleanCache = function () {
    Cache.messageQueue = [];
    Cache.messageRecallEdit = {};
    Message._cache = {};
};

var apiList = [];
Message.loadApi = function () {
    var dataModel = RongIM.dataModel;
    orgApi = dataModel.Organization;
    userApi = dataModel.User;
    groupApi = dataModel.Group;
    conversationApi = dataModel.Conversation;
    statusApi = dataModel.Status;
    pinApi = dataModel.Pin;
    bindApi = dataModel.Bind;
    remoteCtrlApi = dataModel.RemoteControl;
    groupNtcApi = dataModel.GroupNotice;
    friendApi = dataModel.Friend;
    starApi = dataModel.Star;
    accountApi = dataModel.Account;
    apiList = [orgApi, userApi, groupApi, conversationApi, statusApi];
    apiList = apiList.concat([pinApi, groupNtcApi, friendApi, starApi]);
    apiList = apiList.concat([accountApi, remoteCtrlApi, bindApi]);

    config = dataModel.config;
    utils = RongIM.utils;
    getCurrentConnectionStatus = statusApi.getCurrentConnectionStatus;
};

function getCacheMessageById(cacheList, messageId) {
    for (var i = 0, len = cacheList.length; i < len; i += 1) {
        if (cacheList[i].messageId === messageId) {
            return cacheList[i];
        }
    }
    return null;
}

function getCacheMessageByUId(type, id, uid) {
    var key = type + '_' + id;
    var list = Message._cache[key] || [];
    for (var i = list.length - 1; i >= 0; i -= 1) {
        if (list[i].messageUId === uid) {
            return list[i];
        }
    }
    return null;
}

function getCacheKey(obj) {
    return obj.conversationType + '_' + obj.targetId;
}

Message.getCacheKey = getCacheKey;

function spliceMessage(cacheList, messageId, message) {
    if (!cacheList) {
        return null;
    }
    var index = null;
    for (var i = 0, len = cacheList.length; i < len; i += 1) {
        var cacheMsg = cacheList[i];
        if (cacheMsg.messageId === messageId || cacheMsg.messageUId === messageId) {
            index = i;
        }
    }

    if (index === null) {
        return null;
    }
    var result = cacheList[index];
    if (message) {
        Message.addSendUserInfo(message, function (errorCode, msg) {
            cacheList.splice(index, 1, msg);
        });
    } else {
        cacheList.splice(index, 1);
    }
    return result;
}

/*
说明： 记录删除的最早的一条消息
       在获取 server 的历史消息时对比时间戳，如果比此时间戳大则使用此时间戳。
       (删除本地最早一条消息时，再次从 server 端拉去会将此消息拉取下来)
*/
function getRemovedEarliestMessageTime(params) {
    var key = 'removed_earliest_message_' + params.conversationType + '_' + params.targetId;
    var timestamp = store.get(key) || 0;
    return timestamp;
}
function saveRemovedEarliestMessageTime(params) {
    var timestamp = getRemovedEarliestMessageTime(params);
    var firstSave = timestamp === 0;
    var earliestTime = params.sentTime < timestamp && +params.sentTime !== 0;
    if (earliestTime || firstSave) {
        timestamp = params.sentTime;
    }
    var key = 'removed_earliest_message_' + params.conversationType + '_' + params.targetId;
    store.set(key, timestamp);
}
Message.saveRemovedEarliestMessageTime = saveRemovedEarliestMessageTime;
/*
说明：修改消息体 Web SDK 与 C++ SDK 定义消息结构不一致业务逻辑层不容易处理这里统一数据格式
      1. PublicServiceRichContentMessage 公众号单图文
      2. PublicServiceMultiRichContentMessage 公众号多图文
*/
function hackWebSDKMessage(messageList) {
    if (!$.isArray(messageList)) {
        messageList = [messageList];
    }
    messageList.forEach(function (message) {
        var content = message.content;
        if (message.messageType === 'PublicServiceRichContentMessage') {
            if (!utils.isEmpty(content.richContentMessage)) {
                content.articles = content.richContentMessage.articles || [];
            }
        }
        if (message.messageType === 'PublicServiceMultiRichContentMessage') {
            if (!utils.isEmpty(content.richContentMessages)) {
                content.articles = content.richContentMessages.articles || [];
            }
        }
    });
}

Message.hackWebSDKMessage = hackWebSDKMessage;

Message.addSendUserInfo = function (message, callback) {
    callback = callback || $.noop;
    var userId = message.senderUserId;
    if ($.isArray(message)) {
        var userList = message.map(function (item) {
            return item.senderUserId;
        });
        userApi.getUsers(userList, function () {
            message.forEach(function (item) {
                item.user = Cache.user[item.senderUserId];
            });
            callback(null, message);
        });
    } else {
        userApi.getUsers([userId], function (errorCode, list) {
            if (errorCode) {
                callback(errorCode);
                return;
            }
            message.user = list[0];
            callback(null, message);
        });
    }
};

/*
params.position 1 从缓存获取 2 从服务器获取
params.timestamp
params.count
params.conversationType
params.targetId
params.type 要获取的消息类型  **仅本地消息支持
params.before true:获取比指定时间戳早发的消息,false: 获取比指定时间戳晚发的消息 **仅本地消息支持 false
*/
Message.get = function (params, callback, imageHistory) {
    callback = callback || $.noop;
    var key = params.conversationType + '_' + params.targetId;
    var cacheList = Message._cache[key] = Message._cache[key] || [];

    if (+params.position === 2 || cacheList.length === 0) {
        params.timestamp = Number(params.timestamp) || 0;
        params.count = params.count || config.dataModel.getHistoryMessagesNumber;
        var isFirstGetHistory = cacheList.length === 0;
        var undef;
        if (params.before === undef) {
            params.before = true;
        }
        if (imageHistory) {
            getImageHistoryMessage(params, function (errorCode, messageList, hasMore) {
                if (errorCode) {
                    callback(errorCode);
                    return;
                }
                // C++ 本地获取不到，需要从服务端获取
                var notSearchMessage = !params.type && params.before;
                if (messageList.length < params.count && notSearchMessage) {
                    params.count -= messageList.length;
                    // SDK 和服务器不允许获取一条历史消息
                    var count = params.count;
                    if (params.count === 1) {
                        params.count = 2;
                    }
                    var earliestMessage = messageList[0];
                    if (earliestMessage) {
                        params.timestamp = earliestMessage.sentTime;
                    }
                    getRemoteHistoryMessages(params, function (error, remoteMessageList, remoteHasMore) {
                        remoteMessageList = remoteMessageList.slice(-count);
                        remoteMessageList = remoteMessageList.concat(messageList);
                        if (cacheList.length === 0 && +params.position === 1) {
                            Message._cache[key] = remoteMessageList;
                        }
                        callback(error, remoteMessageList, remoteHasMore);
                    });
                } else {
                    if (cacheList.length === 0 && +params.position === 1) {
                        Message._cache[key] = messageList;
                    }
                    callback(errorCode, messageList, hasMore);
                }
            });
        } else {
            getHistoryMessages(params, function (errorCode, messageList, hasMore) {
                if (errorCode) {
                    callback(errorCode);
                    return;
                }
                console.log(1111,messageList);
                // C++ 本地获取不到，需要从服务端获取
                var notSearchMessage = !params.type && params.before;
                if (messageList.length < params.count && notSearchMessage) {
                    params.count -= messageList.length;
                    // SDK 和服务器不允许获取一条历史消息
                    var count = params.count;
                    if (params.count === 1) {
                        params.count = 2;
                    }
                    var earliestMessage = messageList[0];
                    if (earliestMessage) {
                        params.timestamp = earliestMessage.sentTime;
                    }
                    var timestamp = getRemovedEarliestMessageTime(params);
                    var isBefore = timestamp < params.timestamp;
                    var isInitialValue = timestamp === 0;
                    var isFirstGet = params.timestamp === 0;
                    if ((isBefore && !isInitialValue) || isFirstGet) {
                        params.timestamp = timestamp;
                    }
                    getRemoteHistoryMessages(params, function (error, remoteMessageList, remoteHasMore) {
                        remoteMessageList = remoteMessageList.slice(-count);
                        remoteMessageList = remoteMessageList.concat(messageList);
                        if (cacheList.length === 0 && +params.position === 1) {
                            Message._cache[key] = remoteMessageList;
                        }
                        // 首次进入会话获取历史消息过程中收到新消息返回新消息不使用历史消息
                        if (isFirstGetHistory && cacheList.length > 0) {
                            callback(error, cacheList, true);
                        } else {
                            callback(error, remoteMessageList, remoteHasMore);
                        }
                    });
                } else {
                    if (cacheList.length === 0 && +params.position === 1) {
                        Message._cache[key] = messageList;
                    }
                    // 首次进入会话获取历史消息过程中收到新消息返回新消息不使用历史消息
                    if (isFirstGetHistory && cacheList.length > 0) {
                        callback(errorCode, cacheList, true);
                    } else {
                        callback(errorCode, messageList, hasMore);
                    }
                }
            });
        }
    } else {
        if (cacheList.length > 50) {
            var length = cacheList.length - 50;
            cacheList.splice(0, length);
        }
        // 更新用户信息
        Message.addSendUserInfo(cacheList);
        callback(null, cacheList, true);
    }
};

/**
@param {object}      params
@param {number}      params.conversationType - 会话类型
@param {string}      params.targetId         - 会话Id
@param {number|null} params.timestamp        - 起始时间戳
@param {number}      params.count            - 获取消息条数
@param {string}      params.type             - 获取消息类型
@param {boolean}     params.before           - true: 获取比指定时间戳早的消息，false：获取比指定时间戳晚的消息

@param {function}    callback                - 回调函数
*/
function getHistoryMessages(params, callback) {
    var objectName = '';
    if (params.type) {
        objectName = RongIMClient.MessageParams[params.type].objectName;
    }
    RongIMClient.getInstance().getHistoryMessages(params.conversationType, params.targetId, params.timestamp, params.count, {
        onSuccess: function (list, hasMore) {
            hackWebSDKMessage(list);
            for (var i = 0, len = list.length; i < len; i += 1) {
                bindResponseToMessage(list[i]);
                if (list[i].messageType === RongIMClient.MessageType.LocalFileMessage) {
                    list[i].progress = list[i].progress || 0;
                    list[i].uploadStatus = list[i].uploadStatus || '';
                }
            }
            Message.addSendUserInfo(list, function (errorCode, messageList) {
                callback(null, messageList, hasMore);
            });
        },
        onError: function (errorCode) {
            callback(getLibErrorCode(errorCode));
        }
    }, objectName, params.before);
}

/**
@param {object}      params
@param {number}      params.conversationType - 会话类型
@param {string}      params.targetId         - 会话Id
@param {number|null} params.timestamp        - 起始时间戳
@param {number}      params.count            - 获取消息条数
@param {string}      params.type             - 获取消息类型
@param {boolean}     params.before           - true: 获取比指定时间戳早的消息，false：获取比指定时间戳晚的消息

@param {function}    callback                - 回调函数
*/
function getImageHistoryMessage(params, callback) {
    var objectName;
    if (params.type === 'ImageMessage') {
        objectName = ['RC:ImgMsg', 'RC:SightMsg'];
    } else {
        objectName = 'RC:ImgMsg';
    }
    getHistoryMessagesByObjectNames(params.conversationType, params.targetId, params.timestamp, params.count, {
        onSuccess: function (list, hasMore) {
            for (var i = 0, len = list.length; i < len; i += 1) {
                bindResponseToMessage(list[i]);
            }
            Message.addSendUserInfo(list, function (errorCode, messageList) {
                callback(null, messageList, hasMore);
            });
        },
        onError: function (errorCode) {
            callback(getLibErrorCode(errorCode));
        }
    }, objectName, params.before);
}

/**
方法： 获取只有 图片和小视频 的历史消息数据
参数：
    @param {number}      conversationType - 会话类型
    @param {string}      targetId         - 会话Id
    @param {number|null} timestamp        - 起始时间戳
    @param {number}      count            - 获取消息条数
    @param {function}    callback         - 回调函数
    @param {array}       objectName       - 获取消息类型
    @param {boolean}     direction        - true: 获取比指定时间戳早的消息，false：获取比指定时间戳晚的消息
*/
function getHistoryMessagesByObjectNames(conversationType, targetId, timestamp, count, callback, objectName, direction) {
    objectName = objectName || '';
    direction = typeof direction === 'undefined' || direction;
    var provider = RongIM.lib.getDataProvider();
    try {
        var ret = provider.getHistoryMessagesByObjectNames(conversationType, targetId, timestamp || 0, count, objectName, direction);
        var list = ret ? JSON.parse(ret).list : [];
        var msgs = [];
        list.reverse();
        var len = list.length;
        for (var i = 0; i < len; i += 1) {
            var message = buildMessage(list[i].obj);
            // message.sentStatus = RongIMLib.SentStatus.READ;
            msgs[i] = message;
        }
        callback.onSuccess(msgs, len === count);
    } catch (e) {
        callback.onError(RongIMLib.ErrorCode.TIMEOUT);
    }
}

/**
方法： 将获取到的历史消息数据转换成需要的数据格式
参数：
    @param {object}      result     - 消息数据
*/
function buildMessage(result) {
    var typeMapping = {
        'RC:ImgMsg': 'ImageMessage',
        'RC:SightMsg': 'SightMessage'
    };
    var message = new RongIMLib.Message();
    var ret = JSON.parse(result);
    message.conversationType = ret.conversationType;
    message.targetId = ret.targetId;
    message.senderUserId = ret.senderUserId;
    var msgDirection = +ret.direction;
    message.messageDirection = msgDirection;
    if (msgDirection === RongIMLib.MessageDirection.RECEIVE) {
        message.receivedStatus = ret.status;
    } else if (msgDirection === RongIMLib.MessageDirection.SEND) {
        message.sentStatus = ret.status;
    }
    message.sentTime = ret.sentTime;
    message.objectName = ret.objectName;
    var content = ret.content ? JSON.parse(ret.content) : ret.content;
    var messageType = typeMapping[ret.objectName];
    if (content) {
        content.messageName = messageType;
    }
    message.content = content;
    message.messageId = ret.messageId;
    message.messageUId = ret.messageUid;
    message.messageType = messageType;
    return message;
}

/**
@param {object}      params
@param {number}      params.conversationType - 会话类型
@param {string}      params.targetId         - 会话Id
@param {number|null} params.timestamp        - 起始时间戳
@param {number}      params.count            - 获取消息条数

@param {function}    callback                - 回调函数
*/
function getRemoteHistoryMessages(params, callback) {
    RongIMClient.getInstance().getRemoteHistoryMessages(params.conversationType, params.targetId, params.timestamp, params.count, {
        onSuccess: function (list, hasMore) {
            hackWebSDKMessage(list);
            for (var i = 0, len = list.length; i < len; i += 1) {
                bindResponseToMessage(list[i]);
            }
            Message.addSendUserInfo(list, function (errorCode, messageList) {
                callback(null, messageList, hasMore);
            });
        },
        onError: function (errorCode) {
            callback(getLibErrorCode(errorCode));
        }
    });
}

/*
conversationType
targetId
messageUId
*/
Message.getMessageNearList = function (idOrUId, callback) {
    callback = callback || $.noop;
    Message.getOne(idOrUId, function (errorCode, message) {
        if (errorCode) {
            callback(getLibErrorCode(errorCode));
            return;
        }
        var targetId = message.targetId;
        var conversationType = message.conversationType;
        var timestamp = message.sentTime;
        var params = {
            targetId: targetId,
            conversationType: conversationType,
            timestamp: timestamp,
            position: 2
        };
        messageNearList(params, message, callback);
    });
};
function messageNearList(params, message, callback) {
    Message.get(params, function (errorCode, messageList) {
        if (errorCode) {
            callback(getLibErrorCode(errorCode));
            return;
        }
        Message.addSendUserInfo([message], function (error, msgList) {
            if (error) {
                callback(error);
                return;
            }
            var msg = msgList[0];
            messageList.push(msg);
            params.before = false;
            Message.get(params, function (err, list) {
                if (err) {
                    callback(getLibErrorCode(err));
                    return;
                }
                list.reverse();
                messageList = messageList.concat(list);
                callback(null, messageList, message, msg);
            });
        });
    });
}

Message.getOne = function (idOrUId, callback) {
    callback = callback || $.noop;
    RongIMClient.getInstance().getMessage(idOrUId, {
        onSuccess: function (message) {
            callback(null, message);
        },
        onError: function (errorCode) {
            callback(getLibErrorCode(errorCode));
        }
    });
};

// Message.getAllImageMessage = function (messageList, callback) {
//     var platform = RongIM.utils.getPlatform();
//     if (platform.startsWith('web')) {
//         return messageList;
//     }
//     messageList = messageList || [];
//     var firstMessage = messageList.length > 0 ? messageList[0] : {};
//     var params = {
//         conversationType: firstMessage.conversationType,
//         targetId: firstMessage.targetId,
//         position: 2,
//         timestamp: firstMessage.sentTime,
//         type: 'ImageMessage',
//         count: 20
//     };
//     Message.get(params, function (errorCode, list, hasMore) {
//         messageList = list.concat(messageList);
//         if (hasMore) {
//             Message.getAllImageMessage(messageList, callback);
//         } else {
//             callback(messageList);
//         }
//     });
// };

var getLocalMessage = function (params) {
    var msg = new RongIMLib.Message();
    msg.content = params.content;
    msg.conversationType = params.conversationType;
    msg.targetId = params.targetId;
    msg.senderUserId = Cache.auth.id;
    msg.sentStatus = RongIMLib.SentStatus.SENDING;
    msg.messageId = params.messageId;
    msg.messageDirection = RongIMLib.MessageDirection.SEND;
    msg.messageType = params.content.messageName;
    msg.sentTime = getServerTime();
    msg.receivedStatus = RongIMLib.ReceivedStatus.UNREAD;
    return msg;
};

function sendMessage(params, callback) {
    callback = callback || $.noop;
    // 发送消息前将 localPath 删除防止发给其他端
    var content = $.extend({}, params.content);
    delete content.localPath;
    RongIMClient.getInstance().sendMessage(+params.conversationType, params.targetId, content, {
        onBefore: function (messageId) {
            params.messageId = messageId;
            var message = getLocalMessage(params);
            callback(null, message);
            // Conversation.getList(function (errorCode, list) {
            //     if (errorCode) {
            //         return;
            //     }
            //     converObserverList.notify(list);
            // });
            var convesationCallback = function () {
                var list = conversationApi.getLocalList();
                conversationApi.splitTopConversation(list, function (sortList) {
                    conversationApi.observerList.notify(sortList);
                });
            };
            var conversation = conversationApi.getLocalOne(params.conversationType, params.targetId);
            if (conversation) {
                conversation.latestMessage = message;
                conversation.sentTime = message.sentTime;
                convesationCallback();
            } else {
                params.latestMessage = message;
                params.sentTime = message.sentTime;
                conversationApi.add(params, function () {
                    convesationCallback();
                });
            }
        },
        onSuccess: function (message) {
            fixSendMessageBug(message);
            callback(null, message);
            var conversation = conversationApi.getLocalOne(params.conversationType, params.targetId);
            conversation.latestMessage = message;
            var list = conversationApi.getLocalList();
            conversationApi.observerList.notify(list);
        },
        onError: function (errorCode, message) {
            fixSendMessageBug(message);
            message.sentTime = message.sentTime || new Date().getTime();
            // fix: rcx 为客户定制敏感词返回对应 code 码 rce 不需要提示，这里做特殊处理
            // 敏感词替换、屏蔽，显示发送成功
            var arr = [
                common.ErrorCode.SENSITIVE_WORDS_REPLACED,
                common.ErrorCode.SENSITIVE_WORDS_INTERCEPT
            ];
            var isReplaceOrIntercept = arr.indexOf(errorCode) > -1;
            if (isReplaceOrIntercept) {
                message.sentStatus = RongIMLib.SentStatus.SENT;
                errorCode = null;
                Message.setMessageSentStatus({
                    messageId: message.messageId,
                    status: message.sentStatus
                });
            }
            callback(errorCode, message);
            var conversation = conversationApi.getLocalOne(params.conversationType, params.targetId);
            conversation.latestMessage = message;
            var list = conversationApi.getLocalList();
            conversationApi.observerList.notify(list);
        }
    }, params.mentiondMsg);
}

function sendCommandMessage(params) {
    var status = getCurrentConnectionStatus();
    var CONNECTED = 0;
    if (status !== CONNECTED) {
        Cache.messageQueue.push(params);
    }
    RongIMClient.getInstance().sendMessage(params.conversationType, params.targetId, params.content, {
        onSuccess: function () {

        },
        onError: function () {

        }
    });
}
Message.sendCommandMessage = sendCommandMessage;

function fixSendMessageBug(message) {
    if (+message.messageDirection === RongIMLib.MessageDirection.SEND) {
        message.senderUserId = Cache.auth.id;
        // Web SDK 发送消息 receivedStatus 为 undefined
        var undef;
        if (message.receivedStatus === undef) {
            message.receivedStatus = RongIMLib.ReceivedStatus.UNREAD;
        }
    }
    if (!message.sentTime) {
        // c++ error 回调返回 message 没有sentTime
        message.sentTime = getServerTime();
    }
    return message;
}

Message.fixSendMessageBug = fixSendMessageBug;

Message.send = function (params, callback) {
    callback = callback || $.noop;
    var isPrivate = Number(params.conversationType) === 1;
    if (isPrivate) {
        var canNotChat = !userApi.validateCanChat(params.targetId);
        if (canNotChat) {
            friendApi.requestFriendVerification(params);
            return;
        }
    }
    sendMessage(params, function (errorCode, message) {
        if (message.messageType === utils.messageType.FileMessage) {
            message.sentStatus = RongIMLib.SentStatus.SENT;
            message.content.localPath = params.content.localPath;
            Message.setContent(message);
        }
        Message._sendPush(message, function (errorCode2, msg) {
            msgObserverList.notify(msg);
            callback(getLibErrorCode(errorCode), msg);
        });
    });
};

// params.conversationType
// params.targetId
// params.user = {id:'userId', name: 'username', imageUrl:'portaitUri'}
Message.sendCard = function (params, callback) {
    params.content = new RongIMClient.RegisterMessage.CardMessage(params.user);
    Message.send(params, callback);
};

Message.sendSyncReadStatusMessage = function (message) {
    var lastMessageSendTime = message.sentTime;
    var msg = new RongIMLib.SyncReadStatusMessage({ lastMessageSendTime: lastMessageSendTime });

    var params = {
        conversationType: Number(message.conversationType),
        targetId: message.targetId,
        content: msg
    };
    sendCommandMessage(params);
};

Message.sendReadStatus = function (lastMessage) {
    var content = {
        lastMessageSendTime: lastMessage.sentTime,
        messageUId: lastMessage.messageUId,
        type: 1
    };
    var msg = new RongIMLib.ReadReceiptMessage(content);

    var params = {
        conversationType: Number(lastMessage.conversationType),
        targetId: lastMessage.targetId,
        content: msg
    };
    sendCommandMessage(params);
};

Message.setMessageSentStatus = function (params, callback) {
    callback = callback || $.noop;
    RongIMClient.getInstance().setMessageSentStatus(params.messageId, params.status, {
        onSuccess: function (isUpdate) {
            callback(null, isUpdate);
        },
        onError: function (error) {
            callback(error);
        }
    });
};

Message.setMessageReceivedStatus = function (params, callback) {
    callback = callback || $.noop;
    RongIMClient.getInstance().setMessageReceivedStatus(params.messageId, params.status, {
        onSuccess: function () {
            callback(null);
        },
        onError: function (errorCode) {
            callback(errorCode);
        }
    });
};

Message.setMessageStatus = function (message) {
    var status = RongIMLib.SentStatus.READ;
    var key = getCacheKey(message);
    var cache = Message._cache[key];
    var timespan = message.content.lastMessageSendTime;
    if (cache) {
        cache.forEach(function (item) {
            var isSend = +item.messageDirection === RongIMLib.MessageDirection.SEND;
            var isBefore = item.sentTime <= timespan;
            var isFailed = +item.sentStatus === RongIMLib.SentStatus.FAILED;
            // 上传中的文件不改变状态
            var isUploading = item.uploadStatus && item.uploadStatus !== utils.uploadStatus.SUCCESS;
            if (isSend && isBefore && !isFailed && !isUploading) {
                item.sentStatus = status;
            }
        });
    }
    var type = message.conversationType;
    var id = message.targetId;
    var conv = conversationApi.getLocalOne(type, id) || {};
    var msg = conv.latestMessage;
    if (msg) {
        var isSend = +msg.messageDirection === RongIMLib.MessageDirection.SEND;
        var isBefore = msg.sentTime <= timespan;
        var isFailed = +msg.sentStatus === RongIMLib.SentStatus.FAILED;
        if (isSend && isBefore && !isFailed) {
            msg.sentStatus = status;
        }
    }
    RongIMClient.getInstance().setMessageStatus(type, id, timespan, '1', {
        onSuccess: function () {
            if (PullMessageStatus.get()) {
                conversationApi.notifyConversation();
            }
        },
        onError: function () {

        }
    });
};

Message.sendGroupResponse = function (type, id, req) {
    type = Number(type);
    // 注意:更新会话列表
    // 更新本地会话缓存
    var key = getStoreKey('req');
    var conversationKey = generatorKey([type, id]);
    var request = store.get(key) || {};
    delete request[conversationKey];
    if (!$.isEmptyObject(request)) {
        store.set(key, request);
    } else {
        store.remove(key);
    }

    var msg = new RongIMLib.ReadReceiptResponseMessage({ receiptMessageDic: req });
    var params = {
        conversationType: type,
        targetId: id,
        content: msg
    };
    sendCommandMessage(params);
};

Message.sendGroupRequest = function (message) {
    var messageUId = message.messageUId;
    var msg = new RongIMLib.ReadReceiptRequestMessage({ messageUId: messageUId });
    var key = getStoreKey('res_' + messageUId);
    store.set(key, []);

    var params = {
        conversationType: Number(message.conversationType),
        targetId: message.targetId,
        content: msg
    };
    sendCommandMessage(params);
};

/**
 * @params.conversationType
 * @params.targetId
 * @params.sentTime
 * @params.messageUId
 * @params.sendUserId
 * @params.extra
 * @params.user
 */
Message.recall = function (message, callback) {
    callback = callback || $.noop;
    var params = {
        conversationType: message.conversationType,
        targetId: message.targetId,
        messageUId: message.messageUId,
        messageId: message.messageId,
        sentTime: message.sentTime,
        senderUserId: message.senderUserId
    };
    Message.setRecallContent(message.messageId, message);
    if (message.user && message.user.name) {
        // 消息撤回添加推送消息
        var locale = RongIM.instance.locale;
        params.push = utils.templateFormat(locale.message.recallOther, message.user.name);
    }
    RongIMClient.getInstance().sendRecallMessage(params, {
        onSuccess: function (msg) {
            utils.console.log('消息撤回：', msg);
            msg.sentTime = message.sentTime;
            msgObserverList.notify(msg);
            callback(null, msg);
            Message.addSendUserInfo(msg, function (errorCode, msgInfo) {
                if (errorCode) {
                    callback(errorCode);
                    return;
                }
                var key = getCacheKey(message);
                var list = Message._cache[key];
                msgInfo.messageId = message.messageId;
                msgInfo.sentTime = message.sentTime;
                spliceMessage(list, message.messageId, msgInfo);
                var objectName = RongIMClient.MessageParams[msgInfo.messageType].objectName;
                RongIMClient.getInstance().setMessageContent(message.messageId, params, objectName);
                var lconv = conversationApi.getLocalOne(message.conversationType, message.targetId);
                if (message.messageId === lconv.latestMessage.messageId || message.messageUId === lconv.latestMessage.messageUId) {
                    lconv.latestMessage = msgInfo;
                }
                var lconvList = conversationApi.getLocalList();
                conversationApi.observerList.notify(lconvList);
            });
        },
        onError: function (code) {
            callback(code);
        }
    });
};

Message.setRecallContent = function (messageId, msg) {
    if (msg && msg.messageType === 'TextMessage') {
        var mentionedInfo = msg.content.mentionedInfo;
        var atIdList = [];
        if (mentionedInfo) {
            // type==1 为 @all
            if (mentionedInfo.type === 1) {
                atIdList = [0];
            } else {
                atIdList = [].concat(mentionedInfo.userIdList);
            }
        }
        Cache.messageRecallEdit[messageId] = {
            recallTime: Date.now(),
            text: msg.content.content,
            atIdList: atIdList
        };
    }
};
Message.getRecallContent = function (messageId) {
    return Cache.messageRecallEdit[messageId];
};

/**
 * @params.conversationType
 * @params.targetId
 * @params.keyword
 * @params.timestamp ： 时间戳，默认 0
 * @params.count : 0-20
 * @params.total ：是否返回总数
 */
Message.search = function (params, callback) {
    callback = callback || $.noop;
    var defer = $.Deferred();
    var _instance = RongIMClient.getInstance();
    var isGetTotle = 1;
    _instance.searchMessageByContent(params.conversationType, params.targetId, params.keyword, params.timestamp, params.count, isGetTotle, {
        onSuccess: function (msgList, matched) {
            hackWebSDKMessage(msgList);
            Message.addSendUserInfo(msgList, function (errorCode, messageList) {
                callback(errorCode, messageList, matched);
                defer.resolve({ list: messageList, count: matched });
            });
        },
        onError: function (code) {
            callback(code);
        }
    });
    return defer.promise();
};

/**
 * @params.conversationType
 * @params.targetId
 * @params.messageIds // messageId 的数据，不是 messageUId
 * @params.notNotify  不更新会话列表 (在消息发送失败重新发送时先删除再重新发送，删除的更新-延后通知导致最后一条消息为上一条消息)
 */
Message.removeLocal = function (params, callback) {
    callback = callback || $.noop;
    // saveRemovedEarliestMessageTime(params);
    RongIMClient.getInstance().deleteLocalMessages(params.conversationType, params.targetId, params.messageIds, {
        onSuccess: function (isRemove) {
            callback(null, isRemove);
            var key = getCacheKey(params);
            var list = Message._cache[key];
            params.messageIds.forEach(function (id) {
                spliceMessage(list, id);
            });
            if (!params.notNotify) {
                // conversationApi.getList(function (errorCode, list) {
                //     if (errorCode) {
                //         return callback(getLibErrorCode(errorCode));
                //     }
                //     conversationApi.observerList.notify(list);
                // });
                conversationApi.observerList.notify();
            }
        },
        onError: function (code) {
            callback(code);
        }
    });
};

/**
 * @params.conversationType
 * @params.targetId
 * @params.messageIds // messageId 的数据，不是 messageUId
 */
Message.clearMessages = function (params, callback) {
    callback = callback || $.noop;
    RongIMClient.getInstance().clearMessages(params.conversationType, params.targetId, {
        onSuccess: function (isClear) {
            callback(null, isClear);
            var key = getCacheKey(params);
            var list = Message._cache[key];
            if (list) list.length = 0;
        },
        onError: function (code) {
            callback(code);
        }
    });
};

/*
    向本地插入一条 Message

    params.conversationType
    params.targetId
    params.objectName
    params.content
    params.senderUserId
    params.direction
    params.sentStatus

    // 举例：取消文件消息内容
    params.content = {
        name:'core.js',
        type:'js',
        status: 0 // 0 取消
    }
*/
Message.insertMessage = function (params, callback) {
    callback = callback || $.noop;
    if (utils.isEmpty(params.senderUserId)) {
        params.senderUserId = Cache.auth.id;
    }
    if (utils.isEmpty(params.objectName)) {
        params.objectName = RongIMClient.MessageParams[params.messageType].objectName;
    }
    if (isNaN(params.direction)) {
        params.direction = RongIMLib.MessageDirection.SEND;
    }
    if (isNaN(params.sentStatus)) {
        params.sentStatus = RongIMLib.SentStatus.FAILED;
    }
    if (isNaN(params.receivedStatus)) {
        params.receivedStatus = RongIMLib.ReceivedStatus.READ;
    }
    RongIMClient.getInstance().insertMessage(+params.conversationType, params.targetId, params, {
        onSuccess: function (message) {
            var arg = {
                messageId: message.messageId
            };
            // patch web 端无 insertMessage 返回 message 中无 messageId messageDirection
            message.messageId = message.messageId || ('insert' + RongIM.utils.createUid());
            message.messageDirection = params.direction;
            if (message.messageDirection === RongIMLib.MessageDirection.SEND) {
                var sentStatus = params.sentStatus;
                arg.status = sentStatus;
                Message.setMessageSentStatus(arg);
            } else {
                var receivedStatus = params.receivedStatus;
                arg.status = receivedStatus;
                Message.setMessageReceivedStatus(arg);
            }
            // 这里返回消息没有 发送时间等重新调用接口获取一遍
            Message.getOne(message.messageId, function (errorCode, msg) {
                if (errorCode) {
                    console.warn('insertMessage', params, msg);
                    callback(errorCode);
                    return;
                }
                if (msg.messageType === RongIMClient.MessageType.LocalFileMessage || msg.messageType === RongIMClient.MessageType.SightMessage) {
                    msg.progress = params.progress || 0;
                    msg.uploadStatus = params.uploadStatus || RongIM.utils.uploadStatus.CANCELLED;
                    msg.content.status = 1;
                }
                // patch web 端无 getOne 方法(RongIMClient.getInstance().getMessage) 返回 msg 为空
                msg = $.extend({}, message, msg);
                Message._push(msg);
                callback(null, msg);
                // conversationApi.getList(function (errorCode, list) {
                //     if (errorCode === null) {
                //         conversationApi.observerList.notify(list);
                //     }
                // });
                conversationApi.observerList.notify();
            });
            // 插入消息 可能会创建新的会话需要重新调用接口获取不可直接使用内存
            // var conversation = Conversation.getLocalOne(params.conversationType, params.targetId);
            // if (conversation) {
            //     conversation.latestMessage = message;
            //     if (message.receivedStatus === RongIMLib.ReceivedStatus.UNREAD) {
            //         conversation.unreadMessageCount += 1;
            //     }
            //     var list = Conversation.getLocalList();
            //     converObserverList.notify(list);
            // }
        },
        onError: function (error) {
            callback(error);
        }
    });
};

/*
    messageId: 要替换的原消息Id
    content: 要替换的消息体
    objectName: 修改消息类型为
*/
Message.setContent = function (message) {
    var objectName = message.objectName || '';
    var messageId = Number(message.messageId);
    var content = message.content;
    RongIMClient.getInstance().setMessageContent(messageId, content, objectName);
};

Message.watch = function (listener) {
    msgObserverList.add(listener);
};

Message.unwatch = function (listener) {
    msgObserverList.remove(listener);
};
var offlineMsgfinishObserverList = new ObserverList();
Message.watchOfflineReceivefinish = function (listener) {
    offlineMsgfinishObserverList.add(listener);
};
Message.unwatchOfflineReceivefinish = function (listener) {
    offlineMsgfinishObserverList.remove(listener);
};

Message.registerMessage = function () {
    var messageName = 'PresenceNotificationMessage';
    var objectName = 'RCE:PresNtf';
    var messageTag = new RongIMLib.MessageTag(false, false);
    // windows, OSX, Web, iOS, Android
    /*
    type : 默认为 0, 0 表示在线状态，没有其他值，待后续扩展
    targetId： 被订阅人 Id
    title: Login_Status_PC 或 Login_Status_Mobile 或 Login_Status_Web
    value: online 或者 offline
    updateDt: 更新时间
    */
    var properties = ['type', 'targetId', 'title', 'value', 'updateDt'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    // UpdateStatus
    messageName = 'RCEUpdateStatusMessage';
    objectName = 'RCE:UpdateStatus';
    messageTag = new RongIMLib.MessageTag(false, false);
    properties = ['uid', 'updateType', 'version'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    // 帐号被禁用
    messageName = 'InactiveCommandMessage';
    objectName = 'RCE:InactiveCmd';
    messageTag = new RongIMLib.MessageTag(false, false);
    properties = ['userId'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    // CardMessage
    messageName = 'CardMessage';
    objectName = 'RC:CardMsg';
    messageTag = new RongIMLib.MessageTag(true, true);
    properties = ['userId', 'name', 'portraitUri', 'sendUserId', 'sendUserName', 'extra', 'type'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    // LocalFileMessage  uId唯一标识,用于文件上传的标识
    messageName = 'LocalFileMessage';
    objectName = 'LRC:fileMsg';
    messageTag = new RongIMLib.MessageTag(true, true);
    properties = ['type', 'name', 'localPath', 'status', 'uploadId'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    // LocalImageMessage
    messageName = 'LocalImageMessage';
    objectName = 'LRC:imageMsg';
    messageTag = new RongIMLib.MessageTag(true, true);
    properties = ['localPath', 'status', 'content', 'imageUri'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    // 小视频
    messageName = 'SightMessage';
    objectName = 'RC:SightMsg';
    messageTag = new RongIMLib.MessageTag(true, true);
    properties = ['content', 'sightUrl', 'duration', 'localPath', 'name', 'size'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    // 位置共享
    messageName = 'RealTimeLocationStartMessage';
    objectName = 'RC:RLStart';
    messageTag = new RongIMLib.MessageTag(false, true);
    properties = ['content', 'extra'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    messageName = 'KickoffMsg';
    objectName = 'RCE:KickoffMsg';
    messageTag = new RongIMLib.MessageTag(false, false);
    properties = ['content'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    messageName = 'RealTimeLocationQuitMessage';
    objectName = 'RC:RLQuit';
    messageTag = new RongIMLib.MessageTag(false, false);
    properties = ['content', 'extra'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    messageName = 'RealTimeLocationJoinMessage';
    objectName = 'RC:RLJoin';
    messageTag = new RongIMLib.MessageTag(false, false);
    properties = ['content', 'extra'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    messageName = 'RealTimeLocationStatusMessage';
    objectName = 'RC:RL';
    messageTag = new RongIMLib.MessageTag(false, false);
    properties = ['latitude', 'longitude'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    // 音视频会话总结 SummaryMessage 插入有问题新自定义了一种消息
    messageName = 'VideoSummaryMessage';
    objectName = 'RC:VideoSummary';
    messageTag = new RongIMLib.MessageTag(true, true);
    properties = ['caller', 'inviter', 'mediaType', 'memberIdList', 'startTime', 'duration', 'status'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    // 会议状态更新消息
    messageName = 'ConferenceUpdateMessage';
    objectName = 'RCE:CnfStatus';
    messageTag = new RongIMLib.MessageTag(false, false);
    properties = ['type', 'call_id', 'end_reason', 'initiator', 'participantIds', 'participantNumbers', 'summaryGroupId', 'summaryGroupName'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    // 会议参与者状态更新消息
    messageName = 'ConferenceParticipantUpdateMessage';
    objectName = 'RCE:CnfPartStatus';
    messageTag = new RongIMLib.MessageTag(false, false);
    properties = ['status', 'call_id', 'participantId', 'participantNumber'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    // 会议ping消息
    messageName = 'ConferencePingMessage';
    objectName = 'RCE:CnfPing';
    messageTag = new RongIMLib.MessageTag(false, false);
    properties = ['call_id', 'participantIds', 'participantNumbers'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    // 审批通知
    // "applyType": 1.外出审批
    // "status": 1.审批中；2.审批通过；3.审批拒绝
    messageName = 'ApprovalMessage';
    objectName = 'RCE:ApprovalMsg';
    messageTag = new RongIMLib.MessageTag(true, true);
    properties = ['title', 'url', 'userId', 'content', 'applyType', 'startTime', 'endTime', 'status', 'extra'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    // 朋友圈 有人评论或点赞
    messageName = 'MomentsCommentMessage';
    objectName = 'RC:MomentsComment';
    messageTag = new RongIMLib.MessageTag(true, true);
    properties = ['content'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    // 朋友圈 有人提到你
    messageName = 'MomentsMentionMessage';
    objectName = 'RC:MomentsMention';
    messageTag = new RongIMLib.MessageTag(true, true);
    properties = ['content'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    // 朋友圈 有更新
    messageName = 'MomentsUpdateMessage';
    objectName = 'RC:MomentsUpdate';
    messageTag = new RongIMLib.MessageTag(true, true);
    properties = ['content'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    // 锁屏消息
    messageName = 'MultiClientMessage';
    objectName = 'RCE:MultiClient';
    messageTag = new RongIMLib.MessageTag(false, false);
    properties = ['action', 'platform'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    // 设备锁定消息
    messageName = 'DeviMonitorMessage';
    objectName = 'RCE:DeviMonitor';
    messageTag = new RongIMLib.MessageTag(false, false);
    properties = ['action', 'device_id'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    // 公众号点击菜单消息
    messageName = 'ClickMenuMessage';
    objectName = 'RC:PSCmd';
    messageTag = new RongIMLib.MessageTag(false, false);
    properties = ['cmd', 'id', 'type', 'name', 'data'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    // 公众号关闭消息
    messageName = 'AppNotifyMessage';
    objectName = 'RCE:AppNotify';
    messageTag = new RongIMLib.MessageTag(false, false);
    properties = ['id', 'cmd_type', 'content'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    // 引用消息
    messageName = 'ReferenceMessage';
    objectName = 'RCE:ReferenceMsg';
    messageTag = new RongIMLib.MessageTag(true, true);
    properties = ['content', 'objName', 'text', 'userId'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    // 外部联系人转内部员工通知消息
    messageName = 'UserTypeChangedMessage';
    objectName = 'RCE:UserTypeChanged';
    messageTag = new RongIMLib.MessageTag(false, false);
    properties = ['company_name'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    // 密聊消息
    messageName = 'EncryptedMessage';
    objectName = 'RC:EncryptedMsg';
    messageTag = new RongIMLib.MessageTag(false, false);
    properties = ['message'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    messageName = 'EncryptRequestMessage';
    objectName = 'RC:EncryptRequestMsg';
    messageTag = new RongIMLib.MessageTag(false, false);
    properties = ['message'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    messageName = 'EncryptResponseMessage';
    objectName = 'RC:EncryptResponseMsg';
    messageTag = new RongIMLib.MessageTag(false, false);
    properties = ['message'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    messageName = 'EncryptConfirmMessage';
    objectName = 'RC:EncryptConfirmMsg';
    messageTag = new RongIMLib.MessageTag(false, false);
    properties = ['message'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    messageName = 'EncryptCancelMessage';
    objectName = 'RC:EncryptCancelMsg';
    messageTag = new RongIMLib.MessageTag(false, false);
    properties = ['message'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    messageName = 'EncryptTerminateMessage';
    objectName = 'RC:EncryptTerminateMsg';
    messageTag = new RongIMLib.MessageTag(false, false);
    properties = ['message'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    messageName = 'SCBurnTimeMessage';
    objectName = 'RCE:SCBurnTimeMsg';
    messageTag = new RongIMLib.MessageTag(false, false);
    properties = ['message'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    messageName = 'BurnNoticeMessage';
    objectName = 'RC:BurnNoticeMsg';
    messageTag = new RongIMLib.MessageTag(false, false);
    properties = ['message'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);
    // 密聊消息 End

    // 扩展消息
    messageName = 'ExtendMessage';
    objectName = 'RCE:ExtendMessage';
    messageTag = new RongIMLib.MessageTag(true, true);
    properties = ['content'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    // 红包消息
    // messageName = 'JrmfRedPacketMessage';
    // objectName = 'RCJrmf:RpMsg';
    // messageTag = new RongIMLib.MessageTag(true, true);
    // /*
    // content         消息文本内容
    // Bribery_ID      红包id
    // Bribery_Name    红包名称
    // Bribery_Message 红包描述
    // */
    // properties = ['content', 'Bribery_ID', 'Bribery_Name', 'Bribery_Message'];
    // RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    // // 打开红包消息
    // messageName = 'JrmfRedPacketOpenedMessage';
    // objectName = 'RCJrmf:RpOpendMsg';
    // messageTag = new RongIMLib.MessageTag(true, true);
    // /*
    // sendPacketId    红包发送者id
    // sendNickName    发送者昵称
    // openPacketId    打开红包者id
    // packetId        红包id
    // isGetDone       是否为最后一个红包 1 是最后一个，0不是
    // openNickName    打开红包者昵称
    // */
    // properties = ['sendPacketId', 'sendNickName', 'openPacketId', 'packetId', 'isGetDone', 'openNickName'];
    // RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    // // 公众号图文消息
    // messageName = 'PSImageTextMessage';
    // objectName = 'RC:PSImgTxtMsg';
    // messageTag = new RongIMLib.MessageTag(true, true);
    // properties = ['articles'];
    // RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    // // 公众号多图文消息
    // messageName = 'PSMultiImageTextMessage';
    // objectName = 'RC:PSMultiImgTxtMsg';
    // messageTag = new RongIMLib.MessageTag(true, true);
    // properties = ['articles'];
    // RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);
    apiList.forEach(function (api) {
        if (typeof api.registerMessage === 'function') {
            api.registerMessage();
        }
    });
};

function bindResponseToMessage(message, list) {
    if (list) {
        var userId = Cache.auth.id;
        var messageUIds = message.content.receiptMessageDic[userId] || [];
        for (var i = 0, len = list.length; i < len; i += 1) {
            var cacheMessage = list[i];
            if (messageUIds.indexOf(cacheMessage.messageUId) !== -1) {
                var receiptArr = cacheMessage.receiptResponse = cacheMessage.receiptResponse || [];
                if (receiptArr.indexOf(message.senderUserId) === -1) {
                    receiptArr.push(message.senderUserId);
                }
            }
        }

        messageUIds.forEach(function (uid) {
            var key = getStoreKey('res_' + uid);
            var receipt = store.get(key) || [];
            if (receipt.indexOf(message.senderUserId) === -1) {
                receipt.push(message.senderUserId);
                store.set(key, receipt);
            }
        });
    } else {
        var reskey = getStoreKey('res_' + message.messageUId);
        message.receiptResponse = store.get(reskey);
    }
}

Message.getTypes = function () {
    var type = RongIMClient.MessageType;
    var msgTypes = [
        type.TextMessage,
        type.ImageMessage,
        type.FileMessage,
        type.VoiceMessage,
        type.CardMessage,
        type.LocationMessage,
        type.SightMessage
    ];
    return msgTypes;
};
/*
    params.messageType
    params.content
 */
Message.create = function (params) {
    var messageType = params.messageType;
    var content = params.content;

    var TMessage = RongIMLib[messageType] || RongIMClient.RegisterMessage[messageType];

    if (!TMessage) {
        throw new Error('unknown message type! params => ' + JSON.stringify(params));
    }
    return new TMessage(content);
};

Message.addForwardFaildMessage = function (message) {
    var im = RongIM.instance;
    var loginUser = im.loginUser;
    var groupId = message.targetId;
    var group = Cache.group[groupId];
    var adminId = group.admin_id;
    var admin = Cache.user[adminId];
    var conversationList = Cache.conversation.list || [];
    var messageType = 'GroupNotifyMessage';
    var params = {
        targetId: message.targetId,
        conversationType: message.conversationType,
        objectName: 'RCE:GrpNtfy',
        messageType: messageType,
        senderUserId: message.senderUserId,
        sentStatus: RongIMLib.SentStatus.SENT,
        content: {
            action: 23,
            operatorUser: {
                id: adminId,
                name: admin.name
            },
            targetGroup: {
                id: message.targetId,
                name: group.name
            },
            targetUsers: [{
                id: loginUser.id,
                name: loginUser.name
            }],
            messageName: messageType
        }
    };
    Message.insertMessage(params, function (err, msg) {
        if (!err) {
            conversationList.forEach(function (item) {
                if (item.targetId === msg.targetId) {
                    item.latestMessage = msg;
                }
            });
        }
    });
};

function clearUnreadCountByTimestamp(conversationType, targetId, timestamp, callback) {
    callback = callback || $.noop;
    if (!RongIM.isAvailableData()) {
        var status = getCurrentConnectionStatus();
        var errorCode = 'status-' + status;
        callback(errorCode);
        return;
    }
    var params = {
        conversationType: conversationType,
        targetId: targetId,
        timestamp: timestamp
    };
    RongIM.lib.clearUnreadCountByTimestamp(params, function (error) {
        if (error) {
            callback(getLibErrorCode(error));
            return;
        }
        if (PullMessageStatus.get()) {
            conversationApi.notifyConversation();
        }
        callback();
    });
}

Message.setMessageListener = function () {
    // var notifyConversation = RongIM.utils.throttle(debounceConversation, 600);
    var messageCtrol = {
        // 音视频
        AcceptMessage: function () {
            // utils.console.log('TODO: 此消息需要处理', arguments);
        },
        RingingMessage: function () {
            // utils.console.log('TODO: 此消息需要处理', arguments);
        },
        SummaryMessage: function () {
            // utils.console.log('TODO: 此消息需要处理', arguments);
        },
        HungupMessage: function () {
            $.noop();
        },
        // RCE Server 订阅状态通知,此条消息在应用层，im.js中处理
        PresenceNotificationMessage: function (message) {
            if (message.offLineMessage) {
                return;
            }
            msgObserverList.notify(message);
        },
        InviteMessage: function (message) {
            Message._push(message);
        },
        MediaModifyMessage: function () {
            // utils.console.log('TODO: 此消息需要处理', arguments);
        },
        MemberModifyMessage: function () {
            // utils.console.log('TODO: 此消息需要处理', arguments);
        },
        DeviMonitorMessage: function (message) {
            if (message.offLineMessage) {
                return;
            }
            msgObserverList.notify(message);
        },
        // 资料通知消息
        ProfileNotificationMessage: function (/* message */) {
            // utils.console.log('TODO: 此消息需要处理', message);
        },
        // 命令通知
        CommandMessage: function (/* message */) {
            // utils.console.log('TODO: 此消息需要处理', message);
        },
        // RCE Server
        RCEUpdateStatusMessage: updateStatusMessageHandle,
        InactiveCommandMessage: function (message) {
            if (message.offLineMessage) {
                return;
            }
            accountApi.observerList.notify(10111);
        },
        TypingStatusMessage: function (/* message */) {
            // utils.console.log('TODO: 此消息需要处理', message);
        },
        // 同步已读状态
        SyncReadStatusMessage: function (message) {
            clearUnreadCountByTimestamp(message.conversationType, message.targetId, message.content.lastMessageSendTime, function () {});
        },
        // 撤回消息
        RecallCommandMessage: function (message) {
            var messageUId = message.content.messageUId;
            Message.getOne(messageUId, function (errorCode, msg) {
                if (errorCode) {
                    return;
                }
                if (message.content.isDelete) {
                    Message.removeLocal({
                        targetId: msg.targetId,
                        conversationType: msg.conversationType,
                        messageIds: [msg.messageId]
                    });
                } else {
                    var messageId = msg.messageId || messageUId;
                    var direction = RongIMLib.MessageDirection;
                    var content = message.messageDirection === direction.SEND ? message.content : message;
                    var key = getCacheKey(content);
                    var list = Message._cache[key];
                    if (list) {
                        message.messageId = messageId;
                        message.sentTime = msg.sentTime;
                        spliceMessage(list, messageId, message);
                        // 撤回消息重新编辑不做多端同步 bug 27029
                        // var originMsg = spliceMessage(list, messageId, message);
                        // if (originMsg.messageDirection === direction.SEND) {
                        //     Message.setRecallContent(messageId, originMsg);
                        // }
                    }
                    var objectName = RongIMClient.MessageParams[message.messageType].objectName;
                    RongIMClient.getInstance().setMessageContent(messageId, message.content, objectName);

                    RongIMClient.getInstance().setMessageReceivedStatus(messageId, RongIMLib.ReceivedStatus.READ, {
                        onSuccess: function () {
                        },
                        onError: function (error) {
                            utils.console.log('RecallCommandMessage-setMessageSentStatus', error);
                        }
                    });
                }
                var delMentionedInfo = function (_msg) {
                    var cacheKey = 'rong_mentioneds_' + Cache.auth.id + '_' + _msg.conversationType + '_' + _msg.targetId;
                    var mentioneds = localStorage.getItem(cacheKey);
                    if (mentioneds) {
                        mentioneds = JSON.parse(mentioneds);
                        var contentKey = _msg.conversationType + '_' + _msg.targetId;
                        if (mentioneds[contentKey].uid === _msg.content.messageUId) {
                            localStorage.removeItem(cacheKey);
                            localStorage.removeItem('rong_cu' + Cache.auth.id + _msg.conversationType + _msg.targetId);
                            conversationApi.getOne(_msg.conversationType, _msg.targetId, function (error, conversation) {
                                if (error) {
                                    console.warn(getLibErrorCode(error));
                                    return;
                                }
                                conversation.mentionedMsg = null;
                            });
                        }
                    }
                };
                delMentionedInfo(message);

                var lconv = conversationApi.getLocalOne(message.conversationType, message.targetId);
                var isLastMessage = utils.isEmpty(lconv) || (lconv.lastMessage && lconv.lastMessage.messageId === msg.messageId);
                if (!message.offLineMessage || isLastMessage) {
                    msgObserverList.notify(message);
                    conversationApi.notifyConversation();
                }

                // var lconv = conversationApi.getLocalOne(message.conversationType, message.targetId);
                // if (lconv) {
                //     lconv.unreadMessageCount && lconv.unreadMessageCount --;
                //     lconv.latestMessage = message;
                //     var lconvList = conversationApi.getLocalList();
                //     conversationApi.observerList.notify(lconvList);
                // }
                // web 无法取到消息,暂不做条件判断
                // if(msg.messageType === utils.messageType.ImageMessage){
                if (RongIM.imageViewer) {
                    RongIM.imageViewer.recall(msg.messageUId || message.content.messageUId);
                }
                // }
            });
        },
        // 私聊已读回执
        ReadReceiptMessage: function (message) {
            var isReceive = message.messageDirection === RongIMLib.MessageDirection.RECEIVE;
            if (isReceive) {
                Message.setMessageStatus(message);
            } else {
                conversationApi.clearUnreadCount(message.conversationType, message.targetId, function () {});
            }
        },
        ReadReceiptRequestMessage: function (message) {
            var messageUId = message.content.messageUId;
            if (message.messageDirection === RongIMLib.MessageDirection.SEND) {
                // 多端同步的消息
                var key = getStoreKey('res_' + messageUId);
                store.set(key, []);
                var msg = getCacheMessageByUId(message.conversationType, message.targetId, messageUId);
                if (msg) {
                    if (message.offLineMessage) {
                        return;
                    }
                    msg.receiptResponse = [];
                    msgObserverList.notify(message);
                }
            } else {
                var storeKey = getStoreKey('req');
                var conversationKey = generatorKey([message.conversationType, message.targetId]);

                var data = store.get(storeKey) || {};
                var ret = data[conversationKey] || {};
                var uIds = ret.uIds || [];

                uIds.push(messageUId);
                var result = ret[message.senderUserId] || [];
                ret[message.senderUserId] = result.concat(uIds);
                data[conversationKey] = ret;

                store.set(storeKey, data);
                var type = message.conversationType;
                var id = message.targetId;
                var lconv = conversationApi.getLocalOne(type, id);
                if (lconv) {
                    lconv.requestMsgs = ret;
                    if (lconv.unreadMessageCount === 0) {
                        Message.sendGroupResponse(type, id, ret);
                    }
                    if (message.offLineMessage) {
                        return;
                    }
                    var lconvList = conversationApi.getLocalList();
                    conversationApi.observerList.notify(lconvList);
                }
            }
        },
        ReadReceiptResponseMessage: function (message) {
            var key = getCacheKey(message);
            var cacheList = Message._cache[key] = Message._cache[key] || [];
            bindResponseToMessage(message, cacheList);
        },
        RealTimeLocationJoinMessage: function () {
        },
        RealTimeLocationQuitMessage: function () {
        },
        RealTimeLocationStatusMessage: function () {
        },
        KickoffMsg: function (message) {
            if (message.offLineMessage) {
                return;
            }
            // 判断消息有效性
            var authId = (RongIM.instance.auth || {}).id;
            var valid = message.targetId === authId;
            if (valid) {
                var status = 'logout-by-otherclient';
                statusApi.observerList.notify(status);
            }
        },
        ConferenceUpdateMessage: function (/* message */) {
            // utils.console.log('TODO: 此消息需要处理 ConferenceUpdateMessage', message);
        },
        ConferenceParticipantUpdateMessage: function (/* message */) {
            // utils.console.log('TODO: ConferenceParticipantUpdateMessage此消息需要处理', message);
        },
        ConferencePingMessage: function (/* message */) {
            // utils.console.log('TODO: ConferencePingMessage此消息需要处理', message);
        },
        GroupVerifyNotifyMessage: function (message) {
            groupApi.observerList.notify(message);
        },
        AppNotifyMessage: function (message) {
            // 公众号关闭通知消息
            if (message.offLineMessage) {
                return;
            }
            msgObserverList.notify(message);
        },
        MultiClientMessage: function (message) {
            // 多端登陆，不处理离线消息
            if (message.offLineMessage) {
                return;
            }
            // 增加判断消息有效性
            var deviceRobotId = RongIM.serverConfig.device.device_manage_robot_id;
            if (message.targetId === deviceRobotId) {
                msgObserverList.notify(message);
            }
        },
        RealTimeLocationStartMessage: function (message) {
            var params = {
                conversation: message.conversationType,
                targetId: message.targetId,
                messageIds: [message.messageId],
                notNotify: true
            };
            Message.removeLocal(params, function () {
                saveRemovedEarliestMessageTime(message);
            });
        },
        UserTypeChangedMessage: function (message) {
            // 外部联系人转内部员工通知消息
            if (message.offLineMessage) {
                return;
            }
            msgObserverList.notify(message);
        },
        EncryptedMessage: function () {
            // 密聊消息，直接过滤
        },
        EncryptRequestMessage: function () {
            // 密聊消息，直接过滤
        },
        EncryptResponseMessage: function () {
            // 密聊消息，直接过滤
        },
        EncryptConfirmMessage: function () {
            // 密聊消息，直接过滤
        },
        EncryptCancelMessage: function () {
            // 密聊消息，直接过滤
        },
        EncryptTerminateMessage: function () {
            // 密聊消息，直接过滤
        },
        SCBurnTimeMessage: function () {
            // 密聊消息，直接过滤
        },
        BurnNoticeMessage: function () {
            // 密聊消息，直接过滤
        },
        otherMessage: function (message) {
            Message._push(message);
            if (isSameConversation(message) || PullMessageStatus.get()) {
                msgObserverList.notify(message);
            }
        }
    };

    apiList.forEach(function (api) {
        if (typeof api.messageCtrol === 'object') {
            $.extend(messageCtrol, api.messageCtrol);
        }
    });

    // var messageFilter = [
    //     'PresenceNotificationMessage',
    //     'SyncReadStatusMessage',
    //     'RecallCommandMessage',
    //     'ReadReceiptRequestMessage',
    //     'ReadReceiptResponseMessage',
    //     'ReadReceiptMessage'
    // ];

    var lastMessageFilter = [
        'GroupMemChangedNotifyMessage',
        'PinNotifyMessage',
        'PinCommentMessage',
        'PinConfirmMessage',
        'PinNewReciverMessage',
        'PinDeletedMessage',
        'PinCommentReadMessage',
        'PinConfirmMultiMessage',
        'PinRevokeMessage'
    ];

    var messageObserList = {};
    var isNotifyConvers = false;
    var receiveMessage = function (message, leftCount, hasMore) {
        console.log("message------->",message);
        // 撤回消息偶尔未同步增加日志记录撤回消息
        if (message.messageType === 'RecallCommandMessage') {
            RongIM.system.appLogger('debug', '[ReceiveMessage][RecallCommandMessage]' + JSON.stringify(message));
        }

        RongIM.messageTotal += 1;
        var isHandler = leftCount === 0 && !hasMore;
        PullMessageStatus.set(isHandler);
        // 接收离线消息中接收完给出通知
        var offlineMessageReceiving = !isHandler;
        if (RongIM.offlineMessageReceiving !== offlineMessageReceiving) {
            RongIM.offlineMessageReceiving = offlineMessageReceiving;
            if (!RongIM.offlineMessageReceiving) {
                offlineMsgfinishObserverList.notify();
            }
        }

        // var isCommandMsg = messageFilter.indexOf(message.messageType) >= 0;
        var isLastOpeMsg = lastMessageFilter.indexOf(message.messageType) >= 0;
        var presence = messageCtrol[message.messageType];
        // 群通知专门处理
        var isNotifyGroupMsg = groupApi.getNotifyGroupMsg(message);
        if (!presence || isNotifyGroupMsg) {
            isNotifyConvers = true;
        }
        if (isLastOpeMsg) {
            var isGroupMessage = message.messageType === 'GroupMemChangedNotifyMessage';
            var key = message.messageType + '-' + message.targetId;
            if (isGroupMessage) {
                Message._push(message);
            }
            if (!isGroupMessage) {
                key = message.messageType + '-targetId';
            }
            messageObserList[key] = message;
        } else {
            handleMessage(message);
        }

        if (isHandler) {
            if (!$.isEmptyObject(messageObserList)) {
                Object.keys(messageObserList).forEach(function (item) {
                    handleMessage(messageObserList[item]);
                });
                messageObserList = {};
            }
            if (isNotifyConvers) {
                conversationApi.notifyConversation();
                isNotifyConvers = false;
            }
        }
    };

    // 实际处理方法
    function handleMessage(message) {
        // C++ SDK offLineMessage 标识不准确根据连接返回服务时间和消息发送时间对比判断
        // C++ SDK 已修复屏蔽此行代码 (connect 连接成功为异步先触发接收消息 connectedTime 为 0)
        // message.offLineMessage = message.sentTime < RongIM.instance.connectedTime;
        hackWebSDKMessage(message);
        var notLogin = $.isEmptyObject(Cache.auth) || $.isEmptyObject(Cache.auth.id);
        if (notLogin) {
            return;
        }
        // Web SDK patch 多端同步收到自己发的消息 设置发送状态为发送成功
        if (!message.offLineMessage && message.senderUserId === Cache.auth.id) {
            message.sentStatus = RongIMLib.SentStatus.SENT;
        }
        var messageType = message.messageType;
        var presence = messageCtrol[messageType];
        if (presence) presence(message); else messageCtrol.otherMessage(message);
    }

    RongIMClient.setOnReceiveMessageListener({
        onReceived: receiveMessage
    });
};

function messageUnexist(message, list) {
    var messageUId = message.messageUId;
    if (typeof messageUId === 'undefined') {
        return true;
    }

    var arr = list.filter(function (item) {
        return item.messageUId === messageUId;
    });
    return arr.length === 0;
}

// 删除本地存储的消息
Message.ClearData = function () {
    RongIMClient.getInstance().clearData();
};

RongIM.dataModel.Message = Message;
}(RongIM, {
    jQuery: jQuery,
    RongIMClient: RongIMClient,
    RongIMLib: RongIMLib
}));
