/* 远程控制 */
(function (RongIM, dependencies) {
'use strict';

var $ = dependencies.jQuery;
var RongIMClient = dependencies.RongIMClient;
var Cache = RongIM.dataModel._Cache;
var ObserverList = RongIM.dataModel._ObserverList;
var common = RongIM.common;

var messageApi = RongIM.dataModel.Message;
var conversationApi = RongIM.dataModel.Conversation;

var RemoteEnvironmentType = {
    PUBLIC: 'public',
    PRIVATE: 'private'
};

var RemoteCtrlOpt = RongIM.remoteControl;
var RemoteControlStatus = common.RemoteControlStatus;
var RemoteControlHangupReason = common.RemoteControlHangupReason;

var RemoteControl = {
    observerList: new ObserverList()
};

var remoteCtrlObserverList = RemoteControl.observerList;

function sendRemoteControlMessage(params, msg, callback, pushData) {
    RongIMClient.getInstance().sendMessage(+params.conversationType, params.targetId, msg, {
        onSuccess: function (message) {
            if (callback) callback(null, message);
        },
        onError: callback
    }, false, pushData);
}
// 相关自定义消息注册
RemoteControl.registerMessage = function () {
    // 请求控制对方电脑，发起方可以是电脑或手机
    var messageName = 'RemoteControlRequestMessage';
    var objectName = 'RC:RCRequest';
    var messageTag = new RongIMLib.MessageTag(false, false);
    var properties = ['sessionId'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    // 邀请对方控制已方电脑，发起方只能是电脑
    messageName = 'RemoteControlInviteMessage';
    objectName = 'RC:RCInvite';
    messageTag = new RongIMLib.MessageTag(false, false);
    properties = ['sessionId'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    // 远程控制发起后，对方收到Invite消息后立刻回复的消息,表示对方已收到请求但还未做接收或者拒绝处理
    messageName = 'RemoteControlRingingMessage';
    objectName = 'RC:RCRinging';
    messageTag = new RongIMLib.MessageTag(false, false);
    properties = ['sessionId'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    // 远程控制发起后，对方接受
    messageName = 'RemoteControlAcceptMessage';
    objectName = 'RC:RCAccept';
    messageTag = new RongIMLib.MessageTag(false, false);
    properties = ['sessionId'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    // 远程控制请求发起后，发送消息拒绝请求或者挂断
    messageName = 'RemoteControlHangupMessage';
    objectName = 'RC:RCHangup';
    messageTag = new RongIMLib.MessageTag(false, false);
    properties = ['sessionId', 'reason'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    // 远程控制信令层Session建立后，被控制端发送连接信息和会话信息给主控制端，用于主控制端媒体层session建立
    messageName = 'RemoteControlCofigMessage';
    objectName = 'RC:RCConfig';
    messageTag = new RongIMLib.MessageTag(false, false);
    properties = ['sessionId', 'mediaAddress', 'mediaId', 'platform'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);

    // 远程控制会话结束后，显示在会话页面的统计信息
    messageName = 'RemoteControlSummaryMessage';
    objectName = 'RC:RCSummary';
    messageTag = new RongIMLib.MessageTag(true, true);
    properties = ['reason', 'direction', 'duration', 'isControl'];
    RongIMClient.registerMessageType(messageName, objectName, messageTag, properties);
};

// 相关消息解析
RemoteControl.messageCtrol = {
    RemoteControlRequestMessage: function (message) {
        notifyRemoteControl(message);
    },
    RemoteControlInviteMessage: function (message) {
        notifyRemoteControl(message);
    },
    RemoteControlAcceptMessage: function (message) {
        notifyRemoteControl(message);
    },
    RemoteControlHangupMessage: function (message) {
        notifyRemoteControl(message);
    },
    RemoteControlCofigMessage: function (message) {
        notifyRemoteControl(message);
    },
    RemoteControlSummaryMessage: function (message) {
        notifyRemoteControl(message);
    },
    RemoteControlRingingMessage: function () {
    }
};
/*
请求远程控制对方
params 需包含 targetId, conversationType
 */
RemoteControl.request = function (params, callback) {
    var msg = new RongIMClient.RegisterMessage.RemoteControlRequestMessage({
        sessionId: getRemoteCtrlSessionId(params.targetId)
    });
    sendRemoteControlMessage(params, msg, callback);
};

/*
邀请对方控制
 */
RemoteControl.invite = function (params, callback) {
    var im = RongIM.instance;
    var locale = im.locale;
    var remoteLocale = locale.components.remoteControl.pushInvite;
    var name = im.loginUser.name;
    var pushData = RongIM.utils.templateFormat(remoteLocale, name);
    var msg = new RongIMClient.RegisterMessage.RemoteControlInviteMessage({
        sessionId: getRemoteCtrlSessionId(params.targetId)
    });
    sendRemoteControlMessage(params, msg, callback, pushData);
};

/*
接受 控制 或 被控制
 */
RemoteControl.accept = function (params, sessionId, callback) {
    var msg = new RongIMClient.RegisterMessage.RemoteControlAcceptMessage({
        sessionId: sessionId
    });
    sendRemoteControlMessage(params, msg, callback);
};

/*
挂断 远程控制
 */
RemoteControl.hangup = function (params, sessionId, reason, callback) {
    // 发起挂断消息时, 如果挂断原因是己方正在忙碌, 则只发消息, 不挂断
    if (reason !== RemoteControlHangupReason.BusyLine) {
        RemoteCtrlOpt.hangup();
    }
    var im = RongIM.instance;
    var locale = im.locale;
    var pushData = locale.components.remoteControl.pushHangup;
    var msg = new RongIMClient.RegisterMessage.RemoteControlHangupMessage({
        sessionId: sessionId,
        reason: reason
    });
    sendRemoteControlMessage(params, msg, callback, pushData);
};

// 判断自己是否是发起方
RemoteControl.isInitiator = function (sessionId) {
    sessionId = sessionId || '';
    var auth = RongIM.instance.auth;
    var index = sessionId.indexOf(';;;');
    var initiator = sessionId.substring(0, index);
    var targetId = auth.id;
    return initiator !== targetId;
};

/*
获取配置, 被控端发送给主控端
endEvent: 结束远程控制时的回调
 */
RemoteControl.setConfig = function (params, sessionId, callback, endEvent) {
    /* TODO openId, openKey 需要从服务端获取 */
    var acceptConfig = getAcceptParams();
    var openId = acceptConfig.openId;
    var openKey = acceptConfig.openKey;
    var platform = common.RemotePlatform[RongIM.system.platform];
    RemoteCtrlOpt.accept(openId, openKey, {
        onConnect: $.noop,
        onDisconnected: $.noop,
        onLogin: function (msg) {
            var remoteStatus = RemoteControl.get(params.targetId).status;
            // node 包返回状态码混乱, 暂时避免
            if (remoteStatus === RemoteControlStatus.InviteBuilded) {
                return;
            }
            var option = {
                sessionId: sessionId,
                mediaAddress: msg.address,
                mediaId: msg.sessionName,
                platform: platform
            };
            msg = new RongIMClient.RegisterMessage.RemoteControlCofigMessage(option);
            sendRemoteControlMessage(params, msg, callback);
        },
        onLoginFail: function () {
            callback(true);
        },
        onEnd: endEvent,
        onClose: function () {
            console.debug('remote onclose');
        }
    });
};

/**
 * 远程控制会话结束后, 显示在会话页面的统计信息, 为本地插入的消息, 不发送给其他端
 * @param  {object}   params, 包含 conversationType 和 targetId
 * @param  {object}   content
 *                    content.reason  挂断原因
 *                    content.direction 方向, 用于区分插入消息显示在会话左侧还是右侧
 *                    content.duration  通话时长
 */
RemoteControl.summary = function (params, content, callback) {
    var authId = Cache.auth.id;
    var isSelfSend = content.direction === RongIMLib.MessageDirection.SEND;
    var senderUserId = isSelfSend ? authId : params.targetId;
    params = {
        objectName: 'RC:RCSummary',
        content: content,
        direction: content.direction,
        senderUserId: senderUserId,
        conversationType: params.conversationType,
        targetId: params.targetId,
        sentStatus: RongIMLib.SentStatus.READ,
        receivedStatus: RongIMLib.ReceivedStatus.READ
    };
    messageApi.insertMessage(params, callback);
};

/**
 * 设置 本地 Cache conversation 中的 remoteControl
 * @param {string} message
 * @param {int} status
 */
RemoteControl.setLocalRemote = function (message, callback) {
    var targetId = message.targetId;
    // 设置 cache 中 conversation 的 remoteControl
    var setRemote = function (handle) {
        var conversationList = Cache.conversation.list || [];
        var status = getRemoteControlStatus(message);
        if (message.isOtherPlatSend) {
            status = RemoteControlStatus.Idle;
        }
        var remoteControl = {
            message: message,
            status: status
        };
        if (conversationList) {
            conversationList.forEach(function (conversation) {
                if (conversation.targetId === targetId) {
                    conversation.remoteControl = remoteControl;
                }
            });
        }
        message.remoteControl = remoteControl;
        if (handle) handle(remoteControl);
    };
    // 如果 conversationList 中没有该 conversation, 增加一条
    var conversationList = Cache.conversation.list || [];
    var conversationIdList = conversationList.map(function (con) {
        return con.targetId;
    });
    if (conversationIdList.indexOf(targetId) === -1) {
        var params = {
            targetId: message.targetId,
            conversationType: message.conversationType
        };
        conversationApi.add(params, function () {
            setRemote(callback);
        });
    } else {
        setRemote(callback);
    }
};

RemoteControl.get = function (targetId) {
    var conversationList = Cache.conversation.list || [];
    conversationList = conversationList.filter(function (con) {
        return con.targetId === targetId;
    });
    if (conversationList.length) {
        return conversationList[0].remoteControl || {};
    }
    return {};
};

/* 是否正在进行远程控制 */
RemoteControl.isBusyline = function () {
    var conversationList = Cache.conversation.list || [];
    if (!conversationList) {
        return false;
    }
    var busyList = conversationList.filter(function (con) {
        var remoteControl = con.remoteControl;
        return remoteControl && remoteControl.status !== RemoteControlStatus.Idle;
    });
    return !!busyList.length;
};

RemoteControl.watch = function (listener) {
    remoteCtrlObserverList.add(listener);
};

RemoteControl.unwatch = function (listener) {
    remoteCtrlObserverList.remove(listener);
};

/**
 * 对 onReceived 中收到的消息进行处理以及 notify
 * @param  {object} message onRecieved 中收到的消息
 */
function notifyRemoteControl(message) {
    var isOffLine = message.offLineMessage;
    if (!RongIM.instance.isSupportRemoteControl || isOffLine) {
        return;
    }
    var isSelfSend = message.messageDirection === RongIMLib.MessageDirection.SEND;
    if (isSelfSend) {
        message.isOtherPlatSend = true;
    }
    remoteCtrlObserverList.notify(message);
}

/*
处理接受到的remoteControl message
返回转化后的 status
 */
function getRemoteControlStatus(message) {
    var status = RemoteControlStatus.Idle;
    var isSelfSend = message.messageDirection === 1;
    switch (message.messageType) {
        case 'RemoteControlRequestMessage':
            status = isSelfSend ? RemoteControlStatus.Request : RemoteControlStatus.ReceiveRequest;
            break;
        case 'RemoteControlInviteMessage':
            status = isSelfSend ? RemoteControlStatus.Invite : RemoteControlStatus.ReceiveInvite;
            break;
        case 'RemoteControlAcceptMessage':
            status = message.isOtherPlatSend ? RemoteControlStatus.Idle : RemoteControlStatus.Building;
            break;
        case 'RemoteControlCofigMessage':
            status = isSelfSend ? RemoteControlStatus.InviteBuilded : RemoteControlStatus.RequestBuilded;
            break;
        case 'RemoteControlHangupMessage':
            status = RemoteControlStatus.Idle;
            break;
        default:
            break;
    }
    return status;
}

/*
(targetId + currentTime)
 */
function getRemoteCtrlSessionId(targetId) {
    var currentTime = +new Date();
    return targetId + ';;;' + currentTime;
}

// 根据环境, 获取 openId 和 openKey
function getAcceptParams() {
    var remoteConfig = RongIM.config.modules.remoteControl;
    var openId;
    var privateSessions = remoteConfig.private_sessions;
    if (privateSessions && privateSessions.length) {
        openId = privateSessions[0];
    }
    var params = {
        openId: openId,
        openKey: remoteConfig.private_addr
    };
    if (remoteConfig.type === RemoteEnvironmentType.PUBLIC) {
        params = {
            openId: remoteConfig.public_app_id,
            openKey: remoteConfig.public_app_secret
        };
    }
    return params;
}

RongIM.dataModel.RemoteControl = RemoteControl;
}(RongIM, {
    jQuery: jQuery,
    RongIMClient: RongIMClient
}));
