(function (RongIM, dependencies, components) {
'use strict';

var utils = RongIM.utils;
var Base64Util = utils.Base64;
var common = RongIM.common;
var drag = RongIM.drag;
var UploadClient = dependencies.UploadClient;

// 修改头像组件
components.editAvatar = function (resolve, reject) {
    var im = RongIM.instance;
    var dataModel = im.dataModel;
    var MAX_SCALE = 5;
    var options = {
        name: 'edit-avatar',
        template: 'templates/edit-avatar.html',
        data: function () {
            return {
                showField: true,
                user: im.loginUser,
                src: im.loginUser.portrait_big_url || im.loginUser.avatar,
                canvasSize: 198,
                imgWidth: 0,
                imgHeight: 0,
                offsetX: 0,
                offsetY: 0,
                loadDone: false,
                minScale: 0.1,
                percent: 0,
                rotate: 0,
                isBusy: false
            };
        },
        components: {
            avatar: components.getAvatar
        },
        computed: {
            scale: {
                get: function () {
                    return this.percent / 100 * MAX_SCALE + this.minScale;
                },
                set: function (value) {
                    this.percent = (value - this.minScale) / MAX_SCALE * 100;
                }
            }
        },
        methods: {
            toastError: function (errorCode) {
                var el = null;
                if (this.$el) {
                    el = this.$el.firstChild;
                }
                common.toastError(errorCode, el);
            },
            close: function () {
                this.$emit('close');
            },
            exchangeNewAvatar: function (event) {
                exchangeNewAvatar(this, event);
            },
            loadedImg: function (event) {
                loadedImg(this, event.target);
            },
            loadError: function () {
                this.src = '';
            },
            dragImg: function (event) {
                dragImg(this, event);
            },
            dragTrackButton: function (event) {
                dragTrackButton(this, event);
            },
            zoomOut: function () {
                var STEP = 1;
                this.percent = Math.max(0, this.percent - STEP);
            },
            zoomIn: function () {
                var STEP = 1;
                this.percent = Math.min(100, this.percent + STEP);
            },
            getImgData: function (callback) {
                getImgData(this, callback);
            },
            upload: function (callback) {
                uploadNewAvatar(this, callback);
            },
            saveNewAvatar: function () {
                saveNewAvatar(this, dataModel.User, im);
            }
        }
    };
    utils.asyncComponent(options, resolve, reject);
};

function exchangeNewAvatar(context, event) {
    var file = event.target.files[0];
    if (utils.isEmpty(file)) {
        return;
    }
    var fr = new FileReader();
    fr.onload = function (e) {
        var src = e.target.result;
        /*
        exif Orientation 说明参考文档
        http://www.exif.org/Exif2-2.PDF
        */
        EXIF.getData({ src: src }, function () {
            var Orientation = EXIF.getTag(this, 'Orientation');
            var rotate = 0;
            switch (Orientation) {
                case 1:
                    rotate = 0;
                    break;
                case 6:
                    rotate = 90;
                    break;
                case 3:
                    rotate = 180;
                    break;
                case 8:
                    rotate = 270;
                    break;
                default:
                    rotate = 0;
            }
            context.rotate = rotate;
            context.src = src;
        });
    };
    fr.readAsDataURL(file);
}

function loadedImg(context, img) {
    $(img).css({
        'margin-left': 0,
        'margin-top': 0
    });
    context.imgWidth = img.width;
    context.imgHeight = img.height;
    var canvasSize = context.canvasSize;
    var imgMinSize = Math.min(img.width, img.height);
    context.minScale = canvasSize / imgMinSize;
    context.scale = context.minScale;
    context.loadDone = true;
}

function dragImg(context, event) {
    if (!context.loadDone) {
        return;
    }
    var el = event.target;
    var $el = $(el);
    var oldPosition = {
        left: parseFloat($el.css('left')),
        top: parseFloat($el.css('top'))
    };
    drag(el, event, function (position) {
        var deltaX = position.left - oldPosition.left;
        var deltaY = position.top - oldPosition.top;
        context.offsetX = deltaX;
        context.offsetY = deltaY;
        $el.css({
            'margin-left': deltaX,
            'margin-top': deltaY
        });
    }, el.parentElement);
}

function dragTrackButton(context, event) {
    if (!context.loadDone) {
        return;
    }

    var el = event.target;
    var $el = $(el);

    var trackWidth = $el.parent().width();
    drag(el, event, function (position) {
        var left = Math.min(Math.max(0, position.left), trackWidth);
        var ratio = left / trackWidth;
        context.percent = ratio * 100;
    });
}

function getImgData(context, callback) {
    var MAX = 960;
    var MIN = 200;
    var img = context.$refs.img;
    var scale = context.scale;
    var leftCorner = context.imgWidth / 2 - context.canvasSize / 2 / scale;
    var topCorner = context.imgHeight / 2 - context.canvasSize / 2 / scale;
    var sx = leftCorner - context.offsetX / scale;
    var sy = topCorner - context.offsetY / scale;
    var sWidth = context.canvasSize / scale;
    var sHeight = context.canvasSize / scale;
    var dWidth = 160;
    var dHeight = 160;

    var smallBase64 = clipImage(img, sx, sy, sWidth, sHeight, dWidth, dHeight, context.rotate, true);

    dWidth = Math.min(MAX, Math.max(MIN, sWidth));
    dHeight = Math.min(MAX, Math.max(MIN, sHeight));
    var bigBase64 = clipImage(img, sx, sy, sWidth, sHeight, dWidth, dHeight, context.rotate);

    callback(null, {
        big: bigBase64,
        small: smallBase64
    });
}

function clipImage(img, sx, sy, sWidth, sHeight, dWidth, dHeight, rotate, isSmall) {
    var canvas = document.createElement('canvas');
    canvas.width = dWidth;
    canvas.height = dHeight;
    var ctx = canvas.getContext('2d');
    ctx.translate(dWidth / 2, dHeight / 2);
    ctx.rotate(rotate * Math.PI / 180);
    ctx.drawImage(img, sx, sy, sWidth, sHeight, -dWidth / 2, -dHeight / 2, dWidth, dHeight);
    var url = '';
    if (isSmall) {
        url = canvas.toDataURL('image/jpeg', 1.0);
    } else {
        url = canvas.toDataURL();
    }
    url = Base64Util.replace(url);
    return url;
}

function uploadNewAvatar(context, callback) {
    var bigSrc = '';
    var smallSrc = '';
    context.getImgData(function (errorCode, base64) {
        if (errorCode) {
            callback(errorCode);
            return;
        }
        upload('base64', base64.big, 'png', function (error, src) {
            if (error) {
                callback(error);
                return;
            }
            bigSrc = src;
            if (smallSrc) {
                callback(null, {
                    bigSrc: bigSrc,
                    smallSrc: smallSrc
                });
            }
        });

        upload('base64', base64.small, 'jpeg', function (error, src) {
            if (error) {
                callback(error);
                return;
            }
            smallSrc = src;
            if (bigSrc) {
                callback(null, {
                    bigSrc: bigSrc,
                    smallSrc: smallSrc
                });
            }
        });
    });
}

function saveNewAvatar(context, userApi, im) {
    var currentUserId = im.loginUser.id;
    if (utils.isEmpty(context.src)) {
        context.close();
        return;
    }
    if (context.isBusy) {
        common.messageToast({
            message: im.locale.tips.uploadAvatar,
            type: 'error'
        });
        return;
    }
    context.isBusy = true;
    context.upload(function (errorCode, data) {
        // 用户已变更
        if (!im.loginUser || currentUserId !== im.loginUser.id) {
            return;
        }
        context.isBusy = false;
        if (errorCode) {
            context.toastError(errorCode);
            return;
        }
        userApi.setAvatar(data.smallSrc, data.bigSrc, function (error) {
            if (error) {
                context.toastError(error);
                return;
            }
            im.loginUser.avatar = data.smallSrc;
            im.loginUser.portrait_big_url = data.bigSrc;
            context.$emit('srcchange', data.smallSrc, data.bigSrc);
            context.close();
        });
    });
}

/**
 * @param type - 'file' or 'image' or 'base64'
 * @param fileData
 * @param callback
 */
function upload(type, fileData, imgType, callback) {
    var config = RongIM.config.upload[type] || RongIM.config.upload.file;
    config.ext = imgType || 'png';
    var domain = '';
    if (type === 'base64') {
        config.data = UploadClient.dataType.data;
    }
    config.getToken = function (done) {
        var dataModel = RongIM.instance.dataModel;
        dataModel.User.getAvatarToken(function (errorCode, result) {
            if (errorCode) {
                utils.console.warn('获取上传 token 失败');
                return;
            }
            domain = result.domain;
            done(result.token);
        });
    };

    var actionMap = {
        file: 'initFile',
        image: 'initImage',
        base64: 'initImgBase64'
    };
    var action = actionMap[type];
    var uploadCallback = {
        onBeforeUpload: function () {
        },
        onProgress: function () {
        },
        onCompleted: function (data) {
            var url = common.getDownloadUrl(RongIM.config, data);
            url = url || location.protocol + '//' + domain + '/' + data.key;
            callback(null, url);
        },
        onError: callback
    };
    UploadClient[action](config, function (uploadFile) {
        uploadFile.upload(fileData, uploadCallback);
    });
}
}(RongIM, {
    UploadClient: UploadClient
}, RongIM.components));
