(function (RongIM, dependencies, components) {
'use strict';

var common = RongIM.common;
var utils = RongIM.utils;
var $ = dependencies.jQuery;
var pageNum = RongIM.config.profile.pageNum;
var im;

/*
说明：创建群组
功能：通过企业通讯录、星标联系人、我的好友选择联系人创建群组
参数：
    @param {string}           groupId       群组Id
    @param {array<object>}    members
*/
RongIM.dialog.createGroup = function (groupInfo, members) {
    members = members.map(function (item) {
        return common.unifyUser(item);
    });
    im = RongIM.instance;
    var dataModel = im.dataModel;
    var groupPermission = utils.groupPermission;

    // 模块化屏蔽入口
    // 好友
    var modules = im.config.modules;
    var enabledFriend = modules.friend;
    var serverConfig = im.serverConfig;
    var maxCount = serverConfig.group.max_member_count || im.config.maxGroupMemberNum;
    // 星标联系人
    var enabledStar = modules.star;

    var options = {
        name: 'group-create',
        template: 'templates/group/create.html',
        data: function () {
            return {
                maxCount: maxCount,
                enabledFriend: enabledFriend,
                enabledStar: enabledStar,
                group: groupInfo,
                groupId: groupInfo ? groupInfo.id : null,
                show: true,
                groupName: '',
                // 'star' or 'org'
                tab: 'org',
                defaultSelected: $.extend(true, [], members),
                busy: false,
                selected: [],
                selectedPage: 1,
                pageNum: pageNum,
                loadBusy: false,
                isStaff: im.auth.isStaff,
                disableExecutive: false
            };
        },
        components: {
            avatar: components.getAvatar,
            star: components.group.getStar,
            org: components.group.getOrg,
            friend: components.group.getFriend
        },
        computed: {
            groupIdExisted: function () {
                return !utils.isEmpty(this.groupId);
            },
            isOwnerManage: function () {
                var group = this.group || {};
                return group.invite_member === groupPermission.Owner && group.publish_group_notice === groupPermission.Owner;
            },
            selectedPageList: function () {
                var end = this.selectedPage * this.pageNum;
                return this.selected.slice(0, end);
            }
        },
        created: function () {
            this.selected = [].concat(this.defaultSelected);
            if (!this.isStaff) {
                this.tab = 'friend';
            }
        },
        watch: {
            selected: function () {
                this.selectedPage = 1;
            }
        },
        methods: getMethods(dataModel)
    };
    common.mountDialog(options, function (instance) {
        RongIM._groupInstance = instance;
    });
};

function getMethods(dataModel) {
    return {
        reset: function () {
            this.selected.push({});
            this.selected.pop();
        },
        toastError: function (errorCode) {
            common.toastError(errorCode, this.$el.firstChild);
        },
        toast: function (params) {
            params.el = this.$el.firstChild;
            common.messageToast(params);
        },
        close: function () {
            this.show = false;
        },
        setTab: function (tab) {
            this.tab = tab;
        },
        isDefault: function (item) {
            var idList = this.defaultSelected.map(function (_item) {
                return _item.id;
            });
            return idList.indexOf(item.id) >= 0;
        },
        removeMembers: function (members) {
            removeMembers(this, members);
        },
        added: function (members) {
            added(this, members);
        },
        removed: function (members) {
            removed(this, members);
        },
        getGroupName: function () {
            return getGroupName(this);
        },
        getUsername: common.getUsername,
        createGroup: function () {
            var api = {
                group: dataModel.Group,
                conversation: dataModel.Conversation
            };
            createGroup(this, api);
        },
        addMembers: function () {
            addMembers(this, dataModel.Group, this.groupId);
        },
        maxCountLimit: function () {
            var mostReceiveFormat = im.locale.errorCode['10645'];
            var count = this.maxCount;
            if (im.config.locale !== 'zh') {
                count = utils.toThousands(count);
            }
            var hintMessage = this.localeFormat(mostReceiveFormat, count);
            this.toast({
                type: 'error',
                message: hintMessage
            });
            this.reset();
        },
        loadMore: function () {
            loadMore(this);
        }
    };
}

function loadMore(context) {
    var end = context.selectedPage * context.pageNum;
    var list = context.selected;
    if (list && list.length > end) {
        context.loadBusy = true;
        setTimeout(function () {
            context.loadBusy = false;
            context.selectedPage += 1;
        }, 200);
    }
}

/*
说明：创建群组时，勾选要加入群组的成员
参数：
    @param {object}           context       群组信息
    @param {object}           members       勾选的成员信息
*/
function added(context, members) {
    var selectedIdList = context.selected.map(function (item) {
        return item.id;
    });
    var addedList = members.filter(function (item) {
        return selectedIdList.indexOf(item.id) < 0;
    });
    var totalCount = selectedIdList.length + addedList.length;
    context.selected = context.selected.concat(addedList);
    if (totalCount > context.maxCount) {
        var mostReceiveFormat = im.locale.errorCode['10645'];
        var count = context.maxCount;
        if (im.config.locale !== 'zh') {
            count = utils.toThousands(count);
        }
        var hintMessage = context.localeFormat(mostReceiveFormat, count);
        context.toast({
            type: 'error',
            message: hintMessage
        });
        context.removed(members);
    }
}

/*
说明：创建群组时，取消已勾选的成员
参数：
    @param {object}           context       群组信息
    @param {object}           members       已勾选的成员信息
*/
function removed(context, members) {
    var idList = members.map(function (item) {
        return item.id;
    });
    var reservedIdList = context.defaultSelected.map(function (item) {
        return item.id;
    });
    context.selected = context.selected.filter(function (item) {
        var reserved = reservedIdList.indexOf(item.id) >= 0;
        return reserved || idList.indexOf(item.id) < 0;
    });
}

/*
说明：创建群组时，右侧移除已选择的成员
参数：
    @param {object}           context       群组信息
    @param {object}           members       移除的成员信息
*/
function removeMembers(context, members) {
    members = [].concat(members);
    var idList = members.map(function (item) {
        return item.id;
    });
    context.selected = context.selected.filter(function (item) {
        return idList.indexOf(item.id) < 0;
    });
}

/*
说明：获取群组名称
参数：
    @param {object}           context       群组信息
*/
function getGroupName(context) {
    var groupName;
    if (context.groupName) {
        groupName = context.groupName;
    } else {
        var MAX_LENGTH = 16;
        var memberNames = context.selected;
        memberNames = memberNames.slice(0, MAX_LENGTH)
            .map(function (item) {
                return item.name;
            });
        var group = {
            member_names: memberNames
        };
        groupName = common.getGroupName(group);
    }
    return groupName;
}

function checkMemberLength(context, memberIdList, cb) {
    if (memberIdList.length > 99) {
        common.messagebox({
            type: 'confirm',
            message: context.locale.message.confirmLargeGroup,
            callback: cb,
            hashchangeClose: false
        });
    } else {
        cb();
    }
}

/*
说明：创建群组
参数：
    @param {object}           context
    @param {object}           api       ·
*/
function createGroup(context, api) {
    var groupApi = api.group;
    var conversationApi = api.conversation;
    if (context.busy) {
        return;
    }

    var memberIdList = context.selected
        .map(function (item) {
            return item.id;
        });

    if (memberIdList.length === 2) {
        var targetId = memberIdList[1];
        var conversationType = utils.conversationType.PRIVATE;
        var params = {
            targetId: targetId,
            conversationType: conversationType
        };
        im.$router.push({
            name: 'conversation',
            params: params
        });
        conversationApi.add(params);
        context.close();
        return;
    }

    var groupNameLength = context.groupName.length;
    if (groupNameLength > 0 && groupNameLength < 2) {
        common.messageToast({
            type: 'error',
            message: context.locale.groupNameErr
        });
        return;
    }

    var group = {
        // 0: 自建群, 1: 官方群
        type: 0,
        name: context.getGroupName(),
        member_ids: memberIdList
    };

    checkMemberLength(context, memberIdList, function() {
        context.busy = true;
        groupApi.create(group, function (errorCode, result) {
            context.busy = false;
            if (errorCode) {
                RongIM.system.appLogger('error', '创建群组失败 ' + JSON.stringify(errorCode));
                context.toastError(errorCode);
                return;
            }
            context.show = false;
            var path = {
                name: 'conversation',
                params: {
                    targetId: result.id,
                    conversationType: utils.conversationType.GROUP
                }
            };
            RongIM.system.appLogger('info', '创建群组成功 ' + result.id);
            im.$router.push(path);
            common.messageToast({
                message: context.locale.createSuccess
            });
        });
    });
}

/*
说明：已经存在的群，群设置中添加新的群组成员
参数：
    @param {object}           context       群组信息
    @param {object}           groupApi      群组 API 接口
    @param {sring}            groupId       群组 ID
*/
function addMembers(context, groupApi, groupId) {
    if (context.busy) {
        return;
    }
    if (context.isOwnerManage && !context.group.is_creator) {
        context.toast({
            message: context.locale.onlyOwnerManageTips,
            type: 'error'
        });
        return;
    }
    var memberIdList = common.without(context.selected, context.defaultSelected)
        .map(function (item) {
            return item.id;
        });
    if (memberIdList.length < 1) {
        context.close();
        return;
    }
    var auth = im.auth || {};
    var group = getGroupDetail(groupId);
    var loginId = auth.id;
    var adminId = group.admin_id;
    var needJoinPermit = group.need_join_permit;
    var isAdmin = adminId === loginId;
    var isSuccess = isAdmin || !needJoinPermit;
    var toast = isSuccess ? context.locale.addMemberSuccess : context.locale.addMemberPermit;
    var locale = context.locale.createNotice;

    var callback = function () {
        context.busy = true;
        groupApi.addMembers(groupId, memberIdList, function (errorCode) {
            context.busy = false;
            if (errorCode) {
                common.toastError(errorCode);
                return;
            }
            common.messageToast({
                message: toast,
                type: 'success'
            });
        });
        context.close();
    };

    if (isSuccess) {
        checkMemberLength(context, context.selected, callback);
    } else {
        common.messagebox(({
            message: locale,
            type: 'confirm',
            callback: callback
        }));
    }
}

function getGroupDetail(groupId) {
    var dataModel = RongIM.dataModel;
    var cache = dataModel._Cache;
    var group = cache.group;
    return group[groupId];
}
}(RongIM, {
    jQuery: jQuery
}, RongIM.components));
