(function (RongIM, dependencies, components) {
'use strict';

var common = RongIM.common;
var utils = RongIM.utils;
var $ = dependencies.jQuery;

/*
说明：转发时选择群组
功能：消息转发时，选择 我的群组 模块
*/
function getGroup(resolve, reject) {
    var im = RongIM.instance;
    var dataModel = im.dataModel;
    var groupApi = dataModel.Group;
    var userApi = dataModel.User;
    var options = {
        name: 'friend',
        template: 'templates/group/group.html',
        data: function () {
            return {
                keyword: '',
                searchResult: [],
                groups: []
            };
        },
        props: ['selected', 'defaultSelected', 'expandMembers'],
        computed: {
            defaultIdList: function () {
                return (this.defaultSelected || []).map(function (item) {
                    return item.id;
                });
            },
            checkedAll: {
                get: function () {
                    return getCheckedAll(this);
                },
                set: function (value) {
                    setCheckedAll(this, value);
                }
            },
            indeterminate: function () {
                if (this.isCardSelect) {
                    return false;
                }
                return (typeof this.checkedAll) !== 'boolean';
            },
            checked: {
                get: function () {
                    var context = this;
                    if (this.expandMembers) {
                        var checkedGroups = [];
                        this.searchResult.forEach(function (item) {
                            if (item.member_id_list) {
                                var checkMembers = item.member_id_list.every(function (memberId) {
                                    return (
                                        memberId === im.loginUser.id
                                    ) || context.selected.some(function (selectedItem) {
                                        return selectedItem.id === memberId;
                                    });
                                });
                                if (checkMembers) checkedGroups.push(item.id);
                            }
                        });
                        return checkedGroups;
                    } else {
                        return this.selected.filter(function (item) {
                            return context.isGroup(item);
                        }).map(function (item) {
                            return item.id;
                        });
                    }
                },
                set: function (newGroupIds) {
                    var context = this;
                    newGroupIds = mapIds(newGroupIds);
                    groupApi.getBatch(newGroupIds, function (errorCode, newGroups) {
                        newGroups = getGroups(newGroups);
                        if (errorCode) {
                            context.toastError(errorCode);
                            return;
                        }

                        var checkedIds = mapIds(context.checked);
                        groupApi.getBatch(checkedIds, function (UerrorCode, oldGroups) {
                            oldGroups = getGroups(oldGroups);
                            if (UerrorCode) {
                                context.toastError(UerrorCode);
                                return;
                            }
                            context.emitEvent('added', common.without(newGroups, oldGroups));
                            context.emitEvent('removed', common.without(oldGroups, newGroups));
                        });
                    });
                    if (context.keyword) context.clear();
                }
            },
            isCardSelect: function () {
                return this.$parent.$options.name === 'card';
            }
        },
        components: {
            avatar: components.getAvatar
        },
        watch: {
            keyword: function (keyword) {
                keywordChanged(this, keyword);
            }
        },
        created: function () {
            created(this, groupApi);
        },
        methods: {
            toastError: function (errorCode) {
                var el = null;
                if (this.$parent) {
                    el = this.$parent.$el.firstChild;
                }
                common.toastError(errorCode, el);
            },
            getUsername: function (item) {
                var name = common.getUsernameHighlight(item);
                return common.filterMark(name);
            },
            getGroupType: common.getGroupType,
            isDefault: function (item) {
                return this.defaultIdList.indexOf(item.id) >= 0;
            },
            isbanned: function (item) {
                if (this.isGroup(item)) {
                    var group = item.group;
                    if (group) {
                        var members = group.members || [];
                        var theMember = members.filter(function (member) {
                            return member.id === im.auth.id;
                        })[0] || {};
                        if (group.is_all_mute && theMember.mute_status !== 2) {
                            return true;
                        }
                        if (theMember.mute_status === 1) {
                            return true;
                        }
                    }
                }
                return false;
            },
            isValidGroup: function (item) {
                if (this.isGroup(item)) {
                    return checkGroupValid(item, im.auth.id);
                }
                return true;
            },
            isDisabled: function (item) {
                return this.isDefault(item) || this.isbanned(item) || !this.isValidGroup(item);
            },
            clear: function () {
                this.keyword = '';
            },
            isGroup: function (item) {
                // return item.conversationType === 3;
                return item.id.startsWith('group_');
            },
            isIndeterminate: function (item) {
                if (this.expandMembers && item.member_id_list) {
                    var context = this;
                    var checkCount = 0;
                    item.member_id_list.forEach(function (memberId) {
                        var checkMember = (
                            memberId !== im.loginUser.id
                        ) && context.selected.some(function (selectedItem) {
                            return selectedItem.id === memberId;
                        });
                        if (checkMember) ++checkCount;
                    });
                    return (checkCount > 0 && checkCount + 1 < item.member_id_list.length);
                }
                return false;
            },
            emitEvent: function (eventid, list) {
                if (list.length > 0) {
                    if (this.expandMembers) {
                        var groupMembers = [];
                        list.forEach(function (item) {
                            if (item.group && item.group.groupMembers) {
                                item.group.groupMembers.forEach(function (member) {
                                    if (member.id !== im.loginUser.id) {
                                        groupMembers.push(member);
                                    }
                                });
                            }
                        });
                        this.$emit(eventid, groupMembers);
                    } else {
                        this.$emit(eventid, list);
                    }
                }
            }
        }
    };
    utils.asyncComponent(options, resolve, reject);
}

function checkGroupValid(conversation, authId) {
    var group = conversation.group || {};
    var members = group.member_id_list || [];
    var inGroup = members.indexOf(authId) >= 0;
    var isDismiss = (group.group_status === 2);
    return inGroup && !isDismiss;
}

function mapIds(ids) {
    return ids.map(function (id) {
        return id.replace('group_', '');
    });
}

function getCheckedAll(context) {
    var starChecked = [];
    var checkedIdList = context.checked;
    context.searchResult.forEach(function (item) {
        var existed = checkedIdList.indexOf(item.id) >= 0;
        if (existed) starChecked.push(item);
    });
    var length = starChecked.length;
    var result;
    if (length > 0) {
        var isAll = length === context.searchResult.length;
        result = isAll ? true : null;
    } else {
        result = false;
    }
    return result;
}

function setCheckedAll(context, value) {
    var memberIds = context.searchResult.filter(function (item) {
        return !context.isbanned(item) && context.isValidGroup(item);
    }).map(function (item) {
        return item.id;
    });
    if (value) {
        context.checked = [].concat(memberIds, context.checked);
    } else {
        var array = context.checked.filter(function (id) {
            return memberIds.indexOf(id) < 0;
        });
        context.checked = array;
    }
}

/*
说明：转发消息，我的群组搜索框 内容变化时，获取满足条件的群组列表
参数：
    @param {object}     context         我的群组信息
    @param {string}     keyword         搜索框关键字
*/
function keywordChanged(context, keyword) {
    if (keyword.length === 0) {
        if (context.searchResult.length === 0) {
            context.searchResult = context.groups;
        }
        // context.searchResult = context.groups;
    } else {
        context.searchResult = [];
        var searchResult = [];
        context.groups.forEach(function (item) {
            var nameRange = utils.searchStrRange(item.name, context.keyword);
            var aliasRange = utils.searchStrRange(item.alias, context.keyword);
            if (nameRange || aliasRange) {
                var result = $.extend({
                    range: nameRange,
                    aliasRange: aliasRange
                }, item);
                searchResult.push(result);
            } else if (item.group && item.group.groupMembers) {
                var hasMember = item.group.groupMembers.some(function (member) {
                    return utils.searchStrRange(member.name, context.keyword) ||
                        utils.searchStrRange(member.alias, context.keyword);
                })
                if (hasMember) searchResult.push(item);
            }
        });
        // common.sortUsers(searchResult);
        context.searchResult = searchResult;
    }
}

/*
说明：转发消息时，获取 我的群组 列表
参数：
    @param {object}     context
    @param {object}     groupApi
*/
function created(context, groupApi) {
    groupApi.getList(function (errorCode, list) {
        if (errorCode) {
            context.toastError(errorCode);
            return;
        }

        function initContextGroups() {
            var groups = getGroups(list);
            groups.sort(function (a, b) {
                return b.type - a.type;
            });
            context.searchResult = context.groups = groups;
        }

        if (context.expandMembers) {
            var groupCount = list.length;
            list.forEach(function (item) {
                groupApi.getAllMembers(item.id, function(err, members, group) {
                    $.extend(item, group);
                    if (--groupCount == 0) initContextGroups();
                });
            });
        } else {
            initContextGroups();
        }
    });
}

function getGroups(list) {
    return list.map(function (group) {
        return {
            id: 'group_' + group.id,
            name: group.name,
            avatar: group.avatar,
            groupMembers: group.groupMembers,
            member_avatars: group.member_avatars,
            member_names: group.member_names,
            member_id_list: group.member_id_list,
            group_status: group.group_status,
            isFriend: true,
            type: group.type,
            group: group
        };
    });
}

$.extend(true, components, {
    group: {
        getGroup: getGroup
    }
});
}(RongIM, {
    jQuery: jQuery
}, RongIM.components));
