(function (RongIM, dependencies, components) {
'use strict';

var common = RongIM.common;
var utils = RongIM.utils;
var cache = utils.cache;

components.getSignup = function (resolve, reject) {
    var im = RongIM.instance;
    var userApi = im.dataModel.User;
    var enabledSms = im.serverConfig.sms.verification_state;
    var options = {
        name: 'signup',
        template: 'templates/signup.html',
        data: function () {
            return {
                enabledSms: enabledSms,
                busy: false,
                nickname: null,
                phone: null,
                password: null,
                captcha: null,
                zip: im.config.zip,
                captchaButtonBusy: false,
                captchaButtonText: ''
            };
        },
        mixins: [
            components.getValidate()
        ],
        mounted: function () {
            this.captchaButtonText = this.locale.sendCaptcha;
        },
        computed: {
            invalid: function () {
                return utils.isEmpty(this.phone) || utils.isEmpty(this.nickname) || utils.isEmpty(this.captcha) || utils.isEmpty(this.password);
            }
        },
        methods: {
            sendCaptcha: function () {
                sendCaptcha(this);
            },
            sendCode: function () {
                return sendCode(this, userApi);
            },
            submit: function () {
                submit(this, userApi, im);
            }
        }
    };
    utils.asyncComponent(options, resolve, reject);
};

function sendCaptcha(context) {
    if (!context.valid('[name=phone]')) {
        return;
    }

    context.captchaButtonBusy = true;
    context.sendCode()
        .then(function () {
            var captchaButtonTextBackup = context.captchaButtonText;
            var count = 59;
            var timer = setInterval(function () {
                if (count > 0) {
                    count = count < 10 ? ('0' + count) : count;
                    context.captchaButtonText = context.localeFormat(context.locale.sentCaptcha, count);
                    count -= 1;
                } else {
                    context.captchaButtonText = captchaButtonTextBackup;
                    context.captchaButtonBusy = false;
                    clearInterval(timer);
                }
            }, 1000);
        })
        .fail(function () {
            context.captchaButtonBusy = false;
        })
        .always(function () {
            sendCaptcha.done = true;
        });
}

function sendCode(context, userApi) {
    var defer = $.Deferred();
    var params = {
        zip: context.zip,
        phone: context.phone
    };
    userApi.sendCode('register', params, function (errorCode) {
        if (errorCode) {
            context.$set(context.errors, 'phone', common.getErrorMessage(errorCode));
            defer.reject(errorCode);
        } else {
            defer.resolve();
        }
    });
    return defer.promise();
}

function checkCode(context, userApi) {
    var defer = $.Deferred();
    var params = {
        zip: context.zip,
        phone: context.phone,
        code: context.captcha
    };
    userApi.checkCode(params, function (errorCode, result) {
        if (errorCode) {
            defer.reject(errorCode);
        } else {
            defer.resolve(result);
        }
    });
    return defer.promise();
}

function register(context, userApi, params, im) {
    // 2019-06-12 密码加密传输 userApi.register 改为 userApi.securityRegister
    userApi.securityRegister(params, function (errorCode) {
        if (errorCode) {
            context.$set(context.errors, 'phone', common.getErrorMessage(errorCode));
            context.busy = false;
        } else {
            cache.set('account-number', context.phone);
            var path = {
                name: 'login',
                params: {
                    selected: 'password'
                }
            };
            im.$router.push(path);
        }
    });
}

function submit(context, userApi, im) {
    if (!context.valid() || context.busy) {
        return;
    }
    context.busy = true;

    checkCode(context, userApi)
        .then(function (result) {
            var params = {
                name: context.nickname,
                zip: context.zip,
                tel: context.phone,
                verifyToken: result.token,
                password: context.password
            };
            register(context, userApi, params, im);
        })
        .fail(function (errorCode) {
            context.$set(context.errors, 'captcha', common.getErrorMessage(errorCode));
            context.busy = false;
        });
}
}(RongIM, {
    jQuery: jQuery
}, RongIM.components));
