(function (RongAppversion) {
'use strict';

var cache = (function () {
    /*
    说明：
    1: JSON.stringfy --> set --> get --> JSON.parse
    2: data format well return as set`s
    3: undefined in array will be null after stringfy+parse
    4: NS --> namespace 缩写
    */
    var keyNS = 'rong-default-';

    function get(key) {
        /*
        legal data: "" [] {} null flase true

        illegal: undefined
            1: key not set
            2: key is cleared
            3: key removed
            4: wrong data format
        */
        key = keyNS + key;

        if (!isKeyExist(key)) {
            return undefined;
        }

        // maybe keyNS could avoid conflict
        var val = localStorage.getItem(key) || sessionStorage.getItem(key);
        val = JSON.parse(val);

        var hasOwnProperty = Object.prototype.hasOwnProperty;
        // val format check
        if (val !== null && hasOwnProperty.call(val, 'type') && hasOwnProperty.call(val, 'data')) {
            return val.data;
        }
        return undefined;
    }

    // isPersistent
    function set(key, val, isTemp) {
        var store = localStorage;
        if (isTemp) {
            store = sessionStorage;
        }

        key = keyNS + key;
        var type = (typeof val);
        val = {
            data: val,
            type: type
        };

        store[key] = JSON.stringify(val);
    }

    function remove(key) {
        key = keyNS + key;
        localStorage.removeItem(key);
        sessionStorage.removeItem(key);
    }

    function isKeyExist(key) {
        var hasOwnProperty = Object.prototype.hasOwnProperty;
        // do not depend on value cause of ""和0
        return hasOwnProperty.call(localStorage, key) || hasOwnProperty.call(sessionStorage, key);
    }

    function setKeyNS(NS) {
        var isString = typeof NS === 'string';
        if (isString && NS !== '') {
            keyNS = NS;
        }
    }

    function onchange(callback) {
        callback = callback || $.noop;
        $(window).on('storage', function (e) {
            var event = e.originalEvent;
            if ($.isEmpty(event.key)) {
                return;
            }
            var key = event.key.slice(keyNS.length);
            var value = get(key);
            callback(key, value);
        });
    }

    return {
        setKeyNS: setKeyNS,
        get: get,
        set: set,
        remove: remove,
        onchange: onchange
    };
}());

function templateFormat(source, params) {
    var undef;
    if (arguments.length === 1) {
        return function () {
            var args = $.makeArray(arguments);
            args.unshift(source);
            return $.validator.format.apply(this, args);
        };
    }
    if (params === undef) {
        return source;
    }
    if (arguments.length > 2 && params.constructor !== Array) {
        params = $.makeArray(arguments).slice(1);
    }
    if (params.constructor !== Array) {
        params = [params];
    }
    $.each(params, function (i, n) {
        source = source.replace(new RegExp('\\{{' + i + '\\}}', 'g'), function () {
            return n;
        });
    });
    return source;
}

function kebabToCamel(string) {
    var firstLetter = string[0] || '';
    return firstLetter + string.slice(1).replace(/-\w/g, function (match) {
        return match[1].toUpperCase();
    });
}

function loadTemplate(template) {
    var version = RongAppversion.instance.config.product.version;
    var promise;
    var pathRegex = new RegExp(/^([a-z_\-\s0-9./]+)+\.html$/);
    var isTemplateUrl = pathRegex.test(template);
    if (isTemplateUrl) {
        promise = $.get(template + '?v=' + version);
    } else {
        var html = $(template).html();
        promise = $.Deferred().resolve(html).promise();
    }
    return promise;
}

function asyncComponent(options, resolve, reject) {
    var promise = loadTemplate(options.template);
    promise.then(function (html) {
        options.mixins = options.mixins || [];
        var locale = {
            computed: {
                locale: function () {
                    var localeInfo = RongAppversion.instance.locale;
                    var name = kebabToCamel(options.name);
                    return $.extend(true, {}, localeInfo, localeInfo.components[name]);
                }
            },
            methods: {
                localeFormat: templateFormat
            }
        };
        options.mixins.push(locale);
        var component = $.extend({}, options, { template: html });
        resolve(component);
    }).fail(function (xhr, status, error) {
        reject(error);
    });
}

function without(origin, out) {
    return origin.filter(function (item) {
        return out.indexOf(item) === -1;
    });
}

function dateFormat(timestamp, options) {
    options = $.extend({
        alwaysShowTime: false,
        timeHour: 24 /* 24小时制 */
    }, options);

    if (dateFormat._init !== 'done') {
        updateLocale();
        dateFormat._init = 'done';
    }
    var now = moment();
    var date = moment(timestamp);
    var hourTime = options.timeHour === 24 ? 'H' : ' A h';
    hourTime = ' H:mm'.replace(/H/ig, hourTime);
    var time = options.alwaysShowTime ? date.format(hourTime) : '';

    if (date.isSameOrBefore(now, 'month') && date.isSame(now, 'year')) {
        return date.format('M月D日') + time;
    }

    return date.format('YYYY年M月D日') + time;
}

function updateLocale() {
    moment.updateLocale(moment.locale(), {
        week: {
            dow: 7
        },
        meridiem: function (hour, minute) {
            var locale = RongIM.instance.locale;
            var hm = hour * 100 + minute;
            if (hm < 600) {
                return locale.time.morning;
            // 文档中没有早上这个时间段
            // } else if (hm < 900) {
            //     return '早上';
            } if (hm < 1130) {
                return locale.time.forenoon;
            } if (hm < 1230) {
                return locale.time.noon;
            } if (hm < 1800) {
                return locale.time.afternoon;
            }
            return locale.time.evening;
        }
    });
}

function tplEngine(tpl, data) {
    var reg = /<%([^%>]+)?%>/g;
    var regOut = /(^( )?(if|for|else|switch|case|break|{|}))(.*)?/g;
    var code = 'var r=[];\n';
    var cursor = 0;
    var match;

    var add = function (line, js) {
        if (js) {
            code += line.match(regOut) ? line + '\n' : 'r.push(' + line + ');\n';
        } else {
            code += line !== '' ? 'r.push("' + line.replace(/"/g, '\\"') + '");\n' : '';
        }
        return add;
    };
    // eslint-disable-next-line no-cond-assign
    while (match = reg.exec(tpl)) {
        add(tpl.slice(cursor, match.index))(match[1], true);
        cursor = match.index + match[0].length;
    }
    add(tpl.substr(cursor, tpl.length - cursor));
    code += 'return r.join("");';
    // eslint-disable-next-line no-new-func
    return new Function(code.replace(/[\r\t\n]/g, '')).apply(data);
}

function render(id, data) {
    var el = document.getElementById(id);
    var html = el.innerHTML;
    return tplEngine(html, data);
}

function getBrowser() {
    var userAgent = navigator.userAgent;
    var version;
    var type;
    var condition = {
        IE: /rv:([\d.]+)\) like Gecko|MSIE ([\d.]+)/,
        Edge: /Edge\/([\d.]+)/,
        Firefox: /Firefox\/([\d.]+)/,
        Opera: /(?:OPERA|OPR).([\d.]+)/,
        QQBrowser: /QQBrowser\/([\d.]+)/,
        Chrome: /Chrome\/([\d.]+)/,
        Safari: /Version\/([\d.]+).*Safari/
    };
    // eslint-disable-next-line no-restricted-syntax
    for (var key in condition) {
        if (Object.prototype.hasOwnProperty.call(condition, key)) {
            var browserContent = userAgent.match(condition[key]);
            if (browserContent) {
                type = key;
                version = browserContent[1] || browserContent[2];
                break;
            }
        }
    }
    return {
        type: type || 'UnKonw',
        version: version || 'UnKonw'
    };
}

function getPlatform() {
    var platform = RongAppversion.browserWindow.platform;
    if (platform === 'darwin') {
        return 'OSX';
    } if (platform.indexOf('web') === 0) {
        return 'web';
    }
    return 'windows';
}

RongAppversion.utils = {
    without: without,
    cache: cache,
    asyncComponent: asyncComponent,
    dateFormat: dateFormat,
    tplEngine: tplEngine,
    render: render,
    getBrowser: getBrowser,
    getPlatform: getPlatform
};
}(window.RongAppversion));
