(function (RongIM, dependencies, components) {
    'use strict';

    var common = RongIM.common;
    var utils = RongIM.utils;
    var $ = dependencies.jQuery;
    var copyToClipboard = utils.copyToClipboard;
    var collectApi = RongIM.dataModel.Collect;

    function getCollect(resolve, reject) {
        var im = RongIM.instance;
        var currentLocale = im.locale.components.collect;
        var options = {
            name: 'collect',
            template: 'modules/collect/templates/collect-list.html',
            data: function () {
                return {
                    title: '',
                    collectList: [],
                    keyword: '',
                    Busy: false,
                    loadingNextPage: false,
                    pageNum: 10,
                    pageIndex: 0,
                    all: 0,
                    params: {
                        text: 0,
                        voice: 0,
                        position: 0,
                        video: 0,
                        file: 0,
                        link: 0
                    },
                    status: 0
                };
            },
            computed: {
                isBusy: function () {
                    return this.Busy;
                },
                isEmpty: function () {
                    var hasInputValue = this.keyword.length > 0;
                    var matchCount = this.collectList.length === 0;
                    return hasInputValue && matchCount;
                }
            },
            mounted: function () {
                var context = this;
                var route = context.$route.params.id;
                this.title = currentLocale[route];
                im.$on('keywordsearch', function (ids, keyword) {
                    context.Busy = true;
                    context.keyword = keyword;
                    if (ids.length === 0) {
                        context.collectList = [];
                        context.Busy = false;
                    } else {
                        var params = {
                            ids: ids
                        };
                        collectApi.getIdList(params, function (errorcode, list) {
                            context.collectList = list;
                            context.Busy = false;
                            context.keyword = keyword;
                        });
                    }
                });
                function getMessageList(params, list) {
                    context.all = list.length;
                    context.params.text = params['RC:TxtMsg'] ? params['RC:TxtMsg'] : 0;
                    context.params.voice = params['RC:VcMsg'] ? params['RC:VcMsg'] : 0;
                    context.params.position = params['RC:LBSMsg'] ? params['RC:LBSMsg'] : 0;
                    context.params.video = (params['RC:SightMsg'] ? params['RC:SightMsg'] : 0) + (params['RC:ImgMsg'] ? params['RC:ImgMsg'] : 0);
                    context.params.file = (params['RC:FileMsg'] ? params['RC:FileMsg'] : 0) + (params['LRC:fileMsg'] ? params['LRC:fileMsg'] : 0);
                    context.params.link = params['RC:ImgTextMsg'] ? params['RC:ImgTextMsg'] : 0;
                }
                im.$emit('collectlength', getMessageList);
                getList(context);
            },
            mixins: [getContextMenu()],
            filters: {
                dateFormat: function (timestamp) {
                    return utils.dateFormat(timestamp, {
                        alwaysShowTime: true
                    });
                }
            },
            watch: {
                $route: function (data) {
                    this.status = 0;
                    this.Busy = true;
                    this.pageIndex = 0;
                    var context = this;
                    var str = data.path.split('/');
                    var newdata = str[str.length - 1];
                    this.title = currentLocale[newdata];
                    getList(context);
                }
            },
            methods: {
                showDetail: function (item) {
                    if (item.objectName === 'RC:TxtMsg' || item.objectName === 'RC:VcMsg') {
                        RongIM.dialog.collectDetail(item);
                    }
                },
                forward: function (message) {
                    if (message.messageType === 'VoiceMessage') {
                        var param = {
                            message: im.locale.tips.voiceForward,
                            callback: function () { },
                            el: this.$el.firstChild,
                            type: 'error'
                        };
                        common.messageToast(param);
                        this.closeContextmenu();
                        return;
                    }
                    forward(this, message);
                },
                save: function (message) {
                    im.$emit('filesaveas', message);
                    this.closeContextmenu();
                },
                copy: function (message) {
                    copyToClipboard(message.content.url);
                    this.closeContextmenu();
                },
                remove: function (message) {
                    var context = this;
                    var index = 0;
                    for (var i = 0; i < this.collectList.length; i += 1) {
                        if (this.collectList[i].messageUId === message.messageUId) {
                            index = i;
                        }
                    }
                    collectApi.remove(message.uid, function () {
                        context.collectList.splice(index, 1);
                        context.closeContextmenu();
                        context.Busy = false;
                        im.$emit('collectList');
                        return false;
                    });
                },
                getMessageType: function (item) {
                    var messageType;
                    if (item.messageType === 'RC:SightMsg') {
                        messageType = 'SightMessage';
                    } else {
                        messageType = item.messageType;
                    }
                    if (item.messageType === 'LRC:fileMsg') {
                        messageType = 'FileMessage';
                    }
                    return messageType;
                },
                showImage: function (message) {
                    common.showImage(getImageMessageList(this.collectList), message.messageUId, im.config.locale);
                },
                showSight: function (message) {
                    this.showImage(message);
                },
                localeFormat: utils.templateFormat,
                scrollBottom: utils.throttle(function (event) {
                    scrollBottom(this, event);
                }, 800)
            },
            components: {
                customSelect: components.customSelect,
                TextMessage: components.getTextMessage,
                ImageMessage: components.getImageMessage,
                LocalImageMessage: components.getImageMessage,
                FileMessage: components.getFileMessage,
                LocalFileMessage: components.getFileMessage,
                VoiceMessage: components.getVoiceMessage,
                LocationMessage: components.getLocationMessage,
                CardMessage: components.getCardMessage,
                SightMessage: components.getSightMessage,
                RichContentMessage: components.getRichContentMessage,
                ApprovalMessage: components.getApprovalMessage,
                UnknownMessage: components.getUnknownMessage,
                GroupNoticeNotifyMessage: components.getGroupNoticeNotifyMessage,
                BQMMEmojiMessage: components.getBQMMEmojiMessage,
                BQMMGifMessage: components.getBQMMGifMessage,
                JrmfRedPacketMessage: components.getJrmfRedPacketMessage,
                PublicServiceRichContentMessage: components.getPSImageTextMessage,
                PublicServiceMultiRichContentMessage: components.getPSMultiImageTextMessage,
                VideoSummaryMessage: components.getSummaryMessage,
                RemoteControlSummaryMessage: components.getRemoteSummaryMessage,
                ReferenceMessage: components.getQuoteMessage

            }
        };
        utils.asyncComponent(options, resolve, reject);
    }
    function getImageMessageList(list) {
        var imageMsgList = list.filter(function (item) {
            if (item.messageType === 'LocalImageMessage') {
                item.content.imageUri = utils.Base64.concat(item.content.content);
            }
            var url = item.content.imageUri || item.content.sightUrl || (item.content.content || {}).imageUri;
            var isImage = item.messageType === 'ImageMessage' || item.messageType === 'LocalImageMessage';
            var isSight = item.messageType === 'SightMessage';
            var isQuoteimage = item.messageType === 'ReferenceMessage' && item.content.objName === 'RC:ImgMsg';
            if (!url && !isImage) {
                return false;
            }
            return isImage || isSight || isQuoteimage;
        });
        return imageMsgList;
    }
    function getList(context) {
        context.Busy = true;
        var route = context.$route.params.id;
        var start = context.pageIndex * context.pageNum;
        var pageNum = context.pageNum;

        var params = {
            version: -1,
            scope: 'message',
            offset: start,
            limit: pageNum
        };
        if (route === 'all') {
            if (context.all < start) {
                context.loadingNextPage = false;
                return;
            }
        } else {
            if (context.params[route] < start) {
                context.loadingNextPage = false;
                return;
            }
            params.type = reName(route);
        }
        collectApi.getList(params, (function (currentPath, page) {
            return function (errorcode, result) {
                if (errorcode) {
                    common.toastError(errorcode);
                    context.Busy = false;
                    context.status = 1;
                    return;
                }
                var path = context.$route.params.id;
                var flag = path === currentPath;
                if (page === 0) context.collectList = [];
                if (!flag) return;
                context.Busy = false;
                context.collectList = context.collectList.concat(result);
            };
        }(route, context.pageIndex)));
    }
    function getContextMenu() {
        var options = {
            template: 'modules/collect/templates/collect-menu.html',
            computed: {
                showForward: function () {
                    var excludeList = [
                        //  'TextMessage',
                        // 'LocationMessage',
                        // 'SightMessage',
                        'VoiceMessage'
                        // 'ImageMessage',
                        // 'FileMessage',
                    ];
                    var show = excludeList.indexOf(this.context.message.messageType) < 0;
                    return show;
                },
                showRemove: function () {
                    return true;
                },
                showSave: function () {
                    var excludeList = [
                        'TextMessage',
                        'LocationMessage',
                        'SightMessage',
                        'RC:SightMsg',
                        'VoiceMessage',
                        'ImageMessage',
                        'RichContentMessage'
                        //  'FileMessage',
                    ];
                    var show = excludeList.indexOf(this.context.message.messageType) < 0;
                    return show;
                },
                showCopy: function () {
                    var excludeList = [
                        'TextMessage',
                        'LocationMessage',
                        'SightMessage',
                        'RC:SightMsg',
                        'VoiceMessage',
                        'ImageMessage',
                        'FileMessage'
                    ];
                    var show = excludeList.indexOf(this.context.message.messageType) < 0;
                    return show;
                }
            },
            methods: {
                save: function () {
                    this.$emit('save', this.context.message);
                },
                copy: function () {
                    this.$emit('copy', this.context.message);
                },
                remove: function () {
                    this.$emit('remove', this.context.message);
                },
                forward: function () {
                    this.$emit('forward', this.context.message);
                }
            }
        };
        return components.getContextMenu(options);
    }
    function forward(context, message) {
        if (message.messageType === 'RichContentMessage') {
            message = common.buildMessage.RichContentMessage(message);
        }
        if (message.objectName === 'RC:SightMsg') {
            message.messageName = 'SightMessage';
        }
        // RongIM.dialog.forward(message);
        RongIM.dialog.forwardDialog(message,"forward");
        context.closeContextmenu();
    }
    function scrollBottom(context, event) {
        // 判断如果向下滚动并滚动到最下方，就加载更多会话列表
        var down = event.deltaY > 0;
        var $content = $(context.$refs.list);
        var height = $content ? $content[0].scrollHeight : 0;
        var bottom = ($content.scrollTop() + $content.height()) >= height - 15;

        if (down && bottom && !context.keyword) {
            loadMore(context);
        }
    }
    function loadMore(context) {
        var start = context.pageIndex * context.pageNum;
        if (context.all <= start) {
            return;
        }
        context.loadingNextPage = true;
        context.pageIndex += 1;
        getList(context);
    }
    function reName(obj) {
        switch (obj) {
            case 'all':
                obj = ''; break;
            case 'text':
                obj = 'RC:TxtMsg'; break;
            case 'voice':
                obj = 'RC:VcMsg'; break;
            case 'position':
                obj = 'RC:LBSMsg'; break;
            case 'video':
                obj = ['RC:ImgMsg', 'RC:SightMsg']; break;
            case 'file':
                obj = ['RC:FileMsg', 'FileMessage', 'LRC:fileMsg']; break;
            case 'link':
                obj = 'RC:ImgTextMsg'; break;
            default: return undefined;
        }
        return obj;
    }
    $.extend(true, components, {
        collect: {
            getCollect: getCollect
        }
    });
}(RongIM, {
    jQuery: jQuery
}, RongIM.components));
