(function (ImageViewer, dependencies) {
'use strict';

var $ = dependencies.jQuery;

var keyCode = {
    backspace: 8,
    tab: 9,
    enter: 13,
    shift: 16,
    ctrl: 17,
    alt: 18,
    caps: 20,
    esc: 27,
    left: 37,
    up: 38,
    right: 39,
    down: 40,
    delete: 46,
    win: 91,
    q: 81
};

function loadTemplate(template) {
    var version = Math.random();
    var promise;
    var pathRegex = new RegExp(/^([a-z_\-\s0-9./]+)+\.html$/);
    var isTemplateUrl = pathRegex.test(template);
    if (isTemplateUrl) {
        promise = $.get(template + '?v=' + version);
    } else {
        var html = $(template).html();
        promise = $.Deferred().resolve(html).promise();
    }
    return promise;
}

function templateFormat(source, params) {
    var undef;
    if (arguments.length === 1) {
        return function () {
            var args = $.makeArray(arguments);
            args.unshift(source);
            return $.validator.format.apply(this, args);
        };
    }
    if (params === undef) {
        return source;
    }
    if (arguments.length > 2 && params.constructor !== Array) {
        params = $.makeArray(arguments).slice(1);
    }
    if (params.constructor !== Array) {
        params = [params];
    }
    $.each(params, function (i, n) {
        source = source.replace(new RegExp('\\{{' + i + '\\}}', 'g'), function () {
            return n;
        });
    });
    return source;
}

function kebabToCamel(string) {
    var firstLetter = string[0] || '';
    return firstLetter + string.slice(1).replace(/-\w/g, function (match) {
        return match[1].toUpperCase();
    });
}

function asyncComponent(options, resolve, reject) {
    var promise = loadTemplate(options.template);
    promise.then(function (html) {
        var component = $.extend({}, options, { template: html });
        resolve(component);
    }).fail(function (xhr, status, error) {
        reject(error);
    });
}

var Base64 = {
    replace: function (str) {
        var reg = new RegExp('^data:image/[^;]+;base64,');
        if (str) {
            return str.replace(reg, '');
        }
        return '';
    },
    concat: function (base64, type) {
        type = type || 'data:image/jpg;base64,';
        base64 = this.replace(base64);
        return type + base64;
    }
};
// see http://underscorejs.org/#throttle
function throttle(func, wait, options) {
    var timeout;
    var context;
    var args;
    var result;
    var previous = 0;
    if (!options) {
        options = {};
    }
    var later = function () {
        previous = options.leading === false ? 0 : (new Date()).getTime();
        timeout = null;
        result = func.apply(context, args);
        if (!timeout) {
            context = args = null;
        }
    };
    var throttled = function () {
        var now = (new Date()).getTime();
        if (!previous && options.leading === false) {
            previous = now;
        }
        var remaining = wait - (now - previous);
        context = this;
        args = arguments;
        if (remaining <= 0 || remaining > wait) {
            if (timeout) {
                clearTimeout(timeout);
                timeout = null;
            }
            previous = now;
            result = func.apply(context, args);
            if (!timeout) {
                context = args = null;
            }
        } else if (!timeout && options.trailing !== false) {
            timeout = setTimeout(later, remaining);
        }
        return result;
    };
    throttled.cancel = function () {
        clearTimeout(timeout);
        previous = 0;
        timeout = context = args = null;
    };
    return throttled;
}
var delay = restArgs(function (func, wait, args) {
    return setTimeout(function () {
        return func.apply(null, args);
    }, wait);
});
// see http://underscorejs.org/#debounce
function debounce(func, wait, immediate) {
    var timeout;
    var result;
    var later = function (context, args) {
        timeout = null;
        if (args) {
            result = func.apply(context, args);
        }
    };
    var debounced = restArgs(function (args) {
        if (timeout) {
            clearTimeout(timeout);
        }
        if (immediate) {
            var callNow = !timeout;
            timeout = setTimeout(later, wait);
            if (callNow) {
                result = func.apply(this, args);
            }
        } else {
            timeout = delay(later, wait, this, args);
        }
        return result;
    });
    debounced.cancel = function () {
        clearTimeout(timeout);
        timeout = null;
    };
    return debounced;
}

function restArgs(func, startIndex) {
    startIndex = startIndex ? +startIndex : func.length - 1;
    return function () {
        var length = Math.max(arguments.length - startIndex, 0);
        var rest = Array(length);
        var index = 0;
        for (; index < length; index += 1) {
            rest[index] = arguments[index + startIndex];
        }
        switch (startIndex) {
            case 0:
                return func.call(this, rest);
            case 1:
                return func.call(this, arguments[0], rest);
            case 2:
                return func.call(this, arguments[0], arguments[1], rest);
            default:
        }
        var args = Array(startIndex + 1);
        for (index = 0; index < startIndex; index += 1) {
            args[index] = arguments[index];
        }
        args[startIndex] = rest;
        return func.apply(this, args);
    };
}

/*
说明：统一弹窗格式
参数：
    @param {object}      params
    @param {string}      params.type          - 'confirm':有2个按钮的弹窗，'alert':有1个按钮的弹窗，默认为'alert'
    @param {string}      params.title         - 弹窗title，不传则为空
    @param {string}      params.message       - 弹窗内容，不传则为空
    @param {string}      params.submitText    - 确定按钮的文案
    @param {boolean}     params.isAlignLeft   - 弹窗内容不都左对齐
    @param {function}    params.closeCallback - 取消按钮回调函数
    @param {function}    params.callback      - 确认按钮回调函数
*/
function messagebox(params) {
/*    var r = confirm(params.message)
    if (r === true) {
        params.callback && params.callback();
    }
    return; */
    var options = {
        name: 'messagebox',
        template: $('#rong-messagebox').html(),
        data: function () {
            return {
                type: params.type || 'alert',
                title: params.title,
                message: params.message,
                submitText: params.submitText,
                show: true,
                isAlignLeft: params.isAlignLeft
            };
        },
        created: function () {
            var context = this;
            var im = ImageViewer.instance;
            context.title = params.title || context.locale.tips.msgboxTitle;
            context.submitText = params.submitText || context.locale.tips.msgboxSubmitText;
            context.imLogined = function () {
                context.close();
                im.$off(context.imLogined);
            };
            im.$on('imlogined', context.imLogined);
            im.$on('messageboxHide', context.close);
            window.addEventListener('hashchange', context.close);
        },
        methods: {
            close: function () {
                if (params.closeCallback) params.closeCallback();
                window.removeEventListener('hashchange', this.close);
                this.show = false;
            },
            confirm: function () {
                if (params.callback) params.callback();
                this.show = false;
            }
        },
        directives: {
            autofocus: {
                inserted: function (el) {
                    Vue.nextTick(function () {
                        el.focus();
                    });
                }
            }
        }
    };
    var locale = {
        computed: {
            locale: function () {
                var localeInfo = ImageViewer.instance.locale;
                var name = kebabToCamel(options.name);
                return $.extend(true, {}, localeInfo, localeInfo.components[name]);
            }
        }
    };
    options.mixins = [locale];
    var Messagebox = Vue.extend(options);
    var instance = new Messagebox({
        el: document.createElement('div')
    });
    var wrap = ImageViewer.instance.$el.firstChild;
    $(wrap).append(instance.$el);
    return instance;
}

/*
说明：统一弹窗格式
参数：
    @param {string}      params.message       - toast内容
    @param {string}      params.type          - toast类型，'success'绿色，'error'红色，默认为'success'
*/
function messageToast(params) {
    var options = {
        name: 'messagetoast',
        template: $('#rong-messagetoast').html(),
        data: function () {
            return {
                type: params.type || 'success',
                message: params.message,
                show: true
            };
        },
        created: function () {
            var context = this;
            setTimeout(function () {
                context.show = false;
                if (typeof params.callback === 'function') {
                    params.callback();
                }
            }, 1500);
        }
    };
    var Messagetoast = Vue.extend(options);
    var instance = new Messagetoast({
        el: document.createElement('div')
    });
    var wrap = ImageViewer.instance.$el.firstChild;
    $(wrap).append(instance.$el);
    return instance;
}

ImageViewer.utils = {
    keyCode: keyCode,
    loadTemplate: loadTemplate,
    templateFormat: templateFormat,
    asyncComponent: asyncComponent,
    kebabToCamel: kebabToCamel,
    Base64: Base64,
    debounce: debounce,
    throttle: throttle,
    messagebox: messagebox,
    messageToast: messageToast
};
}(ImageViewer, {
    jQuery: jQuery
}));
