(function (RongIM, dependencies, components) {
'use strict';

var $ = dependencies.jQuery;
var utils = RongIM.utils;
var common = RongIM.common;
var RemoteControlStatus = common.RemoteControlStatus;
var RemoteControlHangupReason = common.RemoteControlHangupReason;
var browserWindow = RongIM.browserWindow;
var remoteWindowHandler = browserWindow.remoteWindowHandler;
var im;
var remoteApi;
var dataModel;
var remotePanelWidth = 288;
/*
远程控制组件
 */
components.getRemoteControl = function (resolve, reject) {
    im = RongIM.instance;
    dataModel = im.dataModel;
    remoteApi = dataModel.RemoteControl;
    var options = {
        name: 'remoteControl',
        template: 'modules/remote-control/templates/remote-control.html',
        // remoteControl 来自 conversation.js
        props: ['remoteControl'],
        data: function () {
            return {
                RemoteControlStatus: RemoteControlStatus,
                user: {},
                isBusy: false
            };
        },
        computed: {
            remoteControlStatus: function () {
                return this.remoteControl.status || RemoteControlStatus.Idle;
            },
            userName: function () {
                var user = this.user;
                return user ? user.name : '';
            },
            locale: function () {
                return im.locale.components.remoteControl;
            },
            isWeb: function () {
                return RongIM.system.platform.indexOf('web') !== -1;
            }
        },
        methods: {
            // 己方取消发出的请求
            cancel: function () {
                var reason = RemoteControlHangupReason.Cancel;
                hangup(this, reason);
            },
            // 己方挂断
            hangup: function () {
                var reason = RemoteControlHangupReason.Hangup;
                hangup(this, reason);
            },
            // 己方拒绝
            refuse: function () {
                var reason = RemoteControlHangupReason.Reject;
                hangup(this, reason);
            },
            // 接受
            accept: function () {
                // 主控, 发送 accept, 等待接受 config
                // 被控, 先发送 accept, 再发送 config
                var context = this;
                var remoteStatus = context.remoteControl.status;
                accept(context, function (/* err, message */) {
                    if (remoteStatus === RemoteControlStatus.ReceiveRequest) {
                        setConfig(context);
                    }
                });
            }
        },
        watch: {
            $route: function () {
                setUser(this);
            },
            remoteControlStatus: function (newVal, oldVal) {
                var isValid = function (val) {
                    return val && val !== RemoteControlStatus.Idle;
                };
                if (isValid(newVal) && !isValid(oldVal)) {
                    var isExtend = isExtendWindowSize();
                    setWindowMinWidth(isExtend);
                    extendWindowSize(isExtend);
                } else if (!isValid(newVal) && isValid(oldVal)) {
                    remoteWindowHandler.setDefaultMinWidth();
                    restoreWindowSize(remoteWindowHandler.isExtend);
                }
            }
        },
        mounted: function () {
            setUser(this);
        }
    };
    utils.asyncComponent(options, resolve, reject);
};

// 当远程控制进行中时, 避免拉动会话列表造成消息显示混乱
function setWindowMinWidth(/* isExtend */) {
    var mainWidth = $('#rong-main').width();
    var winSize = remoteWindowHandler.getWindowSize();
    // 500 为消息列表的最小宽度
    var minWidth = winSize[0] - (mainWidth - 500 - remotePanelWidth);
    remoteWindowHandler.setMinWidth(minWidth);
}

// 扩大窗口大小
// 注: 需要扩大窗口宽度, 并不能有空白, 闪烁的效果. 所以先固定 rong-main 的宽度, 1s 后, 窗口扩大完毕, 再把固定取消
// 临时需求, 这样做不需要改动其他位置, 风险最小
function extendWindowSize(isExtend) {
    var $conversation = $('#rong-main');
    if (isExtend) {
        var width = $conversation.width();
        $conversation.css('width', width);
        remoteWindowHandler.extend();
        setTimeout(function () {
            $conversation.css('width', 'auto');
        }, 1000);
    }
}

function restoreWindowSize(isExtend) {
    remoteWindowHandler.restore();
    if (isExtend) {
        var $conversation = $('#rong-main');
        var width = $conversation.width();
        $conversation.css('width', width);
        setTimeout(function () {
            $conversation.css('width', 'auto');
        }, 1000);
    }
}

function isExtendWindowSize() {
    var $conversation = $('#rong-main');
    var winSize = remoteWindowHandler.getWindowSize();
    var winWidth = winSize[0];
    var width = $conversation.width();
    // 当拖拽会话列表, 导致消息列表宽度小于 590 时, 也要扩大窗口
    return winWidth < remoteWindowHandler.WidthLimit || width < 590;
}

function setUser(context) {
    var params = context.$route.params;
    var targetId = params.targetId;
    var conversationType = params.conversationType;
    var userApi = dataModel.User;
    if (conversationType === 1 && targetId) {
        userApi.get(targetId, function (err, user) {
            context.user = err ? null : user;
        });
    } else {
        context.user = null;
    }
}

/*
挂断, 挂断后, 修改 conversation组件remoteControl 的值
 */
function hangup(context, reason) {
    var params = getRemoteControlParams(context);
    var conversation = params.conversation;
    var currentStatus = remoteApi.get(conversation.targetId).status;
    browserWindow.closeRemoteControl();
    // 避免鼠标快速连续点两次拒绝
    if (context.isBusy) {
        return;
    }
    context.isBusy = true;
    remoteApi.hangup(conversation, params.sessionId, reason, function (err, message) {
        if (!err) {
            // setRemoteControl 声明位置在 conversation.js -> watchRemoteControl 函数
            var summaryParams = getSummaryParams(reason, currentStatus, params.sessionId);
            var remote = remoteApi.get(params.conversation.targetId);
            console.debug(remote.status);
            if (RemoteControlStatus.Idle !== remote.status) {
                remoteApi.summary(params.conversation, summaryParams);
            }
            im.$emit('setRemoteControl', message);
        }
        context.isBusy = false;
    });
}

function getSummaryParams(reason, status, sessionId) {
    var direction = RongIMLib.MessageDirection.RECEIVE;
    var isControl = false;
    var showSelfReason = [
        RemoteControlHangupReason.RemoteHangup,
        RemoteControlHangupReason.NoResponse,
        RemoteControlHangupReason.RemoteNoResponse,
        RemoteControlHangupReason.Cancel
    ];
    var hangupReasons = [
        RemoteControlHangupReason.Hangup,
        RemoteControlHangupReason.RemoteHangup
    ];
    var isHangup = hangupReasons.indexOf(reason) !== -1;
    var MessageDirection = RongIMLib.MessageDirection;
    // 己方主动发出的挂断, 只有连接后的挂断, 未应答, 取消, 气泡在右侧, 其他都在左侧
    if (showSelfReason.indexOf(reason) !== -1) {
        direction = MessageDirection.SEND;
    }
    if (isHangup) {
        var isInitiator = remoteApi.isInitiator(sessionId);
        direction = isInitiator ? MessageDirection.SEND : MessageDirection.RECEIVE;
    }
    // 挂断前, 为这些状态时, 表示自己为主控端, 此时提示 远程控制, 否则提示 远程协助
    var controlStatusList = [
        'Request', 'ReceiveInvite', 'Building', 'RequestBuilded'
    ];
    status = RemoteControlStatus[status];
    if (controlStatusList.indexOf(status) !== -1) {
        isControl = true;
    }
    return {
        reason: reason,
        direction: direction,
        isControl: isControl
    };
}

/*
同意远程控制, 同意后, 修改
 */
function accept(context, callback) {
    var params = getRemoteControlParams(context);
    browserWindow.closeRemoteControl();
    remoteApi.accept(params.conversation, params.sessionId, function (err, message) {
        if (!err) {
            im.$emit('setRemoteControl', message);
        }
        callback(err, message);
    });
}

/**
 * callback.onConnect  开始连接
 * callback.onDisconnected  断开连接
 * callback.onLogin  登录成功, 返回 address 和 sessionName
 * callback.onLoginFail  登录失败
 */
function setConfig(context) {
    var params = getRemoteControlParams(context);
    remoteApi.setConfig(params.conversation, params.sessionId, function (err, message) {
        var remote = remoteApi.get(params.conversation.targetId);
        if (remote.status !== RemoteControlStatus.Idle) {
            im.$emit('setRemoteControl', message);
        }
    }, function () {
        // 挂断的回调, 对方杀进程也会执行
        var remote = remoteApi.get(params.conversation.targetId);
        console.debug('setConfig status:', remote.status);
        if (remote.status !== RemoteControlStatus.Idle) {
            var reason = RemoteControlHangupReason.Hangup;
            var summaryParams = getSummaryParams(reason, remote.status, params.sessionId);
            remoteApi.summary(params.conversation, summaryParams, function (err, message) {
                im.$emit('setRemoteControl', message);
            });
        }
    });
}

/*
* 获取 调用 remoteApi 所需的参数
* @param {object} context  vue组件
*/
function getRemoteControlParams(context) {
    var conversation = context.$route.params || {};
    var message = context.remoteControl.message;
    var sessionId = message.content.sessionId;
    return {
        conversation: conversation,
        sessionId: sessionId
    };
}
}(RongIM, {
    jQuery: jQuery
}, RongIM.components));
